-- =============================================
-- Gartle Planning
-- Version 5.3, April 10, 2023
--
-- Copyright 2017-2023 Gartle LLC
--
-- License: MIT
-- =============================================

SET NOCOUNT ON
GO

CREATE USER planning_app WITHOUT LOGIN;
GO

CREATE SCHEMA dbo25 AUTHORIZATION planning_app;
GO
CREATE SCHEMA xls25 AUTHORIZATION planning_app;
GO
CREATE SCHEMA xls25a AUTHORIZATION planning_app;
GO
CREATE SCHEMA xls25b AUTHORIZATION planning_app;
GO

CREATE TABLE [dbo25].[axis_types] (
      [id] tinyint NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(100) NOT NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_axis_types_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_axis_types_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_axis_types_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_axis_types_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_axis_types] PRIMARY KEY ([id])
    , CONSTRAINT [IX_axis_types_code] UNIQUE ([code])
    , CONSTRAINT [IX_axis_types_name] UNIQUE ([name])
);
GO

CREATE TABLE [dbo25].[calc_types] (
      [id] tinyint NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(100) NOT NULL
    , [sort_order] tinyint NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_calc_types_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_calc_types_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_calc_types_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_calc_types_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_calc_types] PRIMARY KEY ([id])
    , CONSTRAINT [IX_calc_types_code] UNIQUE ([code])
    , CONSTRAINT [IX_calc_types_name] UNIQUE ([name])
);
GO

CREATE TABLE [dbo25].[companies] (
      [id] int IDENTITY(1,1) NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(100) NOT NULL
    , [default_language] varchar(10) NULL CONSTRAINT [DF_companies_default_language] DEFAULT('en')
    , [sort_order] int NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_companies_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_companies_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_companies_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_companies_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_companies] PRIMARY KEY ([id])
    , CONSTRAINT [IX_companies_code] UNIQUE ([code])
    , CONSTRAINT [IX_companies_name] UNIQUE ([name])
);
GO

CREATE TABLE [dbo25].[name_formats] (
      [id] tinyint NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(50) NOT NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_name_formats_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_name_formats_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_name_formats_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_name_formats_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_name_formats] PRIMARY KEY ([id])
    , CONSTRAINT [IX_name_formats_code] UNIQUE ([code])
    , CONSTRAINT [IX_name_formats_name] UNIQUE ([name])
);
GO

CREATE TABLE [dbo25].[translated_tables] (
      [id] tinyint NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(100) NOT NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_translated_tables_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_translated_tables_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_translated_tables_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_translated_tables_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_translated_tables] PRIMARY KEY ([id])
    , CONSTRAINT [IX_translated_tables_code] UNIQUE ([code])
    , CONSTRAINT [IX_translated_tables_name] UNIQUE ([name])
);
GO

CREATE TABLE [dbo25].[dimensions] (
      [id] tinyint NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(100) NOT NULL
    , [parameter_name] nvarchar(50) NOT NULL
    , [sort_order] tinyint NOT NULL
    , [join_order] tinyint NOT NULL
    , [name_format_id] tinyint NOT NULL CONSTRAINT [DF_dimensions_name_format_id] DEFAULT((0))
    , [is_protected] bit NOT NULL CONSTRAINT [DF_dimensions_is_protected] DEFAULT((0))
    , [default_select_permission] bit NOT NULL
    , [default_update_permission] bit NOT NULL
    , [is_active] bit NOT NULL CONSTRAINT [DF_dimensions_is_active] DEFAULT((0))
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_dimensions_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_dimensions_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_dimensions_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_dimensions_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_dimensions] PRIMARY KEY ([id])
    , CONSTRAINT [IX_dimensions_code] UNIQUE ([code])
    , CONSTRAINT [IX_dimensions_name] UNIQUE ([name])
);
GO

ALTER TABLE [dbo25].[dimensions] ADD CONSTRAINT [FK_dimensions_name_formats] FOREIGN KEY ([name_format_id]) REFERENCES [dbo25].[name_formats] ([id]);
GO

CREATE TABLE [dbo25].[formats] (
      [ID] int IDENTITY(1,1) NOT NULL
    , [TABLE_SCHEMA] nvarchar(128) NOT NULL
    , [TABLE_NAME] nvarchar(128) NOT NULL
    , [TABLE_EXCEL_FORMAT_XML] xml NULL
    , [company_id] int NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_formats_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_formats_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_formats_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_formats_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_formats] PRIMARY KEY ([ID])
    , CONSTRAINT [IX_formats] UNIQUE ([company_id], [TABLE_NAME], [TABLE_SCHEMA])
);
GO

ALTER TABLE [dbo25].[formats] ADD CONSTRAINT [FK_formats_companies] FOREIGN KEY ([company_id]) REFERENCES [dbo25].[companies] ([id]);
GO

CREATE TABLE [dbo25].[forms] (
      [id] int IDENTITY(1,1) NOT NULL
    , [company_id] int NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(100) NOT NULL
    , [sort_order] int NULL
    , [is_active] bit NOT NULL CONSTRAINT [DF_forms_is_active] DEFAULT((1))
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_forms_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_forms_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_forms_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_forms_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_forms] PRIMARY KEY ([id])
    , CONSTRAINT [IX_forms_code] UNIQUE ([company_id], [code])
    , CONSTRAINT [IX_forms_name] UNIQUE ([company_id], [name])
);
GO

ALTER TABLE [dbo25].[forms] ADD CONSTRAINT [FK_forms_companies] FOREIGN KEY ([company_id]) REFERENCES [dbo25].[companies] ([id]);
GO

CREATE TABLE [dbo25].[tax_rates] (
      [id] int IDENTITY(1,1) NOT NULL
    , [company_id] int NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(50) NOT NULL
    , [sort_order] tinyint NULL
    , [factor] float NULL
    , [is_translatable] bit NULL CONSTRAINT [DF_tax_rates_is_translatable] DEFAULT((1))
    , [is_active] bit NOT NULL CONSTRAINT [DF_tax_rates_is_active] DEFAULT((1))
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_tax_rates_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_tax_rates_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_tax_rates_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_tax_rates_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_tax_rates] PRIMARY KEY ([id])
    , CONSTRAINT [IX_tax_rates_code] UNIQUE ([company_id], [code])
    , CONSTRAINT [IX_tax_rates_name] UNIQUE ([company_id], [name])
);
GO

ALTER TABLE [dbo25].[tax_rates] ADD CONSTRAINT [FK_tax_rates_companies] FOREIGN KEY ([company_id]) REFERENCES [dbo25].[companies] ([id]);
GO

CREATE TABLE [dbo25].[translations] (
      [company_id] int NOT NULL
    , [table_id] tinyint NOT NULL
    , [member_id] int NOT NULL
    , [language] varchar(10) NOT NULL
    , [name] nvarchar(400) NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_translations_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_translations_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_translations_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_translations_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_translations] PRIMARY KEY ([company_id], [table_id], [member_id], [language])
);
GO

ALTER TABLE [dbo25].[translations] ADD CONSTRAINT [FK_translations_companies] FOREIGN KEY ([company_id]) REFERENCES [dbo25].[companies] ([id]);
GO

ALTER TABLE [dbo25].[translations] ADD CONSTRAINT [FK_translations_translated_tables] FOREIGN KEY ([table_id]) REFERENCES [dbo25].[translated_tables] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

CREATE TABLE [dbo25].[units] (
      [id] int IDENTITY(1,1) NOT NULL
    , [company_id] int NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(100) NOT NULL
    , [sort_order] tinyint NULL
    , [factor] float NOT NULL CONSTRAINT [DF_units_factor] DEFAULT((1))
    , [is_currency] bit NOT NULL CONSTRAINT [DF_units_is_currency] DEFAULT((0))
    , [is_functional_currency] bit NOT NULL CONSTRAINT [DF_units_is_functional_currency] DEFAULT((0))
    , [is_translatable] bit NULL CONSTRAINT [DF_units_is_translatable] DEFAULT((1))
    , [is_active] bit NOT NULL CONSTRAINT [DF_units_is_active] DEFAULT((1))
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_units_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_units_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_units_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_units_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_units] PRIMARY KEY ([id])
    , CONSTRAINT [IX_units_code] UNIQUE ([company_id], [code])
    , CONSTRAINT [IX_units_name] UNIQUE ([company_id], [name])
);
GO

ALTER TABLE [dbo25].[units] ADD CONSTRAINT [FK_units_companies] FOREIGN KEY ([company_id]) REFERENCES [dbo25].[companies] ([id]);
GO

CREATE TABLE [dbo25].[users] (
      [id] int IDENTITY(1,1) NOT NULL
    , [company_id] int NOT NULL
    , [username] nvarchar(128) NOT NULL
    , [name] nvarchar(100) NOT NULL
    , [is_admin] bit NOT NULL CONSTRAINT [DF_users_is_admin] DEFAULT((0))
    , [is_default] bit NOT NULL CONSTRAINT [DF_users_is_default] DEFAULT((1))
    , [is_active] bit NOT NULL CONSTRAINT [DF_users_is_active] DEFAULT((0))
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_users_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_users_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_users_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_users_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_users] PRIMARY KEY ([id])
    , CONSTRAINT [IX_users_name] UNIQUE ([company_id], [name])
    , CONSTRAINT [IX_users_username] UNIQUE ([company_id], [username])
);
GO

ALTER TABLE [dbo25].[users] ADD CONSTRAINT [FK_users_companies] FOREIGN KEY ([company_id]) REFERENCES [dbo25].[companies] ([id]);
GO

CREATE TABLE [dbo25].[dimension_calc_types] (
      [dimension_id] tinyint NOT NULL
    , [calc_type_id] tinyint NOT NULL
    , [is_active] bit NOT NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_dimension_calc_types_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_dimension_calc_types_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_dimension_calc_types_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_dimension_calc_types_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_dimension_calc_types] PRIMARY KEY ([dimension_id], [calc_type_id])
);
GO

ALTER TABLE [dbo25].[dimension_calc_types] ADD CONSTRAINT [FK_dimension_calc_types_calc_types] FOREIGN KEY ([calc_type_id]) REFERENCES [dbo25].[calc_types] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[dimension_calc_types] ADD CONSTRAINT [FK_dimension_calc_types_dimensions] FOREIGN KEY ([dimension_id]) REFERENCES [dbo25].[dimensions] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

CREATE TABLE [dbo25].[form_permissions] (
      [form_id] int NOT NULL
    , [username] nvarchar(128) NOT NULL
    , [select_permission] bit NOT NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_form_permissions_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_form_permissions_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_form_permissions_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_form_permissions_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_form_permissions] PRIMARY KEY ([form_id], [username])
);
GO

ALTER TABLE [dbo25].[form_permissions] ADD CONSTRAINT [FK_form_permissions_forms] FOREIGN KEY ([form_id]) REFERENCES [dbo25].[forms] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

CREATE TABLE [dbo25].[members] (
      [id] int IDENTITY(1,1) NOT NULL
    , [company_id] int NOT NULL
    , [dimension_id] tinyint NOT NULL
    , [sort_order] int NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(400) NOT NULL
    , [comment] nvarchar(1000) NULL
    , [tax_rate_id] int NULL
    , [unit_id] int NULL
    , [calc_type_id] tinyint NOT NULL CONSTRAINT [DF_members_calc_types] DEFAULT((0))
    , [previous_period_id] int NULL
    , [same_period_id] int NULL
    , [decimal_places] tinyint NULL
    , [is_percent] bit NULL
    , [row_color] tinyint NULL
    , [row_bold] tinyint NULL
    , [row_indent] tinyint NULL
    , [show_line] bit NULL CONSTRAINT [DF_members_show_line] DEFAULT((1))
    , [show_line_before] bit NULL CONSTRAINT [DF_members_show_line_before] DEFAULT((0))
    , [show_line_after] bit NULL CONSTRAINT [DF_members_show_line_after] DEFAULT((0))
    , [is_translatable] bit NULL CONSTRAINT [DF_members_is_translatable] DEFAULT((1))
    , [is_active] bit NOT NULL CONSTRAINT [DF_members_is_active] DEFAULT((1))
    , [external_id] int NULL
    , [external_code] nvarchar(50) NULL
    , [excel_formula] nvarchar(400) NULL
    , [cube_formula] nvarchar(max) NULL
    , [is_calculated] tinyint NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_members_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_members_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_members_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_members_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_members] PRIMARY KEY ([id])
    , CONSTRAINT [IX_members_code] UNIQUE ([company_id], [code])
);
GO

CREATE INDEX [IX_members_external_code] ON [dbo25].[members] ([external_code]);
GO

CREATE INDEX [IX_members_external_id] ON [dbo25].[members] ([external_id]);
GO

CREATE INDEX [IX_members_name] ON [dbo25].[members] ([company_id], [name]);
GO

ALTER TABLE [dbo25].[members] ADD CONSTRAINT [FK_members_calc_types] FOREIGN KEY ([calc_type_id]) REFERENCES [dbo25].[calc_types] ([id]) ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[members] ADD CONSTRAINT [FK_members_companies] FOREIGN KEY ([company_id]) REFERENCES [dbo25].[companies] ([id]);
GO

ALTER TABLE [dbo25].[members] ADD CONSTRAINT [FK_members_dimensions] FOREIGN KEY ([dimension_id]) REFERENCES [dbo25].[dimensions] ([id]) ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[members] ADD CONSTRAINT [FK_members_members_same_period_id] FOREIGN KEY ([same_period_id]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[members] ADD CONSTRAINT [FK_members_previous_period] FOREIGN KEY ([previous_period_id]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[members] ADD CONSTRAINT [FK_members_tax_rates] FOREIGN KEY ([tax_rate_id]) REFERENCES [dbo25].[tax_rates] ([id]) ON DELETE SET NULL ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[members] ADD CONSTRAINT [FK_members_units] FOREIGN KEY ([unit_id]) REFERENCES [dbo25].[units] ([id]) ON DELETE SET NULL ON UPDATE CASCADE;
GO

CREATE TABLE [dbo25].[currency_rates] (
      [base_currency_id] int NOT NULL
    , [quote_currency_id] int NOT NULL
    , [time_id] int NOT NULL
    , [category_id] int NOT NULL
    , [currency_rate] float NOT NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_currency_rates_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_currency_rates_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_currency_rates_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_currency_rates_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_currency_rates] PRIMARY KEY ([base_currency_id], [quote_currency_id], [time_id], [category_id])
);
GO

ALTER TABLE [dbo25].[currency_rates] ADD CONSTRAINT [FK_currency_rates_members_category_id] FOREIGN KEY ([category_id]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[currency_rates] ADD CONSTRAINT [FK_currency_rates_members_time_id] FOREIGN KEY ([time_id]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[currency_rates] ADD CONSTRAINT [FK_currency_rates_units_from_id] FOREIGN KEY ([base_currency_id]) REFERENCES [dbo25].[units] ([id]);
GO

ALTER TABLE [dbo25].[currency_rates] ADD CONSTRAINT [FK_currency_rates_units_to_id] FOREIGN KEY ([quote_currency_id]) REFERENCES [dbo25].[units] ([id]);
GO

CREATE TABLE [dbo25].[dimension_properties] (
      [company_id] int NOT NULL
    , [id] tinyint NOT NULL
    , [code] nvarchar(50) NOT NULL
    , [name] nvarchar(50) NOT NULL
    , [parameter_name] nvarchar(50) NOT NULL
    , [sort_order] tinyint NOT NULL
    , [join_order] tinyint NOT NULL
    , [name_format_id] tinyint NOT NULL CONSTRAINT [DF_dimension_properties_name_format_id] DEFAULT((0))
    , [root_member_id] int NULL
    , [default_member_id] int NULL
    , [is_protected] bit NOT NULL CONSTRAINT [DF_dimension_properties_is_protected] DEFAULT((0))
    , [default_select_permission] bit NOT NULL
    , [default_update_permission] bit NOT NULL
    , [is_active] bit NOT NULL CONSTRAINT [DF_dimension_properties_is_active] DEFAULT((0))
    , [external_id] int NULL
    , [external_code] nvarchar(50) NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_dimension_properties_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_dimension_properties_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_dimension_properties_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_dimension_properties_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_dimension_properties] PRIMARY KEY ([company_id], [id])
    , CONSTRAINT [IX_dimension_properties_code] UNIQUE ([company_id], [code])
    , CONSTRAINT [IX_dimension_properties_name] UNIQUE ([company_id], [name])
);
GO

CREATE INDEX [IX_dimension_properties_external_code] ON [dbo25].[dimension_properties] ([external_code]);
GO

CREATE INDEX [IX_dimension_properties_external_id] ON [dbo25].[dimension_properties] ([external_id]);
GO

ALTER TABLE [dbo25].[dimension_properties] ADD CONSTRAINT [FK_dimension_properties_companies] FOREIGN KEY ([company_id]) REFERENCES [dbo25].[companies] ([id]);
GO

ALTER TABLE [dbo25].[dimension_properties] ADD CONSTRAINT [FK_dimension_properties_dimensions] FOREIGN KEY ([id]) REFERENCES [dbo25].[dimensions] ([id]);
GO

ALTER TABLE [dbo25].[dimension_properties] ADD CONSTRAINT [FK_dimension_properties_members_default_member_id] FOREIGN KEY ([default_member_id]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[dimension_properties] ADD CONSTRAINT [FK_dimension_properties_members_root_member_id] FOREIGN KEY ([root_member_id]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[dimension_properties] ADD CONSTRAINT [FK_dimension_properties_name_formats] FOREIGN KEY ([name_format_id]) REFERENCES [dbo25].[name_formats] ([id]);
GO

CREATE TABLE [dbo25].[factors] (
      [member_id] int NOT NULL
    , [parent_id] int NOT NULL
    , [calc_type_id] tinyint NOT NULL CONSTRAINT [DF_factors_calc_type_id] DEFAULT((0))
    , [factor] float NOT NULL CONSTRAINT [DF_factors_factor] DEFAULT((1))
    , [is_calculated] AS (case when [member_id]=[parent_id] then (0) else (1) end)
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_factors_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_factors_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_factors_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_factors_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_factors] PRIMARY KEY ([member_id], [parent_id], [calc_type_id])
);
GO

ALTER TABLE [dbo25].[factors] ADD CONSTRAINT [FK_factors_calc_types_calc_type_id] FOREIGN KEY ([calc_type_id]) REFERENCES [dbo25].[calc_types] ([id]);
GO

ALTER TABLE [dbo25].[factors] ADD CONSTRAINT [FK_factors_members_member_id] FOREIGN KEY ([member_id]) REFERENCES [dbo25].[members] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[factors] ADD CONSTRAINT [FK_factors_members_parent_id] FOREIGN KEY ([parent_id]) REFERENCES [dbo25].[members] ([id]);
GO

CREATE TABLE [dbo25].[facts] (
      [id] int IDENTITY(1,1) NOT NULL
    , [id1] int NOT NULL
    , [id2] int NOT NULL
    , [id3] int NOT NULL
    , [id4] int NOT NULL
    , [id5] int NOT NULL
    , [id6] int NOT NULL
    , [id7] int NOT NULL
    , [value] float NULL
    , [source_value] float NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_facts_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_facts_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_facts_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_facts_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_facts] PRIMARY KEY ([id])
);
GO

ALTER TABLE [dbo25].[facts] ADD CONSTRAINT [FK_facts_members_id1] FOREIGN KEY ([id1]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[facts] ADD CONSTRAINT [FK_facts_members_id2] FOREIGN KEY ([id2]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[facts] ADD CONSTRAINT [FK_facts_members_id3] FOREIGN KEY ([id3]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[facts] ADD CONSTRAINT [FK_facts_members_id4] FOREIGN KEY ([id4]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[facts] ADD CONSTRAINT [FK_facts_members_id5] FOREIGN KEY ([id5]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[facts] ADD CONSTRAINT [FK_facts_members_id6] FOREIGN KEY ([id6]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[facts] ADD CONSTRAINT [FK_facts_members_id7] FOREIGN KEY ([id7]) REFERENCES [dbo25].[members] ([id]);
GO

CREATE TABLE [dbo25].[form_dimensions] (
      [form_id] int NOT NULL
    , [dimension_id] tinyint NOT NULL
    , [axis_type_id] tinyint NOT NULL
    , [parameter_index] tinyint NULL
    , [header_order] tinyint NULL
    , [root_member_id] int NULL
    , [parameter_start_level] tinyint NULL
    , [parameter_end_level] tinyint NULL
    , [form_start_level] tinyint NULL
    , [form_end_level] tinyint NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_form_dimensions_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_form_dimensions_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_form_dimensions_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_form_dimensions_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_form_dimensions] PRIMARY KEY ([form_id], [dimension_id])
);
GO

ALTER TABLE [dbo25].[form_dimensions] ADD CONSTRAINT [FK_form_dimensions_axis_types] FOREIGN KEY ([axis_type_id]) REFERENCES [dbo25].[axis_types] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[form_dimensions] ADD CONSTRAINT [FK_form_dimensions_dimensions] FOREIGN KEY ([dimension_id]) REFERENCES [dbo25].[dimensions] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[form_dimensions] ADD CONSTRAINT [FK_form_dimensions_forms] FOREIGN KEY ([form_id]) REFERENCES [dbo25].[forms] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[form_dimensions] ADD CONSTRAINT [FK_form_dimensions_members] FOREIGN KEY ([root_member_id]) REFERENCES [dbo25].[members] ([id]);
GO

CREATE TABLE [dbo25].[form_rows] (
      [id] int IDENTITY(1,1) NOT NULL
    , [rowset_id] int NOT NULL
    , [sort_order] int NOT NULL
    , [member_id] int NULL
    , [code] nvarchar(50) NULL
    , [name] nvarchar(400) NULL
    , [decimal_places] tinyint NULL
    , [is_percent] bit NULL
    , [row_color] tinyint NULL
    , [row_bold] tinyint NULL
    , [row_indent] tinyint NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_form_rows_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_form_rows_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_form_rows_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_form_rows_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_form_rows] PRIMARY KEY ([id])
);
GO

CREATE INDEX [IX_form_rows_rowset_id_member_id] ON [dbo25].[form_rows] ([rowset_id], [member_id]);
GO

CREATE INDEX [IX_form_rows_rowset_id_sort_order] ON [dbo25].[form_rows] ([rowset_id], [sort_order]);
GO

ALTER TABLE [dbo25].[form_rows] ADD CONSTRAINT [FK_form_rows_members] FOREIGN KEY ([member_id]) REFERENCES [dbo25].[members] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[form_rows] ADD CONSTRAINT [FK_form_rows_members_rowset_id] FOREIGN KEY ([rowset_id]) REFERENCES [dbo25].[members] ([id]);
GO

CREATE TABLE [dbo25].[form_subtotals] (
      [id] int IDENTITY(1,1) NOT NULL
    , [form_id] int NOT NULL
    , [dimension_id1] tinyint NULL
    , [member_id1] int NULL
    , [dimension_id2] tinyint NULL
    , [member_id2] int NULL
    , [dimension_id3] tinyint NULL
    , [member_id3] int NULL
    , [show] bit NOT NULL CONSTRAINT [DF_form_subtotals_show] DEFAULT((0))
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_form_subtotals_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_form_subtotals_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_form_subtotals_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_form_subtotals_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_form_subtotals] PRIMARY KEY ([id])
);
GO

ALTER TABLE [dbo25].[form_subtotals] ADD CONSTRAINT [FK_form_subtotals_dimensions_dimension_id1] FOREIGN KEY ([dimension_id1]) REFERENCES [dbo25].[dimensions] ([id]);
GO

ALTER TABLE [dbo25].[form_subtotals] ADD CONSTRAINT [FK_form_subtotals_dimensions_dimension_id2] FOREIGN KEY ([dimension_id2]) REFERENCES [dbo25].[dimensions] ([id]);
GO

ALTER TABLE [dbo25].[form_subtotals] ADD CONSTRAINT [FK_form_subtotals_dimensions_dimension_id3] FOREIGN KEY ([dimension_id3]) REFERENCES [dbo25].[dimensions] ([id]);
GO

ALTER TABLE [dbo25].[form_subtotals] ADD CONSTRAINT [FK_form_subtotals_forms] FOREIGN KEY ([form_id]) REFERENCES [dbo25].[forms] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[form_subtotals] ADD CONSTRAINT [FK_form_subtotals_members_member_id1] FOREIGN KEY ([member_id1]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[form_subtotals] ADD CONSTRAINT [FK_form_subtotals_members_member_id2] FOREIGN KEY ([member_id2]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[form_subtotals] ADD CONSTRAINT [FK_form_subtotals_members_member_id3] FOREIGN KEY ([member_id3]) REFERENCES [dbo25].[members] ([id]);
GO

CREATE TABLE [dbo25].[hierarchies] (
      [member_id] int NOT NULL
    , [parent_id] int NOT NULL
    , [level] int NOT NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_hierarchies_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_hierarchies_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_hierarchies_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_hierarchies_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_hierarchies] PRIMARY KEY ([member_id], [parent_id])
);
GO

CREATE INDEX [IX_hierarchies_parent_id] ON [dbo25].[hierarchies] ([parent_id]);
GO

ALTER TABLE [dbo25].[hierarchies] ADD CONSTRAINT [FK_hierarchies_members_member_id] FOREIGN KEY ([member_id]) REFERENCES [dbo25].[members] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[hierarchies] ADD CONSTRAINT [FK_hierarchies_members_parent_id] FOREIGN KEY ([parent_id]) REFERENCES [dbo25].[members] ([id]);
GO

CREATE TABLE [dbo25].[member_permissions] (
      [member_id] int NOT NULL
    , [username] nvarchar(128) NOT NULL
    , [select_permission] bit NOT NULL
    , [update_permission] bit NOT NULL
    , [is_inherited] bit NOT NULL
    , [permission] AS (case when [update_permission]=(1) then case when [is_inherited]=(1) then 'w' else 'W' end when [select_permission]=(1) then case when [is_inherited]=(1) then 'r' else 'R' end when [select_permission]=(0) then case when [is_inherited]=(1) then 'd' else 'D' end else '?' end)
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_member_permissions_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_member_permissions_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_member_permissions_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_member_permissions_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_member_permissions] PRIMARY KEY ([member_id], [username])
);
GO

ALTER TABLE [dbo25].[member_permissions] ADD CONSTRAINT [FK_member_permissions_members_member_id] FOREIGN KEY ([member_id]) REFERENCES [dbo25].[members] ([id]);
GO

CREATE TABLE [dbo25].[member_relations] (
      [member_id] int NOT NULL
    , [partner_id] int NOT NULL
    , [select_permission] bit NULL
    , [update_permission] bit NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_member_relations_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_member_relations_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_member_relations_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_member_relations_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_member_relations] PRIMARY KEY ([member_id], [partner_id])
);
GO

CREATE INDEX [IX_member_relations_partner_id_select_permission] ON [dbo25].[member_relations] ([partner_id], [select_permission]);
GO

CREATE INDEX [IX_member_relations_partner_id_update_permission] ON [dbo25].[member_relations] ([partner_id], [update_permission]);
GO

ALTER TABLE [dbo25].[member_relations] ADD CONSTRAINT [FK_member_relations_members_member_id] FOREIGN KEY ([member_id]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[member_relations] ADD CONSTRAINT [FK_member_relations_members_partner_id] FOREIGN KEY ([partner_id]) REFERENCES [dbo25].[members] ([id]);
GO

CREATE TABLE [dbo25].[parents] (
      [member_id] int NOT NULL
    , [parent_id] int NOT NULL
    , [factor] float NOT NULL CONSTRAINT [DF_parents_factor] DEFAULT((1))
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_parents_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_parents_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_parents_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_parents_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_parents] PRIMARY KEY ([member_id], [parent_id])
);
GO

ALTER TABLE [dbo25].[parents] ADD CONSTRAINT [FK_parents_members_member_id] FOREIGN KEY ([member_id]) REFERENCES [dbo25].[members] ([id]) ON DELETE CASCADE ON UPDATE CASCADE;
GO

ALTER TABLE [dbo25].[parents] ADD CONSTRAINT [FK_parents_members_parent_id] FOREIGN KEY ([parent_id]) REFERENCES [dbo25].[members] ([id]);
GO

CREATE TABLE [dbo25].[strings] (
      [id] int IDENTITY(1,1) NOT NULL
    , [id1] int NOT NULL
    , [id2] int NOT NULL
    , [id3] int NOT NULL
    , [id4] int NOT NULL
    , [id5] int NOT NULL
    , [id6] int NOT NULL
    , [id7] int NOT NULL
    , [value] nvarchar(4000) NULL
    , [created_by] nvarchar(128) NULL CONSTRAINT [DF_strings_created_by] DEFAULT(user_name())
    , [created_on] datetime NULL CONSTRAINT [DF_strings_created_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , [modified_by] nvarchar(128) NULL CONSTRAINT [DF_strings_modified_by] DEFAULT(user_name())
    , [modified_on] datetime NULL CONSTRAINT [DF_strings_modified_on] DEFAULT(CONVERT([datetime2](0),getdate()))
    , CONSTRAINT [PK_strings] PRIMARY KEY ([id])
);
GO

ALTER TABLE [dbo25].[strings] ADD CONSTRAINT [FK_strings_members_id1] FOREIGN KEY ([id1]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[strings] ADD CONSTRAINT [FK_strings_members_id2] FOREIGN KEY ([id2]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[strings] ADD CONSTRAINT [FK_strings_members_id3] FOREIGN KEY ([id3]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[strings] ADD CONSTRAINT [FK_strings_members_id4] FOREIGN KEY ([id4]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[strings] ADD CONSTRAINT [FK_strings_members_id5] FOREIGN KEY ([id5]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[strings] ADD CONSTRAINT [FK_strings_members_id6] FOREIGN KEY ([id6]) REFERENCES [dbo25].[members] ([id]);
GO

ALTER TABLE [dbo25].[strings] ADD CONSTRAINT [FK_strings_members_id7] FOREIGN KEY ([id7]) REFERENCES [dbo25].[members] ([id]);
GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Returns the translated string
-- =============================================

CREATE FUNCTION [dbo25].[get_translated_string]
(
    @string nvarchar(128) = NULL
    , @data_language varchar(10) = NULL
)
RETURNS nvarchar(128)
AS
BEGIN

RETURN COALESCE((
    SELECT
        COALESCE(TRANSLATED_DESC, TRANSLATED_NAME)
    FROM
        xls.translations
    WHERE
        TABLE_SCHEMA = 'dbo25' AND TABLE_NAME = 'strings'
        AND COLUMN_NAME = @string AND LANGUAGE_NAME = COALESCE(@data_language, 'en')
    ), @string)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Returns JSON array values
-- =============================================

CREATE FUNCTION [dbo25].[get_json_array_values]
(
    @c nvarchar(max) = NULL
)
RETURNS @t TABLE(id int IDENTITY(1,1) NOT NULL PRIMARY KEY, value nvarchar(max))
AS
BEGIN

-- SET NOCOUNT ON
-- DECLARE @c nvarchar(max) = '[1,null,"a","b,c","d,\\\",e","f",1000]'
-- DECLARE @t TABLE(id int IDENTITY(1,1) NOT NULL PRIMARY KEY, value nvarchar(max))

DECLARE @start_index AS int = CHARINDEX('[', @c)

IF @start_index = 0
    BEGIN
    INSERT INTO @t (value) VALUES (@c)
    RETURN
    END

DECLARE @value nvarchar(max)

DECLARE @quote_index int = CHARINDEX('"', @c)
DECLARE @comma_index int = CHARINDEX(',', @c)

DECLARE @end_index AS int
DECLARE @buffered_index AS int
DECLARE @quote_opened int
DECLARE @slash_index int
DECLARE @pos int = 0
DECLARE @array_level int = 0
DECLARE @object_level int = 0
-- DECLARE @i int = 0

-- PRINT '   i  s  ,  "  ar    1234567890123456789012345678901234567890'

WHILE @start_index > 0 AND @start_index < LEN(@c) -- AND @i < 20
    BEGIN

    -- SET @i = @i + 1

    -- PRINT '   ' + CAST(@i AS char(3)) + CAST(@start_index AS char(3)) + CAST(@comma_index AS char(3)) + CAST(@quote_index AS char(3)) + CAST(@array_level AS char(3)) + '  ' + REPLICATE(' ', @start_index) + SUBSTRING(@c, @start_index, LEN(@c))

    IF @quote_index > 0 AND @quote_index < @comma_index
        BEGIN
        SET @quote_opened = 0
        WHILE @quote_index > 0 AND @quote_index < @comma_index AND @comma_index > 0
            BEGIN
            SET @quote_opened = CASE @quote_opened WHEN 1 THEN 0 ELSE 1 END

            SET @slash_index = @quote_index - 1
            WHILE @slash_index > 0 AND SUBSTRING(@c, @slash_index, 1) = '\'
                BEGIN
                SET @quote_opened = CASE @quote_opened WHEN 1 THEN 0 ELSE 1 END
                SET @slash_index = @slash_index - 1
                END

            SET @quote_index = CHARINDEX('"', @c, @quote_index + 1)

            IF @quote_index > @comma_index AND @quote_opened = 1
                SET @comma_index = CHARINDEX(',', @c, @comma_index + 1)
            END
        END

    IF @comma_index = 0
        SET @end_index = LEN(@c)
    ELSE
        BEGIN
        SET @end_index = @comma_index
        SET @comma_index = CHARINDEX(',', @c, @comma_index + 1)
        END

    SET @value = LTRIM(RTRIM(SUBSTRING(@c, @start_index + 1, @end_index - @start_index - 1)))

    -- PRINT '->                   ' + REPLICATE(' ', @start_index) + RTRIM(SUBSTRING(@c, @start_index + 1, @end_index - @start_index - 1))

    SET @pos = 1
    WHILE SUBSTRING(@value, @pos, 1) = '['
        BEGIN
        SET @array_level = @array_level + 1
        IF @array_level = 1
            SET @buffered_index = @start_index
        SET @pos = @pos + 1
        END

    SET @pos = LEN(@value)
    WHILE SUBSTRING(@value, @pos, 1) = ']'
        BEGIN
        SET @array_level = @array_level - 1
        IF @array_level = 0
            SET @value = LTRIM(RTRIM(SUBSTRING(@c, @buffered_index + 1, @end_index - @buffered_index - 1)))
        SET @pos = @pos - 1
        END

    SET @pos = 1
    WHILE SUBSTRING(@value, @pos, 1) = '{'
        BEGIN
        SET @object_level = @object_level + 1
        IF @object_level = 1
            SET @buffered_index = @start_index
        SET @pos = @pos + 1
        END

    SET @pos = LEN(@value)
    WHILE SUBSTRING(@value, @pos, 1) = '}'
        BEGIN
        SET @object_level = @object_level - 1
        IF @object_level = 0
            SET @value = LTRIM(RTRIM(SUBSTRING(@c, @buffered_index + 1, @end_index - @buffered_index - 1)))
        SET @pos = @pos - 1
        END

    IF @array_level = 0 AND @object_level = 0
        BEGIN
        IF LEFT(@value, 1) = '"'
            BEGIN
            SET @value = SUBSTRING(@value, 2, LEN(@value) - 2)

            IF CHARINDEX('\', @value) > 0
                SET @value = REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(@value, '\\', CHAR(1)+CHAR(1)),
                    '\r', CHAR(13)), '\n', CHAR(10)), '\f', CHAR(12)), '\t', CHAR(9)), '\/', '/'), '\"', '"'),
                    CHAR(1)+CHAR(1), '\')
            END

        INSERT INTO @t (value) VALUES (@value)
        END

    SET @start_index = @end_index
    END

RETURN
END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Returns JSON object values
-- =============================================

CREATE FUNCTION [dbo25].[get_json_object_values]
(
    @c nvarchar(max) = NULL
)
RETURNS @t TABLE(name nvarchar(100) NOT NULL PRIMARY KEY, value nvarchar(max))
AS
BEGIN

DECLARE @start_index AS int = CHARINDEX('{', @c)

IF @start_index = 0
    BEGIN
    INSERT INTO @t (value) VALUES (@c)
    RETURN
    END

DECLARE @value nvarchar(max)
DECLARE @name nvarchar(100)

DECLARE @quote_index int = CHARINDEX('"', @c)
DECLARE @comma_index int = CHARINDEX(',', @c)

DECLARE @end_index AS int
DECLARE @buffered_index AS int
DECLARE @quote_opened int
DECLARE @slash_index int
DECLARE @pos int = 0
DECLARE @array_level int = 0
DECLARE @object_level int = 0
-- DECLARE @i int = 0

-- PRINT '   i  s  ,  "  ar    1234567890123456789012345678901234567890'

WHILE @start_index > 0 AND @start_index < LEN(@c) -- AND @i < 10
    BEGIN

    -- SET @i = @i + 1

    -- PRINT '   ' + CAST(@i AS char(3)) + CAST(@start_index AS char(3)) + CAST(@comma_index AS char(3)) + CAST(@quote_index AS char(3)) + CAST(@array_level AS char(3)) + '  ' + REPLICATE(' ', @start_index) + SUBSTRING(@c, @start_index, LEN(@c))

    IF @quote_index > 0 AND @quote_index < @comma_index
        BEGIN
        SET @quote_opened = 0
        WHILE @quote_index > 0 AND @quote_index < @comma_index AND @comma_index > 0
            BEGIN
            SET @quote_opened = CASE @quote_opened WHEN 1 THEN 0 ELSE 1 END

            SET @slash_index = @quote_index - 1
            WHILE @slash_index > 0 AND SUBSTRING(@c, @slash_index, 1) = '\'
                BEGIN
                SET @quote_opened = CASE @quote_opened WHEN 1 THEN 0 ELSE 1 END
                SET @slash_index = @slash_index - 1
                END
            IF @quote_opened = 1
                BEGIN
                SET @quote_index = CHARINDEX('"', @c, @quote_index + 1)
                SET @quote_opened = 0
                CONTINUE
                END

            SET @quote_index = CHARINDEX('"', @c, @quote_index + 1)

            IF @quote_index > @comma_index AND @quote_opened = 1
                SET @comma_index = CHARINDEX(',', @c, @comma_index + 1)
            END
        END

    IF @comma_index = 0
        SET @end_index = LEN(@c)
    ELSE
        BEGIN
        SET @end_index = @comma_index
        SET @comma_index = CHARINDEX(',', @c, @comma_index + 1)
        END

    SET @value = LTRIM(RTRIM(SUBSTRING(@c, @start_index + 1, @end_index - @start_index - 1)))

    -- PRINT '->                   ' + REPLICATE(' ', @start_index) + RTRIM(SUBSTRING(@c, @start_index + 1, @end_index - @start_index - 1))

    SET @pos = CHARINDEX('[', @value)
    WHILE @pos > 0
        BEGIN
        SET @array_level = @array_level + 1
        IF @array_level = 1
            SET @buffered_index = @start_index
        SET @pos = CHARINDEX('[', @value, @pos + 1)
        END

    SET @pos = LEN(@value)
    WHILE SUBSTRING(@value, @pos, 1) = ']'
        BEGIN
        SET @array_level = @array_level - 1
        IF @array_level = 0
            SET @value = LTRIM(RTRIM(SUBSTRING(@c, @buffered_index + 1, @end_index - @buffered_index - 1)))
        SET @pos = @pos - 1
        END

    SET @pos = CHARINDEX('{', @value)
    WHILE @pos > 0
        BEGIN
        SET @object_level = @object_level + 1
        IF @object_level = 1
            SET @buffered_index = @start_index
        SET @pos = CHARINDEX('{', @value, @pos + 1)
        END

    SET @pos = LEN(@value)
    WHILE SUBSTRING(@value, @pos, 1) = '}'
        BEGIN
        SET @object_level = @object_level - 1
        IF @object_level = 0
            SET @value = LTRIM(RTRIM(SUBSTRING(@c, @buffered_index + 1, @end_index - @buffered_index - 1)))
        SET @pos = @pos - 1
        END

    IF @array_level = 0 AND @object_level = 0
        BEGIN
        SET @pos = CHARINDEX(':', @value)

        SET @name = RTRIM(LEFT(@value, @pos - 1))

        IF LEFT(@name, 1) = '"'
            BEGIN
            SET @name = SUBSTRING(@name, 2, LEN(@name) - 2)

            IF CHARINDEX('\', @name) > 0
                SET @name = REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(@name, '\\', CHAR(1)+CHAR(1)),
                    '\r', CHAR(13)), '\n', CHAR(10)), '\f', CHAR(12)), '\t', CHAR(9)), '\/', '/'), '\"', '"'),
                    CHAR(1)+CHAR(1), '\')
            END

        SET @value = SUBSTRING(@value, @pos + 1, LEN(@value))

        IF LEFT(@value, 1) = '"'
            BEGIN
            SET @value = SUBSTRING(@value, 2, LEN(@value) - 2)

            IF CHARINDEX('\', @value) > 0
                SET @value = REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(@value, '\\', CHAR(1)+CHAR(1)),
                    '\r', CHAR(13)), '\n', CHAR(10)), '\f', CHAR(12)), '\t', CHAR(9)), '\/', '/'), '\"', '"'),
                    CHAR(1)+CHAR(1), '\')
            END

        INSERT INTO @t (name, value) VALUES (@name, @value)
        END

    SET @start_index = @end_index
    END

RETURN
END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Returns quarter members for the year
-- =============================================

CREATE FUNCTION [dbo25].[get_quarter_members]
(
    @year int = NULL
)
RETURNS @t TABLE(
    code nvarchar(50) NOT NULL PRIMARY KEY,
    name nvarchar(400) NOT NULL,
    dimension_id tinyint NOT NULL,
    sort_order int NOT NULL,
    calc_type_id tinyint NOT NULL,
    row_color tinyint NULL,
    previous_period nvarchar(50) NULL,
    same_period nvarchar(50) NULL,
    parent1 nvarchar(50) NULL,
    parent2 nvarchar(50) NULL,
    parent3 nvarchar(50) NULL
)
AS
BEGIN

INSERT INTO @t (code, name, dimension_id, sort_order, calc_type_id, row_color, previous_period, same_period, parent1, parent2, parent3)
SELECT
    c.code, c.code AS name, 2 AS dimension_id, c.sort_order, c.calc_type_id, c.row_color, previous_period, same_period, parent1, parent2, parent3
FROM
    (
        SELECT
            -- 2018.00
            CAST(@year AS nvarchar) + '.00' AS code
            , 1000000 + @year * 100 AS sort_order
            , 1 AS calc_type_id
            , NULL AS row_color
            , NULL AS previous_period
            , NULL AS same_period
            , 'OPENING.BALANCE' AS parent1
            , NULL AS parent2
            , NULL AS parent3
        UNION ALL
        SELECT
            -- 2018.Q
            CAST(@year AS nvarchar) + '.Q' AS code
            , 4000000 + @year * 100 AS sort_order
            , 1 AS calc_type_id
            , 11 AS row_color
            , CAST(@year AS nvarchar) + '.00' AS previous_period
            , CAST(@year - 1 AS nvarchar) + '.Q' AS same_period
            , 'YEARS.Q' AS parent1
            , NULL AS parent2
            , NULL AS parent3
        UNION ALL
        SELECT
            -- 2018.Q1 .. 2018.Q4
            CAST(@year AS nvarchar) + '.Q' + mv.m AS code
            , 4000000 + @year * 100 + CAST(mv.m AS int) AS sort_order
            , 1 AS calc_type_id
            , NULL AS row_color
            , CASE WHEN mv.m = '1' THEN CAST(@year AS nvarchar) + '.00' ELSE CAST(@year AS nvarchar) + '.Q' + RIGHT(CAST(100 + CAST(mv.m AS int) - 1 AS nvarchar), 1) END AS previous_period
            , CAST(@year - 1 AS nvarchar) + '.Q' + mv.m AS same_period
            , CAST(@year AS nvarchar) + '.Q' AS parent1
            , NULL AS parent2
            , NULL AS parent3
        FROM
            (VALUES ('1'), ('2'), ('3'), ('4')) mv(m)
    ) c

RETURN
END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Returns standard members
-- =============================================

CREATE FUNCTION [dbo25].[get_standard_members]
(
)
RETURNS @t TABLE(
    code nvarchar(50) NOT NULL PRIMARY KEY,
    name nvarchar(400) NOT NULL,
    dimension_id tinyint NOT NULL,
    sort_order int NOT NULL,
    calc_type_id tinyint NOT NULL,
    row_color tinyint NULL,
    previous_period nvarchar(50) NULL,
    same_period nvarchar(50) NULL,
    parent1 nvarchar(50) NULL,
    factor1 float NULL
)
AS
BEGIN

INSERT INTO @t (dimension_id, sort_order, code, name, calc_type_id, previous_period, same_period, row_color, parent1, factor1)
SELECT
    dimension_id, sort_order, code, name, calc_type_id, previous_period, same_period, row_color, parent1, factor1
FROM
    (VALUES
        -- root members
        (1, 1, N'ACCOUNTS', N'Accounts', 0, NULL, NULL, 11, NULL, NULL),
        (2, 1, N'TIMES', N'TIMES', 0, NULL, NULL, 11, NULL, NULL),
        (3, 1, N'CATEGORIES', N'CATEGORIES', 0, NULL, NULL, 11, NULL, NULL),
        (4, 1, N'ENTITIES', N'ENTITIES', 0, NULL, NULL, 11, NULL, NULL),
        (5, 1, N'DIM5', N'DIM5', 0, NULL, NULL, NULL, NULL, NULL),
        (6, 1, N'DIM6', N'DIM6', 0, NULL, NULL, NULL, NULL, NULL),
        (7, 1, N'DIM7', N'DIM7', 0, NULL, NULL, NULL, NULL, NULL),

        -- standard time members
        (2, 1200000, N'OPENING.BALANCE', N'Opening Balance', 0, NULL, NULL, 12, N'TIMES', 0),
        (2, 2200000, N'YEARS', N'Years', 0, NULL, NULL, 12, N'TIMES', 0),
        (2, 3200000, N'YEARS.YTD', N'YTD', 0, NULL, NULL, 12, N'TIMES', 0),
        (2, 4200000, N'YEARS.Q', N'Years by Quarters', 0, NULL, NULL, 12, N'TIMES', 0),

        -- standard time rowsets
        (2, 101, N'years*', N'Years*', 9, NULL, NULL, NULL, NULL, NULL),
        (2, 102, N'quarters*', N'Quarters*', 9, NULL, NULL, NULL, NULL, NULL),

        -- standard categories
        (3, 2, N'CATEGORIES.ALL', N'All Categories', 0, NULL, NULL, NULL, N'CATEGORIES', 0),
        (3, 3, N'SAME_PERIOD', N'Same Period', 6, NULL, NULL, NULL, N'CATEGORIES.ALL', 0),
        (3, 4, N'BUDGET', N'Budget', 1, NULL, NULL, NULL, N'CATEGORIES.ALL', 0),
        (3, 5, N'FORECAST', N'Forecast', 1, NULL, NULL, NULL, N'CATEGORIES.ALL', 0),
        (3, 7, N'ACTUAL', N'Actual', 1, NULL, N'SAME_PERIOD', NULL, N'CATEGORIES.ALL', 0),

        -- standard differences
        (3, 8, N'BUDGET-SAME_PERIOD', N'Budget vs Same Period', 12, N'SAME_PERIOD', N'BUDGET', NULL, N'CATEGORIES.ALL', 0),
        (3, 9, N'BUDGET/SAME_PERIOD,%', N'Budget vs Same Period,%', 13, N'SAME_PERIOD', N'BUDGET', NULL, N'CATEGORIES.ALL', 0),
        (3, 10, N'FORECAST-BUDGET', N'Budget vs Forecast', 12, N'BUDGET', N'FORECAST', NULL, N'CATEGORIES.ALL', 0),
        (3, 11, N'FORECAST/BUDGET,%', N'Budget vs Forecast, %', 13, N'BUDGET', N'FORECAST', NULL, N'CATEGORIES.ALL', 0),
        (3, 12, N'FORECAST-SAME_PERIOD', N'Budget vs Forecast', 12, N'SAME_PERIOD', N'FORECAST', NULL, N'CATEGORIES.ALL', 0),
        (3, 13, N'FORECAST/SAME_PERIOD,%', N'Budget vs Forecast, %', 13, N'SAME_PERIOD', N'FORECAST', NULL, N'CATEGORIES.ALL', 0),
        (3, 14, N'ACTUAL-SAME_PERIOD', N'Actual vs Same Period', 12, N'SAME_PERIOD', N'ACTUAL', NULL, N'CATEGORIES.ALL', 0),
        (3, 15, N'ACTUAL-SAME_PERIOD,%', N'Actual vs Same Period,%', 13, N'SAME_PERIOD', N'ACTUAL', NULL, N'CATEGORIES.ALL', 0),
        (3, 16, N'ACTUAL-BUDGET', N'Budget vs Actual', 12, N'BUDGET', N'ACTUAL', NULL, N'CATEGORIES.ALL', 0),
        (3, 17, N'ACTUAL-BUDGET,%', N'Budget vs Actual, %', 13, N'BUDGET', N'ACTUAL', NULL, N'CATEGORIES.ALL', 0),
        (3, 18, N'ACTUAL-FORECAST', N'Actual vs Forecast', 12, N'FORECAST', N'ACTUAL', NULL, N'CATEGORIES.ALL', 0),
        (3, 19, N'ACTUAL-FORECAST,%', N'Actual vs Forecast,%', 13, N'FORECAST', N'ACTUAL', NULL, N'CATEGORIES.ALL', 0),

        -- standard category rowsets
        (3, 101, N'BUDGET_VS_SAME_PERIOD*', N'Budget vs Same Period', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 102, N'BUDGET_VS_SAME_PERIOD**', N'Budget vs Same Period+', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 103, N'BUDGET_VS_FORECAST*', N'Budget vs Forecast', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 104, N'BUDGET_VS_FORECAST**', N'Budget vs Forecast+', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 105, N'BUDGET_VS_ACTUAL*', N'Budget vs Actual', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 106, N'BUDGET_VS_ACTUAL**', N'Budget vs Actual+', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 107, N'ACTUAL_VS_SAME_PERIOD*', N'Actual vs Same Period', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 108, N'ACTUAL_VS_SAME_PERIOD**', N'Actual vs Same Period+', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 109, N'ACTUAL_VS_FORECAST*', N'Actual vs Forecast', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 110, N'ACTUAL_VS_FORECAST**', N'Actual vs Forecast+', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 111, N'FORECAST_VS_SAME_PERIOD*', N'Forecast vs Same Period', 9, NULL, NULL, NULL, NULL, NULL),
        (3, 112, N'FORECAST_VS_SAME_PERIOD**', N'Forecast vs Same Period+', 9, NULL, NULL, NULL, NULL, NULL),

        -- standard entity members
        (4, 2, N'ENTITIES.ALL', N'All Entities', 0, NULL, NULL, NULL, N'ENTITIES', 0),
        (4, 3, N'TOTAL', N'Company Total', 1, NULL, NULL, 11, N'ENTITIES.ALL', 0),

        -- standard dim5 members
        (5, 2, N'DIM5.ALL', N'DIM5.ALL', 0, NULL, NULL, NULL, N'DIM5', 0),
        (5, 3, N'DIM5.NONE', N'.', 1, NULL, NULL, NULL, N'DIM5.ALL', 1),

        -- standard dim6 members
        (6, 2, N'DIM6.ALL', N'DIM6.ALL', 0, NULL, NULL, NULL, N'DIM6', 0),
        (6, 3, N'DIM6.NONE', N'.', 1, NULL, NULL, NULL, N'DIM6.ALL', 1),

        -- standard dim7 members
        (7, 2, N'DIM7.ALL', N'DIM7.ALL', 0, NULL, NULL, NULL, N'DIM7', 0),
        (7, 3, N'DIM7.NONE', N'.', 1, NULL, NULL, NULL, N'DIM7.ALL', 1)

    ) t(dimension_id, sort_order, code, name, calc_type_id, previous_period, same_period, row_color, parent1, factor1)

RETURN
END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Returns time members for the year
-- =============================================

CREATE FUNCTION [dbo25].[get_year_members]
(
    @year int = NULL
)
RETURNS @t TABLE(
    code nvarchar(50) NOT NULL PRIMARY KEY,
    name nvarchar(400) NOT NULL,
    dimension_id tinyint NOT NULL,
    sort_order int NOT NULL,
    calc_type_id tinyint NOT NULL,
    row_color tinyint NULL,
    previous_period nvarchar(50) NULL,
    same_period nvarchar(50) NULL,
    parent1 nvarchar(50) NULL,
    parent2 nvarchar(50) NULL,
    parent3 nvarchar(50) NULL
)
AS
BEGIN

INSERT INTO @t (code, name, dimension_id, sort_order, calc_type_id, row_color, previous_period, same_period, parent1, parent2, parent3)
SELECT
    c.code, c.code AS name, 2 AS dimension_id, c.sort_order, c.calc_type_id, c.row_color, previous_period, same_period, parent1, parent2, parent3
FROM
    (
        SELECT
            -- 2018.00
            CAST(@year AS nvarchar) + '.00' AS code
            , 1000000 + @year * 100 AS sort_order
            , 1 AS calc_type_id
            , NULL AS row_color
            , NULL AS previous_period
            , NULL AS same_period
            , 'OPENING.BALANCE' AS parent1
            , NULL AS parent2
            , NULL AS parent3
        UNION ALL
        SELECT
            -- 2018
            CAST(@year AS nvarchar) AS code
            , 2000000 + @year * 100 AS sort_order
            , 1 AS calc_type_id
            , 11 AS row_color
            , CAST(@year AS nvarchar) + '.00' AS previous_period
            , CAST(@year - 1 AS nvarchar) AS same_period
            , 'YEARS' AS parent1
            , NULL AS parent2
            , NULL AS parent3
        UNION ALL
        SELECT
            -- 2018.01 .. 2018.12
            CAST(@year AS nvarchar) + '.' + mv.m AS code
            , 2000000 + @year * 100 + CAST(mv.m AS int) AS sort_order
            , 1 AS calc_type_id
            , NULL AS row_color
            , CAST(@year AS nvarchar) + '.' + RIGHT(CAST(100 + CAST(mv.m AS int) - 1 AS nvarchar), 2) AS previous_period
            , CAST(@year - 1 AS nvarchar) + '.' + mv.m AS same_period
            , CAST(@year AS nvarchar) AS parent1
            , CAST(@year AS nvarchar) + '.' + mv.m + '.YTD' AS parent2
            , CAST(@year AS nvarchar) + '.Q' + CAST(CEILING((CAST(mv.m AS int) + 2) / 3) AS nvarchar) AS parent3
        FROM
            (VALUES ('01'), ('02'), ('03'), ('04'), ('05'), ('06'), ('07'), ('08'), ('09'), ('10'), ('11'), ('12')) mv(m)
        UNION ALL
        SELECT
            -- 2018.YTD
            CAST(@year AS nvarchar) + '.YTD' AS code
            , 3000000 + @year * 100 AS sort_order
            , 0 AS calc_type_id
            , 11 AS row_color
            , NULL AS previous_period
            , NULL AS same_period
            , 'YEARS.YTD' AS parent1
            , NULL AS parent2
            , NULL AS parent3
        UNION ALL
        SELECT
            -- 2018.01.YTD .. 2018.12.YTD
            CAST(@year AS nvarchar) + '.' + mv.m + '.YTD' AS code
            , 3000000 + @year * 100 + CAST(mv.m AS int) AS sort_order
            , 5 AS calc_type_id
            , NULL AS row_color
            , CAST(@year AS nvarchar) + '.00' AS previous_period
            , CAST(@year - 1 AS nvarchar) + '.' + mv.m + '.YTD' AS same_period
            , CAST(@year AS nvarchar) + '.YTD' AS parent1
            , CASE WHEN mv.m = '12' THEN NULL ELSE CAST(@year AS nvarchar) + '.' + RIGHT(CAST(100 + CAST(mv.m AS int) + 1 AS nvarchar), 2) + '.YTD' END AS parent2
            , NULL AS parent3
        FROM
            (VALUES ('01'), ('02'), ('03'), ('04'), ('05'), ('06'), ('07'), ('08'), ('09'), ('10'), ('11'), ('12')) mv(m)
        UNION ALL
        SELECT
            -- 2018.Q
            CAST(@year AS nvarchar) + '.Q' AS code
            , 4000000 + @year * 100 AS sort_order
            , 1 AS calc_type_id
            , 11 AS row_color
            , CAST(@year AS nvarchar) + '.00' AS previous_period
            , CAST(@year - 1 AS nvarchar) + '.Q' AS same_period
            , 'YEARS.Q' AS parent1
            , NULL AS parent2
            , NULL AS parent3
        UNION ALL
        SELECT
            -- 2018.Q1 .. 2018.Q4
            CAST(@year AS nvarchar) + '.Q' + mv.m AS code
            , 4000000 + @year * 100 + CAST(mv.m AS int) AS sort_order
            , 1 AS calc_type_id
            , NULL AS row_color
            , CASE WHEN mv.m = '1' THEN CAST(@year AS nvarchar) + '.00' ELSE CAST(@year AS nvarchar) + '.Q' + RIGHT(CAST(100 + CAST(mv.m AS int) - 1 AS nvarchar), 1) END AS previous_period
            , CAST(@year - 1 AS nvarchar) + '.Q' + mv.m AS same_period
            , CAST(@year AS nvarchar) + '.Q' AS parent1
            , NULL AS parent2
            , NULL AS parent3
        FROM
            (VALUES ('1'), ('2'), ('3'), ('4')) mv(m)
    ) c

RETURN
END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects base fields of the dynamic forms
-- =============================================

CREATE VIEW [dbo25].[view_dynamic_form_fields]
AS

SELECT
    m.id AS id
    , m.id AS id2
    , m.id AS id3
    , m.sort_order AS sort_order
    , m.sort_order AS sort_order2
    , m.sort_order AS sort_order3
    , m.id AS member_id
    , m.id AS member_id2
    , m.id AS member_id3
    , 0 AS is_editable
    , m.decimal_places
    , m.is_percent
    , m.row_color
    , m.row_bold
    , m.row_indent
    , m.code AS code
    , m.name AS name
    , m.comment
    , m.code AS code1
    , m.code AS code2
    , m.name AS name2
    , m.code AS code3
    , m.name AS name3
    , c.name AS calc_type
    , tr.name AS tax_rate
    , u.name AS unit
    , NULL AS data_format
FROM
    dbo25.members m
    INNER JOIN dbo25.calc_types c ON c.id = m.calc_type_id
    LEFT OUTER JOIN dbo25.tax_rates tr ON tr.id = m.tax_rate_id
    INNER JOIN dbo25.units u ON u.id = m.unit_id
WHERE
    m.id IS NULL


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects calculated cube data
-- =============================================

CREATE VIEW [dbo25].[view_facts]
AS

SELECT
    f.company_id
    , f.id1
    , f.id2
    , f.id3
    , f.id4
    , f.id5
    , f.id6
    , f.id7
    , f.calc_type_id
    , MAX(f.unit_id) AS unit_id
    , CASE WHEN f.calc_type_id = 2 THEN AVG(source_value) ELSE SUM(source_value) END AS source_value
    , CASE WHEN f.calc_type_id = 2 THEN AVG(value) ELSE SUM(value) END AS value
FROM
    (
        SELECT
            m1.company_id
            , f1.parent_id AS id1
            , f2.parent_id AS id2
            , f3.parent_id AS id3
            , f4.parent_id AS id4
            , f5.parent_id AS id5
            , f6.parent_id AS id6
            , f7.parent_id AS id7
            , d1.calc_type_id
            , d1.unit_id
            , CASE
                WHEN m1.calc_type_id = 2 THEN NULL
                WHEN f1.parent_id = f1.member_id THEN f.source_value
                WHEN m1.unit_id = d1.unit_id THEN f.source_value * f1.factor
                WHEN m1.calc_type_id = 11 THEN 0
                ELSE f.value * f1.factor END AS source_value
            , f.value * f1.factor AS value
        FROM
            dbo25.facts f

            INNER JOIN dbo25.factors f1 ON f1.member_id = f.id1
            INNER JOIN dbo25.factors f2 ON f2.member_id = f.id2
            INNER JOIN dbo25.factors f3 ON f3.member_id = f.id3
            INNER JOIN dbo25.factors f4 ON f4.member_id = f.id4
            INNER JOIN dbo25.factors f5 ON f5.member_id = f.id5
            INNER JOIN dbo25.factors f6 ON f6.member_id = f.id6
            INNER JOIN dbo25.factors f7 ON f7.member_id = f.id7

            INNER JOIN dbo25.members m1 ON m1.id = f.id1
            INNER JOIN dbo25.members d1 ON d1.id = f1.parent_id

            INNER JOIN dbo25.users u ON u.company_id = m1.company_id AND u.username = USER_NAME() AND u.is_active = 1
        WHERE
            (f1.calc_type_id = 1 AND f2.calc_type_id = 1
            OR f1.calc_type_id = 3 AND f2.calc_type_id = 3)
            AND f.value IS NOT NULL
    ) f
GROUP BY
    f.calc_type_id
    , f.company_id
    , f.id5
    , f.id6
    , f.id3
    , f.id4
    , f.id1
    , f.id2
    , f.id7


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects source cube data
-- =============================================

CREATE VIEW [dbo25].[view_facts_data]
AS

SELECT
    f.id
    , m1.company_id
    , f.id1
    , f.id2
    , f.id3
    , f.id4
    , f.id5
    , f.id6
    , f.id7
    , m1.unit_id
    , m1.calc_type_id
    , f.source_value
    , f.value
FROM
    dbo25.facts f

    INNER JOIN dbo25.members m1 ON m1.id = f.id1
    INNER JOIN dbo25.members m2 ON m2.id = f.id2
    INNER JOIN dbo25.members m3 ON m3.id = f.id3
    INNER JOIN dbo25.members m4 ON m4.id = f.id4
    INNER JOIN dbo25.members m5 ON m5.id = f.id5
    INNER JOIN dbo25.members m6 ON m6.id = f.id6
    INNER JOIN dbo25.members m7 ON m7.id = f.id7

    LEFT OUTER JOIN dbo25.units n ON n.id = m1.unit_id
    INNER JOIN dbo25.calc_types t ON t.id = m1.calc_type_id

    INNER JOIN dbo25.users u ON u.company_id = m1.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = m1.company_id
WHERE
    f.value IS NOT NULL


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects database objects to connect for analysts
-- =============================================

CREATE VIEW [dbo25].[view_query_list]
AS

SELECT
    t.TABLE_SCHEMA
    , t.TABLE_NAME
    , t.TABLE_TYPE
    , t.TABLE_CODE
    , t.INSERT_PROCEDURE
    , t.UPDATE_PROCEDURE
    , t.DELETE_PROCEDURE
    , t.PROCEDURE_TYPE
FROM
    xls.queries t
WHERE
    t.TABLE_SCHEMA IN ('dbo25', 'xls25')
    AND NOT t.TABLE_NAME LIKE 'usp_export_%'
    AND NOT t.TABLE_NAME LIKE 'usp_import_%'


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects database objects to connect for analysts
-- =============================================

CREATE VIEW [xls25].[xl_query_list]
AS

SELECT
    t.TABLE_SCHEMA
    , t.TABLE_NAME
    , t.TABLE_TYPE
    , t.TABLE_CODE
    , t.INSERT_PROCEDURE
    , t.UPDATE_PROCEDURE
    , t.DELETE_PROCEDURE
    , t.PROCEDURE_TYPE
FROM
    xls.queries t
WHERE
    t.TABLE_SCHEMA IN ('xls25')
    AND NOT t.TABLE_NAME = 'usp_run_json_form'


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parents, hierarchies, and factors
-- =============================================

CREATE VIEW [xls25].[view_hierarchies]
AS

SELECT
    m.company_id
    , c.name AS company
    , d.id AS dimension_id
    , d.name AS dimension_name
    , 'parents' AS table_name
    , f.member_id
    , f.parent_id
    , p.calc_type_id
    , CASE d.name_format_id WHEN 0 THEN m.code WHEN 1 THEN m.name WHEN 2 THEN CAST(m.id AS nvarchar) WHEN 3 THEN CAST(m.id AS nvarchar) + ' - ' + m.name ELSE m.code + ' - ' + m.name END AS member
    , CASE d.name_format_id WHEN 0 THEN p.code WHEN 1 THEN p.name WHEN 2 THEN CAST(p.id AS nvarchar) WHEN 3 THEN CAST(p.id AS nvarchar) + ' - ' + p.name ELSE p.code + ' - ' + p.name END AS parent
    , t.name AS calc_type
    , factor
    , NULL AS level
FROM
    dbo25.parents f
    INNER JOIN dbo25.members m ON m.id = f.member_id
    INNER JOIN dbo25.members p ON p.id = f.parent_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
    INNER JOIN dbo25.calc_types t ON t.id = p.calc_type_id
    INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = m.company_id

UNION ALL
SELECT
    m.company_id
    , c.name AS company
    , d.id AS dimension_id
    , d.name AS dimension_name
    , 'hierarchies' AS table_name
    , f.member_id
    , f.parent_id
    , NULL AS calc_type_id
    , CASE d.name_format_id WHEN 0 THEN m.code WHEN 1 THEN m.name WHEN 2 THEN CAST(m.id AS nvarchar) WHEN 3 THEN CAST(m.id AS nvarchar) + ' - ' + m.name ELSE m.code + ' - ' + m.name END AS member
    , CASE d.name_format_id WHEN 0 THEN p.code WHEN 1 THEN p.name WHEN 2 THEN CAST(p.id AS nvarchar) WHEN 3 THEN CAST(p.id AS nvarchar) + ' - ' + p.name ELSE p.code + ' - ' + p.name END AS parent
    , NULL AS calc_type
    , NULL AS factor
    , f.level
FROM
    dbo25.hierarchies f
    INNER JOIN dbo25.members m ON m.id = f.member_id
    INNER JOIN dbo25.members p ON p.id = f.parent_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
    INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = m.company_id

UNION ALL
SELECT
    m.company_id
    , c.name AS company
    , d.id AS dimension_id
    , d.name AS dimension_name
    , 'factors' AS table_name
    , f.member_id
    , f.parent_id
    , f.calc_type_id
    , CASE d.name_format_id WHEN 0 THEN m.code WHEN 1 THEN m.name WHEN 2 THEN CAST(m.id AS nvarchar) WHEN 3 THEN CAST(m.id AS nvarchar) + ' - ' + m.name ELSE m.code + ' - ' + m.name END AS member
    , CASE d.name_format_id WHEN 0 THEN p.code WHEN 1 THEN p.name WHEN 2 THEN CAST(p.id AS nvarchar) WHEN 3 THEN CAST(p.id AS nvarchar) + ' - ' + p.name ELSE p.code + ' - ' + p.name END AS parent
    , t.name AS calc_type
    , factor
    , NULL AS level
FROM
    dbo25.factors f
    INNER JOIN dbo25.members m ON m.id = f.member_id
    INNER JOIN dbo25.members p ON p.id = f.parent_id
    INNER JOIN dbo25.calc_types t ON t.id = f.calc_type_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
    INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = m.company_id


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects Excel formats of user forms
-- =============================================

CREATE VIEW [xls25].[xl_formats]
AS

SELECT
    v.TABLE_SCHEMA
    , c.code + '_' + f.code AS TABLE_NAME
    , COALESCE(tf.TABLE_EXCEL_FORMAT_XML, t1.TABLE_EXCEL_FORMAT_XML, t2.TABLE_EXCEL_FORMAT_XML) AS TABLE_EXCEL_FORMAT_XML
FROM
    dbo25.forms f
    CROSS JOIN (VALUES ('xls25a'), ('xls25b')) v(TABLE_SCHEMA)
    INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    LEFT OUTER JOIN dbo25.formats tf ON tf.company_id = f.company_id AND tf.TABLE_SCHEMA = v.TABLE_SCHEMA AND tf.TABLE_NAME = f.code
    LEFT OUTER JOIN xls.formats t1 ON t1.TABLE_SCHEMA = v.TABLE_SCHEMA AND t1.TABLE_NAME = c.code + '_' + f.code
    LEFT OUTER JOIN xls.formats t2 ON t2.TABLE_SCHEMA = 'xls25' AND t2.TABLE_NAME = 'usp_run_form'
WHERE
    tf.TABLE_EXCEL_FORMAT_XML IS NOT NULL
    OR t1.TABLE_EXCEL_FORMAT_XML IS NOT NULL
    OR t2.TABLE_EXCEL_FORMAT_XML IS NOT NULL


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects the configuration of event handlers for user forms
-- =============================================

CREATE VIEW [xls25].[xl_handlers]
AS

SELECT
    'xls25a' AS TABLE_SCHEMA
    , c.code + '_' + f.code AS TABLE_NAME
    , h.COLUMN_NAME AS COLUMN_NAME
    , h.EVENT_NAME AS EVENT_NAME
    , h.HANDLER_SCHEMA AS HANDLER_SCHEMA
    , h.HANDLER_NAME AS HANDLER_NAME
    , h.HANDLER_TYPE AS HANDLER_TYPE
    , h.HANDLER_CODE AS HANDLER_CODE
    , h.TARGET_WORKSHEET AS TARGET_WORKSHEET
    , h.MENU_ORDER AS MENU_ORDER
    , h.EDIT_PARAMETERS AS EDIT_PARAMETERS
FROM
    dbo25.forms f
    INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    INNER JOIN xls.handlers h ON h.TABLE_SCHEMA = 'xls25' AND h.TABLE_NAME = 'usp_run_form'
WHERE
    h.EVENT_NAME IN ('Actions', 'DoubleClick', 'ContextMenu', 'Change', 'ProtectRows', 'DoNotSort', 'Format')

UNION ALL
SELECT
    'xls25a' AS TABLE_SCHEMA
    , c.code + '_' + f.code AS TABLE_NAME
    , 'form_p' + CAST(fd.parameter_index AS nvarchar(2)) AS COLUMN_NAME
    , 'ParameterValues' AS EVENT_NAME
    , 'xls25a' AS HANDLER_SCHEMA
    , 'xl_parameter_values_run_form_p' + CAST(fd.parameter_index AS nvarchar(2)) AS HANDLER_NAME
    , 'CODE' AS HANDLER_TYPE
    , 'EXEC xls25.xl_parameter_values_run_form_p' + CAST(fd.parameter_index AS nvarchar(2)) + ' ' + CAST(f.id AS nvarchar(128)) + ', @data_language' AS HANDLER_CODE
    , CAST(NULL AS nvarchar(128)) AS TARGET_WORKSHEET
    , CAST(NULL AS int) AS MENU_ORDER
    , CAST(NULL AS bit) AS EDIT_PARAMETERS
FROM
    dbo25.form_dimensions fd
    INNER JOIN dbo25.forms f ON f.id = fd.form_id
    INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = f.company_id AND d.id = fd.dimension_id
WHERE
    fd.parameter_index IS NOT NULL
    AND d.is_active = 1

UNION ALL
SELECT
    'xls25b' AS TABLE_SCHEMA
    , c.code + '_' + f.code AS TABLE_NAME
    , h.COLUMN_NAME AS COLUMN_NAME
    , h.EVENT_NAME AS EVENT_NAME
    , h.HANDLER_SCHEMA AS HANDLER_SCHEMA
    , h.HANDLER_NAME AS HANDLER_NAME
    , h.HANDLER_TYPE AS HANDLER_TYPE
    , h.HANDLER_CODE AS HANDLER_CODE
    , h.TARGET_WORKSHEET AS TARGET_WORKSHEET
    , h.MENU_ORDER AS MENU_ORDER
    , h.EDIT_PARAMETERS AS EDIT_PARAMETERS
FROM
    dbo25.forms f
    INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    INNER JOIN xls.handlers h ON h.TABLE_SCHEMA = 'xls25' AND h.TABLE_NAME = 'usp_run_offline_form'
WHERE
    h.EVENT_NAME IN ('Actions', 'DoubleClick', 'ContextMenu', 'Change', 'ProtectRows', 'DoNotSort', 'Format')

UNION ALL
SELECT
    'xls25b' AS TABLE_SCHEMA
    , c.code + '_' + f.code AS TABLE_NAME
    , 'form_p' + CAST(fd.parameter_index AS nvarchar(2)) AS COLUMN_NAME
    , 'ParameterValues' AS EVENT_NAME
    , 'xls25b' AS HANDLER_SCHEMA
    , 'xl_parameter_values_run_form_p' + CAST(fd.parameter_index AS nvarchar(2)) AS HANDLER_NAME
    , 'CODE' AS HANDLER_TYPE
    , 'EXEC xls25.xl_parameter_values_run_form_p' + CAST(fd.parameter_index AS nvarchar(2)) + ' ' + CAST(f.id AS nvarchar(128)) + ', @data_language' AS HANDLER_CODE
    , CAST(NULL AS nvarchar(128)) AS TARGET_WORKSHEET
    , CAST(NULL AS int) AS MENU_ORDER
    , CAST(NULL AS bit) AS EDIT_PARAMETERS
FROM
    dbo25.form_dimensions fd
    INNER JOIN dbo25.forms f ON f.id = fd.form_id
    INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = f.company_id AND d.id = fd.dimension_id
WHERE
    fd.parameter_index IS NOT NULL
    AND d.is_active = 1

UNION ALL
SELECT
    v.TABLE_SCHEMA
    , c.code + '_' + f.code AS TABLE_NAME
    , 'form_p' + CAST(fd.parameter_index AS nvarchar(2)) AS COLUMN_NAME
    , 'DataTypeInt' AS EVENT_NAME
    , CAST(NULL AS nvarchar) AS HANDLER_SCHEMA
    , CAST(NULL AS nvarchar) AS HANDLER_NAME
    , 'ATTRIBUTE' AS HANDLER_TYPE
    , CAST(NULL AS nvarchar(max)) AS HANDLER_CODE
    , CAST(NULL AS nvarchar(128)) AS TARGET_WORKSHEET
    , CAST(NULL AS int) AS MENU_ORDER
    , CAST(NULL AS bit) AS EDIT_PARAMETERS
FROM
    dbo25.form_dimensions fd
    INNER JOIN dbo25.forms f ON f.id = fd.form_id
    INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = f.company_id AND d.id = fd.dimension_id
    CROSS JOIN (VALUES ('xls25a'), ('xls25b')) v(TABLE_SCHEMA)
WHERE
    fd.parameter_index IS NOT NULL
    AND d.is_active = 1

UNION ALL
SELECT
    'xls25' AS TABLE_SCHEMA
    , 'usp_data_management' AS TABLE_NAME
    , m.code AS COLUMN_NAME
    , 'ContextMenu' AS EVENT_NAME
    , 'xls25' AS HANDLER_SCHEMA
    , 'Delete the period data for ' + m.code AS HANDLER_NAME
    , 'CODE' AS HANDLER_TYPE
    , 'EXEC xls25.xl_actions_delete_data @code, ''' + REPLACE(m.code, '''', '''''') + ''', @set_1_to_delete'  AS HANDLER_CODE
    , '_reload' AS TARGET_WORKSHEET
    , 11 AS MENU_ORDER
    , CAST(1 AS bit) AS EDIT_PARAMETERS
FROM
    dbo25.members m
    INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1
WHERE
    m.dimension_id = 3
    AND m.calc_type_id IN (1)       -- total
    AND m.id NOT IN (
        SELECT parent_id FROM dbo25.parents
    )

UNION ALL
SELECT
    'xls25' AS TABLE_SCHEMA
    , 'usp_data_management' AS TABLE_NAME
    , m.code AS COLUMN_NAME
    , 'ContextMenu' AS EVENT_NAME
    , 'xls25' AS HANDLER_SCHEMA
    , 'MenuSeparator20' AS HANDLER_NAME
    , 'MENUSEPARATOR' AS HANDLER_TYPE
    , CAST(NULL AS nvarchar(max)) AS HANDLER_CODE
    , CAST(NULL AS nvarchar(255)) AS TARGET_WORKSHEET
    , 20 AS MENU_ORDER
    , CAST(NULL AS bit) AS EDIT_PARAMETERS
FROM
    dbo25.members m
    INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1
WHERE
    m.dimension_id = 3
    AND m.calc_type_id IN (1)       -- total
    AND m.id NOT IN (
        SELECT parent_id FROM dbo25.parents
    )

UNION ALL
SELECT
    'xls25' AS TABLE_SCHEMA
    , 'usp_data_management' AS TABLE_NAME
    , t.code AS COLUMN_NAME
    , 'ContextMenu' AS EVENT_NAME
    , 'xls25' AS HANDLER_SCHEMA
    , 'Copy data to ' + t.code + ' from ' + s.code AS HANDLER_NAME
    , 'CODE' AS HANDLER_TYPE
    , 'EXEC xls25.xl_actions_copy_data @code, ''' + REPLACE(s.code, '''', '''''') + ''', ''' + REPLACE(t.code, '''', '''''') + ''', @set_1_to_copy'  AS HANDLER_CODE
    , '_reload' AS TARGET_WORKSHEET
    , 20 + ROW_NUMBER() OVER (PARTITION BY s.code ORDER BY t.code) AS MENU_ORDER
    , CAST(1 AS bit) AS EDIT_PARAMETERS
FROM
    dbo25.members t
    INNER JOIN dbo25.users u ON u.company_id = t.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.members s ON s.company_id = t.company_id
WHERE
    t.dimension_id = 3
    AND t.calc_type_id IN (1)       -- total
    AND t.id NOT IN (
        SELECT parent_id FROM dbo25.parents
    )
    AND s.dimension_id = 3
    AND s.calc_type_id IN (1)       -- total
    AND s.id NOT IN (
        SELECT parent_id FROM dbo25.parents
    )
    AND NOT s.id = t.id

UNION ALL
SELECT
    t.TABLE_SCHEMA
    , t.TABLE_NAME
    , CAST(NULL AS nvarchar(128)) AS COLUMN_NAME
    , 'Actions' AS EVENT_NAME
    , t.TABLE_SCHEMA AS HANDLER_SCHEMA
    , CASE WHEN v.menu_order = 90 THEN 'MenuSeparator90' ELSE 'Online Database Help - ' + t.TABLE_SCHEMA + '.' + t.TABLE_NAME END AS HANDLER_NAME
    , CASE WHEN v.menu_order = 90 THEN 'MENUSEPARATOR' ELSE 'HTTP' END AS HANDLER_TYPE
    , CASE WHEN v.menu_order = 90 THEN NULL ELSE CASE
        WHEN t.TABLE_TYPE = 'BASE TABLE' THEN REPLACE(t.URL, '.htm', '-tables.htm')
        WHEN t.TABLE_TYPE = 'VIEW' THEN REPLACE(t.URL, '.htm', '-views.htm')
        WHEN t.TABLE_TYPE = 'PROCEDURE' THEN REPLACE(t.URL, '.htm', '-procedures.htm')
        WHEN t.TABLE_TYPE = 'FUNCTION' THEN REPLACE(t.URL, '.htm', '-functions.htm')
        ELSE t.URL
        END + '#' + REPLACE(t.TABLE_SCHEMA, ' ', '') + '.' + REPLACE(t.TABLE_NAME, ' ', '') END HANDLER_CODE
    , CAST(NULL AS nvarchar) AS TARGET_WORKSHEET
    , v.menu_order AS MENU_ORDER
    , CAST(NULL AS bit) AS EDIT_PARAMETERS
FROM
    (
        SELECT
            t.TABLE_SCHEMA
            , t.TABLE_NAME
            , t.TABLE_TYPE
            , 'https://www.savetodb.com/help/planning-application.htm' AS URL
        FROM
            INFORMATION_SCHEMA.TABLES t
        WHERE
            t.TABLE_SCHEMA IN ('dbo25', 'xls25')
        UNION ALL
        SELECT
            t.ROUTINE_SCHEMA AS TABLE_SCHEMA
            , t.ROUTINE_NAME AS TABLE_NAME
            , t.ROUTINE_TYPE AS TABLE_TYPE
            , 'https://www.savetodb.com/help/planning-application.htm' AS URL
        FROM
            INFORMATION_SCHEMA.ROUTINES t
        WHERE
            t.ROUTINE_SCHEMA IN ('dbo25', 'xls25')
            AND t.ROUTINE_TYPE IN ('PROCEDURE')
            AND NOT t.ROUTINE_NAME LIKE 'xl_%'
            AND NOT t.ROUTINE_NAME LIKE 'usp_import_%'
            AND NOT t.ROUTINE_NAME LIKE 'usp_export_%'
            AND NOT t.ROUTINE_NAME LIKE '%_change'
            AND NOT t.ROUTINE_NAME LIKE '%_insert'
            AND NOT t.ROUTINE_NAME LIKE '%_update'
            AND NOT t.ROUTINE_NAME LIKE '%_delete'
    ) t
    CROSS JOIN (VALUES (90), (91)) v(menu_order)
WHERE
    t.URL IS NOT NULL

UNION ALL
SELECT
    p.SPECIFIC_SCHEMA AS TABLE_SCHEMA
    , p.SPECIFIC_NAME AS TABLE_NAME
    , t.COLUMN_NAME
    , 'DefaultValue' AS EVENT_NAME
    , CAST(NULL AS nvarchar) AS HANDLER_SCHEMA
    , CAST(NULL AS nvarchar) AS HANDLER_NAME
    , 'ATTRIBUTE' AS HANDLER_TYPE
    , t.HANDLER_CODE
    , CAST(NULL AS nvarchar) AS TARGET_WORKSHEET
    , CAST(NULL AS int) AS MENU_ORDER
    , CAST(NULL AS bit) AS EDIT_PARAMETERS
FROM
    (
        SELECT
            TOP 1
            'company_id' AS COLUMN_NAME
            , CAST(company_id AS nvarchar) AS HANDLER_CODE
        FROM
            dbo25.users u
            INNER JOIN dbo25.companies c ON c.id = u.company_id
        WHERE
            u.is_default = 1
            AND u.username = USER_NAME()
        ORDER BY
            c.sort_order
            , c.id
    ) t
    INNER JOIN (
        SELECT
            p.SPECIFIC_SCHEMA
            , p.SPECIFIC_NAME
            , SUBSTRING(p.PARAMETER_NAME, 2, 100) AS PARAMETER_NAME
        FROM
            INFORMATION_SCHEMA.PARAMETERS p
        WHERE
            p.SPECIFIC_SCHEMA = 'xls25'
            AND p.SPECIFIC_NAME LIKE 'usp_%'
            AND NOT RIGHT(p.SPECIFIC_NAME, 7) IN ('_insert', '_update', '_delete', '_change')
    ) p ON p.PARAMETER_NAME = t.COLUMN_NAME

UNION ALL
SELECT
    v.TABLE_SCHEMA
    , c.code + '_' + f.code AS TABLE_NAME
    , 'form_p' + CAST(fd.parameter_index AS nvarchar(2)) AS COLUMN_NAME
    , 'DefaultValue' AS EVENT_NAME
    , CAST(NULL AS nvarchar) AS HANDLER_SCHEMA
    , CAST(NULL AS nvarchar) AS HANDLER_NAME
    , 'ATTRIBUTE' AS HANDLER_TYPE
    , CAST(m.id AS nvarchar) AS HANDLER_CODE
    , CAST(NULL AS nvarchar) AS TARGET_WORKSHEET
    , CAST(NULL AS int) AS MENU_ORDER
    , CAST(NULL AS bit) AS EDIT_PARAMETERS
FROM
    dbo25.form_dimensions fd
    INNER JOIN dbo25.forms f ON f.id = fd.form_id
    INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = f.company_id AND d.id = fd.dimension_id
    INNER JOIN dbo25.members m ON m.company_id = f.company_id AND m.dimension_id = fd.dimension_id
    CROSS JOIN (VALUES ('xls25a'), ('xls25b')) v(TABLE_SCHEMA)
WHERE
    fd.parameter_index IS NOT NULL
    AND d.is_active = 1
    AND (
        m.dimension_id = 1 AND m.code = 'Accounts'
        OR m.dimension_id = 2 AND m.code = CAST(YEAR(GETDATE()) AS nvarchar)
        OR m.dimension_id = 3 AND m.name = 'Budget'
        OR m.dimension_id = 4 AND m.name = 'All'
        OR m.dimension_id = 5 AND m.name = 'All'
        OR m.dimension_id = 6 AND m.name = 'All'
        OR m.dimension_id = 7 AND m.name = 'All'
        )

UNION ALL
SELECT
    'xls25' AS TABLE_SCHEMA
    , p.TABLE_NAME
    , 'form_id' AS COLUMN_NAME
    , 'DefaultValue' AS EVENT_NAME
    , CAST(NULL AS nvarchar) AS HANDLER_SCHEMA
    , CAST(NULL AS nvarchar) AS HANDLER_NAME
    , 'ATTRIBUTE' AS HANDLER_TYPE
    , CAST(f.id AS nvarchar) AS HANDLER_CODE
    , CAST(NULL AS nvarchar) AS TARGET_WORKSHEET
    , CAST(NULL AS int) AS MENU_ORDER
    , CAST(NULL AS bit) AS EDIT_PARAMETERS
FROM
    dbo25.forms f
    INNER JOIN (
        SELECT TOP 1
            u.company_id
        FROM
            dbo25.users u
            INNER JOIN dbo25.companies c ON c.id = u.company_id
        WHERE
            u.is_default = 1
            AND u.username = USER_NAME()
        ORDER BY
            c.sort_order
            , c.id
        ) u ON u.company_id = f.company_id
    CROSS JOIN (VALUES ('usp_run_form'), ('usp_run_offline_form')) p(TABLE_NAME)
WHERE
    f.is_active = 1
    AND f.name = 'Accounts*Times'

UNION ALL
SELECT
    'xls25' AS TABLE_SCHEMA
    , p.TABLE_NAME
    , 'form_p' + CAST(fd.parameter_index AS nvarchar(2)) AS COLUMN_NAME
    , 'DefaultValue' AS EVENT_NAME
    , CAST(NULL AS nvarchar) AS HANDLER_SCHEMA
    , CAST(NULL AS nvarchar) AS HANDLER_NAME
    , 'ATTRIBUTE' AS HANDLER_TYPE
    , CAST(m.id AS nvarchar) AS HANDLER_CODE
    , CAST(NULL AS nvarchar) AS TARGET_WORKSHEET
    , CAST(NULL AS int) AS MENU_ORDER
    , CAST(NULL AS bit) AS EDIT_PARAMETERS
FROM
    dbo25.form_dimensions fd
    INNER JOIN dbo25.forms f ON f.id = fd.form_id
    INNER JOIN (
        SELECT TOP 1
            u.company_id
        FROM
            dbo25.users u
            INNER JOIN dbo25.companies c ON c.id = u.company_id
        WHERE
            u.is_default = 1
            AND u.username = USER_NAME()
        ORDER BY
            c.sort_order
            , c.id
        ) u ON u.company_id = f.company_id
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = f.company_id AND d.id = fd.dimension_id
    INNER JOIN dbo25.members m ON m.company_id = f.company_id AND m.dimension_id = fd.dimension_id
    CROSS JOIN (VALUES ('usp_run_form'), ('usp_run_offline_form')) p(TABLE_NAME)
WHERE
    fd.parameter_index IS NOT NULL
    AND d.is_active = 1
    AND f.is_active = 1
    AND f.name = 'Accounts*Times'
    AND (
        m.dimension_id = 1 AND m.code = 'Accounts'
        OR m.dimension_id = 2 AND m.code = CAST(YEAR(GETDATE()) AS nvarchar)
        OR m.dimension_id = 3 AND m.name = 'Budget'
        OR m.dimension_id = 4 AND m.name = 'All'
        OR m.dimension_id = 5 AND m.name = 'All'
        OR m.dimension_id = 6 AND m.name = 'All'
        OR m.dimension_id = 7 AND m.name = 'All'
        )


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects dynamic forms
-- =============================================

CREATE VIEW [xls25].[xl_query_list_offline_forms]
AS

SELECT
    TOP 100 PERCENT
    'xls25b' AS TABLE_SCHEMA
    , c.code + '_' + t.code AS TABLE_NAME
    , 'CODE' AS TABLE_TYPE
    , 'EXEC xls25.usp_run_offline_form'
        + ' ' + CAST(t.company_id AS nvarchar)
        + ', ' + CAST(t.id AS nvarchar)
        + COALESCE((
            SELECT
                ', ' + CASE WHEN fd.parameter_index IS NULL THEN 'NULL' ELSE '@form_p' + CAST(d.id AS nvarchar) END
            FROM
                dbo25.dimension_properties d
                LEFT OUTER JOIN dbo25.form_dimensions fd ON fd.form_id = t.id AND fd.parameter_index = d.id
                LEFT OUTER JOIN dbo25.dimension_properties d2 ON d2.company_id = d.company_id AND d2.id = fd.dimension_id
            WHERE
                d.company_id = t.company_id
                AND COALESCE(d2.is_active, 1) = 1
                AND d.id <= (SELECT MAX(parameter_index) FROM dbo25.form_dimensions WHERE form_id = t.id)
            ORDER BY
                d.id
            FOR XML PATH(''), TYPE
            ).value('.', 'nvarchar(MAX)'), '')
            + ', @data_language= @data_language'
        AS TABLE_CODE
    , 'xls25.usp_run_offline_form_insert' AS INSERT_PROCEDURE
    , 'xls25.usp_run_offline_form_update' AS UPDATE_PROCEDURE
    , 'xls25.usp_run_offline_form_delete' AS DELETE_PROCEDURE
    , 'PROCEDURE' AS PROCEDURE_TYPE
FROM
    dbo25.forms t
    INNER JOIN dbo25.users u ON u.company_id = t.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = t.company_id
    LEFT OUTER JOIN dbo25.form_permissions fp ON fp.form_id = t.id AND fp.username = USER_NAME()
WHERE
    t.is_active = 1
    AND (fp.select_permission = 1 OR COALESCE(IS_MEMBER('planning_app_users'), 0) = 0)
ORDER BY
    t.sort_order
    , t.code


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects dynamic forms
-- =============================================

CREATE VIEW [xls25].[xl_query_list_online_forms]
AS

SELECT
    TOP 100 PERCENT
    'xls25a' AS TABLE_SCHEMA
    , c.code + '_' + t.code AS TABLE_NAME
    , 'CODE' AS TABLE_TYPE
    , 'EXEC xls25.usp_run_form'
        + ' ' + CAST(t.company_id AS nvarchar)
        + ', ' + CAST(t.id AS nvarchar)
        + COALESCE((
            SELECT
                ', ' + CASE WHEN fd.parameter_index IS NULL THEN 'NULL' ELSE '@form_p' + CAST(d.id AS nvarchar) END
            FROM
                dbo25.dimension_properties d
                LEFT OUTER JOIN dbo25.form_dimensions fd ON fd.form_id = t.id AND fd.parameter_index = d.id
                LEFT OUTER JOIN dbo25.dimension_properties d2 ON d2.company_id = d.company_id AND d2.id = fd.dimension_id
            WHERE
                d.company_id = t.company_id
                AND COALESCE(d2.is_active, 1) = 1
                AND d.id <= (SELECT MAX(parameter_index) FROM dbo25.form_dimensions WHERE form_id = t.id)
            ORDER BY
                d.id
            FOR XML PATH(''), TYPE
            ).value('.', 'nvarchar(MAX)'), '')
            + ', @data_language= @data_language'
        AS TABLE_CODE
    , CAST(NULL AS nvarchar(max)) AS INSERT_PROCEDURE
    , CAST(NULL AS nvarchar(max)) AS UPDATE_PROCEDURE
    , CAST(NULL AS nvarchar(max)) AS DELETE_PROCEDURE
    , CAST(NULL AS nvarchar(128)) AS PROCEDURE_TYPE
FROM
    dbo25.forms t
    INNER JOIN dbo25.users u ON u.company_id = t.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = t.company_id
    LEFT OUTER JOIN dbo25.form_permissions fp ON fp.form_id = t.id AND fp.username = USER_NAME()
WHERE
    t.is_active = 1
    AND (fp.select_permission = 1 OR COALESCE(IS_MEMBER('planning_app_users'), 0) = 0)
ORDER BY
    t.sort_order
    , t.code


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects translations for user forms
-- =============================================

CREATE VIEW [xls25].[xl_translations]
AS

WITH cte AS (
    SELECT
        c.id AS company_id
        , c.default_language AS language
    FROM
        dbo25.companies c
        INNER JOIN dbo25.users u ON u.company_id = c.id AND u.username = USER_NAME() AND u.is_active = 1
    WHERE
        c.default_language IS NOT NULL
    UNION
    SELECT
        t.company_id
        , t.language
    FROM
        dbo25.translations t
        INNER JOIN dbo25.users u ON u.company_id = t.company_id AND u.username = USER_NAME() AND u.is_active = 1
)

SELECT
    v.TABLE_SCHEMA
    , c.code + '_' + f.code AS TABLE_NAME
    , CAST(NULL AS nvarchar(128)) AS COLUMN_NAME
    , COALESCE(t.language, 'en') AS LANGUAGE_NAME
    , CASE WHEN ff.count > 1 THEN c.code + ': ' ELSE '' END + COALESCE(t.name, f.name, f.code) AS TRANSLATED_NAME
    , CAST(NULL AS nvarchar(max)) AS TRANSLATED_DESC
    , CAST(NULL AS nvarchar(max)) AS TRANSLATED_COMMENT
FROM
    dbo25.forms f
    INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    INNER JOIN cte ON cte.company_id = c.id
    LEFT OUTER JOIN dbo25.translations t ON t.company_id = f.company_id AND t.table_id = 7 AND t.member_id = f.id AND t.language = cte.language
    LEFT OUTER JOIN dbo25.form_permissions fp ON fp.form_id = f.id AND fp.username = USER_NAME()
    INNER JOIN (
        SELECT
            u.username
            , COUNT(*) AS count
        FROM
            dbo25.users u
        WHERE
            u.username = USER_NAME() AND u.is_active = 1
        GROUP BY
            u.username
        ) ff ON ff.username = u.username
    CROSS JOIN (VALUES ('xls25a'), ('xls25b')) v(TABLE_SCHEMA)

WHERE
    f.is_active = 1
    AND (fp.select_permission = 1 OR COALESCE(IS_MEMBER('planning_app_users'), 0) = 0)

UNION ALL
SELECT
    v.TABLE_SCHEMA
    , c.code + '_' + f.code AS TABLE_NAME
    , 'form_p' + CAST(fd.parameter_index AS nvarchar) AS COLUMN_NAME
    , COALESCE(t.language, 'en') AS LANGUAGE_NAME
    , COALESCE(t.name, d.parameter_name, d.name) AS TRANSLATED_NAME
    , CAST(NULL AS nvarchar(max)) AS TRANSLATED_DESC
    , CAST(NULL AS nvarchar(max)) AS TRANSLATED_COMMENT
FROM
    dbo25.form_dimensions fd
    INNER JOIN dbo25.forms f ON f.id = fd.form_id
    INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = f.company_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = f.company_id AND d.id = fd.dimension_id
    INNER JOIN cte ON cte.company_id = c.id
    LEFT OUTER JOIN dbo25.translations t ON t.company_id = f.company_id AND t.table_id = 6 AND t.member_id = d.id AND t.language = cte.language
    LEFT OUTER JOIN dbo25.form_permissions fp ON fp.form_id = f.id AND fp.username = USER_NAME()
    CROSS JOIN (VALUES ('xls25a'), ('xls25b')) v(TABLE_SCHEMA)
WHERE
    f.is_active = 1
    AND (fp.select_permission = 1 OR COALESCE(IS_MEMBER('planning_app_users'), 0) = 0)
    AND fd.parameter_index IS NOT NULL
    AND d.is_active = 1


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert companies
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_companies]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER (ORDER BY c.code) AS r
        , 'EXEC dbo25.usp_import_companies ' +
               + COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(c.name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(c.default_language, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(c.sort_order AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.companies c
        INNER JOIN dbo25.users u ON u.company_id = c.id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
    WHERE
        c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert currency rates
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_currency_rates]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER (ORDER BY f.code, q.code, g.code, t.code) AS r
        , 'EXEC dbo25.usp_import_rates ' +
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(q.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(t.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(g.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CASE WHEN CAST(CAST(r.currency_rate AS nvarchar) AS float) = r.currency_rate THEN CAST(r.currency_rate AS nvarchar) ELSE STR(r.currency_rate, 20, 10) END, 'NULL')
        + ';' AS command
    FROM
        dbo25.currency_rates r
        INNER JOIN dbo25.units f ON f.id = r.base_currency_id
        INNER JOIN dbo25.units q ON q.id = r.quote_currency_id
        INNER JOIN dbo25.members t ON t.id = r.time_id
        INNER JOIN dbo25.members g ON g.id = r.category_id
        INNER JOIN dbo25.users u ON u.company_id = t.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = t.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert dimension calc types
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_dimension_calc_types]
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        2 AS part
        , ROW_NUMBER() OVER (ORDER BY d.calc_type_id, d.dimension_id) AS r
        , 'EXEC dbo25.usp_import_dimension_calc_types ' +
               + CAST(d.dimension_id AS nvarchar)
        + ', ' + COALESCE(CAST(d.calc_type_id AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_active AS tinyint) AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.dimension_calc_types d
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to update dimension members
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_dimension_members]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        3 AS part
        , ROW_NUMBER() OVER (ORDER BY d.id) AS r
        , 'EXEC dbo25.usp_import_dimension_members '
               + COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + CAST(d.id AS nvarchar)
        + ', ' + COALESCE('N''' + REPLACE(m.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.code, '''', '''''') + '''', 'NULL')
        + ';' AS command
    FROM
        dbo25.dimension_properties d
        LEFT OUTER JOIN dbo25.members m ON m.id = d.root_member_id
        LEFT OUTER JOIN dbo25.members f ON f.id = d.default_member_id
        INNER JOIN dbo25.users u ON u.company_id = d.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = d.company_id AND c.code = COALESCE(@company, c.code)
    WHERE
        m.id IS NOT NULL
        OR f.id IS NOT NULL
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert dimension properties
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_dimension_properties]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER (ORDER BY d.id) AS r
        , 'EXEC dbo25.usp_import_dimension_properties ' +
               + COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + CAST(d.id AS nvarchar)
        + ', ' + COALESCE('N''' + REPLACE(d.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(d.name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(d.parameter_name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(d.sort_order AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(d.join_order AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(d.name_format_id AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_protected AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.default_select_permission AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.default_update_permission AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_active AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(d.external_id AS nvarchar), 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(d.external_code, '''', '''''') + '''', 'NULL')
        + ';' AS command
    FROM
        dbo25.dimension_properties d
        INNER JOIN dbo25.users u ON u.company_id = d.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = d.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert dimensions
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_dimensions]
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER (ORDER BY d.id) AS r
        , 'EXEC dbo25.usp_import_dimensions ' +
               + CAST(id AS nvarchar)
        + ', ' + COALESCE('N''' + REPLACE(d.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(d.name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(d.parameter_name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(d.sort_order AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(d.join_order AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(d.name_format_id AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_protected AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.default_select_permission AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.default_update_permission AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_active AS tinyint) AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.dimensions d
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to import source cube data
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_facts]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    'EXEC dbo25.usp_import_facts '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(CAST(f.value AS money) AS nchar(16)), 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m1.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m2.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m3.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m4.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m5.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m6.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m7.code, '''', '''''') + '''', 'NULL')
        + ';' AS command
FROM
    dbo25.facts f

    INNER JOIN dbo25.members m1 ON m1.id = f.id1
    INNER JOIN dbo25.members m2 ON m2.id = f.id2
    INNER JOIN dbo25.members m3 ON m3.id = f.id3
    INNER JOIN dbo25.members m4 ON m4.id = f.id4
    INNER JOIN dbo25.members m5 ON m5.id = f.id5
    INNER JOIN dbo25.members m6 ON m6.id = f.id6
    INNER JOIN dbo25.members m7 ON m7.id = f.id7
    INNER JOIN dbo25.users u ON u.company_id = m1.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
    INNER JOIN dbo25.companies c ON c.id = m1.company_id AND c.code = COALESCE(@company, c.code)
WHERE
    f.value IS NOT NULL
ORDER BY
    m3.sort_order
    , m3.code
    , m1.sort_order
    , m1.code
    , m2.sort_order
    , m2.code
    , m4.sort_order
    , m4.code
    , m5.sort_order
    , m5.code
    , m6.sort_order
    , m6.code
    , m7.sort_order
    , m7.code

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert form dimensions
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_form_dimensions]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        2 AS part
        , ROW_NUMBER() OVER (ORDER BY f.code, fd.dimension_id) AS r
        , 'EXEC dbo25.usp_import_form_dimensions '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(fd.dimension_id AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(fd.axis_type_id AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(fd.parameter_index AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(fd.header_order AS nvarchar), 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(fd.parameter_start_level AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(fd.parameter_end_level AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(fd.form_start_level AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(fd.form_end_level AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.form_dimensions fd
        INNER JOIN dbo25.forms f ON f.id = fd.form_id
        LEFT OUTER JOIN dbo25.members m ON m.id = fd.root_member_id
        INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = f.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert form permissions
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_form_permissions]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        4 AS part
        , ROW_NUMBER() OVER (ORDER BY fp.username, f.sort_order, f.code) AS r
        , 'EXEC dbo25.usp_import_form_permissions '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(fp.username, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(fp.select_permission AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.form_permissions fp
        INNER JOIN dbo25.forms f ON f.id = fp.form_id
        INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = f.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert form rows
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_form_rows]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        3 AS part
        , ROW_NUMBER() OVER (ORDER BY s.id, r.sort_order, m.sort_order, r.id) AS r
        , 'EXEC dbo25.usp_import_form_rows '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(s.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(r.sort_order AS nvarchar), 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(r.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(r.name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(r.decimal_places AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(r.is_percent AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(r.row_color AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(r.row_bold AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(r.row_indent AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.form_rows r
        INNER JOIN dbo25.members s ON s.id = r.rowset_id
        LEFT OUTER JOIN dbo25.members m ON m.id = r.member_id
        INNER JOIN dbo25.users u ON u.company_id = s.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = s.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert form subtotals
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_form_subtotals]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        5 AS part
        , ROW_NUMBER() OVER (ORDER BY f.code, fs.id) AS r
        , 'EXEC dbo25.usp_import_form_subtotals '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(fs.show AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(fs.dimension_id1 AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(fs.dimension_id2 AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(fs.dimension_id3 AS nvarchar), 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m1.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m2.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m3.code, '''', '''''') + '''', 'NULL')
        + ';' AS command
    FROM
        dbo25.form_subtotals fs
        INNER JOIN dbo25.forms f ON f.id = fs.form_id
        LEFT OUTER JOIN dbo25.members m1 ON m1.id = fs.member_id1
        LEFT OUTER JOIN dbo25.members m2 ON m2.id = fs.member_id2
        LEFT OUTER JOIN dbo25.members m3 ON m3.id = fs.member_id3
        INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = f.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert formats
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_formats]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER (ORDER BY c.code, TABLE_SCHEMA, TABLE_NAME) AS r
        , 'EXEC dbo25.usp_import_formats '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.TABLE_SCHEMA, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.TABLE_NAME, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(CAST(f.TABLE_EXCEL_FORMAT_XML AS nvarchar(max)), '''', '''''') + '''', 'NULL')
        + ';' AS command
    FROM
        dbo25.formats f
        INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = f.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert forms
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_forms]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER (ORDER BY f.sort_order) AS r
        , 'EXEC dbo25.usp_import_forms '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(f.sort_order AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(f.is_active AS tinyint) AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.forms f
        INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = f.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert parents
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_member_parents]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        3 AS part
        , ROW_NUMBER() OVER (ORDER BY m.dimension_id, m.sort_order, m.code, p.sort_order, p.code) AS r
        , 'EXEC dbo25.usp_import_member_parents '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(p.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CASE WHEN CAST(CAST(mp.factor AS nvarchar) AS float) = mp.factor THEN CAST(mp.factor AS nvarchar) ELSE STR(mp.factor, 15, 15) END, 'NULL')
        + ';' AS command
    FROM
        dbo25.parents mp
        INNER JOIN dbo25.members m ON m.id = mp.member_id
        INNER JOIN dbo25.members p ON p.id = mp.parent_id
        INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = m.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert member permissions
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_member_permissions]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        5 AS part
        , ROW_NUMBER() OVER (ORDER BY mp.username, m.dimension_id, m.sort_order, m.code) AS r
        , 'EXEC dbo25.usp_import_member_permissions '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(mp.username, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(mp.select_permission AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(mp.update_permission AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.member_permissions mp
        INNER JOIN dbo25.members m ON m.id = mp.member_id
        INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = m.company_id AND c.code = COALESCE(@company, c.code)
    WHERE
        mp.is_inherited = 0
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert member properties
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_member_properties]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        2 AS part
        , ROW_NUMBER() OVER (ORDER BY m.dimension_id, m.sort_order, m.code) AS r
        , 'EXEC dbo25.usp_import_member_properties '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(p.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(s.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(t.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(n.code, '''', '''''') + '''', 'NULL')
        + ';' AS command
    FROM
        dbo25.members m
        LEFT OUTER JOIN dbo25.members p ON p.id = m.previous_period_id
        LEFT OUTER JOIN dbo25.members s ON s.id = m.same_period_id
        LEFT OUTER JOIN dbo25.tax_rates t ON t.id = m.tax_rate_id
        LEFT OUTER JOIN dbo25.units n ON n.id = m.unit_id
        INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = m.company_id AND c.code = COALESCE(@company, c.code)
    WHERE
        m.previous_period_id IS NOT NULL
        OR m.same_period_id IS NOT NULL
        OR t.id IS NOT NULL
        OR u.id IS NOT NULL
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert member relations
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_member_relations]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        4 AS part
        , ROW_NUMBER() OVER (ORDER BY m.dimension_id, p.sort_order, p.code, m.sort_order, m.code) AS r
        , 'EXEC dbo25.usp_import_member_relations '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(p.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(mp.select_permission AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(mp.update_permission AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.member_relations mp
        INNER JOIN dbo25.members m ON m.id = mp.member_id
        INNER JOIN dbo25.members p ON p.id = mp.partner_id
        INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = m.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert members
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_members]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        -- , ROW_NUMBER() OVER (ORDER BY m.dimension_id, m.sort_order, m.id) AS r
        , ROW_NUMBER() OVER (ORDER BY m.id) AS r
        , 'EXEC dbo25.usp_import_members ' +
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(m.dimension_id AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(m.sort_order AS nvarchar), 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(m.calc_type_id AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(m.decimal_places AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(m.is_percent AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(m.row_color AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(m.row_bold AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(m.row_indent AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(m.show_line AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(m.show_line_before AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(m.show_line_after AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(m.is_translatable AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(m.is_active AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(m.external_id AS nvarchar), 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.external_code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.excel_formula, '''', '''''') + '''', 'NULL')
        + ';' AS command
    FROM
        dbo25.members m
        INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = m.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to import source cube data
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_strings]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    'EXEC dbo25.usp_import_strings '
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(f.value, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m1.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m2.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m3.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m4.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m5.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m6.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m7.code, '''', '''''') + '''', 'NULL')
        + ';' AS command
FROM
    dbo25.strings f

    INNER JOIN dbo25.members m1 ON m1.id = f.id1
    INNER JOIN dbo25.members m2 ON m2.id = f.id2
    INNER JOIN dbo25.members m3 ON m3.id = f.id3
    INNER JOIN dbo25.members m4 ON m4.id = f.id4
    INNER JOIN dbo25.members m5 ON m5.id = f.id5
    INNER JOIN dbo25.members m6 ON m6.id = f.id6
    INNER JOIN dbo25.members m7 ON m7.id = f.id7
    INNER JOIN dbo25.users u ON u.company_id = m1.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
    INNER JOIN dbo25.companies c ON c.id = m1.company_id AND c.code = COALESCE(@company, c.code)
WHERE
    f.value IS NOT NULL
ORDER BY
    m3.sort_order
    , m3.code
    , m1.sort_order
    , m1.code
    , m2.sort_order
    , m2.code
    , m4.sort_order
    , m4.code
    , m5.sort_order
    , m5.code
    , m6.sort_order
    , m6.code
    , m7.sort_order
    , m7.code

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert tax_rates
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_tax_rates]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER (ORDER BY d.id) AS r
        , 'EXEC dbo25.usp_import_tax_rates ' +
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(d.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(d.name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(d.sort_order AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_translatable AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_active AS tinyint) AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.tax_rates d
        INNER JOIN dbo25.users u ON u.company_id = d.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = d.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert translations
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_translations]
    @company nvarchar(50) = NULL
    , @language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int
DECLARE @company_code nvarchar(50)

SELECT TOP 1 @company_id = c.id, @company_code = c.code FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

;WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER(ORDER BY t.language, m.sort_order, m.code) AS r
        , 'EXEC dbo25.usp_import_translations '
        + 'N''' + REPLACE(@company_code, '''', '''''') + ''''
        + ', N''' + REPLACE(tt.code, '''', '''''') + ''''
        + ', N''' + REPLACE(m.code, '''', '''''') + ''''
        + ', N''' + REPLACE(t.language, '''', '''''') + ''''
        + ', N''' + REPLACE(t.name, '''', '''''') + ''''
        + ';' AS command
    FROM
        (
            SELECT 2 AS sort_order, t.id AS company_id, 2 AS table_id, t.id AS member_id, t.code FROM dbo25.companies t
            UNION ALL SELECT 3 AS sort_order, t.company_id, 3 AS table_id, t.id AS member_id, t.code FROM dbo25.units t
            UNION ALL SELECT 4 AS sort_order, t.company_id, 4 AS table_id, t.id AS member_id, t.code FROM dbo25.tax_rates t
            UNION ALL SELECT 5 AS sort_order, t.company_id, 5 AS table_id, t.id AS member_id, t.code FROM dbo25.dimension_properties t
            UNION ALL SELECT 6 AS sort_order, t.company_id, 6 AS table_id, t.id AS member_id, t.parameter_name FROM dbo25.dimension_properties t
            UNION ALL SELECT 7 AS sort_order, t.company_id, 7 AS table_id, t.id AS member_id, t.code FROM dbo25.forms t
            UNION ALL SELECT 8 AS sort_order, m.company_id, 8 AS table_id, MIN(t.id) AS member_id, t.name FROM dbo25.form_rows t INNER JOIN dbo25.members m ON m.id = t.rowset_id WHERE t.name IS NOT NULL GROUP BY m.company_id, t.name
            UNION ALL SELECT 9 AS sort_order, t.company_id, 1 AS table_id, t.id AS member_id, t.code FROM dbo25.members t
        ) m
        INNER JOIN dbo25.translations t ON t.company_id = m.company_id AND t.table_id = m.table_id AND t.member_id = m.member_id
        INNER JOIN dbo25.translated_tables tt ON tt.id = m.table_id
        INNER JOIN dbo25.users u ON u.company_id = m.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = m.company_id AND c.code = COALESCE(@company, c.code)
    WHERE
        t.language = COALESCE(@language, t.language)
        AND t.name IS NOT NULL
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert units
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_units]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER (ORDER BY d.id) AS r
        , 'EXEC dbo25.usp_import_units ' +
        +        COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(d.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(d.name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(d.sort_order AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(d.factor AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_currency AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_functional_currency AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_translatable AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(d.is_active AS tinyint) AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.units d
        INNER JOIN dbo25.users u ON u.company_id = d.company_id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.companies c ON c.id = d.company_id AND c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects SQL commands to insert users
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_users]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON;

WITH cte(part, r, command) AS (
    SELECT
        1 AS part
        , ROW_NUMBER() OVER (ORDER BY c.code) AS r
        , 'EXEC dbo25.usp_import_users ' +
               + COALESCE('N''' + REPLACE(c.code, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.username, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE('N''' + REPLACE(m.name, '''', '''''') + '''', 'NULL')
        + ', ' + COALESCE(CAST(CAST(m.is_admin AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(m.is_default AS tinyint) AS nvarchar), 'NULL')
        + ', ' + COALESCE(CAST(CAST(m.is_active AS tinyint) AS nvarchar), 'NULL')
        + ';' AS command
    FROM
        dbo25.companies c
        INNER JOIN dbo25.users u ON u.company_id = c.id AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1
        INNER JOIN dbo25.users m ON m.company_id = c.id
    WHERE
        c.code = COALESCE(@company, c.code)
)
SELECT
    command
FROM
    cte
ORDER BY
    part
    , r

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Executes all export procedures
-- =============================================

CREATE PROCEDURE [dbo25].[usp_export_all]
    @company nvarchar(50) = NULL
    , @print_go bit = 1
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @goLine nvarchar(2) = CASE WHEN @print_go = 1 THEN 'GO' ELSE '' END

DECLARE @t TABLE (id int IDENTITY(1,1), command nvarchar(max))

INSERT INTO @t (command) VALUES ('-- Uncomment the line below to clear all data first'), ('-- EXEC dbo25.usp_import_clear_all_data ''' + REPLACE(@company, '''', '''''') + ''', ''yes'''), (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_companies @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_users @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_dimension_properties @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_tax_rates @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_units @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_members @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_member_properties @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_member_parents @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_member_relations @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_member_permissions @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_dimension_members @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_currency_rates @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_forms @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_form_dimensions @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_form_rows @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_form_permissions @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_form_subtotals @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_facts @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_strings @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_formats @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) EXEC dbo25.usp_export_translations @company
INSERT INTO @t (command) VALUES (@goLine), ('')

INSERT INTO @t (command) VALUES ('EXEC dbo25.usp_import_finish ''' + REPLACE(@company, '''', '''''') + ''''), (@goLine), ('')

SELECT command FROM @t ORDER BY id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Clears all user's data before importing
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_clear_all_data]
    @company nvarchar(50)
    , @confirm nvarchar(50)
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @confirm IS NULL OR NOT @confirm = 'yes'
    RETURN

DECLARE @message nvarchar(max)

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Incorrect company_id', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

DELETE dbo25.facts                FROM dbo25.facts f INNER JOIN dbo25.members m ON m.id = f.id1 WHERE m.company_id = @company_id
DELETE dbo25.strings            FROM dbo25.strings f INNER JOIN dbo25.members m ON m.id = f.id1 WHERE m.company_id = @company_id

DELETE dbo25.form_permissions    FROM dbo25.form_permissions fp INNER JOIN dbo25.forms f ON f.id = fp.form_id WHERE f.company_id = @company_id
DELETE dbo25.form_rows            FROM dbo25.form_rows fr INNER JOIN dbo25.members m ON m.id = fr.rowset_id WHERE m.company_id = @company_id
DELETE dbo25.form_dimensions    FROM dbo25.form_dimensions fd INNER JOIN dbo25.forms f ON f.id = fd.form_id WHERE f.company_id = @company_id
DELETE dbo25.forms                WHERE company_id = @company_id

DELETE dbo25.member_permissions FROM dbo25.member_permissions mp INNER JOIN dbo25.members m ON m.id = mp.member_id WHERE m.company_id = @company_id
DELETE dbo25.member_relations    FROM dbo25.member_relations   mr INNER JOIN dbo25.members m ON m.id = mr.member_id WHERE m.company_id = @company_id

UPDATE dbo25.dimension_properties SET root_member_id = NULL, default_member_id = NULL WHERE company_id = @company_id

DELETE dbo25.currency_rates        FROM dbo25.currency_rates f INNER JOIN dbo25.members m ON m.id = f.time_id WHERE m.company_id = @company_id

DELETE dbo25.factors            FROM dbo25.factors f INNER JOIN dbo25.members m ON m.id = f.member_id WHERE m.company_id = @company_id
DELETE dbo25.hierarchies        FROM dbo25.hierarchies h INNER JOIN dbo25.members m ON m.id = h.member_id WHERE m.company_id = @company_id
DELETE dbo25.parents            FROM dbo25.parents p INNER JOIN dbo25.members m ON m.id = p.member_id WHERE m.company_id = @company_id
DELETE dbo25.members            WHERE company_id = @company_id

DELETE FROM dbo25.tax_rates        WHERE company_id = @company_id
DELETE FROM dbo25.units            WHERE company_id = @company_id

DECLARE @id int

SET @id = COALESCE((SELECT MAX(id) FROM dbo25.facts), 0);

DBCC CHECKIDENT ('dbo25.facts', RESEED, @id) WITH NO_INFOMSGS;

SET @id = COALESCE((SELECT MAX(id) FROM dbo25.strings), 0);

DBCC CHECKIDENT ('dbo25.strings', RESEED, @id) WITH NO_INFOMSGS;

SET @id = COALESCE((SELECT MAX(id) FROM dbo25.form_rows), 0);

DBCC CHECKIDENT ('dbo25.form_rows', RESEED, @id) WITH NO_INFOMSGS;

SET @id = COALESCE((SELECT MAX(id) FROM dbo25.forms), 0);

DBCC CHECKIDENT ('dbo25.forms', RESEED, @id) WITH NO_INFOMSGS;

SET @id = COALESCE((SELECT MAX(id) FROM dbo25.members), 0);

DBCC CHECKIDENT ('dbo25.members', RESEED, @id) WITH NO_INFOMSGS;

SET @id = COALESCE((SELECT MAX(id) FROM dbo25.tax_rates), 0);

DBCC CHECKIDENT ('dbo25.tax_rates', RESEED, @id) WITH NO_INFOMSGS;

SET @id = COALESCE((SELECT MAX(id) FROM dbo25.units), 0);

DBCC CHECKIDENT ('dbo25.units', RESEED, @id) WITH NO_INFOMSGS;

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports companies
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_companies]
    @code nvarchar(50)
    , @name nvarchar(100)
    , @default_language varchar(10) = NULL
    , @sort_order int = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int = (SELECT id FROM dbo25.companies WHERE code = @code)

IF @company_id IS NOT NULL
    BEGIN

    IF NOT EXISTS (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1 AND is_admin = 1)
        RETURN

    UPDATE dbo25.companies
    SET
        name = @name
        , default_language = COALESCE(@default_language, 'en')
        , sort_order = @sort_order
    WHERE
        code = @code

    END
ELSE
    BEGIN
    INSERT INTO dbo25.companies
        (code, name, default_language, sort_order)
    VALUES
        (@code, @name, @default_language, @sort_order)

    SET @company_id = SCOPE_IDENTITY()

    INSERT INTO dbo25.users
        (company_id, username, name, is_admin, is_active)
    VALUES
        (@company_id, USER_NAME(), USER_NAME(), 1, 1)

    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports currency rates
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_currency_rates]
    @company nvarchar(50)
    , @base_currency_code nvarchar(50)
    , @quote_currency_code nvarchar(50)
    , @time_code nvarchar(50)
    , @category_code nvarchar(50)
    , @currency_rate float
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @base_currency_id int = (SELECT id FROM dbo25.units WHERE company_id = @company_id AND code = @base_currency_code)
DECLARE @quote_currency_id int = (SELECT id FROM dbo25.units WHERE company_id = @company_id AND code = @quote_currency_code)

DECLARE @time_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @time_code)
DECLARE @category_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @category_code)

IF @base_currency_id IS NULL OR @quote_currency_id IS NULL OR @time_id IS NULL OR @category_id IS NULL OR @currency_rate IS NULL
    RETURN

UPDATE dbo25.currency_rates
SET
    currency_rate = @currency_rate
WHERE
    base_currency_id = @base_currency_id
    AND quote_currency_id = @quote_currency_id
    AND time_id = @time_id
    AND category_id = @category_id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.currency_rates
        (base_currency_id, quote_currency_id, time_id, category_id, currency_rate)
    VALUES
        (@base_currency_id, @quote_currency_id, @time_id, @category_id, @currency_rate)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports dimension calculation types
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_dimension_calc_types]
    @dimension_id tinyint
    , @calc_type_id tinyint
    , @is_active bit = NULL
AS
BEGIN

SET NOCOUNT ON

IF @is_active IS NULL SET @is_active = 1

UPDATE dbo25.dimension_calc_types
SET
    is_active = @is_active
WHERE
    dimension_id = @dimension_id
    AND calc_type_id = @calc_type_id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.dimension_calc_types
        (dimension_id, calc_type_id, is_active)
    VALUES
        (@dimension_id, @calc_type_id, @is_active)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports dimension properties
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_dimension_members]
    @company nvarchar(50)
    , @dimension_id tinyint
    , @root_member_code nvarchar(50) = NULL
    , @default_member_code nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @root_member_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @root_member_code)
DECLARE @default_member_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @default_member_code)

UPDATE dbo25.dimension_properties
SET
    root_member_id = @root_member_id
    , default_member_id = @default_member_id
WHERE
    company_id = @company_id
    AND id = @dimension_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports dimension properties
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_dimension_properties]
    @company nvarchar(50)
    , @id tinyint
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @parameter_name nvarchar(50)
    , @sort_order tinyint
    , @join_order tinyint
    , @name_format_id tinyint
    , @is_protected bit = NULL
    , @default_select_permission bit = NULL
    , @default_update_permission bit = NULL
    , @is_active bit = NULL
    , @external_id int = NULL
    , @external_code nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

IF @is_protected IS NULL SET @is_protected = 0
IF @default_select_permission IS NULL SET @default_select_permission = 1
IF @default_update_permission IS NULL SET @default_update_permission = 1
IF @is_active IS NULL SET @is_active = 1

UPDATE dbo25.dimension_properties
SET
    code = @code
    , name = @name
    , parameter_name = @parameter_name
    , sort_order = @sort_order
    , join_order = @join_order
    , name_format_id = @name_format_id
    , is_protected = @is_protected
    , default_select_permission = @default_select_permission
    , default_update_permission = @default_update_permission
    , is_active = @is_active
    , external_id = @external_id
    , external_code = @external_code
WHERE
    company_id = @company_id
    AND id = @id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.dimension_properties
        ( company_id, id, code, name, parameter_name, sort_order, join_order, name_format_id
        , is_protected, default_select_permission, default_update_permission
        , is_active, external_id, external_code)
    VALUES
        ( @company_id, @id, @code, @name, @parameter_name, @sort_order, @join_order, @name_format_id
        , @is_protected, @default_select_permission, @default_update_permission
        , @is_active, @external_id, @external_code)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports dimensions
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_dimensions]
    @id tinyint
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @parameter_name nvarchar(50)
    , @sort_order tinyint
    , @join_order tinyint
    , @name_format_id tinyint
    , @is_protected bit = NULL
    , @default_select_permission bit = NULL
    , @default_update_permission bit = NULL
    , @is_active bit = NULL
AS
BEGIN

SET NOCOUNT ON

IF @is_protected IS NULL SET @is_protected = 0
IF @default_select_permission IS NULL SET @default_select_permission = 1
IF @default_update_permission IS NULL SET @default_update_permission = 1
IF @is_active IS NULL SET @is_active = 1

UPDATE dbo25.dimensions
SET
    code = @code
    , name = @name
    , parameter_name = @parameter_name
    , sort_order = @sort_order
    , join_order = @join_order
    , name_format_id = @name_format_id
    , is_protected = @is_protected
    , default_select_permission = @default_select_permission
    , default_update_permission = @default_update_permission
    , is_active = @is_active
WHERE
    id = @id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.dimensions
        ( id, code, name, parameter_name, sort_order, join_order, name_format_id
        , is_protected, default_select_permission, default_update_permission
        , is_active)
    VALUES
        ( @id, @code, @name, @parameter_name, @sort_order, @join_order, @name_format_id
        , @is_protected, @default_select_permission, @default_update_permission
        , @is_active)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports cube data
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_facts]
    @company nvarchar(50)
    , @value float = NULL
    , @code1 nvarchar(50)
    , @code2 nvarchar(50)
    , @code3 nvarchar(50)
    , @code4 nvarchar(50)
    , @code5 nvarchar(50)
    , @code6 nvarchar(50)
    , @code7 nvarchar(50)
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @parameters TABLE (dimension_id int PRIMARY KEY, member_id int)

INSERT INTO @parameters (dimension_id, member_id)
SELECT
    c.dimension_id, COALESCE(m.id, d.default_member_id)
FROM
    (VALUES (1, @code1), (2, @code2), (3, @code3), (4, @code4), (5, @code5), (6, @code6), (7, @code7)) c(dimension_id, code)
    INNER JOIN dbo25.dimension_properties d ON d.company_id = @company_id AND d.id = c.dimension_id
    LEFT OUTER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code

IF @@ROWCOUNT <> 7
    RETURN

-- Members

DECLARE @members TABLE(id int PRIMARY KEY, id1 int, id2 int, id3 int, id4 int, id5 int, id6 int, id7 int)

INSERT INTO @members (id, id1, id2, id3, id4, id5, id6, id7)
SELECT
    1, [1], [2], [3], [4], [5], [6], [7]
FROM
    (
        SELECT dimension_id, member_id FROM @parameters
    ) s PIVOT (
        MAX(member_id) FOR dimension_id IN ([1], [2], [3], [4], [5], [6], [7])
    ) p

UPDATE dbo25.facts
SET
    value = @value
    , source_value = @value
FROM
    dbo25.facts f
    INNER JOIN @members m ON f.id1 = m.id1 AND f.id2 = m.id2 AND f.id3 = m.id3 AND f.id4 = m.id4 AND  f.id5 = m.id5 AND f.id6 = m.id6 AND f.id7 = m.id7

IF @@ROWCOUNT = 0
    BEGIN
    IF @value IS NULL
        RETURN

    INSERT INTO dbo25.facts (source_value, value, id1, id2, id3, id4, id5, id6, id7)
    SELECT
        @value, @value, id1, id2, id3, id4, id5, id6, id7
    FROM
        @members m
    WHERE
        id1 IS NOT NULL AND id2 IS NOT NULL AND id3 IS NOT NULL AND id4 IS NOT NULL AND id5 IS NOT NULL AND id6 IS NOT NULL AND id7 IS NOT NULL
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports form dimension properties
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_form_dimensions]
    @company nvarchar(50)
    , @form_code nvarchar(50)
    , @dimension_id tinyint
    , @axis_type_id tinyint
    , @parameter_index tinyint = NULL
    , @header_order tinyint = NULL
    , @root_member_code nvarchar(50) = NULL
    , @parameter_start_level tinyint = NULL
    , @parameter_end_level tinyint = NULL
    , @form_start_level tinyint = NULL
    , @form_end_level tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @form_id int = (SELECT id FROM dbo25.forms WHERE company_id = @company_id AND code = @form_code)

IF @form_id IS NULL
    RETURN

DECLARE @root_member_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @root_member_code)

UPDATE dbo25.form_dimensions
SET
    axis_type_id = @axis_type_id
    , parameter_index = @parameter_index
    , header_order = @header_order
    , root_member_id = @root_member_id
    , parameter_start_level = @parameter_start_level
    , parameter_end_level = @parameter_end_level
    , form_start_level = @form_start_level
    , form_end_level = @form_end_level
FROM
    dbo25.form_dimensions fd
WHERE
    form_id = @form_id
    AND dimension_id = @dimension_id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.form_dimensions
        ( form_id
        , dimension_id
        , axis_type_id
        , parameter_index
        , header_order
        , root_member_id
        , parameter_start_level
        , parameter_end_level
        , form_start_level
        , form_end_level
        )
    VALUES
        ( @form_id
        , @dimension_id
        , @axis_type_id
        , @parameter_index
        , @header_order
        , @root_member_id
        , @parameter_start_level
        , @parameter_end_level
        , @form_start_level
        , @form_end_level
        )

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports form permissions
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_form_permissions]
    @company nvarchar(50)
    , @form_code nvarchar(50) = NULL
    , @username nvarchar(128) = NULL
    , @select_permission tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

IF @username IS NULL
    RETURN

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @form_id int = (SELECT id FROM dbo25.forms WHERE company_id = @company_id AND code = @form_code)

IF @form_id IS NULL
    RETURN

UPDATE dbo25.form_permissions
SET
    select_permission = @select_permission
WHERE
    form_id = @form_id
    AND username = @username

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.form_permissions
        (form_id, username, select_permission)
    VALUES
        (@form_id, @username, @select_permission)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports rows
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_form_rows]
    @company nvarchar(50)
    , @rowset_code nvarchar(50)
    , @sort_order int = NULL
    , @member_code nvarchar(50) = NULL
    , @code nvarchar(50) = NULL
    , @name nvarchar(400) = NULL
    , @decimal_places tinyint = NULL
    , @is_percent tinyint = NULL
    , @row_color tinyint = NULL
    , @row_bold tinyint = NULL
    , @row_indent tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @rowset_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @rowset_code)
DECLARE @member_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @member_code)

INSERT INTO dbo25.form_rows
    ( rowset_id
    , sort_order
    , member_id
    , code
    , name
    , decimal_places
    , is_percent
    , row_color
    , row_bold
    , row_indent
    )
VALUES
    ( @rowset_id
    , @sort_order
    , @member_id
    , @code
    , @name
    , @decimal_places
    , @is_percent
    , @row_color
    , @row_bold
    , @row_indent
    )

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports form subtotals
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_form_subtotals]
    @company nvarchar(50)
    , @form_code nvarchar(50)
    , @show bit = NULL
    , @dimension_id1 tinyint = NULL
    , @dimension_id2 tinyint = NULL
    , @dimension_id3 tinyint = NULL
    , @member_code1 nvarchar(50) = NULL
    , @member_code2 nvarchar(50) = NULL
    , @member_code3 nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @form_id int = (SELECT id FROM dbo25.forms WHERE company_id = @company_id AND code = @form_code)
DECLARE @member_id1 int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @member_code1)
DECLARE @member_id2 int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @member_code2)
DECLARE @member_id3 int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @member_code3)

IF @form_id IS NULL
    RETURN

UPDATE dbo25.form_subtotals
SET
    show = @show
WHERE
    form_id = @form_id
    AND COALESCE(dimension_id1, 0) = COALESCE(@dimension_id1, 0)
    AND COALESCE(dimension_id2, 0) = COALESCE(@dimension_id2, 0)
    AND COALESCE(dimension_id3, 0) = COALESCE(@dimension_id3, 0)
    AND COALESCE(member_id1, 0) = COALESCE(@member_id1, 0)
    AND COALESCE(member_id2, 0) = COALESCE(@member_id2, 0)
    AND COALESCE(member_id3, 0) = COALESCE(@member_id3, 0)

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.form_subtotals
        ( form_id
        , dimension_id1
        , dimension_id2
        , dimension_id3
        , member_id1
        , member_id2
        , member_id3
        , show
        )
    VALUES
        ( @form_id
        , @dimension_id1
        , @dimension_id2
        , @dimension_id3
        , @member_id1
        , @member_id2
        , @member_id3
        , @show
        )

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports forms
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_formats]
    @company nvarchar(50)
    , @TABLE_SCHEMA nvarchar(128)
    , @TABLE_NAME nvarchar(128)
    , @TABLE_EXCEL_FORMAT_XML xml
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

UPDATE dbo25.formats
SET
    TABLE_EXCEL_FORMAT_XML = @TABLE_EXCEL_FORMAT_XML
WHERE
    company_id = @company_id
    AND TABLE_SCHEMA = @TABLE_SCHEMA
    AND TABLE_NAME = @TABLE_NAME

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.formats
        (company_id, TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML)
    VALUES
        (@company_id, @TABLE_SCHEMA, @TABLE_NAME, @TABLE_EXCEL_FORMAT_XML);

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports forms
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_forms]
    @company nvarchar(50)
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @sort_order int = NULL
    , @is_active bit = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

UPDATE dbo25.forms
SET
    name = @name
    , sort_order = @sort_order
    , is_active = COALESCE(@is_active, 1)
WHERE
    company_id = @company_id
    AND code = @code

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.forms
        (company_id, code, name, sort_order, is_active)
    VALUES
        (@company_id, @code, @name, @sort_order, COALESCE(@is_active, 1));

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports member parents
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_member_parents]
    @company nvarchar(50)
    , @member_code nvarchar(50)
    , @parent_code nvarchar(50)
    , @factor float = NULL
AS
BEGIN

SET NOCOUNT ON

IF @factor IS NULL
    RETURN

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @member_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @member_code)
DECLARE @parent_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @parent_code)

IF @member_id IS NULL OR @parent_id IS NULL
    RETURN

UPDATE dbo25.parents
SET
    factor = @factor
WHERE
    member_id = @member_id
    AND parent_id = @parent_id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.parents
        (member_id, parent_id, factor)
    VALUES
        (@member_id, @parent_id, @factor)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports member permissions
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_member_permissions]
    @company nvarchar(50)
    , @member_code nvarchar(50)
    , @username nvarchar(128)
    , @select_permission tinyint = NULL
    , @update_permission tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

IF @username IS NULL
    RETURN

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @member_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @member_code)

UPDATE dbo25.member_permissions
SET
    select_permission = @select_permission
    , update_permission = @update_permission
WHERE
    member_id = @member_id
    AND username = @username

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.member_permissions
        (member_id, username, select_permission, update_permission, is_inherited)
    VALUES
        (@member_id, @username, @select_permission, @update_permission, 0)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports member properties
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_member_properties]
    @company nvarchar(50)
    , @member_code nvarchar(50)
    , @previous_period_code nvarchar(50) = NULL
    , @same_period_code nvarchar(50) = NULL
    , @tax_rate_code nvarchar(50) = NULL
    , @unit_code nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @member_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @member_code)
DECLARE @previous_period_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @previous_period_code)
DECLARE @same_period_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @same_period_code)
DECLARE @tax_rate_id int = (SELECT id FROM dbo25.tax_rates WHERE company_id = @company_id AND code = @tax_rate_code)
DECLARE @unit_id int = (SELECT id FROM dbo25.units WHERE company_id = @company_id AND code = @unit_code)

UPDATE dbo25.members
SET
    previous_period_id = @previous_period_id
    , same_period_id = @same_period_id
    , tax_rate_id = @tax_rate_id
    , unit_id = @unit_id
WHERE
    id = @member_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports member relations
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_member_relations]
    @company nvarchar(50)
    , @member_code nvarchar(50)
    , @partner_code nvarchar(50)
    , @select_permission tinyint = NULL
    , @update_permission tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @member_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @member_code)
DECLARE @partner_id int = (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @partner_code)

IF @member_id IS NULL OR @partner_id IS NULL
    RETURN

UPDATE dbo25.member_relations
SET
    select_permission = @select_permission
    , update_permission = @update_permission
WHERE
    member_id = @member_id
    AND partner_id = @partner_id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.member_relations
        (member_id, partner_id, select_permission, update_permission)
    VALUES
        (@member_id, @partner_id, @select_permission, @update_permission)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports members
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_members]
    @company nvarchar(50)
    , @dimension_id tinyint
    , @sort_order int
    , @code nvarchar(50)
    , @name nvarchar(400)
    , @calc_type_id tinyint
    , @decimal_places tinyint = NULL
    , @is_percent bit = NULL
    , @row_color tinyint = NULL
    , @row_bold tinyint = NULL
    , @row_indent tinyint = NULL
    , @show_line bit = NULL
    , @show_line_before bit = NULL
    , @show_line_after bit = NULL
    , @is_translatable bit = NULL
    , @is_active bit = NULL
    , @external_id int = NULL
    , @external_code nvarchar(50) = NULL
    , @excel_formula nvarchar(400) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

IF @is_active IS NULL SET @is_active = 1

UPDATE dbo25.members
SET
    dimension_id = @dimension_id
    , sort_order = @sort_order
    , name = @name
    , calc_type_id = @calc_type_id
    , decimal_places = @decimal_places
    , is_percent = @is_percent
    , row_color = @row_color
    , row_bold = @row_bold
    , row_indent = @row_indent
    , show_line = COALESCE(@show_line, 1)
    , show_line_before = COALESCE(@show_line_before, 0)
    , show_line_after = COALESCE(@show_line_after, 0)
    , is_translatable = COALESCE(@is_translatable, 1)
    , is_active = COALESCE(@is_active, 1)
    , external_id = @external_id
    , external_code = @external_code
    , excel_formula = @excel_formula
WHERE
    company_id = @company_id
    AND code = @code

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.members
        ( company_id, dimension_id, sort_order, code, name, calc_type_id
        , decimal_places, is_percent, row_color, row_bold, row_indent
        , show_line, show_line_before, show_line_after
        , is_translatable
        , is_active, external_id, external_code, excel_formula)
    VALUES
        ( @company_id, @dimension_id, @sort_order, @code, @name, @calc_type_id
        , @decimal_places, @is_percent, @row_color, @row_bold, @row_indent
        , COALESCE(@show_line, 1), COALESCE(@show_line_before, 0), COALESCE(@show_line_after, 0)
        , COALESCE(@is_translatable, 1)
        , COALESCE(@is_active, 1), @external_id, @external_code, @excel_formula)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports cube string data
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_strings]
    @company nvarchar(50)
    , @value nvarchar(4000) = NULL
    , @code1 nvarchar(50)
    , @code2 nvarchar(50)
    , @code3 nvarchar(50)
    , @code4 nvarchar(50)
    , @code5 nvarchar(50)
    , @code6 nvarchar(50)
    , @code7 nvarchar(50)
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @parameters TABLE (dimension_id int PRIMARY KEY, member_id int)

INSERT INTO @parameters (dimension_id, member_id)
SELECT
    c.dimension_id, m.id
FROM
    (VALUES (1, @code1), (2, @code2), (3, @code3), (4, @code4), (5, @code5), (6, @code6), (7, @code7)) c(dimension_id, code)
    INNER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code

IF @@ROWCOUNT <> 7
    RETURN

-- Members

DECLARE @members TABLE(id int PRIMARY KEY, id1 int, id2 int, id3 int, id4 int, id5 int, id6 int, id7 int)

INSERT INTO @members (id, id1, id2, id3, id4, id5, id6, id7)
SELECT
    1, [1], [2], [3], [4], [5], [6], [7]
FROM
    (
        SELECT dimension_id, member_id FROM @parameters
    ) s PIVOT (
        MAX(member_id) FOR dimension_id IN ([1], [2], [3], [4], [5], [6], [7])
    ) p

UPDATE dbo25.strings
SET
    value = @value
FROM
    dbo25.strings f
    INNER JOIN @members m ON f.id1 = m.id1 AND f.id2 = m.id2 AND f.id3 = m.id3 AND f.id4 = m.id4 AND  f.id5 = m.id5 AND f.id6 = m.id6 AND f.id7 = m.id7

IF @@ROWCOUNT = 0
    BEGIN
    IF @value IS NULL
        RETURN

    INSERT INTO dbo25.strings (value, id1, id2, id3, id4, id5, id6, id7)
    SELECT
        @value, id1, id2, id3, id4, id5, id6, id7
    FROM
        @members m
    WHERE
        id1 IS NOT NULL AND id2 IS NOT NULL AND id3 IS NOT NULL AND id4 IS NOT NULL AND id5 IS NOT NULL AND id6 IS NOT NULL AND id7 IS NOT NULL
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports tax rates
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_tax_rates]
    @company nvarchar(50)
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @sort_order tinyint = NULL
    , @is_translatable bit = NULL
    , @is_active bit = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

UPDATE dbo25.tax_rates
SET
    name = @name
    , sort_order = @sort_order
    , is_translatable = COALESCE(@is_translatable, 1)
    , is_active = COALESCE(@is_active, 1)
WHERE
    company_id = @company_id
    AND code = @code

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.tax_rates
        (company_id, code, name, sort_order, is_translatable, is_active)
    VALUES
        (@company_id, @code, @name, @sort_order, COALESCE(@is_translatable, 1), COALESCE(@is_active, 1))

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports translations
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_translations]
    @company nvarchar(50)
    , @table nvarchar(50) = NULL
    , @member nvarchar(50) = NULL
    , @language varchar(10)
    , @name nvarchar(400) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @language IS NULL
    RETURN

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

DECLARE @table_id tinyint = (SELECT TOP 1 t.id FROM dbo25.translated_tables t WHERE t.code = @table)

DECLARE @member_id int

IF @table_id = 8
    BEGIN
    ;MERGE dbo25.translations t
    USING
        (
        SELECT
            m.company_id
            , @table_id AS table_id
            , r.id AS member_id
            , @language AS language
            , @name AS name
        FROM
            dbo25.form_rows r
            INNER JOIN dbo25.members m ON m.id = r.rowset_id
        WHERE
            m.company_id = @company_id
            AND r.name = @member
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    RETURN
    END

IF @table_id = 1
    SELECT TOP 1 @member_id = t.id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.company_id = @company_id AND t.code = @member AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 2
    SET @member_id = @company_id
ELSE IF @table_id = 3
    SELECT TOP 1 @member_id = t.id FROM dbo25.units t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.company_id = @company_id AND t.code = @member AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 4
    SELECT TOP 1 @member_id = t.id FROM dbo25.tax_rates t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.company_id = @company_id AND t.code = @member AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 5
    SELECT TOP 1 @member_id = t.id FROM dbo25.dimension_properties t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.company_id = @company_id AND t.code = @member AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 6
    SELECT TOP 1 @member_id = t.id FROM dbo25.dimension_properties t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.company_id = @company_id AND t.parameter_name = @member AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 7
    SELECT TOP 1 @member_id = t.id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.company_id = @company_id AND t.code = @member AND u.username = USER_NAME() AND u.is_active = 1
ELSE
    RETURN

IF @member_id IS NULL
    RETURN

;MERGE dbo25.translations t
USING
    (
    SELECT company_id = @company_id, @table_id AS table_id, @member_id AS member_id, @language AS language, @name AS name
    ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
    UPDATE SET name = s.name
WHEN NOT MATCHED THEN
    INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports units
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_units]
    @company nvarchar(50)
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @sort_order int
    , @factor float
    , @is_currency bit = NULL
    , @is_functional_currency bit = NULL
    , @is_translatable bit = NULL
    , @is_active bit = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

UPDATE dbo25.units
SET
    name = @name
    , sort_order = @sort_order
    , factor = @factor
    , is_currency = COALESCE(@is_currency, 1)
    , is_functional_currency = COALESCE(@is_functional_currency, 0)
    , is_translatable = COALESCE(@is_translatable, 1)
    , is_active = COALESCE(@is_active, 1)
WHERE
    company_id = @company_id
    AND code = @code

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.units
        ( company_id, code, name, sort_order, factor, is_currency, is_functional_currency, is_translatable, is_active)
    VALUES
        ( @company_id, @code, @name, @sort_order, @factor, COALESCE(@is_currency, 1)
        , COALESCE(@is_functional_currency, 0), @is_translatable, COALESCE(@is_active, 1))

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Imports users
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_users]
    @company nvarchar(50)
    , @username nvarchar(128)
    , @name nvarchar(100)
    , @is_admin bit = NULL
    , @is_default bit = NULL
    , @is_active bit = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

UPDATE dbo25.users
SET
    name = @name
    , is_admin = COALESCE(@is_admin, 0)
    , is_default = COALESCE(@is_default, 0)
    , is_active = COALESCE(@is_active, 1)
WHERE
    company_id = @company_id
    AND username = @username

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.users
        (company_id, username, name, is_admin, is_default, is_active)
    VALUES
        (@company_id, @username, @name, COALESCE(@is_admin, 0), COALESCE(@is_default, 0), COALESCE(@is_active, 1))

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Creates standard forms
-- =============================================

CREATE PROCEDURE [dbo25].[xl_actions_create_standard_forms]
    @company_id int
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Incorrect company_id', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

DECLARE @company nvarchar(50) = (SELECT TOP 1 code FROM dbo25.companies WHERE id = @company_id)

EXEC dbo25.usp_import_forms @company, N'Accounts*Times', N'Accounts*Times', 1, 1;
EXEC dbo25.usp_import_forms @company, N'Accounts*Categories', N'Accounts*Categories', 2, 1;
EXEC dbo25.usp_import_forms @company, N'Accounts*Entities', N'Accounts*Entities', 3, 1;
EXEC dbo25.usp_import_forms @company, N'Accounts*Times+Categories', N'Accounts*Times+Categories', 4, 1;
EXEC dbo25.usp_import_forms @company, N'Accounts+Entities*Times', N'Accounts+Entities*Times', 5, 1;

EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Categories', 1, 2, 1, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Categories', 2, 1, 4, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Categories', 3, 3, 3, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Categories', 4, 1, 2, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Categories', 5, 1, 5, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Categories', 6, 1, 6, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Categories', 7, 1, 7, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Entities', 1, 2, 1, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Entities', 2, 1, 4, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Entities', 3, 1, 3, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Entities', 4, 3, 2, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Entities', 5, 1, 5, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Entities', 6, 1, 6, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Entities', 7, 1, 7, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times', 1, 2, 1, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times', 2, 3, 4, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times', 3, 1, 3, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times', 4, 1, 2, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times', 5, 1, 5, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times', 6, 1, 6, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times', 7, 1, 7, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times+Categories', 1, 2, 1, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times+Categories', 2, 3, 4, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times+Categories', 3, 3, 3, 2, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times+Categories', 4, 1, 2, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times+Categories', 5, 1, 5, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times+Categories', 6, 1, 6, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts*Times+Categories', 7, 1, 7, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts+Entities*Times', 1, 2, 1, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts+Entities*Times', 2, 3, 4, 1, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts+Entities*Times', 3, 1, 3, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts+Entities*Times', 4, 2, 2, 2, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts+Entities*Times', 5, 1, 5, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts+Entities*Times', 6, 1, 6, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_dimensions @company, N'Accounts+Entities*Times', 7, 1, 7, NULL, NULL, NULL, NULL, NULL, NULL;

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Creates standard tax_rates
-- =============================================

CREATE PROCEDURE [dbo25].[xl_actions_create_standard_tax_rates]
    @company_id int
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Incorrect company_id', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

DECLARE @tax_rate_id int = (SELECT TOP 1 id FROM dbo25.tax_rates t WHERE t.company_id = @company_id ORDER BY t.id)

IF @tax_rate_id IS NULL
    INSERT INTO dbo25.tax_rates
        ( company_id, code, name, sort_order, factor, is_active)
    VALUES
        ( @company_id, 'default', '', 1, 1, 1)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Creates standard units
-- =============================================

CREATE PROCEDURE [dbo25].[xl_actions_create_standard_units]
    @company_id int
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Incorrect company_id', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

DECLARE @currency_id int = (SELECT TOP 1 id FROM dbo25.units u WHERE u.company_id = @company_id AND u.is_functional_currency = 1 ORDER BY u.id)

IF @currency_id IS NULL
    BEGIN
    SET @currency_id = (SELECT TOP 1 id FROM dbo25.units u WHERE u.company_id = @company_id AND u.code = 'USD')

    IF @currency_id IS NULL
        SET @currency_id = (SELECT TOP 1 id FROM dbo25.units u WHERE u.company_id = @company_id AND u.is_currency = 1 ORDER BY u.id)

    IF @currency_id IS NULL
        BEGIN
        INSERT INTO dbo25.units
            ( company_id, code, name, sort_order, factor, is_currency, is_functional_currency, is_active)
        VALUES
            ( @company_id, 'USD', 'USD', 1, 1, 1, 1, 1)
        END
    ELSE
        UPDATE dbo25.units SET is_currency = 1, is_functional_currency = 1, is_active = 1 WHERE id = @currency_id
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Sets permissions on Database Help Framework roles
-- =============================================

CREATE PROCEDURE [dbo25].[xl_actions_set_doc_role_permissions]
AS
BEGIN

SET NOCOUNT ON

DECLARE @sql nvarchar(max)

IF USER_ID('doc_readers') IS NOT NULL
    BEGIN
    -- EXEC sp_addrolemember 'doc_readers', 'planning_app_admins';
    -- EXEC sp_addrolemember 'doc_readers', 'planning_app_analysts';
    EXEC sp_addrolemember 'doc_writers', 'planning_app_developers';

    SET @sql ='
GRANT VIEW DEFINITION ON ROLE::doc_readers TO planning_app_developers;
GRANT VIEW DEFINITION ON ROLE::doc_writers TO planning_app_developers;
'
    EXEC (@sql);
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Sets permissions on Change Tracking Framework roles
-- =============================================

CREATE PROCEDURE [dbo25].[xl_actions_set_log_role_permissions]
AS
BEGIN

SET NOCOUNT ON

DECLARE @sql nvarchar(max)

IF USER_ID('log_users') IS NOT NULL
    BEGIN
    EXEC sp_addrolemember 'log_admins', 'planning_app_developers';
    EXEC sp_addrolemember 'log_users',  'planning_app_admins';
    EXEC sp_addrolemember 'log_users',  'planning_app_analysts';

    SET @sql ='
GRANT VIEW DEFINITION ON ROLE::log_admins TO planning_app_developers;
GRANT VIEW DEFINITION ON ROLE::log_users  TO planning_app_developers;
'
    EXEC (@sql);
    END


END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Sets permissions of the planning application roles
-- =============================================

CREATE PROCEDURE [dbo25].[xl_actions_set_role_permissions]
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

-- SaveToDB Framework roles

EXEC sp_addrolemember 'xls_users',          'planning_app';

EXEC sp_addrolemember 'xls_users',          'planning_app_admins';
EXEC sp_addrolemember 'xls_formats',        'planning_app_admins';
EXEC sp_addrolemember 'xls_developers',     'planning_app_developers';
EXEC sp_addrolemember 'xls_users',          'planning_app_analysts';
EXEC sp_addrolemember 'xls_formats',        'planning_app_analysts';
EXEC sp_addrolemember 'xls_users',          'planning_app_users';

IF USER_ID('xls_admins') IS NOT NULL
    BEGIN
    EXEC sp_addrolemember 'xls_admins', 'planning_app_admins';
    END

-- Developers may add/delete users

GRANT ALTER, VIEW DEFINITION ON ROLE::planning_app_admins       TO planning_app_developers;
GRANT ALTER, VIEW DEFINITION ON ROLE::planning_app_analysts     TO planning_app_developers;
GRANT ALTER, VIEW DEFINITION ON ROLE::planning_app_developers   TO planning_app_developers;
GRANT ALTER, VIEW DEFINITION ON ROLE::planning_app_users        TO planning_app_developers;

-- Developers may see SaveToDB Framework users

GRANT VIEW DEFINITION ON ROLE::xls_users                        TO planning_app_developers;
GRANT VIEW DEFINITION ON ROLE::xls_developers                   TO planning_app_developers;
GRANT VIEW DEFINITION ON ROLE::xls_formats                      TO planning_app_developers;

-- Developers may execute any procedure in dbo25 and xls25 and select from any view in xls25
-- Direct SELECT in dbo25 is prohibited to protect data of different companies

GRANT         EXECUTE ON SCHEMA::dbo25                          TO planning_app_developers;
GRANT SELECT, EXECUTE ON SCHEMA::xls25                          TO planning_app_developers;

GRANT EXECUTE ON xls25a.usp_update_table_format                 TO planning_app_developers;
GRANT EXECUTE ON xls25b.usp_update_table_format                 TO planning_app_developers;

-- Administrators may see application users

GRANT VIEW DEFINITION ON ROLE::planning_app_users               TO planning_app_admins;

-- Administrators may execute forms to manage companies, users, and permissions

GRANT EXECUTE ON xls25.usp_companies                            TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_companies_insert                     TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_companies_update                     TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_companies_delete                     TO planning_app_admins;

GRANT EXECUTE ON xls25.usp_users                                TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_users_insert                         TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_users_update                         TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_users_delete                         TO planning_app_admins;

GRANT EXECUTE ON xls25.usp_form_permissions                     TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_form_permissions_change              TO planning_app_admins;

GRANT EXECUTE ON xls25.usp_member_permissions                   TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_member_permissions_change            TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_member_relations                     TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_member_relations_change              TO planning_app_admins;

GRANT EXECUTE ON xls25.usp_closed_periods                       TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_closed_periods_change                TO planning_app_admins;

GRANT EXECUTE ON xls25.usp_role_members                         TO planning_app_admins;
GRANT EXECUTE ON xls25.usp_role_members_change                  TO planning_app_admins;

GRANT EXECUTE ON xls25.xl_parameter_values_company_id           TO planning_app_admins;
GRANT EXECUTE ON xls25.xl_parameter_values_dimension_id         TO planning_app_admins;
GRANT EXECUTE ON xls25.xl_parameter_values_dimension_id_or_null TO planning_app_admins;
GRANT EXECUTE ON xls25.xl_parameter_values_root_member_id_code  TO planning_app_admins;
GRANT EXECUTE ON xls25.xl_parameter_values_username             TO planning_app_admins;

GRANT EXECUTE ON xls25.xl_parameter_values_relation_dimension_id    TO planning_app_admins;
GRANT EXECUTE ON xls25.xl_parameter_values_relation_partner_id      TO planning_app_admins;
GRANT EXECUTE ON xls25.xl_parameter_values_root_member_id_code      TO planning_app_admins;

GRANT EXECUTE ON xls25a.usp_update_table_format                 TO planning_app_admins;
GRANT EXECUTE ON xls25b.usp_update_table_format                 TO planning_app_admins;

-- Analysts may execute any procedure and select from any view in dbo25
-- except for the procedures that changes permissions

GRANT SELECT  ON xls25.view_hierarchies                 TO planning_app_analysts;
GRANT SELECT  ON xls25.xl_formats                       TO planning_app_analysts;
GRANT SELECT  ON xls25.xl_handlers                      TO planning_app_analysts;
GRANT SELECT  ON xls25.xl_translations                  TO planning_app_analysts;
GRANT SELECT  ON xls25.xl_query_list                    TO planning_app_analysts;
GRANT SELECT  ON xls25.xl_query_list_online_forms       TO planning_app_analysts;
GRANT SELECT  ON xls25.xl_query_list_offline_forms      TO planning_app_analysts;

GRANT EXECUTE ON xls25.usp_axis_types                                   TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_calc_types                                   TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_closed_periods                               TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_closed_periods_change                        TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_companies                                    TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_currency_rates                               TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_currency_rates_change                        TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_data_management                              TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_dimensions                                   TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_dimensions_change                            TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_dimensions_delete                            TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_dimensions_insert                            TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_dimensions_update                            TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_facts                                        TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_facts_data                                   TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_facts_strings                                TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_dimensions                              TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_dimensions_delete                       TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_dimensions_insert                       TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_dimensions_update                       TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_permissions                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_permissions_change                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_rows                                    TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_rows_delete                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_rows_insert                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_form_rows_update                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_forms                                        TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_forms_delete                                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_forms_insert                                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_forms_update                                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_member_permissions                           TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_member_permissions_change                    TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_member_relations                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_member_relations_change                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_members                                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_members_delete                               TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_members_insert                               TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_members_update                               TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_name_formats                                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_role_members                                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_rowsets                                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_rowsets_delete                               TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_rowsets_insert                               TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_rowsets_update                               TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_run_form                                     TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_run_form_change                              TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_run_json_form                                TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_run_offline_form                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_run_offline_form_delete                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_run_offline_form_insert                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_run_offline_form_update                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_tax_rates                                    TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_tax_rates_delete                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_tax_rates_insert                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_tax_rates_update                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_translations                                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_translations_change                          TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_translations_common                          TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_translations_xls                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_translations_xls_change                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_units                                        TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_units_delete                                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_units_insert                                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_units_update                                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.usp_users                                        TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_add_language                          TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_add_quarters                          TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_add_year                              TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_copy_data                             TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_delete_data                           TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_delete_year                           TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_run_form_cell_data                    TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_set_functional_currency               TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_update_facts                          TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_update_hierarchies                    TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_actions_update_member_permissions             TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_aliases_members                               TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_0_or_1                       TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_calc_type_id                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_calc_type_id_or_null         TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_company_id                   TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_currency_id                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_dimension_id                 TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_dimension_id_or_null         TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_form_id                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_form_id_or_null              TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_member_id_by_dimension_id    TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_member_id_dim1               TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_member_id_dim2               TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_member_id_dim3               TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_member_id_dim4               TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_member_id_dim5               TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_member_id_dim6               TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_member_id_dim7               TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_relation_dimension_id        TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_relation_partner_id          TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_root_member_id_code          TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_rowset_id                    TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_rowset_id_or_null            TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p                   TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p1                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p2                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p3                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p4                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p5                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p6                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p7                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_unit_id                      TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_parameter_values_username                     TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_axis_type_id                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_calc_type_id                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_calc_type_id_by_dimension_id  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_default_member_id_code        TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_dimension_id                  TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_member_id                     TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_member_id_code_by_dimension_id TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_member_id_code_by_rowset_id   TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_name_format_id                TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_previous_period_id_by_dimension_id TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_root_member_id_code           TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_root_member_or_rowset_id_code TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_rowset_id                     TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_same_period_id_by_dimension_id TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_tax_rate_id_by_dimension_id   TO planning_app_analysts;
GRANT EXECUTE ON xls25.xl_validation_list_unit_id_by_dimension_id       TO planning_app_analysts;

GRANT EXECUTE ON xls25a.usp_update_table_format                         TO planning_app_analysts;
GRANT EXECUTE ON xls25b.usp_update_table_format                         TO planning_app_analysts;

-- Users may execute the configured forms only

GRANT SELECT  ON xls25.xl_formats                       TO planning_app_users;
GRANT SELECT  ON xls25.xl_handlers                      TO planning_app_users;
GRANT SELECT  ON xls25.xl_translations                  TO planning_app_users;
GRANT SELECT  ON xls25.xl_query_list_online_forms       TO planning_app_users;
GRANT SELECT  ON xls25.xl_query_list_offline_forms      TO planning_app_users;

GRANT EXECUTE ON xls25.usp_run_form                     TO planning_app_users;
GRANT EXECUTE ON xls25.usp_run_form_change              TO planning_app_users;

GRANT EXECUTE ON xls25.usp_run_offline_form             TO planning_app_users;
GRANT EXECUTE ON xls25.usp_run_offline_form_insert      TO planning_app_users;
GRANT EXECUTE ON xls25.usp_run_offline_form_update      TO planning_app_users;
GRANT EXECUTE ON xls25.usp_run_offline_form_delete      TO planning_app_users;

GRANT EXECUTE ON xls25.xl_actions_run_form_cell_data    TO planning_app_users;

GRANT EXECUTE ON xls25.xl_parameter_values_company_id   TO planning_app_users;
GRANT EXECUTE ON xls25.xl_parameter_values_form_id      TO planning_app_users;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p1  TO planning_app_users;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p2  TO planning_app_users;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p3  TO planning_app_users;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p4  TO planning_app_users;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p5  TO planning_app_users;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p6  TO planning_app_users;
GRANT EXECUTE ON xls25.xl_parameter_values_run_form_p7  TO planning_app_users;

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_companies
-- =============================================

CREATE PROCEDURE [xls25].[usp_companies_insert]
    @code nvarchar(50)
    , @name nvarchar(100)
    , @translated_name nvarchar(100) = NULL
    , @default_language varchar(10) = NULL
    , @sort_order int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF NOT EXISTS (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_admin = 1)
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

DECLARE @message nvarchar(max)

IF CHARINDEX(@code, ' ') > 0 OR CHARINDEX(@code, '''') > 0 OR CHARINDEX(@code, '"') > 0 OR CHARINDEX(@code, '[') > 0 OR CHARINDEX(@code, ']') > 0
    BEGIN
    SET @message = dbo25.get_translated_string('The code ''%s'' contains prohibited symbols (space, '', ", [ or ])', @data_language)
    RAISERROR (@message, 11, 0, @code)
    RETURN
    END

INSERT INTO dbo25.companies (code, name, default_language, sort_order) VALUES (@code, @name, COALESCE(@default_language, 'en'), @sort_order)

DECLARE @company_id int = SCOPE_IDENTITY()

INSERT INTO dbo25.users (company_id, username, name, is_admin, is_active) VALUES (@company_id, USER_NAME(), USER_NAME(), 1, 1)

INSERT INTO dbo25.dimension_properties
    (company_id, id, code, name, parameter_name, sort_order, join_order, name_format_id, is_protected, default_select_permission, default_update_permission, is_active)
SELECT
    @company_id, id, code, name, parameter_name, sort_order, join_order, name_format_id, is_protected, default_select_permission, default_update_permission, is_active
FROM
    dbo25.dimensions d

EXEC dbo25.xl_actions_create_standard_units @company_id, @data_language

EXEC dbo25.xl_actions_create_standard_tax_rates @company_id, @data_language

EXEC dbo25.xl_actions_create_standard_members @company_id, @data_language

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    INSERT INTO dbo25.translations
        (company_id, table_id, member_id, language, name)
    VALUES
        (@company_id, 2, @company_id, @data_language, @translated_name)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Creates standard members
-- =============================================

CREATE PROCEDURE [dbo25].[xl_actions_create_standard_members]
    @company_id int
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Incorrect company_id', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

DECLARE @company nvarchar(50) = (SELECT TOP 1 code FROM dbo25.companies WHERE id = @company_id)

-- creating absent members first

INSERT INTO dbo25.members (company_id, code, name, dimension_id, sort_order, calc_type_id, row_color)
SELECT
    @company_id, c.code, c.name, c.dimension_id, c.sort_order, c.calc_type_id, c.row_color
FROM
    dbo25.get_standard_members() c
    LEFT OUTER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
WHERE
    m.code IS NULL
ORDER BY
    CASE WHEN c.sort_order = 1 THEN 1 ELSE 2 END
    , c.dimension_id
    , c.sort_order

-- setting root members of dimension properties

UPDATE dbo25.dimension_properties
SET
    root_member_id = m.id
FROM
    (VALUES (1, 'ACCOUNTS'), (2, 'TIMES'), (3, 'CATEGORIES'), (4, 'ENTITIES')
        , (5, 'DIM5'), (6, 'DIM6'), (7, 'DIM7')) v(dimension_id, member_code)
    INNER JOIN dbo25.dimension_properties d ON d.id = v.dimension_id
    INNER JOIN dbo25.members m ON m.company_id = d.company_id AND m.code = v.member_code
WHERE
    d.company_id = @company_id
    AND d.root_member_id IS NULL

-- setting default members of dimension properties

UPDATE dbo25.dimension_properties
SET
    default_member_id = m.id
FROM
    (VALUES (5, 'DIM5.NONE'), (6, 'DIM6.NONE'), (7, 'DIM7.NONE')) v(dimension_id, member_code)
    INNER JOIN dbo25.dimension_properties d ON d.id = v.dimension_id
    INNER JOIN dbo25.members m ON m.company_id = d.company_id AND m.code = v.member_code
WHERE
    d.company_id = @company_id
    AND d.default_member_id IS NULL
    AND d.is_active = 0

-- updating member properties after creating members

UPDATE dbo25.members
SET
    previous_period_id = p.id
    , same_period_id = s.id
FROM
    dbo25.get_standard_members() c
    INNER JOIN dbo25.members ON dbo25.members.company_id = @company_id AND dbo25.members.code = c.code
    LEFT OUTER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.previous_period
    LEFT OUTER JOIN dbo25.members s ON s.company_id = @company_id AND s.code = c.same_period
WHERE
    NOT COALESCE(dbo25.members.previous_period_id, 0) = COALESCE(p.id, 0)
    OR NOT COALESCE(dbo25.members.same_period_id, 0) = COALESCE(s.id, 0)

-- inserting parents by parent1

INSERT INTO dbo25.parents (member_id, parent_id, factor)
SELECT
    m.id, p.id, 1
FROM
    dbo25.get_standard_members() c
    INNER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
    INNER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.parent1
    LEFT OUTER JOIN dbo25.parents mp ON mp.member_id = m.id AND mp.parent_id = p.id
WHERE
    mp.member_id IS NULL

-- updating hierarchies

EXEC xls25.xl_actions_update_hierarchies @company_id

-- importing category rowset rows

EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_SAME_PERIOD*', 1, N'SAME_PERIOD', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_SAME_PERIOD*', 2, N'BUDGET', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_SAME_PERIOD**', 1, N'SAME_PERIOD', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_SAME_PERIOD**', 2, N'BUDGET', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_SAME_PERIOD**', 3, N'BUDGET-SAME_PERIOD', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_SAME_PERIOD**', 4, N'BUDGET/SAME_PERIOD,%', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_FORECAST*', 1, N'BUDGET', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_FORECAST*', 2, N'FORECAST', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_FORECAST**', 1, N'BUDGET', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_FORECAST**', 2, N'FORECAST', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_FORECAST**', 3, N'FORECAST-BUDGET', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_FORECAST**', 4, N'FORECAST/BUDGET,%', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_ACTUAL*', 1, N'BUDGET', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_ACTUAL*', 2, N'ACTUAL', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_ACTUAL**', 1, N'BUDGET', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_ACTUAL**', 2, N'ACTUAL', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_ACTUAL**', 3, N'ACTUAL-BUDGET', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'BUDGET_VS_ACTUAL**', 4, N'ACTUAL-BUDGET,%', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_SAME_PERIOD*', 1, N'SAME_PERIOD', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_SAME_PERIOD*', 2, N'ACTUAL', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_SAME_PERIOD**', 1, N'SAME_PERIOD', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_SAME_PERIOD**', 2, N'ACTUAL', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_SAME_PERIOD**', 3, N'ACTUAL-SAME_PERIOD', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_SAME_PERIOD**', 4, N'ACTUAL-SAME_PERIOD,%', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_FORECAST*', 1, N'FORECAST', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_FORECAST*', 2, N'ACTUAL', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_FORECAST**', 1, N'FORECAST', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_FORECAST**', 2, N'ACTUAL', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_FORECAST**', 3, N'ACTUAL-FORECAST', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'ACTUAL_VS_FORECAST**', 4, N'ACTUAL-FORECAST,%', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'FORECAST_VS_SAME_PERIOD*', 1, N'SAME_PERIOD', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'FORECAST_VS_SAME_PERIOD*', 2, N'FORECAST', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'FORECAST_VS_SAME_PERIOD**', 1, N'SAME_PERIOD', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'FORECAST_VS_SAME_PERIOD**', 2, N'FORECAST', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'FORECAST_VS_SAME_PERIOD**', 3, N'FORECAST-SAME_PERIOD', NULL, NULL, NULL, NULL, NULL, NULL, NULL;
EXEC dbo25.usp_import_form_rows @company, N'FORECAST_VS_SAME_PERIOD**', 4, N'FORECAST/SAME_PERIOD,%', NULL, NULL, NULL, NULL, NULL, NULL, NULL;

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Finishes importing data
-- =============================================

CREATE PROCEDURE [dbo25].[usp_import_finish]
    @company nvarchar(50) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = c.id FROM dbo25.companies c INNER JOIN dbo25.users u ON u.company_id = c.id  WHERE c.code = @company AND u.username = USER_NAME() AND u.is_active = 1 AND u.is_admin = 1

IF @company_id IS NULL
    RETURN

EXEC xls25.xl_actions_update_hierarchies @company_id

EXEC xls25.xl_actions_update_member_permissions @company_id

EXEC xls25.xl_actions_update_facts @company_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects axis_types
-- =============================================

CREATE PROCEDURE [xls25].[usp_axis_types]
    @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    t.id
    , t.code
    , COALESCE(tr.TRANSLATED_NAME, t.name) AS name
FROM
    dbo25.axis_types t
    LEFT OUTER JOIN xls.translations tr ON tr.TABLE_SCHEMA = 'dbo25' AND tr.TABLE_NAME = 'axis_types_name' AND tr.COLUMN_NAME = t.name AND tr.LANGUAGE_NAME = @data_language
ORDER BY
    t.id

END
SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects calc_types
-- =============================================

CREATE PROCEDURE [xls25].[usp_calc_types]
    @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

DECLARE @columns nvarchar(max)

SELECT @columns = STUFF((
        SELECT
            ', [' + d.code + ']' AS code
        FROM
            dbo25.dimensions d
        ORDER BY
            d.id
        FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(MAX)'), 1, 2, '')


DECLARE @sql nvarchar(max)

SET @sql = '
SELECT
    c.id
    , c.code
    , ' + CASE WHEN @data_language IS NULL THEN 'c.name' ELSE 'COALESCE(tr.TRANSLATED_NAME, c.name) AS name' END + '
    , c.sort_order
    , ' + @columns + '
FROM
    dbo25.calc_types c'
    + CASE WHEN @data_language IS NULL THEN '' ELSE '
    LEFT OUTER JOIN xls.translations tr ON tr.TABLE_SCHEMA = ''dbo25'' AND tr.TABLE_NAME = ''calc_types_name'' AND tr.COLUMN_NAME = c.name AND tr.LANGUAGE_NAME = ''' + @data_language + ''''
        END + '
    LEFT OUTER JOIN (
        SELECT
            dc.calc_type_id
            , d.code
            , CAST(dc.is_active AS tinyint) AS value
        FROM
            dbo25.dimension_calc_types dc
            INNER JOIN dbo25.dimensions d ON d.id = dc.dimension_id
    ) s PIVOT (
        MAX(value) FOR code IN (' + @columns + ')
    ) p ON p.calc_type_id = c.id
'

-- PRINT @sql

IF @sql IS NULL
    SELECT c.id, c.name, c.sort_order FROM dbo25.calc_types c
ELSE
    EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects rows to set closed periods
-- =============================================

CREATE PROCEDURE [xls25].[usp_closed_periods]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

EXECUTE AS CALLER

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

DECLARE @columns nvarchar(max)

SELECT @columns = STUFF((
        SELECT
            ', [' + CAST(m.code AS nvarchar) + ']'
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND m.dimension_id = 3
            AND m.calc_type_id IN (1)       -- total
            AND m.id NOT IN (
                SELECT parent_id FROM dbo25.parents
            )
        ORDER BY
            m.sort_order
            , m.code
        FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(MAX)'), 1, 2, '')

-- PRINT @columns

DECLARE @sql nvarchar(max)

SET @sql = 'SELECT
    m.id
    , m.sort_order
    , CASE WHEN h.parent_id IS NOT NULL THEN 0 WHEN m.calc_type_id = 0 THEN 0 ELSE 1 END AS is_editable
    , m.decimal_places
    , CAST(m.is_percent AS tinyint) AS is_percent
    , m.row_color
    , m.row_bold
    , m.row_indent
    , m.code
    , CASE WHEN m.row_indent > 0 THEN REPLICATE('' '', m.row_indent) ELSE '''' END + ' + CASE WHEN @data_language IS NULL THEN 'm.name' ELSE 'COALESCE(tr.name, m.name)' END + ' AS name
    , ' + CASE WHEN @data_language IS NULL THEN 'c.name' ELSE 'COALESCE(tc.TRANSLATED_NAME, c.name)' END + ' AS calc_type
    , NULL AS data_format
    ' + COALESCE(', ' + @columns, '') + '
FROM
    dbo25.members m
    INNER JOIN dbo25.calc_types c ON c.id = m.calc_type_id
    LEFT OUTER JOIN (SELECT DISTINCT parent_id FROM dbo25.parents) h ON h.parent_id = m.id'
    + CASE WHEN @columns IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            m.id
            , c.code
            , CAST(mp.update_permission AS tinyint) AS value
        FROM
            dbo25.member_relations mp
            INNER JOIN dbo25.members m ON m.id = mp.member_id
            INNER JOIN dbo25.members c ON c.id = mp.partner_id
        WHERE
            m.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
            AND m.dimension_id = 2
            AND c.dimension_id = 3
    ) s PIVOT (
        MAX(value) FOR code IN (' + @columns + ')
    ) p ON p.id = m.id' END
    + CASE WHEN @data_language IS NULL THEN '' ELSE '
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + ' AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = ''' + @data_language + '''
    LEFT OUTER JOIN xls.translations tc ON tc.TABLE_SCHEMA = ''dbo25'' AND tc.TABLE_NAME = ''calc_types_name'' AND tc.COLUMN_NAME = c.name AND tc.LANGUAGE_NAME = ''' + @data_language + ''''
        END + '
WHERE
    m.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
    AND m.dimension_id = 2
    AND NOT m.calc_type_id IN (0, 9)
    AND h.parent_id IS NULL
ORDER BY
    m.sort_order
    , m.code
'

-- PRINT @sql

REVERT

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_closed_periods
-- =============================================

CREATE PROCEDURE [xls25].[usp_closed_periods_change]
    @column_name nvarchar(128) = NULL
    , @cell_number_value int = NULL
    , @id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

-- Protecting non-data columns

IF @column_name IN ('id', 'sort_order', 'row_color', 'row_bold', 'row_indent', 'member', 'data_format')
    BEGIN
    SET @message = dbo25.get_translated_string('The column ''%s'' is not updatable', @data_language)
    RAISERROR (@message, 11, 0, @column_name)
    RETURN
    END

IF NOT EXISTS (SELECT id FROM dbo25.members WHERE id = @id)
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find member_id %i', @data_language)
    RAISERROR (@message, 11, 0, @id)
    RETURN
    END

DECLARE @company_id int

SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1)

IF @company_id IS NULL
    RETURN

DECLARE @category_code nvarchar(128) = @column_name
DECLARE @category_id int

SELECT @category_id = id FROM dbo25.members WHERE company_id = @company_id AND code = @category_code

IF @category_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find member_id %s', @data_language)
    RAISERROR (@message, 11, 0, @category_code)
    RETURN
    END

DECLARE @update_permission bit = NULL

IF @cell_number_value IS NULL
    BEGIN
    DELETE FROM dbo25.member_relations WHERE member_id = @id AND partner_id = @category_id
    RETURN
    END
ELSE IF @cell_number_value = 0
    BEGIN
    SET @update_permission = 0
    END
ELSE IF @cell_number_value = 1
    BEGIN
    SET @update_permission = 1
    END
ELSE
    BEGIN
    SET @message = dbo25.get_translated_string('Unknown option: %i (Use 0 or 1)', @data_language)
    RAISERROR (@message, 11, 0, @cell_number_value)
    END

UPDATE dbo25.member_relations
SET
    update_permission = @update_permission
WHERE
    member_id = @id
    AND partner_id = @category_id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.member_relations (member_id, partner_id, update_permission)
    VALUES (@id, @category_id, @update_permission)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects companies
-- =============================================

CREATE PROCEDURE [xls25].[usp_companies]
    @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    c.id
    , c.code
    , c.name
    , tr.name AS translated_name
    , c.default_language
    , c.sort_order
FROM
    dbo25.companies c
    INNER JOIN dbo25.users u ON u.company_id = c.id AND u.username = USER_NAME() AND u.is_active = 1
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = c.id AND tr.table_id = 2 AND tr.member_id = c.id AND tr.language = @data_language
ORDER BY
    c.sort_order
    , c.code

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_companies
-- =============================================

CREATE PROCEDURE [xls25].[usp_companies_delete]
    @id int
AS
BEGIN

SET NOCOUNT ON

IF NOT EXISTS (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @id AND username = USER_NAME() AND is_active = 1 AND is_admin = 1)
    RETURN

SET NOCOUNT OFF

DELETE dbo25.companies WHERE id = @id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_accounts
-- =============================================

CREATE PROCEDURE [xls25].[usp_companies_update]
      @id int
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @translated_name nvarchar(100) = NULL
    , @default_language varchar(10) = NULL
    , @sort_order int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int
DECLARE @current_name nvarchar(100)

SELECT TOP 1 @company_id = u.company_id, @current_name = t.name FROM dbo25.companies t INNER JOIN dbo25.users u ON u.company_id = t.id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

DECLARE @message nvarchar(max)

IF CHARINDEX(@code, ' ') > 0 OR CHARINDEX(@code, '''') > 0 OR CHARINDEX(@code, '"') > 0 OR CHARINDEX(@code, '[') > 0 OR CHARINDEX(@code, ']') > 0
    BEGIN
    SET @message = dbo25.get_translated_string('The code ''%s'' contains prohibited symbols (space, '', ", [ or ])', @data_language)
    RAISERROR (@message, 11, 0, @code)
    RETURN
    END

SET NOCOUNT OFF

UPDATE dbo25.companies
SET
    code = @code
    , name = @name
    , default_language = COALESCE(@default_language, 'en')
    , sort_order = @sort_order
FROM
    dbo25.companies c
WHERE
    c.id = @id
    AND NOT (
        COALESCE(code, '-1') = COALESCE(@code, '-2')
        AND COALESCE(name, '-1') = COALESCE(@name, '-2')
        AND COALESCE(default_language, '-1') = COALESCE(@default_language, '-2')
        AND COALESCE(sort_order, -1) = COALESCE(@sort_order, -2)
        )

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    BEGIN
    MERGE dbo25.translations t
    USING
        (
        SELECT company_id = @company_id, 2 AS table_id, @company_id AS member_id, @data_language AS language, @translated_name AS name
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects currency rates
-- =============================================

CREATE PROCEDURE [xls25].[usp_currency_rates]
    @company_id int = NULL
    , @base_currency_id tinyint = NULL
    , @quote_currency_id tinyint = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

EXECUTE AS CALLER

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

DECLARE @columns nvarchar(max)

SELECT @columns = STUFF((
        SELECT
            ', [' + CAST(m.code AS nvarchar) + ']'
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND m.dimension_id = 3
            AND m.excel_formula IS NULL
            AND m.calc_type_id IN (1, 0)      -- total, none
            AND m.id NOT IN (
                SELECT parent_id FROM dbo25.parents
            )
        ORDER BY
            m.sort_order
            , m.code
        FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(MAX)'), 1, 2, '')

-- PRINT @columns

DECLARE @sql nvarchar(max)

SET @sql = 'SELECT
    m.id
    , m.sort_order
    , CASE WHEN h.parent_id IS NOT NULL THEN 0 WHEN m.calc_type_id = 0 THEN 0 ELSE 1 END AS is_editable
    , m.decimal_places
    , CAST(m.is_percent AS tinyint) AS is_percent
    , m.row_color
    , m.row_bold
    , m.row_indent
    , m.code
    , CASE WHEN m.row_indent > 0 THEN REPLICATE('' '', m.row_indent) ELSE '''' END + ' + CASE WHEN @data_language IS NULL THEN 'm.name' ELSE 'COALESCE(tr.name, m.name)' END + ' AS name
    , ' + CASE WHEN @data_language IS NULL THEN 'c.name' ELSE 'COALESCE(tc.TRANSLATED_NAME, c.name)' END + ' AS calc_type
    , NULL AS data_format
    ' + COALESCE(', ' + @columns, '') + '
FROM
    dbo25.members m
    INNER JOIN dbo25.calc_types c ON c.id = m.calc_type_id
    LEFT OUTER JOIN (SELECT DISTINCT parent_id FROM dbo25.parents) h ON h.parent_id = m.id'
    + CASE WHEN @columns IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            t.id
            , c.code
            , r.currency_rate
        FROM
            dbo25.currency_rates r
            INNER JOIN dbo25.members t ON t.id = r.time_id
            INNER JOIN dbo25.members c ON c.id = r.category_id
        WHERE
            t.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
            AND t.dimension_id = 2
            AND c.dimension_id = 3
            AND r.base_currency_id = ' + COALESCE(CAST(@base_currency_id AS nvarchar), 'NULL') + '
            AND r.quote_currency_id = ' + COALESCE(CAST(@quote_currency_id AS nvarchar), 'NULL') + '
    ) s PIVOT (
        MAX(currency_rate) FOR code IN (' + @columns + ')
    ) p ON p.id = m.id' END
    + CASE WHEN @data_language IS NULL THEN '' ELSE '
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = ''' + @data_language + '''
    LEFT OUTER JOIN xls.translations tc ON tc.TABLE_SCHEMA = ''dbo25'' AND tc.TABLE_NAME = ''calc_types_name'' AND tc.COLUMN_NAME = c.name AND tc.LANGUAGE_NAME = ''' + @data_language + ''''
        END + '
WHERE
    m.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
    AND m.dimension_id = 2
    AND NOT m.calc_type_id IN (0, 9)
    AND h.parent_id IS NULL
ORDER BY
    m.sort_order
    , m.code
'

-- PRINT @sql

REVERT

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_rates
-- =============================================

CREATE PROCEDURE [xls25].[usp_currency_rates_change]
    @column_name nvarchar(128) = NULL
    , @cell_number_value float = NULL
    , @id int = NULL
    , @base_currency_id tinyint = NULL
    , @quote_currency_id tinyint = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

IF @base_currency_id IS NULL OR @quote_currency_id IS NULL OR @id IS NULL
    RETURN

-- Protecting non-data columns

IF @column_name IN ('id', 'sort_order', 'row_color', 'row_bold', 'row_indent', 'member', 'data_format')
    BEGIN
    SET @message = dbo25.get_translated_string('The column ''%s'' is not updatable', @data_language)
    RAISERROR (@message, 11, 0, @column_name)
    RETURN
    END

DECLARE @company_id int

EXECUTE AS CALLER

SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1)

REVERT

IF @company_id IS NULL
    RETURN

IF NOT EXISTS (SELECT id FROM dbo25.members WHERE id = @id)
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find member_id %i', @data_language)
    RAISERROR (@message, 11, 0, @id)
    RETURN
    END

DECLARE @category_code nvarchar(128) = @column_name
DECLARE @category_id int

SELECT @category_id = id FROM dbo25.members WHERE company_id = @company_id AND code = @category_code

IF @category_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find member_id %s', @data_language)
    RAISERROR (@message, 11, 0, @category_code)
    RETURN
    END

EXECUTE AS CALLER

IF @cell_number_value IS NULL
    BEGIN
    DELETE FROM dbo25.currency_rates
    WHERE
        base_currency_id = @base_currency_id
        AND quote_currency_id = @quote_currency_id
        AND category_id = @category_id
        AND time_id = @id
    END
ELSE
    BEGIN
    UPDATE dbo25.currency_rates
    SET
        currency_rate = @cell_number_value
    WHERE
        base_currency_id = @base_currency_id
        AND quote_currency_id = @quote_currency_id
        AND category_id = @category_id
        AND time_id = @id

    IF @@ROWCOUNT = 0
        INSERT INTO dbo25.currency_rates (base_currency_id, quote_currency_id, time_id, category_id, currency_rate)
        VALUES (@base_currency_id, @quote_currency_id, @id, @category_id, @cell_number_value)
    END

REVERT

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects data for management
-- =============================================

CREATE PROCEDURE [xls25].[usp_data_management]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

EXECUTE AS CALLER

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

DECLARE @columns nvarchar(max)

SELECT @columns = STUFF((
        SELECT
            ', [' + CAST(m.code AS nvarchar) + ']'
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND m.dimension_id = 3
            AND m.calc_type_id IN (1)       -- total
            AND m.id NOT IN (
                SELECT parent_id FROM dbo25.parents
            )
        ORDER BY
            m.sort_order
            , m.code
        FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(MAX)'), 1, 2, '')

DECLARE @select_columns nvarchar(max)

SELECT @select_columns = STUFF((
        SELECT
            ', CASE WHEN [' + CAST(m.code AS nvarchar) + '] = 0 THEN NULL ELSE [' + CAST(m.code AS nvarchar) + '] END AS [' + CAST(m.code AS nvarchar) + ']'
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND m.dimension_id = 3
            AND m.calc_type_id IN (1)       -- total
            AND m.id NOT IN (
                SELECT parent_id FROM dbo25.parents
            )
        ORDER BY
            m.sort_order
            , m.code
        FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(MAX)'), 1, 2, '')

-- PRINT @columns

DECLARE @sql nvarchar(max)

SET @sql = 'SELECT
    m.id
    , m.sort_order
    , CASE WHEN h.parent_id IS NOT NULL THEN 0 WHEN m.calc_type_id = 0 THEN 0 ELSE 1 END AS is_editable
    , m.decimal_places
    , CAST(m.is_percent AS tinyint) AS is_percent
    , m.row_color
    , m.row_bold
    , m.row_indent
    , m.code
    , CASE WHEN m.row_indent > 0 THEN REPLICATE('' '', m.row_indent) ELSE '''' END + ' + CASE WHEN @data_language IS NULL THEN 'm.name' ELSE 'COALESCE(tr.name, m.name)' END + ' AS name
    , ' + CASE WHEN @data_language IS NULL THEN 'c.name' ELSE 'COALESCE(tc.TRANSLATED_NAME, c.name)' END + ' AS calc_type
    , NULL AS data_format
    ' + COALESCE(', ' + @select_columns, '') + '
FROM
    dbo25.members m
    INNER JOIN dbo25.calc_types c ON c.id = m.calc_type_id
    LEFT OUTER JOIN (SELECT DISTINCT parent_id FROM dbo25.parents) h ON h.parent_id = m.id'
    + CASE WHEN @columns IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            f.id2
            , m3.code
            , f.id AS value
        FROM
            dbo25.facts f
            INNER JOIN dbo25.members m3 ON m3.id = f.id3
        WHERE
            m3.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
            AND f.value IS NOT NULL
    ) s PIVOT (
        COUNT(value) FOR code IN (' + @columns + ')
    ) p ON p.id2 = m.id' END
    + CASE WHEN @data_language IS NULL THEN '' ELSE '
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = ''' + @data_language + '''
    LEFT OUTER JOIN xls.translations tc ON tc.TABLE_SCHEMA = ''dbo25'' AND tc.TABLE_NAME = ''calc_types_name'' AND tc.COLUMN_NAME = c.name AND tc.LANGUAGE_NAME = ''' + @data_language + ''''
        END + '
WHERE
    m.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
    AND m.dimension_id = 2
    AND NOT m.calc_type_id IN (0, 9)
    AND h.parent_id IS NULL
ORDER BY
    m.sort_order
    , m.code
'

-- PRINT @sql

REVERT

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects dimensions
-- =============================================

CREATE PROCEDURE [xls25].[usp_dimensions]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    d.id
    , d.code
    , d.name
    , tr.name AS translated_name
    , d.parameter_name
    , tp.name AS translated_parameter_name
    , d.sort_order
    , d.name_format_id
    , d.root_member_id
    , d.default_member_id
    , CAST(d.is_protected AS tinyint) AS is_protected
    , CAST(d.default_select_permission AS tinyint) AS default_select_permission
    , CAST(d.default_update_permission AS tinyint) AS default_update_permission
    , CAST(d.is_active AS tinyint) AS is_active
FROM
    dbo25.dimension_properties d
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = d.company_id AND tr.table_id = 5 AND tr.member_id = d.id AND tr.language = @data_language
    LEFT OUTER JOIN dbo25.translations tp ON tp.company_id = d.company_id AND tp.table_id = 6 AND tp.member_id = d.id AND tp.language = @data_language
WHERE
    d.company_id = @company_id
ORDER BY
    d.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_dimensions
-- =============================================

CREATE PROCEDURE [xls25].[usp_dimensions_change]
    @column_name nvarchar(255) = NULL
    , @cell_value nvarchar(255) = NULL
    , @cell_number_value float = NULL
    , @cell_address nvarchar(255) = NULL
    , @company_id int = NULL
    , @id int = NULL
    , @root_member_id int = NULL
    , @default_member_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.dimension_properties t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.company_id = @company_id AND t.id = @id AND u.username = USER_NAME() AND u.is_active = 1)

IF @company_id IS NULL
    RETURN

DECLARE @message nvarchar(max)

IF @column_name IN ('root_member_id')
    BEGIN
    IF @root_member_id IS NOT NULL
        IF NOT (SELECT dimension_id FROM dbo25.members WHERE company_id = @company_id AND id = @root_member_id) = @id
            BEGIN
            DECLARE @code11 nvarchar(50) = (SELECT code FROM dbo25.members WHERE id = @root_member_id)
            DECLARE @code12 nvarchar(50) = (SELECT code FROM dbo25.dimension_properties WHERE company_id = @company_id AND id = @id)
            SET @message = dbo25.get_translated_string('Member ''%s'' is not a member of the ''%s'' dimension', @data_language)
            RAISERROR (@message, 11, 0, @code11, @code12)
            RETURN
            END
    END

ELSE IF @column_name IN ('default_member_id')
    BEGIN
    IF @default_member_id IS NOT NULL
        IF NOT (SELECT dimension_id FROM dbo25.members WHERE company_id = @company_id AND id = @default_member_id) = @id
            BEGIN
            DECLARE @code21 nvarchar(50) = (SELECT code FROM dbo25.members WHERE id = @default_member_id)
            DECLARE @code22 nvarchar(50) = (SELECT code FROM dbo25.dimension_properties WHERE company_id = @company_id AND id = @id)
            SET @message = dbo25.get_translated_string('Member ''%s'' is not a member of the ''%s'' dimension', @data_language)
            RAISERROR (@message, 11, 0, @code21, @code22)
            RETURN
            END
    END

ELSE IF @column_name IN ('is_active')
    BEGIN
    IF COALESCE(@cell_number_value, 0) = 1
        RETURN
    ELSE IF NOT COALESCE(@cell_number_value, 0) = 0
        BEGIN
        SET @message = dbo25.get_translated_string('Use values 1 and 0 to activate or deactivate the dimension', @data_language)
        RAISERROR (@message, 11, 0)
        RETURN
        END

    IF @id IN (1, 2, 3, 4)
        BEGIN
        SET @message = dbo25.get_translated_string('Do not change is_active for Accounts, Times, Categories, and Entities', @data_language)
        RAISERROR (@message, 11, 0)
        RETURN
        END
    ELSE IF @id = 5 AND (SELECT COUNT(DISTINCT id5) FROM dbo25.facts f INNER JOIN dbo25.members m ON m.id = f.id5 WHERE m.company_id = @company_id) <= 1
        RETURN
    ELSE IF @id = 6 AND (SELECT COUNT(DISTINCT id6) FROM dbo25.facts f INNER JOIN dbo25.members m ON m.id = f.id6 WHERE m.company_id = @company_id) <= 1
        RETURN
    ELSE IF @id = 7 AND (SELECT COUNT(DISTINCT id7) FROM dbo25.facts f INNER JOIN dbo25.members m ON m.id = f.id7 WHERE m.company_id = @company_id) <= 1
        RETURN

    DECLARE @code1 nvarchar(50) = (SELECT code FROM dbo25.dimension_properties WHERE company_id = @company_id AND id = @id)
    SET @message = dbo25.get_translated_string('The cube contains multiple members of the %s dimension.', @data_language) + CHAR(13) + CHAR(10) + dbo25.get_translated_string('Change all members to the default member before deactivating.', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_dimensions
-- =============================================

CREATE PROCEDURE [xls25].[usp_dimensions_delete]
    @id tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

SET @message = dbo25.get_translated_string('Do not delete dimensions', @data_language)

RAISERROR (@message, 11, 0)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_dimensions
-- =============================================

CREATE PROCEDURE [xls25].[usp_dimensions_insert]
    @company_id int = NULL
    , @id tinyint = NULL
    , @code nvarchar(50) = NULL
    , @name nvarchar(100) = NULL
    , @translated_name nvarchar(100) = NULL
    , @parameter_name nvarchar(50) = NULL
    , @translated_parameter_name nvarchar(50)
    , @sort_order tinyint = NULL
    , @name_format_id tinyint = NULL
    , @root_member_id int = NULL
    , @default_member_id int = NULL
    , @is_protected tinyint = NULL
    , @default_select_permission tinyint = NULL
    , @default_update_permission tinyint = NULL
    , @is_active tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

SET @message = dbo25.get_translated_string('Do not add dimensions', @data_language)

RAISERROR (@message, 11, 0)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_dimensions
-- =============================================

CREATE PROCEDURE [xls25].[usp_dimensions_update]
    @company_id int = NULL
    , @id tinyint = NULL
    , @code nvarchar(50) = NULL
    , @name nvarchar(100) = NULL
    , @translated_name nvarchar(100) = NULL
    , @parameter_name nvarchar(50) = NULL
    , @translated_parameter_name nvarchar(50)
    , @sort_order tinyint = NULL
    , @name_format_id tinyint = NULL
    , @root_member_id int = NULL
    , @default_member_id int = NULL
    , @is_protected tinyint = NULL
    , @default_select_permission tinyint = NULL
    , @default_update_permission tinyint = NULL
    , @is_active tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @current_name nvarchar(100)
DECLARE @current_parameter_name nvarchar(100)

SELECT TOP 1 @company_id = u.company_id, @current_name = t.name, @current_parameter_name = parameter_name FROM dbo25.dimension_properties t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.company_id = @company_id AND t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

DECLARE @message nvarchar(max)

IF @root_member_id IS NOT NULL AND NOT (SELECT dimension_id FROM dbo25.members WHERE company_id = @company_id AND id = @root_member_id) = @id
    BEGIN
    SET @message = dbo25.get_translated_string('The root member is not a member of the %s dimension', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

IF @default_member_id IS NOT NULL AND NOT (SELECT dimension_id FROM dbo25.members WHERE company_id = @company_id AND id = @default_member_id) = @id
    BEGIN
    SET @message = dbo25.get_translated_string('The default member is not a member of the %s dimension', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

UPDATE dbo25.dimension_properties
SET
    code = @code
    , name = @name
    , parameter_name = @parameter_name
    , sort_order = @sort_order
    , name_format_id = @name_format_id
    , root_member_id = @root_member_id
    , default_member_id = @default_member_id
    , is_protected = COALESCE(@is_protected, is_protected)
    , default_select_permission = @default_select_permission
    , default_update_permission = @default_update_permission
    , is_active = CASE WHEN id IN (1, 2, 3, 4) THEN is_active ELSE COALESCE(@is_active, is_active) END
WHERE
    company_id = @company_id
    AND id = @id
    AND NOT (
        COALESCE(code, '-1') = COALESCE(@code, '-2')
        AND COALESCE(name, '-1') = COALESCE(@name, '-2')
        AND COALESCE(parameter_name, '-1') = COALESCE(@parameter_name, '-2')
        AND COALESCE(sort_order, -1) = COALESCE(@sort_order, -2)
        AND COALESCE(name_format_id, -1) = COALESCE(@name_format_id, -2)
        AND COALESCE(root_member_id, -1) = COALESCE(@root_member_id, -2)
        AND COALESCE(default_member_id, -1) = COALESCE(@default_member_id, -2)
        AND COALESCE(is_protected, -1) = COALESCE(@is_protected, -2)
        AND COALESCE(default_select_permission, -1) = COALESCE(@default_select_permission, -2)
        AND COALESCE(default_update_permission, -1) = COALESCE(@default_update_permission, -2)
        AND COALESCE(is_active, -1) = COALESCE(@is_active, -2)
        )

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    BEGIN
    MERGE dbo25.translations t
    USING
        (
        SELECT company_id = @company_id, 5 AS table_id, @id AS member_id, @data_language AS language, @translated_name AS name
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    END

IF @data_language IS NOT NULL AND @translated_parameter_name IS NOT NULL
    BEGIN
    MERGE dbo25.translations t
    USING
        (
        SELECT company_id = @company_id, 5 AS table_id, @id AS member_id, @data_language AS language, @translated_name AS name
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects facts
-- =============================================

CREATE PROCEDURE [xls25].[usp_facts]
    @company_id int = NULL
    , @id1 int = NULL
    , @id2 int = NULL
    , @id3 int = NULL
    , @id4 int = NULL
    , @id5 int = NULL
    , @id6 int = NULL
    , @id7 int = NULL
    , @unit_id int = NULL
    , @calc_type_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    m1.company_id
    , f.id1
    , f.id2
    , f.id3
    , f.id4
    , f.id5
    , f.id6
    , f.id7
    , m1.sort_order AS sort_order1
    , m2.sort_order AS sort_order2
    , m3.sort_order AS sort_order3
    , m4.sort_order AS sort_order4
    , m5.sort_order AS sort_order5
    , m6.sort_order AS sort_order6
    , m7.sort_order AS sort_order7
    , m1.unit_id
    , m1.calc_type_id
    , c.code AS company_code
    , c.name AS company_name
    , m1.code AS code1
    , COALESCE(tr1.name, m1.name) AS name1
    , m2.code AS code2
    , COALESCE(tr2.name, m2.name) AS name2
    , m3.code AS code3
    , COALESCE(tr3.name, m3.name) AS name3
    , m4.code AS code4
    , COALESCE(tr4.name, m4.name) AS name4
    , m5.code AS code5
    , COALESCE(tr5.name, m5.name) AS name5
    , m6.code AS code6
    , COALESCE(tr6.name, m6.name) AS name6
    , m7.code AS code7
    , COALESCE(tr7.name, m7.name) AS name7
    , n.code AS unit_code
    , n.name AS unit_name
    , t.code AS calc_type_code
    , t.name AS calc_type_name
    , f.source_value
    , f.value
FROM
    dbo25.view_facts f

    INNER JOIN dbo25.members m1 ON m1.id = f.id1
    INNER JOIN dbo25.members m2 ON m2.id = f.id2
    INNER JOIN dbo25.members m3 ON m3.id = f.id3
    INNER JOIN dbo25.members m4 ON m4.id = f.id4
    INNER JOIN dbo25.members m5 ON m5.id = f.id5
    INNER JOIN dbo25.members m6 ON m6.id = f.id6
    INNER JOIN dbo25.members m7 ON m7.id = f.id7

    LEFT OUTER JOIN dbo25.units n ON n.id = m1.unit_id
    INNER JOIN dbo25.calc_types t ON t.id = m1.calc_type_id

    INNER JOIN dbo25.companies c ON c.id = m1.company_id

    LEFT OUTER JOIN dbo25.translations tr1 ON tr1.company_id = m1.company_id AND tr1.table_id = 1 AND tr1.member_id = m1.id AND tr1.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr2 ON tr2.company_id = m2.company_id AND tr2.table_id = 1 AND tr2.member_id = m2.id AND tr2.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr3 ON tr3.company_id = m3.company_id AND tr3.table_id = 1 AND tr3.member_id = m3.id AND tr3.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr4 ON tr4.company_id = m4.company_id AND tr4.table_id = 1 AND tr4.member_id = m4.id AND tr4.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr5 ON tr5.company_id = m5.company_id AND tr5.table_id = 1 AND tr5.member_id = m5.id AND tr5.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr6 ON tr6.company_id = m6.company_id AND tr6.table_id = 1 AND tr6.member_id = m6.id AND tr6.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr7 ON tr7.company_id = m7.company_id AND tr7.table_id = 1 AND tr7.member_id = m7.id AND tr7.language = @data_language

WHERE
    f.company_id = @company_id
    AND f.id1 = COALESCE(@id1, f.id1)
    AND f.id2 = COALESCE(@id2, f.id2)
    AND f.id3 = COALESCE(@id3, f.id3)
    AND f.id4 = COALESCE(@id4, f.id4)
    AND f.id5 = COALESCE(@id5, f.id5)
    AND f.id6 = COALESCE(@id6, f.id6)
    AND f.id7 = COALESCE(@id7, f.id7)
    AND f.calc_type_id = COALESCE(@calc_type_id, f.calc_type_id)
    AND COALESCE(f.unit_id, -1) = COALESCE(@unit_id, f.unit_id, -1)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects source facts
-- =============================================

CREATE PROCEDURE [xls25].[usp_facts_data]
    @company_id int = NULL
    , @id1 int = NULL
    , @id2 int = NULL
    , @id3 int = NULL
    , @id4 int = NULL
    , @id5 int = NULL
    , @id6 int = NULL
    , @id7 int = NULL
    , @unit_id int = NULL
    , @calc_type_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    m1.company_id
    , f.id1
    , f.id2
    , f.id3
    , f.id4
    , f.id5
    , f.id6
    , f.id7
    , m1.sort_order AS sort_order1
    , m2.sort_order AS sort_order2
    , m3.sort_order AS sort_order3
    , m4.sort_order AS sort_order4
    , m5.sort_order AS sort_order5
    , m6.sort_order AS sort_order6
    , m7.sort_order AS sort_order7
    , m1.unit_id
    , m1.calc_type_id
    , c.code AS company_code
    , c.name AS company_name
    , m1.code AS code1
    , COALESCE(tr1.name, m1.name) AS name1
    , m2.code AS code2
    , COALESCE(tr2.name, m2.name) AS name2
    , m3.code AS code3
    , COALESCE(tr3.name, m3.name) AS name3
    , m4.code AS code4
    , COALESCE(tr4.name, m4.name) AS name4
    , m5.code AS code5
    , COALESCE(tr5.name, m5.name) AS name5
    , m6.code AS code6
    , COALESCE(tr6.name, m6.name) AS name6
    , m7.code AS code7
    , COALESCE(tr7.name, m7.name) AS name7
    , n.code AS unit_code
    , n.name AS unit_name
    , t.code AS calc_type_code
    , t.name AS calc_type_name
    , f.source_value
    , f.value
FROM
    dbo25.view_facts_data f

    INNER JOIN dbo25.members m1 ON m1.id = f.id1
    INNER JOIN dbo25.members m2 ON m2.id = f.id2
    INNER JOIN dbo25.members m3 ON m3.id = f.id3
    INNER JOIN dbo25.members m4 ON m4.id = f.id4
    INNER JOIN dbo25.members m5 ON m5.id = f.id5
    INNER JOIN dbo25.members m6 ON m6.id = f.id6
    INNER JOIN dbo25.members m7 ON m7.id = f.id7

    LEFT OUTER JOIN dbo25.units n ON n.id = m1.unit_id
    INNER JOIN dbo25.calc_types t ON t.id = m1.calc_type_id

    INNER JOIN dbo25.companies c ON c.id = m1.company_id

    LEFT OUTER JOIN dbo25.translations tr1 ON tr1.company_id = m1.company_id AND tr1.table_id = 1 AND tr1.member_id = m1.id AND tr1.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr2 ON tr2.company_id = m2.company_id AND tr2.table_id = 1 AND tr2.member_id = m2.id AND tr2.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr3 ON tr3.company_id = m3.company_id AND tr3.table_id = 1 AND tr3.member_id = m3.id AND tr3.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr4 ON tr4.company_id = m4.company_id AND tr4.table_id = 1 AND tr4.member_id = m4.id AND tr4.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr5 ON tr5.company_id = m5.company_id AND tr5.table_id = 1 AND tr5.member_id = m5.id AND tr5.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr6 ON tr6.company_id = m6.company_id AND tr6.table_id = 1 AND tr6.member_id = m6.id AND tr6.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr7 ON tr7.company_id = m7.company_id AND tr7.table_id = 1 AND tr7.member_id = m7.id AND tr7.language = @data_language
WHERE
    f.company_id = @company_id
    AND f.id1 = COALESCE(@id1, f.id1)
    AND f.id2 = COALESCE(@id2, f.id2)
    AND f.id3 = COALESCE(@id3, f.id3)
    AND f.id4 = COALESCE(@id4, f.id4)
    AND f.id5 = COALESCE(@id5, f.id5)
    AND f.id6 = COALESCE(@id6, f.id6)
    AND f.id7 = COALESCE(@id7, f.id7)
    AND f.calc_type_id = COALESCE(@calc_type_id, f.calc_type_id)
    AND COALESCE(f.unit_id, -1) = COALESCE(@unit_id, f.unit_id, -1)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects source cube strings
-- =============================================

CREATE PROCEDURE [xls25].[usp_facts_strings]
    @company_id int = NULL
    , @id1 int = NULL
    , @id2 int = NULL
    , @id3 int = NULL
    , @id4 int = NULL
    , @id5 int = NULL
    , @id6 int = NULL
    , @id7 int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    m1.company_id
    , f.id1
    , f.id2
    , f.id3
    , f.id4
    , f.id5
    , f.id6
    , f.id7
    , m1.sort_order AS sort_order1
    , m2.sort_order AS sort_order2
    , m3.sort_order AS sort_order3
    , m4.sort_order AS sort_order4
    , m5.sort_order AS sort_order5
    , m6.sort_order AS sort_order6
    , m7.sort_order AS sort_order7
    , m1.unit_id
    , m1.calc_type_id
    , c.code AS company_code
    , c.name AS company_name
    , m1.code AS code1
    , COALESCE(tr1.name, m1.name) AS name1
    , m2.code AS code2
    , COALESCE(tr2.name, m2.name) AS name2
    , m3.code AS code3
    , COALESCE(tr3.name, m3.name) AS name3
    , m4.code AS code4
    , COALESCE(tr4.name, m4.name) AS name4
    , m5.code AS code5
    , COALESCE(tr5.name, m5.name) AS name5
    , m6.code AS code6
    , COALESCE(tr6.name, m6.name) AS name6
    , m7.code AS code7
    , COALESCE(tr7.name, m7.name) AS name7
    , n.code AS unit_code
    , n.name AS unit_name
    , t.code AS calc_type_code
    , t.name AS calc_type_name
    , f.value
FROM
    dbo25.strings f

    INNER JOIN dbo25.members m1 ON m1.id = f.id1
    INNER JOIN dbo25.members m2 ON m2.id = f.id2
    INNER JOIN dbo25.members m3 ON m3.id = f.id3
    INNER JOIN dbo25.members m4 ON m4.id = f.id4
    INNER JOIN dbo25.members m5 ON m5.id = f.id5
    INNER JOIN dbo25.members m6 ON m6.id = f.id6
    INNER JOIN dbo25.members m7 ON m7.id = f.id7

    LEFT OUTER JOIN dbo25.units n ON n.id = m1.unit_id
    INNER JOIN dbo25.calc_types t ON t.id = m1.calc_type_id

    INNER JOIN dbo25.users u ON u.company_id = m1.company_id AND u.username = USER_NAME() AND u.is_active = 1
    INNER JOIN dbo25.companies c ON c.id = m1.company_id

    LEFT OUTER JOIN dbo25.translations tr1 ON tr1.company_id = m1.company_id AND tr1.table_id = 1 AND tr1.member_id = m1.id AND tr1.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr2 ON tr2.company_id = m2.company_id AND tr2.table_id = 1 AND tr2.member_id = m2.id AND tr2.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr3 ON tr3.company_id = m3.company_id AND tr3.table_id = 1 AND tr3.member_id = m3.id AND tr3.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr4 ON tr4.company_id = m4.company_id AND tr4.table_id = 1 AND tr4.member_id = m4.id AND tr4.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr5 ON tr5.company_id = m5.company_id AND tr5.table_id = 1 AND tr5.member_id = m5.id AND tr5.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr6 ON tr6.company_id = m6.company_id AND tr6.table_id = 1 AND tr6.member_id = m6.id AND tr6.language = @data_language
    LEFT OUTER JOIN dbo25.translations tr7 ON tr7.company_id = m7.company_id AND tr7.table_id = 1 AND tr7.member_id = m7.id AND tr7.language = @data_language
WHERE
    m1.company_id = @company_id
    AND f.id1 = COALESCE(@id1, f.id1)
    AND f.id2 = COALESCE(@id2, f.id2)
    AND f.id3 = COALESCE(@id3, f.id3)
    AND f.id4 = COALESCE(@id4, f.id4)
    AND f.id5 = COALESCE(@id5, f.id5)
    AND f.id6 = COALESCE(@id6, f.id6)
    AND f.id7 = COALESCE(@id7, f.id7)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects form dimensions and properties
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_dimensions]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    f.id AS form_id
    , d.id AS dimension_id
    , COALESCE(tr.name, f.name) AS form_name
    , COALESCE(td.name, d.name) AS dimension_name
    , a.id AS axis_type_id
    , fd.parameter_index
    , fd.header_order
    , fd.root_member_id
    , fd.parameter_start_level
    , fd.parameter_end_level
    , fd.form_start_level
    , fd.form_end_level
FROM
    dbo25.forms f
    INNER JOIN dbo25.dimension_properties d ON d.company_id = f.company_id
    LEFT OUTER JOIN dbo25.form_dimensions fd ON fd.form_id = f.id AND fd.dimension_id = d.id
    LEFT OUTER JOIN dbo25.axis_types a ON a.id = fd.axis_type_id
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = d.company_id AND tr.table_id = 7 AND tr.member_id = f.id AND tr.language = @data_language
    LEFT OUTER JOIN dbo25.translations td ON td.company_id = d.company_id AND td.table_id = 5 AND td.member_id = d.id AND td.language = @data_language
WHERE
    f.company_id = @company_id
    AND d.is_active = 1
ORDER BY
    f.sort_order
    , f.id
    , d.sort_order
    , d.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_form_dimensions
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_dimensions_delete]
    @form_id int = NULL
    , @dimension_id tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = t.company_id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @form_id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET NOCOUNT OFF

DELETE dbo25.form_dimensions
WHERE
    form_id = @form_id
    AND dimension_id = @dimension_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_form_dimensions
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_dimensions_insert]
      @form_id int = NULL
    , @dimension_id tinyint = NULL
    , @axis_type_id tinyint = NULL
    , @parameter_index tinyint = NULL
    , @header_order tinyint = NULL
    , @root_member_id int = NULL
    , @parameter_start_level tinyint = NULL
    , @parameter_end_level tinyint = NULL
    , @form_start_level tinyint = NULL
    , @form_end_level tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

IF @form_id IS NULL OR @dimension_id IS NULL
    RETURN

DECLARE @company_id int

SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @form_id AND u.username = USER_NAME() AND u.is_active = 1)

IF @company_id IS NULL
    RETURN

SET NOCOUNT OFF

UPDATE dbo25.form_dimensions
SET
    axis_type_id = @axis_type_id
    , parameter_index = @parameter_index
    , header_order = @header_order
    , root_member_id = @root_member_id
    , parameter_start_level = @parameter_start_level
    , parameter_end_level = @parameter_end_level
    , form_start_level = @form_start_level
    , form_end_level = @form_end_level
WHERE
    form_id = @form_id
    AND dimension_id = @dimension_id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.form_dimensions
        (form_id, dimension_id, axis_type_id, parameter_index, header_order, root_member_id, parameter_start_level, parameter_end_level, form_start_level, form_end_level)
    VALUES
        (@form_id, @dimension_id, @axis_type_id, @parameter_index, @header_order, @root_member_id, @parameter_start_level, @parameter_end_level, @form_start_level, @form_end_level)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_form_dimensions
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_dimensions_update]
      @form_id int = NULL
    , @dimension_id tinyint = NULL
    , @axis_type_id tinyint = NULL
    , @parameter_index tinyint = NULL
    , @header_order tinyint = NULL
    , @root_member_id int = NULL
    , @parameter_start_level tinyint = NULL
    , @parameter_end_level tinyint = NULL
    , @form_start_level tinyint = NULL
    , @form_end_level tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

EXEC xls25.usp_form_dimensions_insert @form_id, @dimension_id, @axis_type_id, @parameter_index, @header_order, @root_member_id, @parameter_start_level, @parameter_end_level, @form_start_level, @form_end_level

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects form permissions
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_permissions]
    @company_id int = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

EXECUTE AS CALLER

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

DECLARE @users nvarchar(max)

SELECT @users = STUFF((
        SELECT
            ', [' + CAST(u.username AS nvarchar) + ']'
        FROM
            dbo25.users u
        WHERE
            u.company_id = @company_id
            AND u.is_active = 1
            AND u.is_admin = 0
        ORDER BY
            u.username
        FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(MAX)'), 1, 2, '')

-- PRINT @users

IF @users IS NULL
    BEGIN
    SELECT
        f.id
        , f.sort_order
        , f.code
        , f.name
        , CAST(NULL AS tinyint) AS data_format
    FROM
        dbo25.forms f
    WHERE
        f.company_id = @company_id
    ORDER BY
        f.sort_order
        , f.code
    RETURN
    END

DECLARE @sql nvarchar(max)

SET @sql = 'SELECT
    *
FROM
    (
        SELECT
            p.name AS username
            , f.id AS id
            , f.sort_order
            , f.code
            , f.name
            , CAST(NULL AS tinyint) AS data_format
            , CAST(fp.select_permission AS tinyint) AS permission
        FROM
            sys.database_principals p
            INNER JOIN dbo25.users u ON u.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + ' AND u.username = p.name
            CROSS JOIN dbo25.forms f
            LEFT OUTER JOIN dbo25.form_permissions fp ON fp.form_id = f.id AND fp.username = p.name
        WHERE
            f.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
    ) s PIVOT (
        MAX(permission) FOR username IN (' + @users + ')
    ) p
ORDER BY
    p.sort_order
    , p.code
'

-- PRINT @sql

REVERT

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_form_permissions
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_permissions_change]
    @column_name nvarchar(128) = NULL
    , @cell_number_value int = NULL
    , @id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1)

IF @company_id IS NULL
    RETURN

DECLARE @message nvarchar(max)

-- Protecting non-data columns

IF @column_name IN ('id', 'sort_order', 'code', 'name', 'data_format')
    BEGIN
    SET @message = dbo25.get_translated_string('The column ''%s'' is not updatable', @data_language)
    RAISERROR (@message, 11, 0, @column_name)
    RETURN
    END

IF NOT EXISTS (SELECT id FROM dbo25.forms WHERE id = @id)
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find form id %s', @data_language)
    RAISERROR (@message, 11, 0, @id)
    RETURN
    END

IF @cell_number_value IS NULL
    BEGIN
    DELETE FROM dbo25.form_permissions WHERE form_id = @id AND username = @column_name
    RETURN
    END

DECLARE @username nvarchar(128) = @column_name

IF NOT EXISTS (SELECT u.username FROM dbo25.users u WHERE u.company_id = @company_id AND u.username = @username)
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find the principal ''%s'', because it does not exist or you do not have permission.', @data_language)
    RAISERROR (@message, 11, 0, @username)
    RETURN
    END

-- Getting permissions from the cell value

DECLARE @select_permission bit = NULL

IF @cell_number_value = 0
    BEGIN
    SET @select_permission = 0
    END
ELSE IF @cell_number_value = 1
    BEGIN
    SET @select_permission = 1
    END
ELSE
    BEGIN
    SET @message = dbo25.get_translated_string('Unknown option: %i (Use 0 or 1)', @data_language)
    RAISERROR (@message, 11, 0, @cell_number_value)
    RETURN
    END

UPDATE dbo25.form_permissions
SET
    select_permission = @select_permission
WHERE
    form_id = @id
    AND username = @column_name

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.form_permissions (form_id, username, select_permission)
    VALUES (@id, @username, @select_permission)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects form rows for the specified rowset
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_rows]
    @company_id int = NULL
    , @rowset_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    r.id
    , r.sort_order
    , CASE WHEN m.id IS NULL THEN 0 WHEN h.parent_id IS NOT NULL THEN 0 WHEN m.calc_type_id = 0 THEN 0 ELSE 1 END AS is_editable
    , r.decimal_places
    , CAST(r.is_percent AS tinyint) AS is_percent
    , r.row_color
    , r.row_bold
    , r.row_indent
    , r.code
    , CASE WHEN r.row_indent > 0 THEN REPLICATE(' ', r.row_indent) ELSE '' END + r.name AS name
    , CASE WHEN r.row_indent > 0 THEN REPLICATE(' ', r.row_indent) ELSE '' END + tr.name AS translated_name
    , r.member_id
FROM
    dbo25.form_rows r
    INNER JOIN dbo25.members rm ON rm.id = r.rowset_id
    LEFT OUTER JOIN dbo25.members m ON r.member_id = m.id
    LEFT OUTER JOIN (SELECT DISTINCT parent_id FROM dbo25.parents) h ON h.parent_id = m.id
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = rm.company_id AND tr.table_id = 8 AND tr.member_id = r.id AND tr.language = @data_language
WHERE
    rm.company_id = @company_id
    AND r.rowset_id = @rowset_id
ORDER BY
    r.sort_order
    , m.dimension_id
    , r.code
    , r.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_form_rows
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_rows_delete]
    @id int = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = t.company_id FROM dbo25.form_rows f INNER JOIN dbo25.members t ON t.id = f.rowset_id INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE f.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET NOCOUNT OFF

DELETE FROM dbo25.form_rows
WHERE
    id = @id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_form_rows
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_rows_insert]
    @rowset_id int = NULL
    , @rownum int = NULL
    , @member_id int = NULL
    , @code nvarchar(50) = NULL
    , @name nvarchar(400) = NULL
    , @translated_name nvarchar(400) = NULL
    , @decimal_places tinyint = NULL
    , @is_percent tinyint = NULL
    , @row_color tinyint = NULL
    , @row_bold tinyint = NULL
    , @row_indent tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @rowset_id AND u.username = USER_NAME() AND u.is_active = 1)

IF @company_id IS NULL
    RETURN

IF @member_id IS NOT NULL
    BEGIN
    DECLARE @company_id2 int

    SET @company_id2 = (SELECT TOP 1 u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @member_id AND u.username = USER_NAME() AND u.is_active = 1)

    IF @company_id2 IS NULL OR NOT @company_id2 = @company_id
        RETURN
    END

SET NOCOUNT OFF

INSERT INTO dbo25.form_rows
    (rowset_id, sort_order, member_id, code, name, decimal_places, is_percent, row_color, row_bold, row_indent)
VALUES
    (@rowset_id, @rownum, @member_id, LTRIM(RTRIM(@code)), LTRIM(RTRIM(@name)), @decimal_places, @is_percent, @row_color, @row_bold, @row_indent)

DECLARE @id int = SCOPE_IDENTITY()

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    INSERT INTO dbo25.translations
        (company_id, table_id, member_id, language, name)
    VALUES
        (@company_id, 8, @id, @data_language, @translated_name)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_form_rows
-- =============================================

CREATE PROCEDURE [xls25].[usp_form_rows_update]
    @id int = NULL
    , @rowset_id int = NULL
    , @rownum int = NULL
    , @member_id int = NULL
    , @code nvarchar(50) = NULL
    , @name nvarchar(400) = NULL
    , @translated_name nvarchar(400) = NULL
    , @decimal_places tinyint = NULL
    , @is_percent tinyint = NULL
    , @row_color tinyint = NULL
    , @row_bold tinyint = NULL
    , @row_indent tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int
DECLARE @current_name nvarchar(100)

SELECT TOP 1 @company_id = u.company_id, @current_name = t.name FROM dbo25.form_rows t INNER JOIN dbo25.members m ON m.id = t.rowset_id INNER JOIN dbo25.users u ON u.company_id = m.company_id WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

DECLARE @company_id1 int

SET @company_id1 = (SELECT TOP 1 u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @rowset_id AND u.username = USER_NAME() AND u.is_active = 1)

IF @company_id1 IS NULL OR NOT @company_id1 = @company_id
    RETURN

IF @member_id IS NOT NULL
    BEGIN
    DECLARE @company_id2 int

    SET @company_id2 = (SELECT TOP 1 u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @member_id AND u.username = USER_NAME() AND u.is_active = 1)

    IF @company_id2 IS NULL OR NOT @company_id2 = @company_id
        RETURN
    END

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

SET NOCOUNT OFF

UPDATE dbo25.form_rows
SET
    rowset_id = @rowset_id
    , sort_order = @rownum
    , member_id = @member_id
    , code = @code
    , name = @name
    , decimal_places = @decimal_places
    , is_percent = @is_percent
    , row_color = @row_color
    , row_bold = @row_bold
    , row_indent = @row_indent
WHERE
    id = @id
    AND NOT (
        COALESCE(code, '-1') = COALESCE(@code, '-2')
        AND COALESCE(name, '-1') = COALESCE(@name, '-2')
        AND COALESCE(sort_order, -1) = COALESCE(@rownum, -2)
        AND COALESCE(rowset_id, -1) = COALESCE(@rowset_id, -2)
        AND COALESCE(sort_order, -1) = COALESCE(@rownum, -2)
        AND COALESCE(member_id, -1) = COALESCE(@member_id, -2)
        AND COALESCE(decimal_places, -1) = COALESCE(@decimal_places, -2)
        AND COALESCE(is_percent, -1) = COALESCE(@is_percent, -2)
        AND COALESCE(row_color, -1) = COALESCE(@row_color, -2)
        AND COALESCE(row_bold, -1) = COALESCE(@row_bold, -2)
        AND COALESCE(row_indent, -1) = COALESCE(@row_indent, -2)
        )

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    BEGIN
    MERGE dbo25.translations t
    USING
        (
        SELECT company_id = @company_id, 8 AS table_id, @id AS member_id, @data_language AS language, @translated_name AS name
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects forms
-- =============================================

CREATE PROCEDURE [xls25].[usp_forms]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    f.id
    , f.code
    , f.name
    , tr.name AS translated_name
    , f.sort_order
    , CAST(is_active AS tinyint) AS is_active
FROM
    dbo25.forms f
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = f.company_id AND tr.table_id = 7 AND tr.member_id = f.id AND tr.language = @data_language
WHERE
    f.company_id = @company_id
ORDER BY
    f.sort_order
    , f.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_forms
-- =============================================

CREATE PROCEDURE [xls25].[usp_forms_delete]
    @id int
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = t.company_id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET NOCOUNT OFF

DELETE dbo25.forms WHERE id = @id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_forms
-- =============================================

CREATE PROCEDURE [xls25].[usp_forms_insert]
    @company_id int = NULL
    , @code nvarchar(50) = NULL
    , @name nvarchar(100) = NULL
    , @translated_name nvarchar(100) = NULL
    , @sort_order int = NULL
    , @is_active tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))
SET @is_active = COALESCE(@is_active, 1)

INSERT INTO dbo25.forms (company_id, code, name, sort_order, is_active) VALUES (@company_id, @code, @name, @sort_order, @is_active)

DECLARE @id int = SCOPE_IDENTITY()

SET NOCOUNT OFF

INSERT dbo25.form_dimensions
    (form_id, dimension_id, axis_type_id, parameter_index, header_order, root_member_id, parameter_start_level, parameter_end_level, form_start_level, form_end_level)
SELECT
    @id
    , d.id
    , CASE d.id WHEN 1 THEN 2 WHEN 2 THEN 3 ELSE 1 END AS axis_type_id
    , CASE d.id WHEN 1 THEN 1 WHEN 2 THEN 4 WHEN 3 THEN 3 WHEN 4 THEN 2 ELSE d.id END AS parameter_index
    , CASE d.id WHEN 1 THEN 1 WHEN 2 THEN 1 ELSE NULL END header_order
    , d.root_member_id
    , 1 AS parameter_start_level
    , CASE d.id WHEN 1 THEN 1 ELSE NULL END AS parameter_end_level
    , NULL AS form_start_level
    , NULL AS form_end_level
FROM
    dbo25.dimension_properties d
WHERE
    d.company_id = @company_id

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    INSERT INTO dbo25.translations
        (company_id, table_id, member_id, language, name)
    VALUES
        (@company_id, 7, @id, @data_language, @translated_name)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_forms
-- =============================================

CREATE PROCEDURE [xls25].[usp_forms_update]
      @id int
    , @code nvarchar(50) = NULL
    , @name nvarchar(100) = NULL
    , @translated_name nvarchar(100) = NULL
    , @sort_order int = NULL
    , @is_active tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int
DECLARE @current_name nvarchar(100)

SELECT TOP 1 @company_id = u.company_id, @current_name = t.name FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

SET NOCOUNT OFF

UPDATE dbo25.forms
SET
    code = @code
    , name = @name
    , sort_order = @sort_order
    , is_active = @is_active
WHERE
    id = @id
    AND NOT (
        COALESCE(code, '-1') = COALESCE(@code, '-2')
        AND COALESCE(name, '-1') = COALESCE(@name, '-2')
        AND COALESCE(sort_order, -1) = COALESCE(@sort_order, -2)
        AND COALESCE(is_active, -1) = COALESCE(@is_active, -2)
        )

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    BEGIN
    MERGE dbo25.translations t
    USING
        (
        SELECT company_id = @company_id, 7 AS table_id, @id AS member_id, @data_language AS language, @translated_name AS name
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects member permissions
-- =============================================

CREATE PROCEDURE [xls25].[usp_member_permissions]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
    , @root_id int = NULL
    , @username nvarchar(128) = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

EXECUTE AS CALLER

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

DECLARE @columns nvarchar(max)

SELECT @columns = STUFF((
        SELECT
            ', [' + CAST(u.username AS nvarchar) + ']'
        FROM
            dbo25.users u
        WHERE
            u.company_id = @company_id
            AND u.is_active = 1
            AND u.is_admin = 0
        ORDER BY
            u.username
        FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(MAX)'), 1, 2, '')

-- PRINT @columns

DECLARE @sql nvarchar(max)

SET @sql = 'SELECT
    m.id
    , m.dimension_id
    , m.sort_order
    , CASE WHEN h.parent_id IS NOT NULL THEN 0 WHEN m.calc_type_id = 0 THEN 0 ELSE 1 END AS is_editable
    , m.decimal_places
    , CAST(m.is_percent AS tinyint) AS is_percent
    , m.row_color
    , m.row_bold
    , m.row_indent
    , m.code
    , CASE WHEN m.row_indent > 0 THEN REPLICATE('' '', m.row_indent) ELSE '''' END + ' + CASE WHEN @data_language IS NULL THEN 'm.name' ELSE 'COALESCE(tr.name, m.name)' END + ' AS name
    , ' + CASE WHEN @data_language IS NULL THEN 'c.name' ELSE 'COALESCE(tc.TRANSLATED_NAME, c.name)' END + ' AS calc_type
    , NULL AS data_format'
    + CASE WHEN @columns IS NULL THEN '' ELSE '
    , ' + @columns END + '
FROM
    dbo25.members m
    INNER JOIN dbo25.calc_types c ON c.id = m.calc_type_id
    LEFT OUTER JOIN (SELECT DISTINCT parent_id FROM dbo25.parents) h ON h.parent_id = m.id'
    + CASE WHEN @columns IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            mp.member_id
            , mp.username
            , mp.permission AS value
        FROM
            dbo25.member_permissions mp
    ) s PIVOT (
        MAX(value) FOR username IN (' + @columns + ')
    ) p ON p.member_id = m.id' END
    + CASE WHEN @data_language IS NULL THEN '' ELSE '
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = ''' + @data_language + '''
    LEFT OUTER JOIN xls.translations tc ON tc.TABLE_SCHEMA = ''dbo25'' AND tc.TABLE_NAME = ''calc_types_name'' AND tc.COLUMN_NAME = c.name AND tc.LANGUAGE_NAME = ''' + @data_language + ''''
        END + '
WHERE
    m.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
    AND m.calc_type_id IN (1, 2, 3, 4)'
    + CASE WHEN @dimension_id IS NULL THEN '' ELSE '
    AND m.dimension_id = ' + CAST(@dimension_id AS nvarchar) END
    + CASE WHEN @root_id IS NULL THEN '' ELSE '
    AND m.id IN (
        SELECT
            h.member_id
        FROM
            dbo25.hierarchies h
        WHERE
            h.parent_id = ' + CAST(@root_id AS nvarchar) + '
        )' END + '
ORDER BY
    m.dimension_id
    , m.sort_order
    , m.code
'

-- PRINT @sql

REVERT

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects rows to set member relations
-- =============================================

CREATE PROCEDURE [xls25].[usp_member_relations]
    @company_id int = NULL
    , @dimension_id int = NULL
    , @root_id int = NULL
    , @partner_id int = NULL
    , @field nvarchar(50) = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

EXECUTE AS CALLER

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

DECLARE @columns nvarchar(max)

SELECT @columns = STUFF((
        SELECT
            ', [' + CAST(m.code AS nvarchar) + ']'
        FROM
            dbo25.members m
            LEFT OUTER JOIN (SELECT DISTINCT parent_id FROM dbo25.parents) h ON h.parent_id = m.id
        WHERE
            m.company_id = @company_id
            AND m.dimension_id = @partner_id
            AND m.excel_formula IS NULL
            AND m.calc_type_id IN (1)       -- total
            AND h.parent_id IS NULL
        ORDER BY
            m.sort_order
            , m.code
        FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(MAX)'), 1, 2, '')

-- PRINT @columns

DECLARE @sql nvarchar(max)

SET @sql = 'SELECT
    m.id
    , m.dimension_id
    , m.sort_order
    , CASE WHEN h.parent_id IS NOT NULL THEN 0 WHEN m.calc_type_id = 0 THEN 0 ELSE 1 END AS is_editable
    , m.decimal_places
    , CAST(m.is_percent AS tinyint) AS is_percent
    , m.row_color
    , m.row_bold
    , m.row_indent
    , m.code
    , CASE WHEN m.row_indent > 0 THEN REPLICATE('' '', m.row_indent) ELSE '''' END + ' + CASE WHEN @data_language IS NULL THEN 'm.name' ELSE 'COALESCE(tr.name, m.name)' END + ' AS name
    , ' + CASE WHEN @data_language IS NULL THEN 'c.name' ELSE 'COALESCE(tc.TRANSLATED_NAME, c.name)' END + ' AS calc_type
    , NULL AS data_format'
    + CASE WHEN @columns IS NULL THEN '' ELSE '
    , ' + @columns END + '
FROM
    dbo25.members m
    INNER JOIN dbo25.calc_types c ON c.id = m.calc_type_id
    LEFT OUTER JOIN (SELECT DISTINCT parent_id FROM dbo25.parents) h ON h.parent_id = m.id'
    + CASE WHEN @columns IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            m.id
            , p.code
            , ' + CASE @field WHEN 'update_permission' THEN 'CAST(mp.update_permission AS tinyint)'
                    ELSE 'CAST(NULL AS tinyint)' END + ' AS value
        FROM
            dbo25.member_relations mp
            INNER JOIN dbo25.members m ON m.id = mp.member_id
            INNER JOIN dbo25.members p ON p.id = mp.partner_id
        WHERE
            m.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
            AND m.dimension_id = ' + COALESCE(CAST(@dimension_id AS nvarchar), 'NULL') + '
            AND p.dimension_id = ' + COALESCE(CAST(@partner_id AS nvarchar), 'NULL') + '
    ) s PIVOT (
        MAX(value) FOR code IN (' + @columns + ')
    ) p ON p.id = m.id' END
    + CASE WHEN @data_language IS NULL THEN '' ELSE '
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = ''' + @data_language + '''
    LEFT OUTER JOIN xls.translations tc ON tc.TABLE_SCHEMA = ''dbo25'' AND tc.TABLE_NAME = ''calc_types_name'' AND tc.COLUMN_NAME = c.name AND tc.LANGUAGE_NAME = ''' + @data_language + ''''
        END + '
WHERE
    m.company_id = ' + COALESCE(CAST(@company_id AS nvarchar), 'NULL') + '
    AND m.calc_type_id IN (0, 1, 2, 3, 4, 11)
    AND COALESCE(m.show_line, 1) = 1'
    + CASE WHEN @dimension_id IS NULL THEN '' ELSE '
    AND m.dimension_id = ' + CAST(@dimension_id AS nvarchar) END
    + CASE WHEN @root_id IS NULL THEN '' ELSE '
    AND m.id IN (
        SELECT
            h.member_id
        FROM
            dbo25.hierarchies h
        WHERE
            h.parent_id = ' + CAST(@root_id AS nvarchar) + '
        )' END + '
ORDER BY
    m.sort_order
    , m.code
'

-- PRINT @sql

REVERT

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_member_relations
-- =============================================

CREATE PROCEDURE [xls25].[usp_member_relations_change]
    @column_name nvarchar(128) = NULL
    , @cell_number_value int = NULL
    , @id int = NULL
    , @field nvarchar(50) = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

IF @field IS NULL
    RETURN
ELSE IF NOT @field = 'update_permission'
    RETURN

IF @column_name IN ('id', 'sort_order', 'decimal_places', 'is_percent', 'row_color', 'row_bold', 'row_indent', 'member', 'data_format')
    BEGIN
    SET @message = dbo25.get_translated_string('The column ''%s'' is not updatable', @data_language)
    RAISERROR (@message, 11, 0, @column_name)
    RETURN
    END

IF NOT EXISTS (SELECT id FROM dbo25.members WHERE id = @id)
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find member id %i', @data_language)
    RAISERROR (@message, 11, 0, @id)
    RETURN
    END

DECLARE @company_id int

SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1)

IF @company_id IS NULL
    RETURN

DECLARE @partner_id int

SELECT @partner_id = id FROM dbo25.members WHERE company_id = @company_id AND code = @column_name

IF @partner_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find member id %s', @data_language)
    RAISERROR (@message, 11, 0, @column_name)
    RETURN
    END

DECLARE @value bit = NULL

IF @cell_number_value IS NULL
    BEGIN
    DELETE FROM dbo25.member_relations WHERE member_id = @id AND partner_id = @partner_id
    RETURN
    END
ELSE IF @cell_number_value = 0
    BEGIN
    SET @value = 0
    END
ELSE IF @cell_number_value = 1
    BEGIN
    SET @value = 1
    END
ELSE
    BEGIN
    SET @message = dbo25.get_translated_string('Unknown option: %i (Use 0 or 1)', @data_language)
    RAISERROR (@message, 11, 0, @cell_number_value)
    END

IF EXISTS (SELECT TOP 1 parent_id FROM dbo25.parents WHERE parent_id = @id)
    BEGIN
    DECLARE @code nvarchar(128) = (SELECT code FROM dbo25.members WHERE id = @id)
    SET @message = dbo25.get_translated_string('Member %s cannot be editable as it has children', @data_language)
    RAISERROR (@message, 11, 0, @code)
    END

IF EXISTS (SELECT TOP 1 parent_id FROM dbo25.parents WHERE parent_id = @partner_id)
    BEGIN
    SET @message = dbo25.get_translated_string('Member %s cannot be editable as it has children', @data_language)
    RAISERROR (@message, 11, 0, @column_name)
    END

UPDATE dbo25.member_relations
SET
    update_permission = @value
WHERE
    member_id = @id
    AND partner_id = @partner_id

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.member_relations (member_id, partner_id, update_permission) VALUES (@id, @partner_id, @value)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects members
-- =============================================

CREATE PROCEDURE [xls25].[usp_members]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
    , @root_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @dimension_id IS NULL SELECT @dimension_id = MIN(id) FROM dbo25.dimensions

SELECT
    m.id
    , m.sort_order
    , CASE WHEN h.parent_id IS NOT NULL THEN 0 WHEN m.calc_type_id = 0 THEN 0 ELSE 1 END AS is_editable
    , m.decimal_places
    , CAST(m.is_percent AS tinyint) AS is_percent
    , m.row_color
    , m.row_bold
    , m.row_indent
    , m.show_line
    , m.show_line_before
    , m.show_line_after
    , m.code
    , CASE WHEN m.row_indent > 0 THEN REPLICATE(' ', m.row_indent) ELSE '' END + m.name AS name
    , CASE WHEN m.row_indent > 0 THEN REPLICATE(' ', m.row_indent) ELSE '' END + tr.name AS translated_name
    , m.comment
    , m.tax_rate_id
    , m.unit_id
    , m.calc_type_id
    , m.previous_period_id
    , m.same_period_id
    , p.[1] AS parent1
    , h1.factor AS factor1
    , p.[2] AS parent2
    , h2.factor AS factor2
    , p.[3] AS parent3
    , h3.factor AS factor3
    , p.[4] AS parent4
    , h4.factor AS factor4
    , p.[5] AS parent5
    , h5.factor AS factor5
    , p.[6] AS parent6
    , h6.factor AS factor6
    , p.[7] AS parent7
    , h7.factor AS factor7
    , CAST(m.is_active AS tinyint) AS is_active
    , m.excel_formula
    , m.cube_formula
FROM
    dbo25.members m
    LEFT OUTER JOIN (SELECT DISTINCT parent_id FROM dbo25.parents) h ON h.parent_id = m.id
    LEFT OUTER JOIN (
        SELECT
            h.member_id
            , h.parent_id
            , ROW_NUMBER() OVER(PARTITION BY h.member_id ORDER BY p.sort_order, h.parent_id) AS R
        FROM
            dbo25.parents h
            INNER JOIN dbo25.members m ON m.id = h.member_id
            INNER JOIN dbo25.members p ON p.id = h.parent_id
        WHERE
            m.company_id = @company_id
            AND NOT h.member_id = h.parent_id
            AND m.dimension_id = @dimension_id
    ) s PIVOT (
        MAX(parent_id) FOR R IN ([1], [2], [3], [4], [5], [6], [7])
    ) p ON p.member_id = m.id
    LEFT OUTER JOIN dbo25.parents h1 ON h1.member_id = m.id AND h1.parent_id = p.[1]
    LEFT OUTER JOIN dbo25.parents h2 ON h2.member_id = m.id AND h2.parent_id = p.[2]
    LEFT OUTER JOIN dbo25.parents h3 ON h3.member_id = m.id AND h3.parent_id = p.[3]
    LEFT OUTER JOIN dbo25.parents h4 ON h4.member_id = m.id AND h4.parent_id = p.[4]
    LEFT OUTER JOIN dbo25.parents h5 ON h5.member_id = m.id AND h5.parent_id = p.[5]
    LEFT OUTER JOIN dbo25.parents h6 ON h6.member_id = m.id AND h6.parent_id = p.[6]
    LEFT OUTER JOIN dbo25.parents h7 ON h7.member_id = m.id AND h7.parent_id = p.[7]
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
WHERE
    m.company_id = @company_id
    AND m.dimension_id = @dimension_id
    AND NOT COALESCE(m.calc_type_id, 0) = 9    -- rowset
    AND (@root_id IS NULL OR m.id IN (
            SELECT
                h.member_id
            FROM
                dbo25.hierarchies h
            WHERE
                h.parent_id = @root_id
            )
        )
ORDER BY
    m.sort_order
    , m.code
    , m.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_members
-- =============================================

CREATE PROCEDURE [xls25].[usp_members_delete]
    @id int
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = t.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

DELETE dbo25.hierarchies WHERE parent_id = @id

DELETE dbo25.factors WHERE parent_id = @id

DELETE dbo25.parents WHERE parent_id = @id

DELETE dbo25.member_permissions WHERE member_id = @id

DELETE dbo25.member_relations WHERE member_id = @id OR partner_id = @id

SET NOCOUNT OFF

DELETE dbo25.members
WHERE
    id = @id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_members
-- =============================================

CREATE PROCEDURE [xls25].[usp_members_insert]
    @company_id int
    , @dimension_id tinyint
    , @code nvarchar(50)
    , @name nvarchar(400)
    , @translated_name nvarchar(400) = NULL
    , @comment nvarchar(1000) = NULL
    , @sort_order int = NULL
    , @tax_rate_id tinyint = NULL
    , @unit_id tinyint = NULL
    , @calc_type_id tinyint = NULL
    , @previous_period_id int = NULL
    , @same_period_id int = NULL
    , @decimal_places tinyint = NULL
    , @is_percent tinyint = NULL
    , @row_color tinyint = NULL
    , @row_bold tinyint = NULL
    , @row_indent tinyint = NULL
    , @show_line tinyint = NULL
    , @show_line_before tinyint = NULL
    , @show_line_after tinyint = NULL
    , @is_active tinyint = NULL
    , @excel_formula nvarchar(400) = NULL
    , @parent1 int = NULL
    , @parent2 int = NULL
    , @parent3 int = NULL
    , @parent4 int = NULL
    , @parent5 int = NULL
    , @parent6 int = NULL
    , @parent7 int = NULL
    , @factor1 float = NULL
    , @factor2 float = NULL
    , @factor3 float = NULL
    , @factor4 float = NULL
    , @factor5 float = NULL
    , @factor6 float = NULL
    , @factor7 float = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

DECLARE @message nvarchar(max)

IF CHARINDEX(@code, ' ') > 0 OR CHARINDEX(@code, '''') > 0 OR CHARINDEX(@code, '"') > 0 OR CHARINDEX(@code, '[') > 0 OR CHARINDEX(@code, ']') > 0
    BEGIN
    SET @message = dbo25.get_translated_string('The code ''%s'' contains prohibited symbols (space, '', ", [ or ])', @data_language)
    RAISERROR (@message, 11, 0, @code)
    RETURN
    END

SET NOCOUNT OFF

INSERT INTO dbo25.members
    ( company_id
    , dimension_id
    , code
    , name
    , comment
    , sort_order
    , tax_rate_id
    , unit_id
    , calc_type_id
    , excel_formula
    , previous_period_id
    , same_period_id
    , decimal_places
    , is_percent
    , row_color
    , row_bold
    , row_indent
    , show_line
    , show_line_before
    , show_line_after
    , is_active
    )
VALUES
    ( @company_id
    , @dimension_id
    , @code
    , @name
    , @comment
    , @sort_order
    , @tax_rate_id
    , @unit_id
    , COALESCE(@calc_type_id, 0)
    , @excel_formula
    , @previous_period_id
    , @same_period_id
    , @decimal_places
    , @is_percent
    , @row_color
    , @row_bold
    , @row_indent
    , COALESCE(@show_line, 1)
    , COALESCE(@show_line_before, 0)
    , COALESCE(@show_line_after, 0)
    , COALESCE(@is_active, 1)
    )

DECLARE @id int = SCOPE_IDENTITY()

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    INSERT INTO dbo25.translations
        (company_id, table_id, member_id, language, name)
    VALUES
        (@company_id, 1, @id, @data_language, @translated_name)

INSERT INTO dbo25.parents (member_id, parent_id, factor)
SELECT
    @id
    , p.parent_id
    , COALESCE(p.factor, CASE WHEN m.calc_type_id IN (0, 6, 7, 8, 9) THEN 0 ELSE 1 END) AS factor
FROM
    dbo25.members m
    CROSS JOIN (
        SELECT @parent1 AS parent_id, @factor1 AS factor UNION ALL
        SELECT @parent2 AS parent_id, @factor2 AS factor UNION ALL
        SELECT @parent3 AS parent_id, @factor3 AS factor UNION ALL
        SELECT @parent4 AS parent_id, @factor4 AS factor UNION ALL
        SELECT @parent5 AS parent_id, @factor5 AS factor UNION ALL
        SELECT @parent6 AS parent_id, @factor6 AS factor UNION ALL
        SELECT @parent7 AS parent_id, @factor7 AS factor
    ) p
    INNER JOIN dbo25.members pm ON pm.id = p.parent_id AND pm.company_id = m.company_id
    LEFT OUTER JOIN dbo25.parents h ON h.member_id = m.id AND h.parent_id = p.parent_id
WHERE
    m.id = @id
    AND p.parent_id IS NOT NULL
    AND h.member_id IS NULL


END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_members
-- =============================================

CREATE PROCEDURE [xls25].[usp_members_update]
    @id int
    , @code nvarchar(50)
    , @name nvarchar(400)
    , @translated_name nvarchar(400) = NULL
    , @comment nvarchar(1000) = NULL
    , @sort_order int = NULL
    , @tax_rate_id tinyint = NULL
    , @unit_id tinyint = NULL
    , @calc_type_id tinyint = NULL
    , @previous_period_id int = NULL
    , @same_period_id int = NULL
    , @decimal_places tinyint = NULL
    , @is_percent tinyint = NULL
    , @row_color tinyint = NULL
    , @row_bold tinyint = NULL
    , @row_indent tinyint = NULL
    , @show_line tinyint = NULL
    , @show_line_before tinyint = NULL
    , @show_line_after tinyint = NULL
    , @is_active tinyint = NULL
    , @excel_formula nvarchar(400) = NULL
    , @parent1 int = NULL
    , @parent2 int = NULL
    , @parent3 int = NULL
    , @parent4 int = NULL
    , @parent5 int = NULL
    , @parent6 int = NULL
    , @parent7 int = NULL
    , @factor1 float = NULL
    , @factor2 float = NULL
    , @factor3 float = NULL
    , @factor4 float = NULL
    , @factor5 float = NULL
    , @factor6 float = NULL
    , @factor7 float = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int
DECLARE @current_name nvarchar(400) = NULL

SELECT TOP 1 @company_id = u.company_id, @current_name = t.name FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

DECLARE @message nvarchar(max)

IF CHARINDEX(@code, ' ') > 0 OR CHARINDEX(@code, '''') > 0 OR CHARINDEX(@code, '"') > 0 OR CHARINDEX(@code, '[') > 0 OR CHARINDEX(@code, ']') > 0
    BEGIN
    SET @message = dbo25.get_translated_string('The code ''%s'' contains prohibited symbols (space, '', ", [ or ])', @data_language)
    RAISERROR (@message, 11, 0, @code)
    RETURN
    END

SET NOCOUNT OFF

UPDATE dbo25.members
SET
    code = @code
    , name = @name
    , comment = @comment
    , sort_order = @sort_order
    , tax_rate_id = @tax_rate_id
    , unit_id = @unit_id
    , calc_type_id = COALESCE(@calc_type_id, calc_type_id, 0)
    , excel_formula = @excel_formula
    , previous_period_id = @previous_period_id
    , same_period_id = @same_period_id
    , decimal_places = @decimal_places
    , is_percent = @is_percent
    , row_color = @row_color
    , row_bold = @row_bold
    , row_indent = @row_indent
    , show_line = COALESCE(@show_line, 1)
    , show_line_before = COALESCE(@show_line_before, 0)
    , show_line_after = COALESCE(@show_line_after, 0)
    , is_active = COALESCE(@is_active, is_active, 1)
WHERE
    id = @id
    AND NOT (
        COALESCE(code, '-1') = COALESCE(@code, '-2')
        AND COALESCE(name, '-1') = COALESCE(@name, '-2')
        AND COALESCE(comment, '-1') = COALESCE(@comment, '-2')
        AND COALESCE(sort_order, -1) = COALESCE(@sort_order, -2)
        AND COALESCE(tax_rate_id, -1) = COALESCE(@tax_rate_id, -2)
        AND COALESCE(unit_id, -1) = COALESCE(@unit_id, -2)
        AND COALESCE(calc_type_id, -1) = COALESCE(@calc_type_id, -2)
        AND COALESCE(excel_formula, -1) = COALESCE(@excel_formula, -2)
        AND COALESCE(previous_period_id, -1) = COALESCE(@previous_period_id, -2)
        AND COALESCE(same_period_id, -1) = COALESCE(@same_period_id, -2)
        AND COALESCE(decimal_places, -1) = COALESCE(@decimal_places, -2)
        AND COALESCE(is_percent, -1) = COALESCE(@is_percent, -2)
        AND COALESCE(row_color, -1) = COALESCE(@row_color, -2)
        AND COALESCE(row_bold, -1) = COALESCE(@row_bold, -2)
        AND COALESCE(row_indent, -1) = COALESCE(@row_indent, -2)
        AND COALESCE(show_line, -1) = COALESCE(@show_line, -2)
        AND COALESCE(show_line_before, -1) = COALESCE(@show_line_before, -2)
        AND COALESCE(show_line_after, -1) = COALESCE(@show_line_after, -2)
        AND COALESCE(is_active, -1) = COALESCE(@is_active, is_active, -2)
        )

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    BEGIN
    MERGE dbo25.translations t
    USING
        (
        SELECT company_id = @company_id, 1 AS table_id, @id AS member_id, @data_language AS language, @translated_name AS name
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    END

;MERGE INTO dbo25.parents t
USING (
    SELECT
        m.id AS member_id, v.parent_id, COALESCE(v.factor, CASE WHEN m.calc_type_id IN (0, 6, 7, 8, 9) THEN 0 ELSE 1 END) AS factor
    FROM
        dbo25.members m
        CROSS JOIN (VALUES
            (@parent1, @factor1)
            , (@parent2, @factor2)
            , (@parent3, @factor3)
            , (@parent4, @factor4)
            , (@parent5, @factor5)
            , (@parent6, @factor6)
            , (@parent7, @factor7)
            ) AS v(parent_id, factor)
    WHERE
        m.id = @id
        AND v.parent_id IS NOT NULL
) s ON s.member_id = t.member_id AND s.parent_id = t.parent_id
WHEN MATCHED AND NOT t.factor = s.factor THEN
    UPDATE SET factor = s.factor
WHEN NOT MATCHED THEN
    INSERT (member_id, parent_id, factor) VALUES (s.member_id, s.parent_id, factor)
WHEN NOT MATCHED BY SOURCE AND t.member_id = @id THEN
    DELETE;

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects name_formats
-- =============================================

CREATE PROCEDURE [xls25].[usp_name_formats]
AS
BEGIN

SET NOCOUNT ON

SELECT
    t.id
    , t.code
    , t.name
FROM
    dbo25.name_formats t
ORDER BY
    t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects user roles
-- =============================================

CREATE PROCEDURE [xls25].[usp_role_members]
AS
BEGIN

SET NOCOUNT ON;

DECLARE @list varchar(MAX)

SELECT @list = STUFF((
    SELECT ', [' + name + ']' FROM (
        SELECT DISTINCT p.name, p.is_fixed_role FROM sys.database_principals p
            WHERE name NOT IN ('db_owner', 'public') AND p.[type] IN ('R') AND p.is_fixed_role = 0
        ) AS t ORDER BY t.is_fixed_role, t.name
    FOR XML PATH(''), TYPE
    ).value('.', 'nvarchar(MAX)'), 1, 2, '')

IF @list IS NULL
    SELECT
        LOWER(m.type_desc) AS [type]
        , m.name
        , NULL AS format_column
    FROM
        sys.database_principals m
        LEFT JOIN sys.database_role_members rm ON rm.member_principal_id = m.principal_id
        LEFT JOIN sys.database_principals p ON p.principal_id = rm.role_principal_id
    WHERE
        m.[type] IN ('S', 'U', 'R')
        AND m.is_fixed_role = 0
        AND NOT m.name IN ('dbo', 'sys', 'guest', 'public', 'INFORMATION_SCHEMA', 'xls_users', 'xls_developers', 'xls_formats', 'xls_admins', 'doc_readers', 'doc_writers', 'log_app', 'log_admins', 'log_administrators', 'log_users')
    ORDER BY
        m.type_desc
        , m.name
ELSE
    BEGIN
    DECLARE @sql varchar(MAX)
    SET @sql = '
SELECT
    LOWER(p.[type]) AS [type]
    , p.name
    , NULL AS format_column
    , ' + COALESCE(@list, '') + '
    , NULL AS last_format_column
FROM
    (
    SELECT
        p.name AS [role]
        , m.type_desc AS [type]
        , m.name
        , 1 AS [include]
    FROM
        sys.database_principals m
        LEFT JOIN sys.database_role_members rm ON rm.member_principal_id = m.principal_id
        LEFT JOIN sys.database_principals p ON p.principal_id = rm.role_principal_id
    WHERE
        m.[type] IN (''S'', ''U'', ''R'')
        AND m.is_fixed_role = 0
        AND NOT m.name IN (''dbo'', ''sys'', ''guest'', ''public'', ''INFORMATION_SCHEMA'', ''xls_users'', ''xls_developers'', ''xls_formats'', ''xls_admins'', ''doc_readers'', ''doc_writers'', ''log_app'', ''log_admins'', ''log_administrators'', ''log_users'')
    ) s PIVOT (
        SUM([include]) FOR [role] IN ('+ COALESCE(@list, '') + ')
    ) p
ORDER BY
    p.[type]
    , p.[name]
'
    EXEC(@sql)
    -- PRINT @sql
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_roles
-- =============================================

CREATE PROCEDURE [xls25].[usp_role_members_change]
    @column_name nvarchar(128) = NULL
    , @cell_number_value int = NULL
    , @name nvarchar(128) = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

IF DATABASE_PRINCIPAL_ID(@column_name) IS NULL
    BEGIN
    SET @message = N'Role ''%s'' does not exist'
    SET @message = COALESCE((SELECT TOP 1 TRANSLATED_NAME FROM xls.translations WHERE TABLE_SCHEMA = 'dbo27' AND TABLE_NAME = 'strings' AND LANGUAGE_NAME =  @data_language AND COLUMN_NAME = @message), @message)
    RAISERROR(@message, 11, 0, @column_name);
    RETURN
    END

IF DATABASE_PRINCIPAL_ID(@name) IS NULL
    BEGIN
    SET @message = N'User ''%s'' does not exist'
    SET @message = COALESCE((SELECT TOP 1 TRANSLATED_NAME FROM xls.translations WHERE TABLE_SCHEMA = 'dbo27' AND TABLE_NAME = 'strings' AND LANGUAGE_NAME =  @data_language AND COLUMN_NAME = @message), @message)
    RAISERROR(@message, 11, 0, @name);
    RETURN
    END

IF @cell_number_value = 0
    EXEC sp_droprolemember @column_name, @name
ELSE IF @cell_number_value = 1
    EXEC sp_addrolemember @column_name, @name
ELSE
    BEGIN
    SET @message = N'Set 1 to add and 0 to remove a user from the role'
    SET @message = COALESCE((SELECT TOP 1 TRANSLATED_NAME FROM xls.translations WHERE TABLE_SCHEMA = 'dbo27' AND TABLE_NAME = 'strings' AND LANGUAGE_NAME =  @data_language AND COLUMN_NAME = @message), @message)
    RAISERROR(@message, 11, 0);
    RETURN
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects rowsets
-- =============================================

CREATE PROCEDURE [xls25].[usp_rowsets]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    m.id
    , m.code
    , m.name
    , tr.name AS translated_name
    , m.sort_order
FROM
    dbo25.members m
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
WHERE
    m.company_id = @company_id
    AND m.calc_type_id = 9     -- rowset
    AND m.dimension_id = COALESCE(@dimension_id, m.dimension_id)
ORDER BY
    m.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_rowsets
-- =============================================

CREATE PROCEDURE [xls25].[usp_rowsets_delete]
    @id int
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int

SELECT TOP 1 @company_id = t.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET NOCOUNT OFF

DELETE dbo25.members
WHERE
    id = @id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_rowsets
-- =============================================

CREATE PROCEDURE [xls25].[usp_rowsets_insert]
    @company_id int = NULL
    , @dimension_id tinyint
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @translated_name nvarchar(100) = NULL
    , @sort_order int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

SET NOCOUNT OFF

INSERT INTO dbo25.members
    ( company_id
    , dimension_id
    , code
    , name
    , sort_order
    , calc_type_id
    )
VALUES
    ( @company_id
    , @dimension_id
    , @code
    , @name
    , @sort_order
    , 9
    )

DECLARE @id int = SCOPE_IDENTITY()

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    INSERT INTO dbo25.translations
        (company_id, table_id, member_id, language, name)
    VALUES
        (@company_id, 1, @id, @data_language, @translated_name)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_rowsets
-- =============================================

CREATE PROCEDURE [xls25].[usp_rowsets_update]
      @id int
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @translated_name nvarchar(100) = NULL
    , @sort_order int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int
DECLARE @current_name nvarchar(100)

SELECT TOP 1 @company_id = u.company_id, @current_name = t.name FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

SET NOCOUNT OFF

UPDATE dbo25.members
SET
    code = @code
    , name = @name
    , sort_order = @sort_order
WHERE
    id = @id
    AND NOT (
        COALESCE(code, '-1') = COALESCE(@code, '-2')
        AND COALESCE(name, '-1') = COALESCE(@name, '-2')
        AND COALESCE(sort_order, -1) = COALESCE(@sort_order, -2)
        )

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    BEGIN
    MERGE dbo25.translations t
    USING
        (
        SELECT company_id = @company_id, 1 AS table_id, @id AS member_id, @data_language AS language, @translated_name AS name
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    END


END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Form engine
-- =============================================

CREATE PROCEDURE [xls25].[usp_run_form]
    @company_id int = NULL
    , @form_id int = NULL
    , @form_p1 int = NULL
    , @form_p2 int = NULL
    , @form_p3 int = NULL
    , @form_p4 int = NULL
    , @form_p5 int = NULL
    , @form_p6 int = NULL
    , @form_p7 int = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER--, ENCRYPTION
AS
BEGIN

SET NOCOUNT ON

BEGIN -- Getting a user and checking a company --

EXECUTE AS CALLER

DECLARE @username nvarchar(128) = USER_NAME()

DECLARE @all_permissions tinyint = CASE WHEN COALESCE(IS_MEMBER('planning_app_users'), 0) = 0 THEN 1 ELSE NULL END

DECLARE @company_id1 int = (SELECT TOP 1 u.company_id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @form_id AND u.username = @username AND u.is_active = 1)

IF @company_id IS NULL
    SET @company_id = @company_id1
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = @username AND is_active = 1)

REVERT

IF @company_id IS NULL OR @form_id IS NULL
    BEGIN
    SELECT * FROM dbo25.view_dynamic_form_fields
    RETURN
    END

END

BEGIN -- Common variables --

DECLARE @message nvarchar(max)

DECLARE @company_sid nvarchar(50) = COALESCE(CAST(@company_id AS nvarchar(50)), 'NULL')

DECLARE @functional_currency_id int = (SELECT TOP 1 id FROM dbo25.units WHERE company_id = @company_id AND is_functional_currency = 1 ORDER BY id)
DECLARE @functional_currency_sid nvarchar(10) = COALESCE(CAST(@functional_currency_id AS nvarchar(10)), 'NULL')

END

BEGIN -- Parameters --

    DECLARE @parameters TABLE (
        dimension_id tinyint NOT NULL PRIMARY KEY,
        parameter_index tinyint,
        axis_type_id tinyint,
        header_order tinyint,
        join_order tinyint,
        is_null_value tinyint,
        member_id int,
        rowset_id int,
        root_id int,
        calc_type_id tinyint,
        same_period_id int,
        start_level tinyint,
        end_level tinyint,
        d nvarchar(3),
        m nvarchar(10))

    -- @parameters is a table of parameters including form related properties
    -- Parameters include all dimensions except the dimension used in rows
    -- header_order defines column orders in column names

    INSERT INTO @parameters (dimension_id, parameter_index, axis_type_id, header_order, join_order, is_null_value, member_id, rowset_id, root_id, calc_type_id, same_period_id, start_level, end_level, d, m)
    SELECT
        d.id
        , fd.parameter_index
        , COALESCE(fd.axis_type_id, 1) AS axis_type_id
        , ROW_NUMBER() OVER (PARTITION BY fd.axis_type_id ORDER BY fd.header_order, fd.dimension_id) AS header_order
        , d.join_order
        , CASE WHEN p.id IS NOT NULL AND p.member_id IS NULL AND d.default_member_id IS NULL THEN 1 WHEN m.id IS NULL THEN 1 ELSE 0 END AS is_null_value
        , m.id AS member_id
        , CASE WHEN m.calc_type_id = 9 THEN m.id ELSE NULL END AS rowset_id
        , CASE WHEN m.calc_type_id = 9 THEN NULL ELSE m.id END AS root_id
        , m.calc_type_id
        , CASE WHEN d.id = 2 THEN m.same_period_id ELSE NULL END AS same_period_id  -- for times only
        , fd.form_start_level
        , fd.form_end_level
        , CAST(d.id AS nvarchar) AS d
        , CAST(m.id AS nvarchar) AS m
    FROM
        dbo25.dimension_properties d
        LEFT OUTER JOIN dbo25.form_dimensions fd ON fd.dimension_id = d.id AND fd.form_id = @form_id
        LEFT OUTER JOIN (
            SELECT 1 AS id, @form_p1 AS member_id
            UNION ALL SELECT 2, @form_p2
            UNION ALL SELECT 3, @form_p3
            UNION ALL SELECT 4, @form_p4
            UNION ALL SELECT 5, @form_p5
            UNION ALL SELECT 6, @form_p6
            UNION ALL SELECT 7, @form_p7
        ) p ON p.id = fd.parameter_index
        LEFT OUTER JOIN dbo25.members m ON m.company_id = d.company_id AND m.id = COALESCE(p.member_id, fd.root_member_id, d.default_member_id, d.root_member_id)
    WHERE
        d.company_id = @company_id
        AND d.is_active = 1
    ORDER BY
        d.id

    -- SELECT * FROM @parameters
    -- RETURN
END

BEGIN -- Permissions --

    DECLARE @permissions TABLE (dimension_id tinyint PRIMARY KEY, permission_join nvarchar(255), permission_where nvarchar(255))

    IF @all_permissions IS NULL
        BEGIN

        IF COALESCE((SELECT fp.select_permission FROM dbo25.form_permissions fp WHERE fp.form_id = @form_id AND fp.username = @username), 0) = 0
            BEGIN
            SELECT * FROM dbo25.view_dynamic_form_fields
            RETURN
            END

        INSERT INTO @permissions (dimension_id, permission_join, permission_where)
        SELECT
            d.id
            , 'LEFT OUTER JOIN dbo25.member_permissions mp' + CAST(d.id AS nvarchar)
                + ' ON mp' + CAST(d.id AS nvarchar) + '.username = ''' + @username + ''''
                + ' AND mp' + CAST(d.id AS nvarchar) + '.member_id = ' -- + CAST(d.id AS nvarchar) + '.parent_id'
            , 'COALESCE(mp' + CAST(d.id AS nvarchar) + '.select_permission, '
                + CAST(d.default_select_permission AS nvarchar) + ') = 1'
        FROM
            dbo25.dimension_properties d
        WHERE
            d.company_id = @company_id
            AND d.is_active = 1
            AND d.is_protected = 1

        -- SELECT * FROM @permissions

        END

END

BEGIN -- Flags --

    DECLARE @has_null_parameters int = 0

    -- The form does not show data columns if any of form parameters has the NULL value

    IF EXISTS(SELECT TOP 1 p.dimension_id FROM @parameters p WHERE p.is_null_value = 1)
        SET @has_null_parameters = 1

    -- If any of page parameters is a calculated member (like 2020),
    -- then we cannot save any cell data to a cube

    DECLARE @is_calculated int = 0

    IF EXISTS(SELECT TOP 1 p.dimension_id FROM @parameters p INNER JOIN dbo25.parents h ON h.parent_id = p.member_id WHERE p.axis_type_id = 1)
        SET @is_calculated = 1

    -- If a category parameter is a same period member, then the entire form is the same period

    DECLARE @is_same_period int = 0

    IF EXISTS(SELECT TOP 1 p.dimension_id FROM @parameters p WHERE p.dimension_id = 3 AND p.calc_type_id = 6)    -- same_period
        SET @is_same_period = 1

    -- root_id and rowset_id for the accounts dimension in rows

    DECLARE @rowset_id int, @root_id int, @start_level tinyint, @end_level tinyint

    SELECT
        @rowset_id = p.rowset_id
        , @root_id = p.root_id
        , @start_level = COALESCE(p.start_level, 0)
        , @end_level = COALESCE(p.end_level, 255)
    FROM
        @parameters p
    WHERE
        p.dimension_id = 1 AND p.axis_type_id = 2

    -- If selected accounts have averages, then the form must have an additional select with the AVG aggregate

    DECLARE @has_row_averages AS int = 0

    IF @root_id IS NOT NULL AND EXISTS (
            SELECT
                TOP 1 h.member_id
            FROM
                dbo25.hierarchies h
                INNER JOIN dbo25.members m ON m.id = h.member_id AND m.calc_type_id = 2    -- average
            WHERE
                m.company_id = @company_id
                AND h.parent_id = @root_id AND h.[level] BETWEEN @start_level AND @end_level
        )
        SET @has_row_averages = 1

    ELSE IF @rowset_id IS NOT NULL AND EXISTS (
            SELECT
                TOP 1 r.id
            FROM
                dbo25.form_rows r
                INNER JOIN dbo25.members m ON m.id = r.member_id AND m.calc_type_id = 2     -- average
            WHERE
                m.company_id = @company_id
                AND r.id = @rowset_id
        )
        SET @has_row_averages = 1

    -- PRINT @has_row_averages

    -- If selected accounts have strings, then the form must have an additional select from the strings

    DECLARE @has_strings AS int = 0

    IF @root_id IS NOT NULL AND EXISTS (
            SELECT
                TOP 1 h.member_id
            FROM
                dbo25.hierarchies h
                INNER JOIN dbo25.members m ON m.id = h.member_id AND m.calc_type_id = 8    -- strings
            WHERE
                m.company_id = @company_id
                AND h.parent_id = @root_id AND h.[level] BETWEEN @start_level AND @end_level
        )
        SET @has_strings = 1

    ELSE IF @rowset_id IS NOT NULL AND EXISTS (
            SELECT
                TOP 1 r.id
            FROM
                dbo25.form_rows r
                INNER JOIN dbo25.members m ON m.id = r.member_id AND m.calc_type_id = 8     -- strings
            WHERE
                r.id = @rowset_id
        )
        SET @has_strings = 1

    -- PRINT @has_strings

    -- If selected accounts have rates, then the form must have an additional select from rates

    DECLARE @has_rates AS int = 0

    IF @root_id IS NOT NULL AND EXISTS (
            SELECT
                TOP 1 h.member_id
            FROM
                dbo25.hierarchies h
                INNER JOIN dbo25.members m ON m.id = h.member_id AND m.calc_type_id = 10    -- currency rates
            WHERE
                h.parent_id = @root_id AND h.[level] BETWEEN @start_level AND @end_level
        )
        SET @has_rates = 1

    ELSE IF @rowset_id IS NOT NULL AND EXISTS (
            SELECT
                TOP 1 r.id
            FROM
                dbo25.form_rows r
                INNER JOIN dbo25.members m ON m.id = r.member_id AND m.calc_type_id = 10    -- currency rates
            WHERE
                r.id = @rowset_id
        )
        SET @has_rates = 1

END

BEGIN -- Member relation dimensions

    DECLARE @member_relation_dimensions TABLE (
        member_dimension_id int NOT NULL
        , partner_dimension_id int NOT NULL
        , PRIMARY KEY (member_dimension_id, partner_dimension_id)
        )

    INSERT INTO @member_relation_dimensions
        (member_dimension_id, partner_dimension_id)
    SELECT
        DISTINCT
        m.dimension_id AS member_dimension_id
        , p.dimension_id AS partner_dimension_id
    FROM
        dbo25.member_relations mr
        INNER JOIN dbo25.members m ON m.id = mr.member_id
        INNER JOIN dbo25.members p ON p.id = mr.partner_id
    WHERE
        m.company_id = @company_id

    -- SELECT * FROM @member_relation_dimensions
    -- RETURN

END

BEGIN -- Column members --

    DECLARE @column_members TABLE (member_id int PRIMARY KEY, dimension_id tinyint, code nvarchar(50), calc_type_id tinyint,
        header_order int, sort_order int, is_editable tinyint,
        previous_period_code nvarchar(50), same_period_code nvarchar(50),
        parent_category_id int, parent_category_code nvarchar(50),
        excel_formula nvarchar(400))

    -- @column_members contains members with related properties for all dimensions used in columns

    INSERT INTO @column_members (member_id, dimension_id, code, calc_type_id,
        header_order, sort_order, is_editable,
        previous_period_code, same_period_code,
        parent_category_id, parent_category_code,
        excel_formula)
    SELECT
        p.id, m.dimension_id, m.code, m.calc_type_id, p.header_order, p.sort_order
        , CASE WHEN h.parent_id IS NULL AND m.calc_type_id IN (1, 2, 6, 8, 11) THEN 1 ELSE 0 END AS is_editable
        , pp.code, mp.code
        , sp.id
        , sp.code AS parent_period_code
        , m.excel_formula
    FROM
        (
            SELECT
                m.id, p.header_order, m.sort_order
            FROM
                @parameters p
                INNER JOIN dbo25.hierarchies h ON h.parent_id = p.root_id
                INNER JOIN dbo25.members m ON m.id = h.member_id
                INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
                LEFT OUTER JOIN dbo25.member_permissions mp ON mp.member_id = h.member_id AND mp.username = @username
            WHERE
                p.axis_type_id = 3 AND p.header_order IN (1, 2, 3)
                AND m.calc_type_id IN (1, 2, 3, 4, 5, 6, 7, 8, 10, 11, 12, 13)
                AND (p.header_order = 1 OR NOT m.calc_type_id IN (7))
                AND h.[level] BETWEEN COALESCE(p.start_level, 0) AND COALESCE(p.end_level, 255)
                AND (d.is_protected = 0 OR COALESCE(@all_permissions, mp.select_permission, d.default_select_permission, 0) = 1)
            UNION ALL
            SELECT
                m.id, p.header_order, COALESCE(r.sort_order, m.sort_order)
            FROM
                @parameters p
                INNER JOIN dbo25.form_rows r ON r.rowset_id = p.rowset_id
                INNER JOIN dbo25.members m ON m.id = r.member_id
                INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
                LEFT OUTER JOIN dbo25.member_permissions mp ON mp.member_id = m.id AND mp.username = @username
            WHERE
                p.axis_type_id = 3 AND p.header_order IN (1, 2, 3)
                AND m.calc_type_id IN (1, 2, 3, 4, 5, 6, 7, 8, 10, 11, 12, 13)
                AND (p.header_order = 1 OR NOT m.calc_type_id IN (7))
                AND (d.is_protected = 0 OR COALESCE(@all_permissions, mp.select_permission, d.default_select_permission, 0) = 1)
        ) p
        INNER JOIN dbo25.members m ON m.id = p.id
        LEFT OUTER JOIN (SELECT DISTINCT p.parent_id FROM dbo25.parents p INNER JOIN dbo25.members m ON m.id = p.parent_id WHERE m.company_id = @company_id) h ON h.parent_id = m.id
        LEFT OUTER JOIN dbo25.members sp ON sp.same_period_id = m.id AND sp.dimension_id = 3 AND m.calc_type_id = 6 -- the same period
        LEFT OUTER JOIN dbo25.members pp ON pp.id = m.previous_period_id AND m.calc_type_id IN (12, 13) -- difference, percent
        LEFT OUTER JOIN dbo25.members mp ON mp.id = m.same_period_id AND m.calc_type_id IN (12, 13)     -- difference, percent

    -- SELECT * FROM @column_members
    -- RETURN

END

BEGIN -- Column member filters

    -- Filters like @cf2 contain ids and allow adding additional conditions into the WHERE clause to filter column members
    -- The times and categories dimensions are the most important to limit output row amount
    -- @cf2 and @cf3 contains the current period ids, @sf2 and @sf3 contains the same period ids

    DECLARE @cf2 nvarchar(max) = NULL, @sf2 nvarchar(max) = NULL, @cf3 nvarchar(max) = NULL, @sf3 nvarchar(max) = NULL

    SELECT
        @cf2 = STUFF((
            SELECT ', ' + CAST(c.member_id AS nvarchar) FROM @column_members c
            WHERE
                c.dimension_id = 2                      -- times
                AND c.calc_type_id NOT IN (7, 12, 13)   -- excel, difference, percent
                AND c.parent_category_id IS NULL        -- current period
            ORDER BY c.member_id FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 2, '')
        , @sf2 = STUFF((
            SELECT ', ' + CAST(c.member_id AS nvarchar) FROM @column_members c
            WHERE
                c.dimension_id = 2                      -- times
                AND c.calc_type_id NOT IN (7, 12, 13)   -- excel, difference, percent
                AND c.parent_category_id IS NOT NULL    -- the same period of the previous year
            ORDER BY c.member_id FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 2, '')
        , @cf3 = STUFF((
            SELECT ', ' + CAST(c.member_id AS nvarchar) FROM @column_members c
            WHERE
                c.dimension_id = 3                      -- categories
                AND c.calc_type_id NOT IN (7, 12, 13)   -- excel, difference, percent
                AND c.parent_category_id IS NULL        -- current period
            ORDER BY c.member_id FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 2, '')
        , @sf3 = STUFF((
            SELECT ', ' + CAST(c.parent_category_id AS nvarchar) FROM @column_members c
            WHERE
                c.dimension_id = 3                      -- categories
                AND c.calc_type_id NOT IN (7, 12, 13)   -- excel, difference, percent
                AND c.parent_category_id IS NOT NULL    -- current period
            ORDER BY c.member_id FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 2, '')

     -- PRINT @cf2
     -- PRINT @sf2
     -- PRINT @cf3
     -- PRINT @sf3
     -- RETURN

END

BEGIN -- @select_columns_sql and pivot columns --

    -- @pivot_columns_cp_sql contains column codes for the PIVOT line like
    -- SUM(value) FOR code IN ([2020], [2020.01], [2020.02], ..., [2020.12])
    -- SUM(value) FOR code IN ([2020 BUDGET], [2020 ACTUAL], [2020.01 BUDGET], [2020.01 ACTUAL], ..., [2020.12 BUDGET], [2020.12 ACTUAL])
    -- @select_columns_sql contains column codes for the SELECT clause
    -- It has the same as above but checks that the column is editable
    -- If the column is editable that the name is marked with a single quote
    -- that is used for conditional formatting in Excel

    DECLARE @select_columns_sql nvarchar(max) = NULL
    DECLARE @pivot_columns_cp_sql nvarchar(max) = NULL
    DECLARE @pivot_columns_sp_sql nvarchar(max) = NULL;

    IF @has_null_parameters = 0
        BEGIN

        WITH cte (column_name, d1_column_name, d2_column_name, calc_type_id, is_editable, excel_formula,
                 code1, code2, code3, sort_order1, sort_order2, sort_order3, is_same_period, a) AS (
        SELECT
            c1.code + COALESCE(' ' + c2.code + COALESCE(' ' + c3.code, ''),  '') AS column_name
            , COALESCE(c1.previous_period_code, c1.code) + COALESCE(' ' + COALESCE(c2.previous_period_code, c2.code) + COALESCE(' ' + COALESCE(c3.previous_period_code, c3.code), ''),  '') AS d1_column_name
            , COALESCE(c1.same_period_code, c1.code) + COALESCE(' ' + COALESCE(c2.same_period_code, c2.code) + COALESCE(' ' + COALESCE(c3.same_period_code, c3.code), ''),  '') AS d2_column_name
            , CASE WHEN c1.calc_type_id = 7 THEN c1.calc_type_id        -- excel
                WHEN c1.calc_type_id IN (12, 13) THEN c1.calc_type_id   -- difference, percent
                WHEN c2.calc_type_id IN (12, 13) THEN c2.calc_type_id   -- difference, percent
                WHEN c1.calc_type_id IN (12, 13) THEN c3.calc_type_id   -- difference, percent
                ELSE c1.calc_type_id END AS calc_type_id                -- others are not important
            , CASE WHEN c1.is_editable = 0 OR c2.is_editable = 0 OR c3.is_editable = 0 THEN 0 ELSE 1 END AS is_editable
            , c1.excel_formula AS excel_formula
            , c1.code AS code1
            , c2.code AS code2
            , c3.code AS code3
            , c1.sort_order AS sort_order1
            , c2.sort_order AS sort_order2
            , c3.sort_order AS sort_order3
            , CASE WHEN c1.parent_category_code IS NOT NULL
                     OR c2.parent_category_code IS NOT NULL
                     OR c3.parent_category_code IS NOT NULL
                     OR @is_same_period = 1 THEN 1 ELSE 0 END AS is_same_period
            , CASE WHEN c1.parent_category_code IS NOT NULL
                     OR c2.parent_category_code IS NOT NULL
                     OR c3.parent_category_code IS NOT NULL
                     OR @is_same_period = 1 THEN 's' ELSE 'c' END AS a
        FROM
            (SELECT * FROM @column_members c WHERE c.header_order = 1) c1
            OUTER APPLY (SELECT * FROM @column_members c WHERE c.header_order = 2) c2
            OUTER APPLY (SELECT * FROM @column_members c WHERE c.header_order = 3) c3
        )
        -- SELECT * FROM cte
        -- RETURN

        SELECT
            @select_columns_sql = STUFF((
                SELECT '
    , ' + 'CASE WHEN NOT v.line_order = 0 THEN NULL ELSE '
        +           CASE
                        WHEN c.calc_type_id = 7 THEN
                            COALESCE('CASE WHEN m.calc_type_id = 8 THEN NULL ELSE ''=' + REPLACE(c.excel_formula, '''', '''''') + ''' END', 'NULL')
                        WHEN c.calc_type_id = 12 THEN
                            COALESCE('CASE WHEN ' + COALESCE(c1.a, c.a) + 'p.[' + c.d1_column_name + '] IS NULL AND '
                                                  + COALESCE(c2.a, c.a) + 'p.[' + c.d2_column_name + '] IS NULL THEN NULL ELSE '
                                    + 'COALESCE(' + COALESCE(c2.a, c.a) + 'p.[' + c.d2_column_name
                                    + '], 0) - COALESCE(' + COALESCE(c1.a, c.a) + 'p.[' + c.d1_column_name + '], 0) END', 'NULL')
                        WHEN c.calc_type_id = 13 THEN
                            COALESCE('CASE WHEN COALESCE(' + COALESCE(c1.a, c.a) + 'p.[' + c.d1_column_name
                                + '], 0) = 0 THEN NULL ELSE COALESCE(' + COALESCE(c2.a, c.a) + 'p.[' + c.d2_column_name
                                + '], 0) / COALESCE(' + COALESCE(c1.a, c.a) + 'p.[' + c.d1_column_name + '], 0) - 1 END', 'NULL')
                        WHEN @has_strings = 0 AND @has_rates = 0 THEN
                            c.a + 'p.[' + c.column_name + ']'
                        WHEN @has_strings = 0 THEN
                            'COALESCE(' + c.a + 'p.[' + c.column_name + '], ' + c.a + 'r.[' + c.column_name + '])'
                        WHEN @has_rates = 0 THEN
                            'COALESCE(CAST(' + c.a + 'p.[' + c.column_name + '] AS sql_variant), CAST(CAST(' + c.a + 's.[' + c.column_name + '] AS nvarchar) AS sql_variant))'
                        ELSE
                            'COALESCE(CAST(' + c.a + 'p.[' + c.column_name + '] AS sql_variant), CAST(CAST(' + c.a + 's.[' + c.column_name + '] AS nvarchar) AS sql_variant), CAST(' + c.a + 'r.[' + c.column_name + '] AS sql_variant))'
                        END
                    + ' END'
                    + CASE c.is_editable WHEN 1 THEN ' AS [' + c.column_name + ''']' ELSE ' AS [' + c.column_name + ']' END AS code
                FROM
                    cte c
                    LEFT OUTER JOIN cte c1 ON c1.column_name = c.d1_column_name
                    LEFT OUTER JOIN cte c2 ON c2.column_name = c.d2_column_name
                ORDER BY
                    c.sort_order1, c.code1, c.sort_order2, c.code2, c.sort_order3, c.code3
                FOR XML PATH(''), TYPE
                ).value('.', 'nvarchar(MAX)'), 1, 8, '')

            , @pivot_columns_cp_sql = STUFF((
                SELECT
                    ', [' + column_name + ']' AS code
                FROM
                    cte
                WHERE
                    calc_type_id NOT IN (7, 12, 13)     -- excel, difference, percent
                    AND is_same_period = 0
                ORDER BY
                    sort_order1, code1, sort_order2, code2, sort_order3, code3
                FOR XML PATH(''), TYPE
                ).value('.', 'nvarchar(MAX)'), 1, 2, '')

            , @pivot_columns_sp_sql = STUFF((
                SELECT
                    ', [' + column_name + ']' AS code
                FROM
                    cte
                WHERE
                    calc_type_id NOT IN (7, 12, 13)     -- excel, difference, percent
                    AND is_same_period = 1
                ORDER BY
                    sort_order1, code1, sort_order2, code2, sort_order3, code3
                FOR XML PATH(''), TYPE
                ).value('.', 'nvarchar(MAX)'), 1, 2, '')

        END

     -- PRINT @select_columns_sql + CHAR(13) + CHAR(10)
     -- PRINT @pivot_columns_cp_sql + CHAR(13) + CHAR(10)
     -- PRINT @pivot_columns_sp_sql + CHAR(13) + CHAR(10)
     -- RETURN

END

BEGIN -- Configuration of rows --

    DECLARE @rows TABLE (
        id int NOT NULL PRIMARY KEY
        , d1 tinyint, r1 int, m1 int, c1 tinyint, level1_sql nvarchar(100)
        , d2 tinyint, r2 int, m2 int, c2 tinyint, level2_sql nvarchar(100), mr2_sql nvarchar(100)
        , d3 tinyint, r3 int, m3 int, c3 tinyint, level3_sql nvarchar(100), mr3_sql nvarchar(100)
        , max_level1 tinyint, max_level2 tinyint, max_level3 tinyint)

    -- @rows has all data in one row for all row dimensions to simplify selects
    -- @rows contains dimension_id, rowset_id, root member_id, and calc_type_id for all row axis dimensions
    -- level1_sql, level2_sql, and level3_sql contains filters by levels (can be prepared for IN ...)
    -- mr2_sql and mr3_sql filter members of the second and third dimensions in the row axis

    INSERT INTO @rows (id, d1, r1, m1, c1, d2, r2, m2, c2, d3, r3, m3, c3,
                        level1_sql, level2_sql, level3_sql, mr2_sql, mr3_sql)
    SELECT
        1 AS id     -- the @rows always has one line
        , p1.dimension_id, p1.rowset_id, p1.root_id, p1.calc_type_id
        , p2.dimension_id, p2.rowset_id, p2.root_id, p2.calc_type_id
        , p3.dimension_id, p3.rowset_id, p3.root_id, p3.calc_type_id
        , CASE WHEN p1.start_level IS NULL AND p1.end_level IS NULL THEN '' ELSE
            ' AND h1.level BETWEEN ' + CAST(COALESCE(p1.start_level, 0) AS nvarchar)
                           + ' AND ' + CAST(COALESCE(p1.end_level, 255) AS nvarchar) END AS level1_sql
        , CASE WHEN p2.start_level IS NULL AND p2.end_level IS NULL THEN '' ELSE
            ' AND h2.level BETWEEN ' + CAST(COALESCE(p2.start_level, 0) AS nvarchar)
                           + ' AND ' + CAST(COALESCE(p2.end_level, 255) AS nvarchar) END AS level2_sql
        , CASE WHEN p3.start_level IS NULL AND p3.end_level IS NULL THEN '' ELSE
            ' AND h3.level BETWEEN ' + CAST(COALESCE(p3.start_level, 0) AS nvarchar)
                           + ' AND ' + CAST(COALESCE(p3.end_level, 255) AS nvarchar) END AS level3_sql
        -- 2.2 JOIN added if member_relations has data
        , COALESCE(CASE WHEN p1.dimension_id = 1 AND p2.dimension_id IN (4, 5, 6, 7) THEN (
            SELECT '
    LEFT OUTER JOIN dbo25.member_relations mr2 ON mr2.member_id = m.id AND mr2.partner_id = m2.id'
            FROM
                @member_relation_dimensions mr
            WHERE
                mr.member_dimension_id = p1.dimension_id AND mr.partner_dimension_id = p2.dimension_id
            ) ELSE '' END, '') AS mr2_sql
        -- 2.2 JOIN added if member_relations has data
        , COALESCE(CASE WHEN p1.dimension_id = 1 AND p3.dimension_id IN (4, 5, 6, 7) THEN (
            SELECT '
    LEFT OUTER JOIN dbo25.member_relations mr3 ON mr3.member_id = m.id AND mr3.partner_id = m3.id'
            FROM
                @member_relation_dimensions mr
            WHERE
                mr.member_dimension_id = p1.dimension_id AND mr.partner_dimension_id = p3.dimension_id
            ) ELSE '' END, '') AS mr3_sql
    FROM
        @parameters p1
        LEFT OUTER JOIN @parameters p2 ON p2.axis_type_id = 2 AND p2.header_order = 2
        LEFT OUTER JOIN @parameters p3 ON p3.axis_type_id = 2 AND p3.header_order = 3
    WHERE
        p1.axis_type_id = 2 AND p1.header_order = 1

    -- SELECT * FROM @rows
    -- RETURN

END

BEGIN -- 2.2 Member relations with page parameters --

    -- @page_member_relation_where filters account members that have relations with members in the page axis
    -- For example, if the page has an entity 1 that it filters all applicable accounts
    -- If contains the string like m.id IN (...)
    -- This is much faster than the previous version with JOIN - about 1s against 20s

    DECLARE @page_member_relation_where nvarchar(max) = NULL

    SELECT
        @page_member_relation_where = '
    AND (m' + CASE p1.header_order WHEN 2 THEN '2' WHEN 3 THEN '3' ELSE '' END + '.id IS NULL OR '
        + 'm' + CASE p1.header_order WHEN 2 THEN '2' WHEN 3 THEN '3' ELSE '' END + '.id IN (' + STUFF((
        SELECT
            DISTINCT ', ' + CAST(hp.parent_id AS nvarchar)
        FROM
            @parameters p
            INNER JOIN @member_relation_dimensions r ON r.member_dimension_id = p1.dimension_id AND r.partner_dimension_id = p.dimension_id
            INNER JOIN dbo25.hierarchies h ON h.parent_id = p.member_id
            INNER JOIN dbo25.member_relations mr ON mr.partner_id = h.member_id AND update_permission = 1
            INNER JOIN dbo25.hierarchies hp ON hp.member_id = mr.member_id
        WHERE
            p.axis_type_id = 1 AND p.dimension_id IN (4, 5, 6, 7)
            AND p.member_id IS NOT NULL
            FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 2, '')
            + '))'
    FROM
        @parameters p1
    WHERE
        p1.dimension_id = 1 AND p1.axis_type_id = 2

    -- PRINT @page_member_relation_where
    -- RETURN

END

BEGIN -- 2.2 Subtotals --

    -- Getting max member levels to detect parent (subtotal) levels

    UPDATE @rows
    SET
        max_level1 = (SELECT MAX([level]) FROM dbo25.hierarchies h WHERE h.parent_id = m1)
        , max_level2 = (SELECT MAX([level]) FROM dbo25.hierarchies h WHERE h.parent_id = m2)
        , max_level3 = (SELECT MAX([level]) FROM dbo25.hierarchies h WHERE h.parent_id = m3)
    WHERE
        m2 IS NOT NULL

    DECLARE @subtotals_join nvarchar(max)

    SET @subtotals_join = (SELECT CASE WHEN m2 IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            m.id AS id
            , m2.id AS id2'
            + CASE WHEN m3 IS NOT NULL THEN '
            , m3.id AS id3'
                ELSE '' END    + '
        FROM
            dbo25.members m
            INNER JOIN dbo25.hierarchies h1 ON h1.member_id = m.id AND h1.parent_id = ' + CAST(m1 AS nvarchar) + '
            CROSS JOIN dbo25.members m2
            INNER JOIN dbo25.hierarchies h2 ON h2.member_id = m2.id AND h2.parent_id = ' + CAST(m2 AS nvarchar)
            + CASE WHEN m3 IS NOT NULL THEN '
            CROSS JOIN dbo25.members m3
            INNER JOIN dbo25.hierarchies h3 ON h3.member_id = m3.id AND h3.parent_id = ' + CAST(m3 AS nvarchar)
                ELSE '' END    + '
            INNER JOIN dbo25.form_subtotals st ON
                st.dimension_id1 = ' + CAST(d1 AS nvarchar) + '
                AND st.dimension_id2 = ' + CAST(d2 AS nvarchar)
            + CASE WHEN m3 IS NOT NULL THEN '
                AND st.dimension_id3 = ' + CAST(d3 AS nvarchar)
                    ELSE '
                AND st.dimension_id3 IS NULL AND st.member_id3 IS NULL'
                    END + '
                AND (h1.[level] < ' + CAST(max_level1 AS nvarchar) + ' OR h2.[level] < ' + CAST(max_level2 AS nvarchar)
            + CASE WHEN m3 IS NOT NULL THEN ' OR h3.[level] < ' + CAST(max_level3 AS nvarchar) ELSE '' END + ')
                AND COALESCE(st.member_id1, m.id) = m.id
                AND COALESCE(st.member_id2, m2.id) = m2.id'
            + CASE WHEN m3 IS NOT NULL THEN '
                AND COALESCE(st.member_id3, m3.id) = m3.id'
                    ELSE '' END + '
        WHERE
            m.dimension_id = ' + CAST(d1 AS nvarchar) + '
            AND m2.dimension_id = ' + CAST(d2 AS nvarchar)
            + CASE WHEN m3 IS NOT NULL THEN '
            AND m3.dimension_id = ' + CAST(d3 AS nvarchar)
                    ELSE '' END + '
            AND st.show = 0
        GROUP BY
            m.id
            , m2.id'
            + CASE WHEN m3 IS NOT NULL THEN '
            , m3.id'
                    ELSE '' END    + '
    ) st ON st.id = m.id AND st.id2 = m2.id'
            + CASE WHEN m3 IS NOT NULL THEN ' AND st.id3 = m3.id' ELSE '' END
        END
    FROM
        @rows r
    )

END

BEGIN -- Output SQL --

    DECLARE @sql nvarchar(max)

    SET @sql =
        -- common columns
        (SELECT
'SELECT
    '   + CASE WHEN r1 IS NOT NULL THEN 'r.id'  ELSE 'NULL' END + ' AS id
    , ' + CASE WHEN r2 IS NOT NULL THEN 'r2.id' ELSE 'NULL' END + ' AS id2
    , ' + CASE WHEN r3 IS NOT NULL THEN 'r3.id' ELSE 'NULL' END + ' AS id3
    , ROW_NUMBER() OVER (ORDER BY ' + SUBSTRING(
        + CASE WHEN r1 IS NOT NULL THEN ', r.sort_order, m.sort_order' WHEN m1 IS NOT NULL THEN ', m.sort_order' ELSE '' END
        + ', v.line_order'
        + CASE WHEN r2 IS NOT NULL THEN ', r2.sort_order, m2.sort_order' WHEN m2 IS NOT NULL THEN ', m2.sort_order'  ELSE '' END
        + CASE WHEN r3 IS NOT NULL THEN ', r3.sort_order, m3.sort_order' WHEN m3 IS NOT NULL THEN ', m3.sort_order'  ELSE '' END
        , 3, 255) + ') AS sort_order
    , ' + CASE WHEN r2 IS NOT NULL THEN 'r2.sort_order' WHEN m2 IS NOT NULL THEN 'm2.sort_order'  ELSE 'NULL' END + ' AS sort_order2
    , ' + CASE WHEN r3 IS NOT NULL THEN 'r3.sort_order' WHEN m3 IS NOT NULL THEN 'm3.sort_order'  ELSE 'NULL' END + ' AS sort_order3
    , ' + CASE WHEN r1 IS NOT NULL THEN 'r.member_id'   WHEN m1 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 THEN m.id ELSE NULL END'   ELSE 'NULL' END + ' AS member_id
    , ' + CASE WHEN r2 IS NOT NULL THEN 'r2.member_id'  WHEN m2 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 THEN m2.id ELSE NULL END'  ELSE 'NULL' END + ' AS member_id2
    , ' + CASE WHEN r3 IS NOT NULL THEN 'r3.member_id'  WHEN m3 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 THEN m3.id ELSE NULL END'  ELSE 'NULL' END + ' AS member_id3
    , ' + CASE WHEN @is_calculated = 1 THEN '0' ELSE 'CASE WHEN m.id IS NULL THEN 0 WHEN m.is_calculated = 1 THEN 0 '
            + CASE WHEN r2 IS NOT NULL OR m2 IS NOT NULL THEN 'WHEN m2.is_calculated = 1 THEN 0 ' ELSE '' END
            + CASE WHEN r3 IS NOT NULL OR m3 IS NOT NULL THEN 'WHEN m3.is_calculated = 1 THEN 0 ' ELSE '' END
            + ' WHEN m.calc_type_id IN (0, 10) THEN 0 WHEN NOT v.line_order = 0 THEN 0 ELSE 1 END' END + ' AS is_editable
    , ' + CASE WHEN r1 IS NOT NULL THEN 'r.decimal_places'  WHEN m1 IS NOT NULL THEN 'm.decimal_places' ELSE 'NULL' END + ' AS decimal_places
    , CAST(' + CASE WHEN r1 IS NOT NULL THEN 'r.is_percent'      WHEN m1 IS NOT NULL THEN 'm.is_percent'     ELSE 'NULL' END + ' AS tinyint) AS is_percent
    , ' + CASE WHEN r1 IS NOT NULL THEN 'r.row_color'       WHEN m1 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 THEN m.row_color ELSE NULL END' ELSE 'NULL' END + ' AS row_color
    , ' + CASE WHEN r1 IS NOT NULL THEN 'r.row_bold'        WHEN m1 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 THEN m.row_bold ELSE NULL END' ELSE 'NULL' END + ' AS row_bold
    , ' + CASE WHEN r1 IS NOT NULL THEN 'r.row_indent'      WHEN m1 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 THEN m.row_indent ELSE NULL END' ELSE 'NULL' END + ' AS row_indent
    , ' + CASE WHEN r1 IS NOT NULL THEN 'COALESCE(r.code, m.code)'  WHEN m1 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 THEN m.code ELSE NULL END'   ELSE 'NULL' END + ' AS code
    , ' + CASE WHEN r1 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 THEN CASE WHEN r.row_indent > 0 THEN REPLICATE('' '', r.row_indent) ELSE '''' END + COALESCE(' + CASE WHEN @data_language IS NULL THEN 'r.name' ELSE 'rt.name, r.name, t.name' END + ', m.name) ELSE NULL END'
               WHEN m1 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 THEN CASE WHEN m.row_indent > 0 THEN REPLICATE('' '', m.row_indent) ELSE '''' END + ' + CASE WHEN @data_language IS NULL THEN 'm.name' ELSE 'COALESCE(t.name, m.name)' END + ' ELSE NULL END'
               ELSE 'NULL' END + ' AS name
    , CASE WHEN v.line_order = 0 THEN m.comment ELSE NULL END AS comment
    , CASE WHEN v.line_order = 0 THEN m.code ELSE NULL END AS code1
    , ' + CASE WHEN r2 IS NOT NULL THEN 'COALESCE(r2.code, m2.code)'  WHEN m2 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 AND m.id IS NOT NULL THEN m2.code ELSE NULL END' ELSE 'NULL' END + ' AS code2
    , ' + CASE WHEN r2 IS NOT NULL THEN 'COALESCE(' + CASE WHEN @data_language IS NULL THEN 'r2.name' ELSE 'rt2.name, r2.name, t2.name' END + ', m2.name)'  WHEN m2 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 AND m2.id IS NOT NULL THEN ' + CASE WHEN @data_language IS NULL THEN 'm2.name' ELSE 'COALESCE(t2.name, m2.name)' END + ' ELSE NULL END' ELSE 'NULL' END + ' AS name2
    , ' + CASE WHEN r3 IS NOT NULL THEN 'COALESCE(r3.code, m3.code)'  WHEN m3 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 AND m.id IS NOT NULL THEN m3.code ELSE NULL END' ELSE 'NULL' END + ' AS code3
    , ' + CASE WHEN r3 IS NOT NULL THEN 'COALESCE(' + CASE WHEN @data_language IS NULL THEN 'r3.name' ELSE 'rt3.name, r3.name, t3.name' END + ', m3.name)'  WHEN m3 IS NOT NULL THEN 'CASE WHEN v.line_order = 0 AND m3.id IS NOT NULL THEN ' + CASE WHEN @data_language IS NULL THEN 'm3.name' ELSE 'COALESCE(t3.name, m3.name)' END + ' ELSE NULL END' ELSE 'NULL' END + ' AS name3
    , CASE WHEN v.line_order = 0 THEN c.name ELSE NULL END AS calc_type
    , ' + CASE WHEN d1 = 1 OR d2 = 1 OR d3 = 1 THEN 'CASE WHEN v.line_order = 0 THEN tr.name ELSE NULL END'  ELSE 'NULL' END + ' AS tax_rate
    , ' + CASE WHEN d1 = 1 OR d2 = 1 OR d3 = 1 THEN 'CASE WHEN v.line_order = 0 THEN u.name ELSE NULL END'   ELSE 'NULL' END + ' AS unit
    , NULL AS data_format'

        -- data columns
        + CASE WHEN @select_columns_sql IS NULL THEN '' ELSE '
    , ' + @select_columns_sql END

        -- first row dimension
        + CASE WHEN r1 IS NOT NULL THEN '
FROM
    dbo25.form_rows r'
        + CASE WHEN @data_language IS NULL THEN ''ELSE '
    LEFT OUTER JOIN dbo25.translations rt ON rt.company_id = ' + @company_sid + ' AND rt.table_id = 8 AND rt.member_id = r.id AND rt.language = ''' + @data_language + ''''
            END + '
    LEFT OUTER JOIN dbo25.members m ON m.id = r.member_id'
            ELSE '
FROM
    dbo25.members m'
        + CASE WHEN m1 IS NULL THEN '' ELSE '
    INNER JOIN dbo25.hierarchies h1 ON h1.member_id = m.id AND h1.parent_id = ' + CAST(m1 AS nvarchar) + level1_sql
            END
            END
        + CASE WHEN @data_language IS NULL THEN ''ELSE '
    LEFT OUTER JOIN dbo25.translations t ON t.company_id = ' + @company_sid + ' AND t.table_id = 1 AND t.member_id = m.id AND t.language = ''' + @data_language + ''''
            END + '
    LEFT OUTER JOIN dbo25.calc_types c ON c.id = m.calc_type_id'
        + CASE WHEN d1 = 1 THEN '
    LEFT OUTER JOIN dbo25.tax_rates tr ON tr.id = m.tax_rate_id
    LEFT OUTER JOIN dbo25.units u ON u.id = m.unit_id'
            ELSE '' END

        -- second row dimension
        + CASE WHEN r2 IS NOT NULL THEN '
    CROSS JOIN dbo25.form_rows r2'
        + CASE WHEN @data_language IS NULL THEN ''ELSE '
    LEFT OUTER JOIN dbo25.translations rt2 ON rt2.company_id = ' + @company_sid + ' AND rt2.table_id = 8 AND rt2.member_id = r2.id AND rt2.language = ''' + @data_language + ''''
            END + '
    LEFT OUTER JOIN dbo25.members m2 ON m2.id = r2.member_id' + mr2_sql
               WHEN d1 = 1 AND d2 IN (4, 5, 6, 7) AND c2 = 0 THEN '
    LEFT OUTER JOIN (
        SELECT
            f1.parent_id
            , mr.partner_id
        FROM
            dbo25.member_relations mr
            INNER JOIN dbo25.members m1 ON m1.id = mr.member_id
            INNER JOIN dbo25.members m2 ON m2.id = mr.partner_id
            INNER JOIN dbo25.factors f1 ON f1.member_id = mr.member_id AND f1.calc_type_id = 1
        WHERE
            m1.company_id = ' + @company_sid + '
            AND m2.dimension_id = ' + CAST(d2 AS nvarchar) + '
        GROUP BY
            f1.parent_id
            , mr.partner_id
    ) mr12 ON mr12.parent_id = m.id
    LEFT OUTER JOIN dbo25.members m2 ON m2.id = mr12.partner_id
    LEFT OUTER JOIN dbo25.hierarchies h2 ON h2.member_id = m2.id AND h2.parent_id = ' + CAST(m2 AS nvarchar) + level2_sql + mr2_sql
               WHEN m2 IS NOT NULL THEN '
    CROSS JOIN dbo25.members m2
    INNER JOIN dbo25.hierarchies h2 ON h2.member_id = m2.id AND h2.parent_id = ' + CAST(m2 AS nvarchar) + level2_sql + mr2_sql
               ELSE '' END
        + CASE WHEN @data_language IS NULL OR d2 IS NULL THEN '' ELSE '
    LEFT OUTER JOIN dbo25.translations t2 ON t2.company_id = ' + @company_sid + ' AND t2.table_id = 1 AND t2.member_id = m2.id AND t2.language = ''' + @data_language + ''''
            END
        + CASE WHEN d2 = 1 THEN '
    LEFT OUTER JOIN dbo25.tax_rates tr ON tr.id = m2.tax_rate_id
    LEFT OUTER JOIN dbo25.units u ON u.id = m2.unit_id'
               ELSE '' END

        -- third row dimension
        + CASE WHEN r3 IS NOT NULL THEN '
    CROSS JOIN dbo25.form_rows r3'
        + CASE WHEN @data_language IS NULL THEN ''ELSE '
    LEFT OUTER JOIN dbo25.translations rt3 ON rt3.company_id = ' + @company_sid + ' AND rt3.table_id = 8 AND rt3.member_id = m.id AND rt3.language = ''' + @data_language + ''''
            END + '
    LEFT OUTER JOIN dbo25.members m3 ON m3.id = r3.member_id' + mr3_sql
               WHEN d1 = 1 AND d3 IN (4, 5, 6, 7) AND c3 = 0 THEN '
    LEFT OUTER JOIN (
        SELECT
            f1.parent_id
            , mr.partner_id
        FROM
            dbo25.member_relations mr
            INNER JOIN dbo25.members m1 ON m1.id = mr.member_id
            INNER JOIN dbo25.members m2 ON m2.id = mr.partner_id
            INNER JOIN dbo25.factors f1 ON f1.member_id = mr.member_id AND f1.calc_type_id = 1
        WHERE
            m1.company_id = ' + @company_sid + '
            AND m2.dimension_id = ' + CAST(d3 AS nvarchar) + '
        GROUP BY
            f1.parent_id
            , mr.partner_id
    ) mr13 ON mr13.parent_id = m.id
    LEFT OUTER JOIN dbo25.members m3 ON m3.id = mr13.partner_id
    LEFT OUTER JOIN dbo25.hierarchies h3 ON h3.member_id = m3.id AND h3.parent_id = ' + CAST(m3 AS nvarchar) + level3_sql + mr3_sql
               WHEN m3 IS NOT NULL THEN '
    CROSS JOIN dbo25.members m3
    INNER JOIN dbo25.hierarchies h3 ON h3.member_id = m3.id AND h3.parent_id = ' + CAST(m3 AS nvarchar) + level3_sql + mr3_sql
               ELSE '' END
        + CASE WHEN @data_language IS NULL OR d3 IS NULL THEN '' ELSE '
    LEFT OUTER JOIN dbo25.translations t3 ON t3.company_id = ' + @company_sid + ' AND t3.table_id = 1 AND t3.member_id = m3.id AND t3.language = ''' + @data_language + ''''
            END
        + CASE WHEN d3 = 1 THEN '
    LEFT OUTER JOIN dbo25.tax_rates tr ON tr.id = m3.tax_rate_id
    LEFT OUTER JOIN dbo25.units u ON u.id = m3.unit_id'
               ELSE '' END
        FROM
            @rows r
        )

        -- report data for the current period
        + CASE WHEN @pivot_columns_cp_sql IS NULL THEN '' ELSE '
    LEFT OUTER JOIN ('

            -- two joins for SUM and AVG if the row axis contains averages
            + CASE WHEN @has_row_averages = 1 THEN '
        SELECT * FROM
        (' ELSE '' END + '
        SELECT
            CASE WHEN f1.parent_id = f1.member_id THEN f.source_value
                    WHEN m1.unit_id = d1.unit_id THEN f.source_value * f1.factor
                    ELSE f.value * f1.factor END AS value'
            -- rows
            + (SELECT '
            , f' + d + '.parent_id AS id' + CASE WHEN header_order = 1 THEN '' ELSE CAST(header_order AS nvarchar) END
                FROM @parameters p WHERE p.axis_type_id = 2 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + d' + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'
            + '
        FROM'
            -- source cube data
            + '
            dbo25.facts f
'
            -- calculation factors and filters by rowsets or hierarchies
            + (SELECT '
            INNER JOIN dbo25.factors f' + d + ' ON f' + d + '.member_id = f.id' + d
                + CASE WHEN p.axis_type_id = 1 THEN ' AND f' + d + '.parent_id = ' + m
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 2 AND @cf2 IS NOT NULL THEN ' AND f2.parent_id IN (' + @cf2 + ')'
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 3 AND @cf3 IS NOT NULL THEN ' AND f3.parent_id IN (' + @cf3 + ')'
                       WHEN rowset_id IS NOT NULL THEN '
            INNER JOIN dbo25.form_rows r'+ d
                + ' ON r' + d + '.member_id = f' + d + '.parent_id AND r' + d + '.rowset_id = ' + m
                       ELSE '
            INNER JOIN dbo25.hierarchies h' + d
                + ' ON h' + d + '.member_id = f' + d + '.parent_id AND h' + d + '.parent_id = ' + m
                       END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)') + '

            INNER JOIN dbo25.members m1 ON m1.id = f.id1'

            -- column codes for the column axis and accounts anyway
            + (SELECT CASE WHEN p.axis_type_id = 3 OR p.dimension_id = 1 THEN '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.id = f' + d + '.parent_id'
                    ELSE '' END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- filter for total and opening balance (revaluation is calculated as total)
            + '
        WHERE
            m1.company_id = ' + @company_sid + '
            AND (f1.calc_type_id = 1 AND f2.calc_type_id = 1
              OR f1.calc_type_id = 3 AND f2.calc_type_id = 3
            )'

            -- SUM PIVOT
            + '
        ) s PIVOT (
            SUM(value) FOR code IN (' + @pivot_columns_cp_sql + ')'

            -- AVG part if the row axis contains averages
            + CASE WHEN @has_row_averages = 1 THEN '
        ) t
        UNION ALL SELECT * FROM
        (
        SELECT
            CASE WHEN f1.parent_id = f1.member_id THEN f.source_value
                 WHEN m1.unit_id = d1.unit_id THEN f.source_value * f1.factor
                 ELSE f.value * f1.factor END AS value'
            -- rows
            + (SELECT '
            , f' + d + '.parent_id AS id' + CASE WHEN header_order = 1 THEN '' ELSE CAST(header_order AS nvarchar) END
                FROM @parameters p WHERE p.axis_type_id = 2 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + d' + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'

            -- source cube data
            + '
        FROM
            dbo25.facts f
'
            -- calculation factors and filters by rowsets or hierarchies (without opening balance)
            + (SELECT '
            INNER JOIN dbo25.factors f' + d + ' ON f' + d + '.member_id = f.id' + d
                + CASE WHEN p.axis_type_id = 1 THEN ' AND f' + d + '.parent_id = ' + m
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 2 AND @cf2 IS NOT NULL THEN ' AND f2.parent_id IN (' + @cf2 + ')'
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 3 AND @cf3 IS NOT NULL THEN ' AND f3.parent_id IN (' + @cf3 + ')'
                       WHEN rowset_id IS NOT NULL THEN '
            INNER JOIN dbo25.form_rows r'+ d
                + ' ON r' + d + '.member_id = f' + d + '.parent_id AND r' + d + '.rowset_id = ' + m
                       ELSE '
            INNER JOIN dbo25.hierarchies h' + d
                + ' ON h' + d + '.member_id = f' + d + '.parent_id AND h' + d + '.parent_id = ' + m
                       END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)') + '

            INNER JOIN dbo25.members m1 ON m1.id = f.id1'

            -- column codes for the column axis and accounts anyway
            + (SELECT CASE WHEN p.axis_type_id = 3 OR p.dimension_id = 1 THEN '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.id = f' + d + '.parent_id'
                    ELSE '' END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- AVG members only
            + '
        WHERE
            m1.company_id = ' + @company_sid + '
            AND f1.calc_type_id = 2 AND f2.calc_type_id = 1'

            -- AVG PIVOT
            + '
        ) s PIVOT (
            AVG(value) FOR code IN (' + @pivot_columns_cp_sql + ')
        ) t' ELSE '' END

            -- JOIN members
            + '
    ) cp ON cp.id = m.id'
            + COALESCE((SELECT ' AND cp.id' + CAST(header_order AS nvarchar) + ' = m' + CAST(header_order AS nvarchar) + '.id'
                FROM @parameters p WHERE p.axis_type_id = 2 AND header_order > 1 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), '')
        END

        -- report data for the same period of the previous year
        + CASE WHEN @pivot_columns_sp_sql IS NULL THEN '' ELSE '
    LEFT OUTER JOIN ('

            -- two joins for SUM and AVG if the row axis contains averages
            + CASE WHEN @has_row_averages = 1 THEN '
        SELECT * FROM
        (' ELSE '' END + '
        SELECT
            CASE WHEN f1.parent_id = f1.member_id THEN f.source_value
                 WHEN m1.unit_id = d1.unit_id THEN f.source_value * f1.factor
                 ELSE f.value * f1.factor END AS value'
            -- rows
            + (SELECT '
            , f' + d + '.parent_id AS id' + CASE WHEN header_order = 1 THEN '' ELSE CAST(header_order AS nvarchar) END
                FROM @parameters p WHERE p.axis_type_id = 2 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + ' + CASE WHEN p.dimension_id = 3 THEN 'sp' ELSE 'd' END + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'

            -- source cube data
            + '
        FROM
            dbo25.facts f
'
            -- calculation factors and filters by rowsets or hierarchies
            + (SELECT '
            INNER JOIN dbo25.factors f' + d + ' ON f' + d + '.member_id = f.id' + d
                -- replacing source times (2) and categories (3) to the same period
                + CASE WHEN p.axis_type_id = 1 THEN ' AND f' + d + '.parent_id = ' + CASE
                            WHEN p.dimension_id = 2 THEN COALESCE(CAST(p.same_period_id AS nvarchar), 'NULL')
                            WHEN p.dimension_id = 3 AND calc_type_id = 6 THEN COALESCE(CAST((SELECT id FROM dbo25.members WHERE same_period_id = p.member_id) AS nvarchar), 'NULL')
                            ELSE m
                            END
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 2 AND @sf2 IS NOT NULL THEN ' AND f2.parent_id IN (' + @sf2 + ')'
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 3 AND @sf3 IS NOT NULL THEN ' AND f3.parent_id IN (' + @sf3 + ')'
                       WHEN rowset_id IS NOT NULL THEN '
            INNER JOIN dbo25.form_rows r'+ d
                + ' ON r' + d + '.member_id = f' + d + '.parent_id AND r' + d + '.rowset_id = ' + m
                       ELSE '
            INNER JOIN dbo25.hierarchies h' + d
                + ' ON h' + d + '.member_id = f' + d + '.parent_id AND h' + d + '.parent_id = ' + m
                       END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)') + '

            INNER JOIN dbo25.members m1 ON m1.id = f.id1'

            -- column codes for the column axis and accounts anyway
            + (SELECT CASE WHEN p.axis_type_id = 3 OR p.dimension_id = 1 THEN '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.id = f' + d + '.parent_id'
                    -- filter of categories
                    + CASE WHEN p.dimension_id = 3 THEN '
            INNER JOIN dbo25.members sp' + d + ' ON sp' + d + '.id = d' + d + '.same_period_id'
                        ELSE '' END
                    ELSE '' END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- filter for total and opening balance (revaluation is calculated as total)
            + '
        WHERE
            m1.company_id = ' + @company_sid + '
            AND (f1.calc_type_id = 1 AND f2.calc_type_id = 1
              OR f1.calc_type_id = 3 AND f2.calc_type_id = 3
            )'

            -- SUM PIVOT
            + '
        ) s PIVOT (
            SUM(value) FOR code IN (' + @pivot_columns_sp_sql + ')'


            -- AVG part if the row axis contains averages
            + CASE WHEN @has_row_averages = 1 THEN '
        ) t
        UNION ALL SELECT * FROM
        (
        SELECT
            CASE WHEN f1.parent_id = f1.member_id THEN f.source_value
                 WHEN m1.unit_id = d1.unit_id THEN f.source_value * f1.factor
                 ELSE f.value * f1.factor END AS value'
            -- rows
            + (SELECT '
            , f' + d + '.parent_id AS id' + CASE WHEN header_order = 1 THEN '' ELSE CAST(header_order AS nvarchar) END
                FROM @parameters p WHERE p.axis_type_id = 2 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + ' + CASE WHEN p.dimension_id = 3 THEN 'sp' ELSE 'd' END + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'

            -- source cube data
            + '
        FROM
            dbo25.facts f
'
            -- calculation factors and filters by rowsets or hierarchies (without opening balance)
            + (SELECT '
            INNER JOIN dbo25.factors f' + d + ' ON f' + d + '.member_id = f.id' + d
                -- replacing source times (2) and categories (3) to the same period
                + CASE WHEN p.axis_type_id = 1 THEN ' AND f' + d + '.parent_id = ' + CASE
                            WHEN p.dimension_id = 2 THEN COALESCE(CAST(p.same_period_id AS nvarchar), 'NULL')
                            WHEN p.dimension_id = 3 AND calc_type_id = 6 THEN COALESCE(CAST((SELECT id FROM dbo25.members WHERE same_period_id = p.member_id) AS nvarchar), 'NULL')
                            ELSE m
                            END
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 2 AND @sf2 IS NOT NULL THEN ' AND f2.parent_id IN (' + @sf2 + ')'
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 3 AND @sf3 IS NOT NULL THEN ' AND f3.parent_id IN (' + @sf3 + ')'
                       WHEN rowset_id IS NOT NULL THEN '
            INNER JOIN dbo25.form_rows r'+ d
                + ' ON r' + d + '.member_id = f' + d + '.parent_id AND r' + d + '.rowset_id = ' + m
                       ELSE '
            INNER JOIN dbo25.hierarchies h' + d
                + ' ON h' + d + '.member_id = f' + d + '.parent_id AND h' + d + '.parent_id = ' + m
                       END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)') + '

            INNER JOIN dbo25.members m1 ON m1.id = f.id1'

            -- column codes for the column axis and accounts anyway
            + (SELECT CASE WHEN p.axis_type_id = 3 OR p.dimension_id = 1 THEN '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.id = f' + d + '.parent_id'
                    -- filter of categories
                    + CASE WHEN p.dimension_id = 3 THEN '
            INNER JOIN dbo25.members sp' + d + ' ON sp' + d + '.id = d' + d + '.same_period_id'
                        ELSE '' END
                    ELSE '' END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- AVG members only
            + '
        WHERE
            m1.company_id = ' + @company_sid + '
            AND f1.calc_type_id = 2 AND f2.calc_type_id = 1'

            -- AVG PIVOT
            + '
        ) s PIVOT (
            AVG(value) FOR code IN (' + @pivot_columns_sp_sql + ')
        ) t' ELSE '' END

            -- JOIN members
            + '
    ) sp ON sp.id = m.id'
        + COALESCE((SELECT ' AND sp.id' + CAST(header_order AS nvarchar) + ' = m' + CAST(header_order AS nvarchar) + '.id'
            FROM @parameters p WHERE p.axis_type_id = 2 AND header_order > 1 ORDER BY header_order
            FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), '')
        END

        -- string data for the current period
        + CASE WHEN @has_strings = 0 OR @pivot_columns_cp_sql IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            f.value AS value'
            -- rows
            + (SELECT '
            , f.id' + d + ' AS id' + CASE WHEN header_order = 1 THEN '' ELSE CAST(header_order AS nvarchar) END
                FROM @parameters p WHERE p.axis_type_id = 2 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + d' + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'

            -- source cube data
            + '
        FROM
            dbo25.strings f'

            -- calculation factors and filters by rowsets or hierarchies
            + (SELECT CASE WHEN p.axis_type_id = 1 THEN ''
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 2 AND @cf2 IS NOT NULL THEN ''
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 3 AND @cf3 IS NOT NULL THEN ''
                       WHEN rowset_id IS NOT NULL THEN '
            INNER JOIN dbo25.form_rows r'+ d
                + ' ON r' + d + '.member_id = f.id' + d + ' AND r' + d + '.rowset_id = ' + m
                       ELSE '
            INNER JOIN dbo25.hierarchies h' + d
                + ' ON h' + d + '.member_id = f.id' + d + ' AND h' + d + '.parent_id = ' + m
                       END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)') + '
'
            -- column codes
            + (SELECT '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.id = f.id' + d
                FROM @parameters p WHERE p.axis_type_id = 3 OR p.dimension_id = 1  ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- filter string values
            + '
        WHERE
            d1.company_id = ' + @company_sid + '
            AND d1.calc_type_id = 8'

            -- string members with filters
            + (SELECT
                    CASE WHEN p.axis_type_id = 1 THEN '
            AND f.id' + d + ' = ' + m
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 2 AND @cf2 IS NOT NULL THEN '
            AND f.id2 IN (' + @cf2 + ')'
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 3 AND @cf3 IS NOT NULL THEN '
            AND f.id3 IN (' + @cf3 + ')'
                       ELSE '' END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- MAX PIVOT
            + '
        ) s PIVOT (
            MAX(value) FOR code IN (' + @pivot_columns_cp_sql + ')'

            -- JOIN members
            + '
    ) cs ON cs.id = m.id'
            + COALESCE((SELECT ' AND cs.id' + CAST(header_order AS nvarchar) + ' = m' + CAST(header_order AS nvarchar) + '.id'
                FROM @parameters p WHERE p.axis_type_id = 2 AND header_order > 1 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), '')
        END

        -- string data for the same period
        + CASE WHEN @has_strings = 0 OR @pivot_columns_sp_sql IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            f.value AS value'
            -- rows
            + (SELECT '
            , f.id' + d + ' AS id' + CASE WHEN header_order = 1 THEN '' ELSE CAST(header_order AS nvarchar) END
                FROM @parameters p WHERE p.axis_type_id = 2 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + ' + CASE WHEN p.dimension_id = 3 THEN 'sp' ELSE 'd' END + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'

            -- source cube data
            + '
        FROM
            dbo25.strings f'

            -- calculation factors and filters by rowsets or hierarchies
            + (SELECT CASE WHEN p.axis_type_id = 1 THEN ''
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 2 AND @sf2 IS NOT NULL THEN ''
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 3 AND @cf3 IS NOT NULL THEN ''
                       WHEN p.rowset_id IS NOT NULL THEN '
            INNER JOIN dbo25.form_rows r'+ d
                + ' ON r' + d + '.member_id = f.id' + d + ' AND r' + d + '.rowset_id = ' + m
                       ELSE '
            INNER JOIN dbo25.hierarchies h' + d
                + ' ON h' + d + '.member_id = f.id' + d + ' AND h' + d + '.parent_id = ' + m
                       END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)') + '
'
            -- column codes
            + (SELECT '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.id = f.id' + d
                    -- filter of categories
                    + CASE WHEN p.dimension_id = 3 THEN '
            INNER JOIN dbo25.members sp' + d + ' ON sp' + d + '.id = d' + d + '.same_period_id'
                        ELSE '' END
                FROM @parameters p WHERE p.axis_type_id = 3 OR p.dimension_id = 1  ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- filter string values
            + '
        WHERE
            d1.company_id = ' + @company_sid + '
            AND d1.calc_type_id = 8'

            -- string members with filters
            + (SELECT
                    CASE WHEN p.axis_type_id = 1 THEN '
            AND f.id' + d + ' = ' + CASE
                            WHEN p.dimension_id = 2 THEN COALESCE(CAST(p.same_period_id AS nvarchar), 'NULL')
                            WHEN p.dimension_id = 3 AND calc_type_id = 6 THEN COALESCE(CAST((SELECT id FROM dbo25.members WHERE same_period_id = p.member_id) AS nvarchar), 'NULL')
                            ELSE m
                            END
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 2 AND @cf2 IS NOT NULL THEN '
            AND f.id2 IN (' + @cf2 + ')'
                       WHEN p.axis_type_id = 3 AND p.dimension_id = 3 AND @cf3 IS NOT NULL THEN '
            AND f.id3 IN (' + @cf3 + ')'
                       ELSE '' END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- MAX PIVOT
            + '
        ) s PIVOT (
            MAX(value) FOR code IN (' + @pivot_columns_sp_sql + ')'

            -- JOIN members
            + '
    ) ss ON ss.id = m.id'
            + COALESCE((SELECT ' AND ss.id' + CAST(header_order AS nvarchar) + ' = m' + CAST(header_order AS nvarchar) + '.id'
                FROM @parameters p WHERE p.axis_type_id = 2 AND header_order > 1 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), '')
        END

        -- currency rates for the current period
        + CASE WHEN @has_rates = 0 OR @pivot_columns_cp_sql IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            f.currency_rate AS value
            , f.base_currency_id AS id'
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + d' + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'
            + '
        FROM
            dbo25.currency_rates f
            INNER JOIN dbo25.units u ON u.id = f.base_currency_id'

            -- column codes
            + (SELECT CASE WHEN p.dimension_id = 2 THEN '
            INNER JOIN dbo25.members d2 ON d2.id = f.time_id'
                            WHEN p.dimension_id = 3 THEN '
            INNER JOIN dbo25.members d3 ON d3.id = f.category_id'
                            ELSE '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.company_id = ' + @company_sid
                            END
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            + '
        WHERE
            u.company_id = ' + @company_sid + '
            AND NOT f.base_currency_id = ' + @functional_currency_sid + '
            AND f.quote_currency_id = ' + @functional_currency_sid
            -- filters for column code members
            + (SELECT CASE WHEN p.axis_type_id = 3 THEN
                                CASE WHEN p.dimension_id = 2 THEN ''
                                     WHEN p.dimension_id = 3 AND @cf3 IS NOT NULL THEN '
            AND f.category_id IN (' + @cf3 + ')'
                                     WHEN p.dimension_id = 3 THEN '' ELSE '
            AND d' + d + '.dimension_id = ' + d
                                     END
                            WHEN p.axis_type_id = 1 AND p.dimension_id = 2 THEN '
            AND f.time_id = ' + m
                            WHEN p.dimension_id = 3 THEN '
            AND f.category_id = ' + m
                            ELSE ''
                            END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            + '
        UNION ALL
        SELECT
            f.currency_rate AS value
            , f.quote_currency_id AS id'
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + d' + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'
            + '
        FROM
            dbo25.currency_rates f
            INNER JOIN dbo25.units u ON u.id = f.quote_currency_id'

            -- column codes
            + (SELECT CASE WHEN p.dimension_id = 2 THEN '
            INNER JOIN dbo25.members d2 ON d2.id = f.time_id'
                            WHEN p.dimension_id = 3 THEN '
            INNER JOIN dbo25.members d3 ON d3.id = f.category_id'
                            ELSE '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.company_id = ' + @company_sid
                            END
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            + '
        WHERE
            u.company_id = ' + @company_sid + '
            AND f.base_currency_id = ' + @functional_currency_sid + '
            AND NOT f.quote_currency_id = ' + @functional_currency_sid
            -- filters for column code members
            + (SELECT CASE WHEN p.axis_type_id = 3 THEN
                                CASE WHEN p.dimension_id = 2 THEN ''
                                     WHEN p.dimension_id = 3 AND @cf3 IS NOT NULL THEN '
            AND f.category_id IN (' + @cf3 + ')'
                                     WHEN p.dimension_id = 3 THEN '' ELSE '
            AND d' + d + '.dimension_id = ' + d
                                     END
                            WHEN p.axis_type_id = 1 AND p.dimension_id = 2 THEN '
            AND f.time_id = ' + m
                            WHEN p.dimension_id = 3 THEN '
            AND f.category_id = ' + m
                            ELSE ''
                            END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- MAX PIVOT
            + '
        ) s PIVOT (
            AVG(value) FOR code IN (' + @pivot_columns_cp_sql + ')'

            -- JOIN members
            + '
    ) cr ON cr.id = ' + (SELECT CASE WHEN d2 = 1 THEN 'm2' WHEN d3 = 1 THEN 'm3' ELSE 'm' END FROM @rows) + '.unit_id AND '
        + (SELECT CASE WHEN d2 = 1 THEN 'm2' WHEN d3 = 1 THEN 'm3' ELSE 'm' END FROM @rows) + '.calc_type_id = 10'
        END

        -- currency rates for the same period
        + CASE WHEN @has_rates = 0 OR @pivot_columns_sp_sql IS NULL THEN '' ELSE '
    LEFT OUTER JOIN (
        SELECT
            f.currency_rate AS value
            , f.base_currency_id AS id'
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + ' + CASE WHEN p.dimension_id = 3 THEN 'sp' ELSE 'd' END + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'
            + '
        FROM
            dbo25.currency_rates f
            INNER JOIN dbo25.units u ON u.id = f.base_currency_id'

            -- column codes
            + (SELECT CASE WHEN p.dimension_id = 2 THEN '
            INNER JOIN dbo25.members d2 ON d2.id = f.time_id'
                            WHEN p.dimension_id = 3 THEN '
            INNER JOIN dbo25.members d3 ON d3.id = f.category_id
            INNER JOIN dbo25.members sp3 ON sp3.id = d3.same_period_id'
                            ELSE '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.company_id = ' + @company_sid
                            END
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            + '
        WHERE
            u.company_id = ' + @company_sid + '
            AND NOT f.base_currency_id = ' + @functional_currency_sid + '
            AND f.quote_currency_id = ' + @functional_currency_sid
            -- filters for column code members
            + (SELECT CASE WHEN p.axis_type_id = 3 THEN
                                CASE WHEN p.dimension_id = 2 THEN ''
                                     WHEN p.dimension_id = 3 AND @sf3 IS NOT NULL THEN '
            AND f.category_id IN (' + @sf3 + ')'
                                     WHEN p.dimension_id = 3 THEN '' ELSE '
            AND d' + d + '.dimension_id = ' + d
                                     END
                            WHEN p.axis_type_id = 1 AND p.dimension_id = 2 THEN '
            AND f.time_id = ' + CAST(p.same_period_id AS nvarchar)
                            WHEN p.dimension_id = 3 THEN '
            AND f.category_id = ' + m
                            ELSE ''
                            END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            + '
        UNION ALL
        SELECT
            f.currency_rate AS value
            , f.quote_currency_id AS id'
            -- columns
            + '
            , ' + STUFF((
                SELECT ' + '' '' + ' + CASE WHEN p.dimension_id = 3 THEN 'sp' ELSE 'd' END + d + '.code'
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY header_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 9, '') + ' AS code'
            + '
        FROM
            dbo25.currency_rates f
            INNER JOIN dbo25.units u ON u.id = f.quote_currency_id'

            -- column codes
            + (SELECT CASE WHEN p.dimension_id = 2 THEN '
            INNER JOIN dbo25.members d2 ON d2.id = f.time_id'
                            WHEN p.dimension_id = 3 THEN '
            INNER JOIN dbo25.members d3 ON d3.id = f.category_id
            INNER JOIN dbo25.members sp3 ON sp3.id = d3.same_period_id'
                            ELSE '
            INNER JOIN dbo25.members d' + d + ' ON d' + d + '.company_id = ' + @company_sid
                            END
                FROM @parameters p WHERE p.axis_type_id = 3 ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
            + '
        WHERE
            u.company_id = ' + @company_sid + '
            AND f.base_currency_id = ' + @functional_currency_sid + '
            AND NOT f.quote_currency_id = ' + @functional_currency_sid
            -- filters for column code members
            + (SELECT CASE WHEN p.axis_type_id = 3 THEN
                                CASE WHEN p.dimension_id = 2 THEN ''
                                     WHEN p.dimension_id = 3 AND @sf3 IS NOT NULL THEN '
            AND f.category_id IN (' + @sf3 + ')'
                                     WHEN p.dimension_id = 3 THEN '' ELSE '
            AND d' + d + '.dimension_id = ' + d
                                     END
                            WHEN p.axis_type_id = 1 AND p.dimension_id = 2 THEN '
            AND f.time_id = ' + CAST(p.same_period_id AS nvarchar)
                            WHEN p.dimension_id = 3 THEN '
            AND f.category_id = ' + m
                            ELSE ''
                            END
                FROM @parameters p ORDER BY join_order
                FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

            -- MAX PIVOT
            + '
        ) s PIVOT (
            AVG(value) FOR code IN (' + @pivot_columns_sp_sql + ')'

            -- JOIN members
            + '
    ) sr ON sr.id = ' + (SELECT CASE WHEN d2 = 1 THEN 'm2' WHEN d3 = 1 THEN 'm3' ELSE 'm' END FROM @rows) + '.unit_id AND '
        + (SELECT CASE WHEN d2 = 1 THEN 'm2' WHEN d3 = 1 THEN 'm3' ELSE 'm' END FROM @rows) + '.calc_type_id = 10'
        END

        -- Permission JOIN

        + (SELECT
            COALESCE('
    ' + pm1.permission_join + ' m.id', '')
            + COALESCE('
    ' + pm2.permission_join + ' m2.id', '')
            + COALESCE('
    ' + pm3.permission_join + ' m3.id', '')
    FROM
        @rows r
        LEFT OUTER JOIN @permissions pm1 ON pm1.dimension_id = r.d1
        LEFT OUTER JOIN @permissions pm2 ON pm2.dimension_id = r.d2
        LEFT OUTER JOIN @permissions pm3 ON pm3.dimension_id = r.d3
    )
        -- 2.2 Subtotals
        + COALESCE(@subtotals_join, '')

        -- 4.0 show_line, show_line_before, show_line_after
        + '
    CROSS JOIN (VALUES (-1, NULL), (0, 1), (1, NULL)) v(line_order, line_factor)'

        -- WHERE
        + (SELECT '
WHERE'
            -- first row dimension
            + CASE WHEN r1 IS NOT NULL THEN '
    r.rowset_id = ' + CAST(r1 AS nvarchar)
            + COALESCE('
    AND (' + pm1.permission_where + 'OR r.id IS NOT NULL AND r.member_id IS NULL)', '')
                ELSE '
    m.dimension_id = ' + CAST(d1 AS nvarchar)
            + COALESCE('
    AND ' + pm1.permission_where, '')
                END

            -- second row dimension
            + CASE WHEN r2 IS NOT NULL THEN '
    AND r.rowset_id = ' + CAST(r2 AS nvarchar) ELSE '' END
            -- filters rows with the first dimension parents and the second dimension parent
            -- and rows with update permissions that may contain editable data
            -- when m2 is CROSS JOIN (not INNER JOIN for accounts and d4-d7)
            + CASE WHEN m2 IS NULL THEN ''
                   WHEN d1 = 1 AND d2 IN (4, 5, 6, 7) AND c2 = 0 THEN ''
                   WHEN mr2_sql > '' THEN '
    AND (m.is_calculated = 1 AND m2.id = ' + CAST(m2 AS nvarchar)
        + CASE WHEN r1 IS NOT NULL THEN ' OR r.member_id IS NULL AND m2.id = ' + CAST(m2 AS nvarchar) ELSE '' END
        + ' OR mr2.update_permission = 1)' ELSE '' END
            + COALESCE('
    AND ' + pm2.permission_where, '')

            -- third row dimension
            + CASE WHEN r3 IS NOT NULL THEN '
    AND r.rowset_id = ' + CAST(r3 AS nvarchar) ELSE '' END
            -- filters rows with the first dimension parents and the third dimension parent
            -- and rows with update permissions that may contain editable data
            -- when m3 is CROSS JOIN (not INNER JOIN for accounts and d4-d7)
            + CASE WHEN m3 IS NULL THEN ''
                   WHEN d1 = 1 AND d3 IN (4, 5, 6, 7) AND c3 = 0 THEN ''
                   WHEN mr3_sql > '' THEN '
    AND (m.is_calculated = 1 AND m3.id = ' + CAST(m3 AS nvarchar)
        + CASE WHEN r1 IS NOT NULL THEN ' OR r.member_id IS NULL AND m3.id = ' + CAST(m3 AS nvarchar) ELSE '' END
        + ' OR mr3.update_permission = 1)' ELSE '' END
            + COALESCE('
    AND ' + pm3.permission_where, '')
    -- 4.0 show_line, show_line_before, show_line_after
    + '
    AND (v.line_order = 0 AND m.show_line = 1
        OR v.line_order = -1 AND m.show_line_before = 1
        OR v.line_order = 1 AND m.show_line_after = 1'
            + CASE WHEN r1 IS NOT NULL THEN '
        OR v.line_order = 0 AND r.member_id IS NULL'
                ELSE ''
                END + '
        )'
    FROM
        @rows r
        LEFT OUTER JOIN @permissions pm1 ON pm1.dimension_id = r.d1
        LEFT OUTER JOIN @permissions pm2 ON pm2.dimension_id = r.d2
        LEFT OUTER JOIN @permissions pm3 ON pm3.dimension_id = r.d3
    )
    -- 2.2 Member relations with page parameters
    + COALESCE(@page_member_relation_where, '')

    -- 2.2 Subtotals
    + CASE WHEN COALESCE(@subtotals_join, '') > '' THEN '
    AND st.id IS NULL'
        ELSE '' END

    + '
ORDER BY
    sort_order'

END

BEGIN -- Output

IF @sql IS NULL
    BEGIN
    SELECT * FROM dbo25.view_dynamic_form_fields
    RETURN
    END

DECLARE @print int = 1

IF @print = 0
    BEGIN
    PRINT SUBSTRING(@sql, 1, 4000)
    IF LEN(@sql) > 4000  PRINT SUBSTRING(@sql, 4001, 4000)
    IF LEN(@sql) > 8000  PRINT SUBSTRING(@sql, 8001, 4000)
    IF LEN(@sql) > 12000 PRINT SUBSTRING(@sql, 12001, 4000)
    END

EXEC (@sql)

END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_run_form
--
-- This procedure requires @table_name or @form_id.
-- =============================================

CREATE PROCEDURE [xls25].[usp_run_form_change]
    @column_name nvarchar(255) = NULL
    , @table_name nvarchar(255) = NULL
    , @cell_value nvarchar(255) = NULL
    , @cell_number_value float = NULL
    , @cell_address nvarchar(255) = NULL
    , @changed_cell_count int = NULL
    , @changed_cell_index int = NULL
    , @id int = NULL
    , @member_id int = NULL
    , @member_id2 int = NULL
    , @member_id3 int = NULL
    , @form_id int = NULL
    , @form_p1 int = NULL
    , @form_p2 int = NULL
    , @form_p3 int = NULL
    , @form_p4 int = NULL
    , @form_p5 int = NULL
    , @form_p6 int = NULL
    , @form_p7 int = NULL
    , @quiet_mode tinyint = 0
    , @data_language varchar(10) = NULL
--WITH ENCRYPTION
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

DECLARE @company_id int

IF @quiet_mode IS NULL SET @quiet_mode = 0

-- Updating row properties

IF @column_name IN (
    'id', 'id2', 'id3', 'sort_oder', 'sort_oder2', 'sort_oder3',
    'member_id', 'member_id2', 'member_id3',
    'decimal_places', 'is_percent', 'row_color', 'row_bold', 'row_indent',
    'code', 'name', 'code1', 'code2', 'name2', 'code3', 'name3',
    'calc_type', 'tax_rate', 'unit', 'data_format'
    )
    BEGIN

    -- Only analysts and developers may edit these columns

    IF COALESCE(IS_MEMBER('planning_app_analysts'), 0) = 0
        AND COALESCE(IS_MEMBER('planning_app_developers'), 0) = 0
        RETURN

    IF @id IS NOT NULL
        BEGIN
        SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.form_rows f INNER JOIN dbo25.members t ON t.id = f.rowset_id INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE f.id = @id AND u.username = USER_NAME() AND u.is_active = 1)

        IF @company_id IS NULL
            RETURN

        IF @column_name = 'decimal_places'
            UPDATE dbo25.form_rows SET decimal_places = @cell_number_value WHERE id = @id
        ELSE IF @column_name = 'is_percent'
            UPDATE dbo25.form_rows SET is_percent = @cell_number_value WHERE id = @id
        ELSE IF @column_name = 'row_color'
            UPDATE dbo25.form_rows SET row_color = @cell_number_value WHERE id = @id
        ELSE IF @column_name = 'row_bold'
            UPDATE dbo25.form_rows SET row_bold = @cell_number_value WHERE id = @id
        ELSE IF @column_name = 'row_indent'
            UPDATE dbo25.form_rows SET row_indent = @cell_number_value WHERE id = @id
        ELSE IF @column_name = 'code'
            UPDATE dbo25.form_rows SET code = @cell_value WHERE id = @id
        ELSE IF @column_name = 'name'
            UPDATE dbo25.form_rows SET name = @cell_value WHERE id = @id
        ELSE IF @column_name = 'sort_order'
            UPDATE dbo25.form_rows SET sort_order = @cell_number_value WHERE id = @id
        END
    ELSE IF @member_id IS NOT NULL
        BEGIN
        SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @member_id AND u.username = USER_NAME() AND u.is_active = 1)

        IF @company_id IS NULL
            RETURN

        IF @column_name = 'decimal_places'
            UPDATE dbo25.members SET decimal_places = @cell_number_value WHERE id = @member_id
        ELSE IF @column_name = 'is_percent'
            UPDATE dbo25.members SET is_percent = @cell_number_value WHERE id = @member_id
        ELSE IF @column_name = 'row_color'
            UPDATE dbo25.members SET row_color = @cell_number_value WHERE id = @member_id
        ELSE IF @column_name = 'row_bold'
            UPDATE dbo25.members SET row_bold = @cell_number_value WHERE id = @member_id
        ELSE IF @column_name = 'row_indent'
            UPDATE dbo25.members SET row_indent = @cell_number_value WHERE id = @member_id
        ELSE IF @column_name = 'sort_order'
            UPDATE dbo25.members SET sort_order = @cell_number_value WHERE id = @member_id
        END

    RETURN
    END

-- Getting form_id from name

IF @form_id IS NULL
    BEGIN

    DECLARE @form_code nvarchar(128)

    IF LEFT(@table_name, 1) = '['
        SET @form_code = REPLACE(REPLACE(LEFT(@table_name, LEN(@table_name) - 1),'[xls25a].[', ''), '[xls25b].[', '')
    ELSE
        SET @form_code = REPLACE(REPLACE(@table_name, 'xls25a.', ''), 'xls25b.', '')

    SELECT
        @company_id = f.company_id, @form_id = f.id
    FROM
        dbo25.forms f
        INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
        INNER JOIN dbo25.companies c ON c.id = f.company_id
    WHERE
        c.code + '_' + f.code = @form_code

    IF @form_id IS NULL
        BEGIN
        SET @message = dbo25.get_translated_string('Form not found: %s', @data_language)

        IF @quiet_mode = 0
            RAISERROR (@message, 11, 0, @table_name)

        RETURN
        END
    END
ELSE
    BEGIN
    SELECT TOP 1 @company_id = u.company_id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @form_id AND u.username = USER_NAME() AND u.is_active = 1

    IF @company_id IS NULL
        RETURN
    END

DECLARE @functional_currency_id int = (SELECT TOP 1 id FROM dbo25.units WHERE company_id = @company_id AND is_functional_currency = 1 ORDER BY id)

-- Decoding member codes from column names

DECLARE @code1 nvarchar(128)
DECLARE @code2 nvarchar(128)
DECLARE @code3 nvarchar(128)

DECLARE @index int

DECLARE @str nvarchar(255) = LTRIM(RTRIM(@column_name))

IF RIGHT(@str, 1) = ''''
    SET @str = LEFT(@str, LEN(@str) - 1)

SET @index = CHARINDEX(' ', @str, 1)

IF @index = 0
    SET @code1 = @str
ELSE
    BEGIN
    SET @code1 = SUBSTRING(@str, 1, @index - 1)
    SET @str = LTRIM(SUBSTRING(@str, @index + 1, LEN(@str) - @index))

    SET @index = CHARINDEX(' ', @str, 1)
    IF @index = 0
        SET @code2 = @str
    ELSE
        BEGIN
        SET @code2 = SUBSTRING(@str, 1, @index - 1)
        SET @str = LTRIM(SUBSTRING(@str, @index + 1, LEN(@str) - @index))

        SET @code3 = @str
        END
    END

-- Getting cube members
-- from parameters for the body axis
-- from member_id for the row axis
-- from column codes for the column axis

DECLARE @parameters TABLE (dimension_id int PRIMARY KEY, member_id int, default_update_permission bit)

INSERT INTO @parameters (dimension_id, member_id, default_update_permission)
SELECT
    d.id
    , CASE fd.axis_type_id
        WHEN 1 THEN (SELECT TOP 1 id FROM dbo25.members WHERE company_id = @company_id AND id = COALESCE(p.member_id, fd.root_member_id, d.default_member_id))
        WHEN 2 THEN
            CASE fd.header_order
            WHEN 1 THEN (SELECT TOP 1 id FROM dbo25.members WHERE company_id = @company_id AND id = @member_id)
            WHEN 2 THEN (SELECT TOP 1 id FROM dbo25.members WHERE company_id = @company_id AND id = @member_id2)
            WHEN 3 THEN (SELECT TOP 1 id FROM dbo25.members WHERE company_id = @company_id AND id = @member_id3)
            END
        WHEN 3 THEN CASE fd.header_order
            WHEN 1 THEN (SELECT TOP 1 id FROM dbo25.members WHERE company_id = @company_id AND code = @code1)
            WHEN 2 THEN (SELECT TOP 1 id FROM dbo25.members WHERE company_id = @company_id AND code = @code2)
            WHEN 3 THEN (SELECT TOP 1 id FROM dbo25.members WHERE company_id = @company_id AND code = @code3)
            END
        END AS member_id
    , d.default_update_permission
FROM
    dbo25.dimension_properties d
    LEFT OUTER JOIN dbo25.form_dimensions fd ON fd.dimension_id = d.id AND fd.form_id = @form_id
    LEFT OUTER JOIN (
            SELECT 1 AS id, @form_p1 AS member_id
            UNION ALL SELECT 2, @form_p2
            UNION ALL SELECT 3, @form_p3
            UNION ALL SELECT 4, @form_p4
            UNION ALL SELECT 5, @form_p5
            UNION ALL SELECT 6, @form_p6
            UNION ALL SELECT 7, @form_p7
        ) p ON p.id = fd.parameter_index
WHERE
    d.company_id = @company_id

-- Changing period for the same period column

UPDATE @parameters
SET
    member_id = m.id
FROM
    @parameters p
    INNER JOIN (
        SELECT
            p3.member_id AS category_id
            , d3.id AS new_category_id
            , p2.member_id AS time_id
            , m2.same_period_id AS new_time_id
        FROM
            @parameters p3
            INNER JOIN dbo25.members m3 ON m3.id = p3.member_id
            INNER JOIN dbo25.members d3 ON d3.same_period_id = m3.id
            CROSS JOIN @parameters p2
            INNER JOIN dbo25.members m2 ON m2.id = p2.member_id
        WHERE
            p3.dimension_id = 3                 -- Categories
            AND p2.dimension_id = 2             -- Times
            AND m3.calc_type_id = 6             -- type of the category is the same period
            AND d3.calc_type_id = 1             -- type of parent's category is total (not difference or percent)
            AND m2.same_period_id IS NOT NULL   -- and the time has the same period member
    ) t ON p.member_id = t.category_id OR p.member_id = t.time_id
    INNER JOIN dbo25.members m ON m.id = CASE WHEN p.member_id = t.category_id THEN t.new_category_id ELSE t.new_time_id END

-- Skip if not total, average, string, or revaluation

IF EXISTS (SELECT * FROM @parameters p INNER JOIN dbo25.members m ON m.id = p.member_id AND m.calc_type_id NOT IN (1, 2, 8, 11))
    BEGIN
    RETURN
    END

-- Checking update permissions for business users (analysts and developers may edit any cell)

DECLARE @permission bit
DECLARE @code nvarchar(max)

IF IS_MEMBER('planning_app_users') = 1
    BEGIN

    SELECT @code = STUFF((
            SELECT
                ', ' + m.code
            FROM
                @parameters p
                LEFT OUTER JOIN dbo25.member_permissions mp ON mp.member_id = p.member_id AND mp.username = USER_NAME()
                INNER JOIN dbo25.members m ON m.id = p.member_id
            WHERE
                COALESCE(mp.update_permission, p.default_update_permission, 0) = 0
            ORDER BY
                p.dimension_id
            FOR XML PATH(''), TYPE
        ).value('.', 'nvarchar(MAX)'), 1, 2, '')

    IF @code IS NOT NULL
        BEGIN
        SET @message = dbo25.get_translated_string('You have no permission to change members: %s', @data_language)

        IF @quiet_mode = 0
            RAISERROR (@message, 11, 0, @code)
        RETURN
        END

    END

-- Checking closed periods

SELECT
    @permission = cp.update_permission
FROM
    dbo25.member_relations cp
WHERE
    cp.member_id = (SELECT member_id FROM @parameters WHERE dimension_id = 2)
    AND cp.partner_id = (SELECT member_id FROM @parameters WHERE dimension_id = 3)

IF @permission = 0
    BEGIN
    SELECT @code = code FROM dbo25.members WHERE id = (SELECT member_id FROM @parameters WHERE dimension_id = 2)

    SET @message = dbo25.get_translated_string('The period %s is closed for changes', @data_language)

    IF @quiet_mode = 0
        RAISERROR (@message, 11, 0, @code)

    RETURN
    END

-- Members

DECLARE @members TABLE(id int PRIMARY KEY, id1 int, id2 int, id3 int, id4 int, id5 int, id6 int, id7 int)

INSERT INTO @members (id, id1, id2, id3, id4, id5, id6, id7)
SELECT
    1, [1], [2], [3], [4], [5], [6], [7]
FROM
    (
        SELECT dimension_id, member_id FROM @parameters
    ) s PIVOT (
        MAX(member_id) FOR dimension_id IN ([1], [2], [3], [4], [5], [6], [7])
    ) p

-- Updating or inserting the string

IF EXISTS (SELECT * FROM @parameters p INNER JOIN dbo25.members m ON m.id = p.member_id AND m.calc_type_id = 8)
    BEGIN

    UPDATE dbo25.strings
    SET
        value = @cell_value
    FROM
        dbo25.strings f
        INNER JOIN @members m ON f.id1 = m.id1 AND f.id2 = m.id2 AND f.id3 = m.id3 AND f.id4 = m.id4 AND  f.id5 = m.id5 AND f.id6 = m.id6 AND f.id7 = m.id7

    IF @@ROWCOUNT = 0
        BEGIN
        INSERT INTO dbo25.strings (value, id1, id2, id3, id4, id5, id6, id7)
        SELECT
            @cell_value, id1, id2, id3, id4, id5, id6, id7
        FROM
            @members m
        END

    RETURN
    END

-- Blocking changes for parent members

IF EXISTS(SELECT TOP 1 t.member_id FROM @parameters t INNER JOIN dbo25.parents h ON h.parent_id = t.member_id)
    BEGIN
    IF @cell_address IS NULL
        SET @cell_address = (SELECT code + ' ' + @column_name FROM dbo25.members WHERE id = @member_id)

    SET @message = dbo25.get_translated_string('The cell ''%s'' is not editable', @data_language)

    IF @quiet_mode = 0 AND @changed_cell_count = 1
        RAISERROR (@message, 11, 0, @cell_address)

    RETURN
    END

-- Check to prevent setting NULL with @cell_number_value for wrong numbers

IF @cell_number_value IS NULL AND @cell_value IS NOT NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Value ''%s'' is not a number', @data_language)

    IF @quiet_mode = 0
        RAISERROR (@message, 11, 0, @cell_value)

    RETURN
    END

-- Getting currency rate

DECLARE @currency_id tinyint
DECLARE @calc_type_id tinyint

SELECT @currency_id = m.unit_id, @calc_type_id = m.calc_type_id FROM @parameters p INNER JOIN dbo25.members m ON m.id = p.member_id WHERE p.dimension_id = 1

DECLARE @currency_rate float

IF @currency_id IS NULL OR @currency_id = @functional_currency_id OR @calc_type_id = 11
    SET @currency_rate = 1
ELSE
    BEGIN
    -- Examples:
    -- System Currency    Currency    Currency Rates (Base Currency/Quote Currency)
    -- USD                EUR            EUR/USD
    -- EUR                USD            USD/EUR
    SELECT
        @currency_rate = r.currency_rate
    FROM
        dbo25.currency_rates r
        INNER JOIN @members m ON m.id2 = r.time_id AND m.id3 = r.category_id
        INNER JOIN dbo25.units u ON u.id = @currency_id
    WHERE
        r.base_currency_id = @currency_id AND r.quote_currency_id = @functional_currency_id AND NOT r.currency_rate = 0

    -- Examples:
    -- System Currency    Currency    Currency Rates (Base Currency/Quote Currency)
    -- USD                EUR            USD/EUR
    -- EUR                USD            EUR/USD
    IF @currency_rate IS NULL
    SELECT
        @currency_rate = 1 / r.currency_rate
    FROM
        dbo25.currency_rates r
        INNER JOIN @members m ON m.id2 = r.time_id AND m.id3 = r.category_id
        INNER JOIN dbo25.units u ON u.id = @currency_id
    WHERE
        r.base_currency_id = @functional_currency_id AND r.quote_currency_id = @currency_id AND NOT r.currency_rate = 0

    IF @currency_rate IS NULL
        SET @currency_rate = 1
    END

-- Updating or inserting the fact

UPDATE dbo25.facts
SET
    source_value = @cell_number_value
    , value = @cell_number_value * @currency_rate
FROM
    dbo25.facts f
    INNER JOIN @members m ON f.id1 = m.id1 AND f.id2 = m.id2 AND f.id3 = m.id3 AND f.id4 = m.id4 AND  f.id5 = m.id5 AND f.id6 = m.id6 AND f.id7 = m.id7

IF @@ROWCOUNT = 0
    BEGIN
    IF @cell_number_value IS NULL
        RETURN

    INSERT INTO dbo25.facts (source_value, value, id1, id2, id3, id4, id5, id6, id7)
    SELECT
        @cell_number_value, @cell_number_value * @currency_rate, id1, id2, id3, id4, id5, id6, id7
    FROM
        @members m
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: JSON form engine
--
-- @request must be a valid JSON and has the following format:
-- {"page":{<page value object>},
-- "columns":[<column data array>],
-- "rows":[<row data array>]
-- Page object must contain pairs like "<dimension code>":"<member code>"
-- Column and row data arrays must contain arrays like [<member code>,<member code>,...]
-- =============================================

CREATE PROCEDURE [xls25].[usp_run_json_form]
    @company_id int = NULL
    , @request nvarchar(max) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

EXECUTE AS CALLER

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

-- DECLARE @request nvarchar(max)

-- SET @request = '{"page":{"categories":"BUDGET","DIM5":"DIM5.NONE"},'
--            + '"columns":[["2017.01","BUDGET"],["2017.02","BUDGET"],["2017.03","BUDGET"]],'
--            + '"rows":[["CF.090","SALES"],["CF.110","SALES"],["CF.111","SALES"]]}'

BEGIN -- Parameters --

    DECLARE @rq TABLE(name nvarchar(100), value nvarchar(max))

    INSERT INTO @rq (name, value) SELECT name, value FROM dbo25.get_json_object_values(@request)

    DECLARE @page nvarchar(max)
    DECLARE @columns nvarchar(max)
    DECLARE @rows nvarchar(max)

    SELECT @page = value FROM @rq WHERE name = 'page'
    SELECT @columns = value FROM @rq WHERE name = 'columns'
    SELECT @rows = value FROM @rq WHERE name = 'rows'

    DECLARE @pt TABLE(name nvarchar(100), value nvarchar(max), id int, dimension_id int)
    DECLARE @ct TABLE(id int, value nvarchar(max), code1 nvarchar(50), code2 nvarchar(50), code3 nvarchar(50), id1 int, id2 int, id3 int, dimension_id1 int, dimension_id2 int, dimension_id3 int)
    DECLARE @rt TABLE(id int, value nvarchar(max), code1 nvarchar(50), code2 nvarchar(50), code3 nvarchar(50), id1 int, id2 int, id3 int, dimension_id1 int, dimension_id2 int, dimension_id3 int)

    INSERT INTO @pt (name, value) SELECT name, value FROM dbo25.get_json_object_values(@page)
    INSERT INTO @ct (id, value) SELECT id, value FROM dbo25.get_json_array_values(@columns)
    INSERT INTO @rt (id, value) SELECT id, value FROM dbo25.get_json_array_values(@rows)

    UPDATE @pt
    SET
        id = m.id
        , dimension_id = m.dimension_id
    FROM
        @pt t
        INNER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = t.value

    UPDATE @ct
    SET
        code1 = p.[1]
        , code2 = p.[2]
        , code3 = p.[3]
        , id1 = m1.id
        , id2 = m2.id
        , id3 = m3.id
        , dimension_id1 = m1.dimension_id
        , dimension_id2 = m2.dimension_id
        , dimension_id3 = m3.dimension_id
    FROM
        (
            SELECT
                t.id
                , f.id AS m
                , f.value
            FROM
                @ct t
                CROSS APPLY dbo25.get_json_array_values(t.value) f
        ) s PIVOT (
            MAX(value) FOR m IN ([1], [2], [3])
        ) p
        INNER JOIN @ct t ON t.id = p.id
        LEFT OUTER JOIN dbo25.members m1 ON m1.company_id = @company_id AND m1.code = p.[1]
        LEFT OUTER JOIN dbo25.members m2 ON m2.company_id = @company_id AND m2.code = p.[2]
        LEFT OUTER JOIN dbo25.members m3 ON m3.company_id = @company_id AND m3.code = p.[3]

    UPDATE @rt
    SET
        code1 = p.[1]
        , code2 = p.[2]
        , code3 = p.[3]
        , id1 = m1.id
        , id2 = m2.id
        , id3 = m3.id
        , dimension_id1 = m1.dimension_id
        , dimension_id2 = m2.dimension_id
        , dimension_id3 = m3.dimension_id
    FROM
        (
            SELECT
                t.id
                , f.id AS m
                , f.value
            FROM
                @rt t
                CROSS APPLY dbo25.get_json_array_values(t.value) f
        ) s PIVOT (
            MAX(value) FOR m IN ([1], [2], [3])
        ) p
        INNER JOIN @rt t ON t.id = p.id
        LEFT OUTER JOIN dbo25.members m1 ON m1.company_id = @company_id AND m1.code = p.[1]
        LEFT OUTER JOIN dbo25.members m2 ON m2.company_id = @company_id AND m2.code = p.[2]
        LEFT OUTER JOIN dbo25.members m3 ON m3.company_id = @company_id AND m3.code = p.[3]

    --SELECT * FROM @pt
    --SELECT * FROM @ct
    --SELECT * FROM @rt

    DECLARE @p TABLE(row_id int, col_id int, id1 int, id2 int, id3 int, id4 int, id5 int, id6 int, id7 int, PRIMARY KEY(row_id, col_id))

    INSERT INTO @p (row_id, col_id, id1, id2, id3, id4, id5, id6, id7)
    SELECT
        r.row_id
        , c.col_id
        , COALESCE(r.[1], c.[1], p.[1], d.[1]) AS dim1
        , COALESCE(r.[2], c.[2], p.[2], d.[2]) AS dim2
        , COALESCE(r.[3], c.[3], p.[3], d.[3]) AS dim3
        , COALESCE(r.[4], c.[4], p.[4], d.[4]) AS dim4
        , COALESCE(r.[5], c.[5], p.[5], d.[5]) AS dim5
        , COALESCE(r.[6], c.[6], p.[6], d.[6]) AS dim6
        , COALESCE(r.[7], c.[7], p.[7], d.[7]) AS dim7
    FROM
        (
                      SELECT r.id AS row_id, r.dimension_id1 AS dimension, r.id1 AS id FROM @rt r WHERE r.dimension_id1 IS NOT NULL
            UNION ALL SELECT r.id AS row_id, r.dimension_id2 AS dimension, r.id2 AS id FROM @rt r WHERE r.dimension_id2 IS NOT NULL
            UNION ALL SELECT r.id AS row_id, r.dimension_id3 AS dimension, r.id3 AS id FROM @rt r WHERE r.dimension_id3 IS NOT NULL
        ) s PIVOT (
            MAX(id) FOR dimension IN ([1], [2], [3], [4], [5], [6], [7])
        ) r
        CROSS JOIN (SELECT * FROM (
                      SELECT c.id AS col_id, c.dimension_id1 AS dimension, c.id1 AS id FROM @ct c WHERE c.dimension_id1 IS NOT NULL
            UNION ALL SELECT c.id AS col_id, c.dimension_id2 AS dimension, c.id2 AS id FROM @ct c WHERE c.dimension_id2 IS NOT NULL
            UNION ALL SELECT c.id AS col_id, c.dimension_id3 AS dimension, c.id3 AS id FROM @ct c WHERE c.dimension_id3 IS NOT NULL
            ) s PIVOT (
                MAX(id) FOR dimension IN ([1], [2], [3], [4], [5], [6], [7])
            ) c ) c
        CROSS JOIN (SELECT * FROM (
                    SELECT p.dimension_id AS dimension, p.id AS id FROM @pt p
            ) s PIVOT (
                MAX(id) FOR dimension IN ([1], [2], [3], [4], [5], [6], [7])
            ) p ) p
        CROSS JOIN (SELECT * FROM (
                    SELECT d.id AS dimension, d.default_member_id AS id FROM dbo25.dimension_properties d WHERE d.company_id = @company_id
            ) s PIVOT (
                MAX(id) FOR dimension IN ([1], [2], [3], [4], [5], [6], [7])
            ) d ) d

    -- SELECT * FROM @p

END

BEGIN -- SELECT --

    SELECT
        '[' + STUFF((
            SELECT
                CASE WHEN col_id = 1 THEN '],[' ELSE ',' END
                + COALESCE(CAST(value AS nvarchar), 'null')
            FROM
            (
                SELECT
                    p.row_id
                    , p.col_id
                    , CASE WHEN d.calc_type_id = 2 THEN d.average ELSE d.value END AS value
                FROM
                    @p p
                    LEFT OUTER JOIN (
                        SELECT
                            p.row_id
                            , p.col_id
                            , f1.calc_type_id
                            , SUM(f.value * f1.factor) AS value
                            , AVG(f.value * f1.factor) AS average
                        FROM
                            @p p

                            CROSS JOIN dbo25.facts f

                            INNER JOIN dbo25.factors f5 ON f5.member_id = f.id5 AND f5.parent_id = p.id5
                            INNER JOIN dbo25.factors f6 ON f6.member_id = f.id6 AND f6.parent_id = p.id6
                            INNER JOIN dbo25.factors f3 ON f3.member_id = f.id3 AND f3.parent_id = p.id3
                            INNER JOIN dbo25.factors f4 ON f4.member_id = f.id4 AND f4.parent_id = p.id4
                            INNER JOIN dbo25.factors f1 ON f1.member_id = f.id1 AND f1.parent_id = p.id1
                            INNER JOIN dbo25.factors f2 ON f2.member_id = f.id2 AND f2.parent_id = p.id2
                            INNER JOIN dbo25.factors f7 ON f7.member_id = f.id7 AND f7.parent_id = p.id7
                        WHERE
                            f1.calc_type_id = 1 AND f2.calc_type_id = 1
                            OR f1.calc_type_id = 3 AND f2.calc_type_id = 3
                        GROUP BY
                            p.row_id
                            , p.col_id
                            , f1.calc_type_id
                    ) d ON d.row_id = p.row_id AND d.col_id = p.col_id
            ) p
            ORDER BY
                p.row_id
                , p.col_id
            FOR XML PATH(''), TYPE
            ).value('.', 'nvarchar(MAX)'), 1, 2, '') + ']]' AS response

END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Form engine
-- =============================================

CREATE PROCEDURE [xls25].[usp_run_offline_form]
    @company_id int = NULL
    , @form_id int = NULL
    , @form_p1 int = NULL
    , @form_p2 int = NULL
    , @form_p3 int = NULL
    , @form_p4 int = NULL
    , @form_p5 int = NULL
    , @form_p6 int = NULL
    , @form_p7 int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

EXEC xls25.usp_run_form @company_id, @form_id, @form_p1, @form_p2, @form_p3, @form_p4, @form_p5, @form_p6, @form_p7, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_run_offline_form
--
-- The procedure does nothing
-- =============================================

CREATE PROCEDURE [xls25].[usp_run_offline_form_delete]
    @id int
AS
BEGIN

SET NOCOUNT ON

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_run_offline_form
--
-- The procedure does nothing
-- =============================================

CREATE PROCEDURE [xls25].[usp_run_offline_form_insert]
      @id int
AS
BEGIN

SET NOCOUNT ON

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_run_offline_form
-- =============================================

CREATE PROCEDURE [xls25].[usp_run_offline_form_update]
    @company_id int = NULL
    , @table_name nvarchar(255) = NULL
    , @form_id int = NULL
    , @form_p1 int = NULL
    , @form_p2 int = NULL
    , @form_p3 int = NULL
    , @form_p4 int = NULL
    , @form_p5 int = NULL
    , @form_p6 int = NULL
    , @form_p7 int = NULL
    , @member_id int = NULL
    , @member_id2 int = NULL
    , @member_id3 int = NULL
    , @json_columns nvarchar(max)
    , @json_values nvarchar(max)
AS
BEGIN

-- Test code:
-- DECLARE @json_columns nvarchar(max)
-- DECLARE @json_values nvarchar(max)

-- SET @json_columns = '["member_id1","member_id2","member_id3","is_editable","data_format","2017","2017.01''","2017.02''"]'
-- SET @json_values = '[122,null,null,1,null,2003,1001,1002]'

-- EXEC xls25.usp_run_offline_form_update
--    @form_id = 1, @form_p1 = 117, @form_p2 = 114, @form_p3 = 100, @form_p4 = 16, @form_p5 = NULL, @form_p6 = NULL, @form_p7 = NULL,
--    @member_id = 122, @member_id2 = NULL, @member_id3 = NULL,
--    @json_columns = @json_columns, @json_values = @json_values

SET NOCOUNT ON

IF @json_columns IS NULL OR @json_values IS NULL OR @member_id IS NULL
    RETURN

-- Getting form_id from name

IF @form_id IS NULL
    BEGIN
    DECLARE @form_code nvarchar(128)

    IF LEFT(@table_name, 1) = '['
        SET @form_code = REPLACE(REPLACE(LEFT(@table_name, LEN(@table_name) - 1),'[xls25a].[', ''), '[xls25b].[', '')
    ELSE
        SET @form_code = REPLACE(REPLACE(@table_name, 'xls25a.', ''), 'xls25b.', '')

    SELECT
        @company_id = f.company_id, @form_id = f.id
    FROM
        dbo25.forms f
        INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
        INNER JOIN dbo25.companies c ON c.id = f.company_id
    WHERE
        c.code + '_' + f.code = @form_code

    IF @company_id IS NULL OR @form_id IS NULL
        RETURN
    END
ELSE IF @company_id IS NULL
    BEGIN
    SELECT @company_id = f.company_id FROM dbo25.forms f WHERE f.id = @form_id
    END

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

DECLARE @columns TABLE(id int PRIMARY KEY, value nvarchar(255), UNIQUE(value))
DECLARE @values TABLE(id int PRIMARY KEY, value nvarchar(255))

INSERT INTO @columns (id, value) SELECT id, value FROM dbo25.get_json_array_values(@json_columns)
INSERT INTO @values (id, value) SELECT id, value FROM dbo25.get_json_array_values(@json_values)

-- SELECT c.id, c.value AS column_name, v.value AS value FROM @columns c INNER JOIN @values v ON v.id = c.id

IF NOT (SELECT CAST(v.value AS int) FROM @columns c INNER JOIN @values v ON v.id = c.id WHERE c.value = 'is_editable') = 1
    RETURN

DECLARE @column_name nvarchar(255)
DECLARE @cell_value nvarchar(255)
DECLARE @cell_number_value float

DECLARE @index int = (SELECT c.id FROM @columns c WHERE c.value = 'data_format')

IF @index IS NULL
    RETURN

DECLARE @column_count int = (SELECT COUNT(*) FROM @columns)

WHILE @index < @column_count
    BEGIN
    SET @index = @index + 1

    SELECT @column_name = c.value, @cell_value = v.value FROM @columns c INNER JOIN @values v ON v.id = c.id WHERE c.id = @index

    IF NOT RIGHT(@column_name, 1) = ''''
        CONTINUE

    IF @cell_value = 'null'
        BEGIN
        SET @cell_value = NULL
        SET @cell_number_value = NULL
        END
    ELSE
        BEGIN
        SET @cell_number_value = TRY_PARSE(@cell_value AS float)
        IF @cell_number_value IS NULL AND @cell_value IS NOT NULL
            CONTINUE
        END

    -- PRINT @column_name + ' ' + @cell_value + ' ' + CAST(@cell_number_value AS nvarchar)

    EXEC xls25.usp_run_form_change
        @column_name = @column_name, @table_name = @table_name,
        @cell_value = @cell_value, @cell_number_value = @cell_number_value,
        @member_id = @member_id, @member_id2 = @member_id2, @member_id3 = @member_id3,
        @form_id = @form_id,
        @form_p1 = @form_p1, @form_p2 = @form_p2, @form_p3 = @form_p3,
        @form_p4 = @form_p4, @form_p5 = @form_p5, @form_p6 = @form_p6, @form_p7 = @form_p7,
        @quiet_mode = 1
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects tax rates
-- =============================================

CREATE PROCEDURE [xls25].[usp_tax_rates]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
      t.id
    , t.code
    , t.name
    , tr.name AS translated_name
    , t.sort_order
    , t.factor
    , CAST(is_active AS tinyint) AS is_active
FROM
    dbo25.tax_rates t
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 4 AND tr.member_id = t.id AND tr.language = @data_language
WHERE
    t.company_id = @company_id
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_tax_rates
-- =============================================

CREATE PROCEDURE [xls25].[usp_tax_rates_delete]
    @id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF NOT EXISTS (SELECT TOP 1 u.company_id FROM dbo25.tax_rates t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1)
    RETURN

SET NOCOUNT OFF

DELETE dbo25.tax_rates WHERE id = @id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_tax_rates
-- =============================================

CREATE PROCEDURE [xls25].[usp_tax_rates_insert]
    @company_id int = NULL
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @translated_name nvarchar(100) = NULL
    , @sort_order int = NULL
    , @factor float = NULL
    , @is_active bit = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

SET NOCOUNT OFF

INSERT INTO dbo25.tax_rates
    ( company_id
    , code
    , name
    , sort_order
    , factor
    , is_active
    )
VALUES
    ( @company_id
    , @code
    , @name
    , COALESCE(@sort_order, 0)
    , COALESCE(@factor, 1)
    , COALESCE(@is_active, 1)
    )

DECLARE @id int = SCOPE_IDENTITY()

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    INSERT INTO dbo25.translations
        (company_id, table_id, member_id, language, name)
    VALUES
        (@company_id, 4, @id, @data_language, @translated_name)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_tax_rates
-- =============================================

CREATE PROCEDURE [xls25].[usp_tax_rates_update]
    @id int = NULL
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @translated_name nvarchar(100) = NULL
    , @sort_order int = NULL
    , @factor float = NULL
    , @is_active bit = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int
DECLARE @current_name nvarchar(100)

SELECT TOP 1 @company_id = u.company_id, @current_name = t.name FROM dbo25.tax_rates t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

SET NOCOUNT OFF

UPDATE dbo25.tax_rates
SET
    code = @code
    , name = @name
    , sort_order = COALESCE(@sort_order, 0)
    , factor = COALESCE(@factor, 1)
    , is_active = COALESCE(@is_active, 1)
WHERE
    id = @id
    AND ((
        sort_order IS NULL
        OR factor IS NULL
        OR is_active IS NULL
        ) OR NOT (
        code = @code
        AND name = @name
        AND sort_order = COALESCE(@sort_order, 0)
        AND factor = COALESCE(@factor, 0)
        AND is_active = COALESCE(@is_active, 0)
        ))

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    BEGIN
    MERGE dbo25.translations t
    USING
        (
        SELECT company_id = @company_id, 4 AS table_id, @id AS member_id, @data_language AS language, @translated_name AS name
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects translations
-- =============================================

CREATE PROCEDURE [xls25].[usp_translations]
    @company_id int = NULL
    , @is_translatable bit = NULL
    , @is_complete bit = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON;

EXECUTE AS CALLER

IF @company_id IS NULL
    SELECT TOP 1 @company_id = company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1
ELSE
    SELECT TOP 1 @company_id = company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1

REVERT

IF @company_id IS NULL
    BEGIN
    SELECT
        t.table_id
        , t.member_id
        , t.name AS table_name
        , 0 AS is_translatable
        , 0 AS is_complete
        , t.name
        , t.name AS [en]
    FROM
        dbo25.translations t
    WHERE
        t.company_id IS NULL

    RETURN
    END

DECLARE @company_sid nvarchar(max) = CAST(@company_id AS nvarchar(max))
DECLARE @sql nvarchar(max)
DECLARE @languages nvarchar(max)
DECLARE @complete nvarchar(max)

SELECT @languages = STUFF((
    SELECT
        t.name
    FROM
        (
        SELECT
            DISTINCT
            ', [' + t.language + ']' AS name
            , CASE
                WHEN t.language = 'en' THEN '1'
                WHEN t.language = 'fr' THEN '2'
                WHEN t.language = 'it' THEN '3'
                WHEN t.language = 'es' THEN '4'
                WHEN t.language = 'pt' THEN '5'
                WHEN t.language = 'de' THEN '6'
                WHEN t.language = 'ru' THEN '7'
                ELSE t.language
                END AS sort_order
        FROM
            dbo25.translations t
        WHERE
            t.company_id = @company_id
        ) t
    ORDER BY
        t.sort_order
    FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 2, '')

IF @languages IS NULL SET @languages = '[en]'

DECLARE @default_language varchar(10) = (SELECT default_language FROM dbo25.companies WHERE id = @company_id)

IF @default_language IS NULL SET @languages = '  '

SELECT @complete = STUFF((
    SELECT
        t.name
    FROM
        (
        SELECT
            DISTINCT
            ' OR [' + t.language + '] IS NULL' AS name
            , CASE
                WHEN t.language = 'en' THEN '1'
                WHEN t.language = 'fr' THEN '2'
                WHEN t.language = 'it' THEN '3'
                WHEN t.language = 'es' THEN '4'
                WHEN t.language = 'pt' THEN '5'
                WHEN t.language = 'de' THEN '6'
                WHEN t.language = 'ru' THEN '7'
                ELSE t.language
                END AS sort_order
        FROM
            dbo25.translations t
        WHERE
            t.company_id = @company_id
            AND NOT t.language = @default_language
        ) t
    ORDER BY
        t.sort_order
    FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 4, '')

IF @complete IS NULL SET @complete = '[en] IS NULL'

SET @sql = 'SELECT
    t.table_id
    , t.member_id
    , tt.name AS table_name
    , t.is_translatable
    , CASE WHEN t.is_translatable = 0 THEN 1 WHEN ' + @complete + ' THEN 0 ELSE 1 END AS is_complete
    , t.name
    , ' + @languages + '
FROM
    (
        SELECT
            m.sort_order
            , m.table_id
            , m.member_id
            , m.is_translatable
            , m.name
            , t.language
            , t.name AS translated_name
        FROM
            (
                SELECT 2 AS sort_order, 2 AS table_id, t.id AS member_id, t.name, 1 AS is_translatable FROM dbo25.companies t WHERE t.id = ' + @company_sid + '
                UNION ALL SELECT 3 AS sort_order, 3 AS table_id, t.id AS member_id, t.name, t.is_translatable FROM dbo25.units t WHERE t.company_id = ' + @company_sid + '
                UNION ALL SELECT 4 AS sort_order, 4 AS table_id, t.id AS member_id, t.name, t.is_translatable FROM dbo25.tax_rates t WHERE t.company_id = ' + @company_sid + '
                UNION ALL SELECT 5 AS sort_order, 5 AS table_id, t.id AS member_id, t.name, 1 AS is_translatable FROM dbo25.dimension_properties t WHERE t.company_id = ' + @company_sid + '
                UNION ALL SELECT 6 AS sort_order, 6 AS table_id, t.id AS member_id, t.parameter_name, 1 AS is_translatable FROM dbo25.dimension_properties t WHERE t.company_id = ' + @company_sid + '
                UNION ALL SELECT 7 AS sort_order, 7 AS table_id, t.id AS member_id, t.name, 1 AS is_translatable FROM dbo25.forms t WHERE t.company_id = ' + @company_sid + '
                UNION ALL SELECT 8 AS sort_order, 8 AS table_id, t.id AS member_id, t.name, 1 AS is_translatable FROM dbo25.form_rows t INNER JOIN dbo25.members m ON m.id = t.rowset_id WHERE m.company_id = ' + @company_sid + '
                UNION ALL SELECT 9 AS sort_order, 1 AS table_id, t.id AS member_id, t.name, t.is_translatable FROM dbo25.members t WHERE t.company_id = ' + @company_sid + '
            ) m
            LEFT OUTER JOIN dbo25.translations t ON t.company_id = ' + @company_sid + ' AND t.table_id = m.table_id AND t.member_id = m.member_id
        WHERE
            m.name IS NOT NULL
    ) s PIVOT (
        MAX(translated_name) FOR language IN (' + @languages + ')
    ) t
    INNER JOIN dbo25.translated_tables tt ON tt.id = t.table_id'
    + CASE WHEN @is_complete IS NULL AND @is_translatable IS NULL THEN '' ELSE '
WHERE'
        + CASE WHEN @is_complete IS NULL THEN '' WHEN @is_complete = 1 THEN '
    NOT (' + @complete + ')'
            ELSE '
    (' + @complete + ')'
            END
        + CASE WHEN @is_translatable IS NULL THEN '' ELSE '
    '        + CASE WHEN @is_complete IS NULL THEN '' ELSE 'AND ' END + 't.is_translatable = ' + CASE WHEN @is_translatable = 1 THEN '1' ELSE '0' END
            END
        END    + '
ORDER BY
    t.sort_order
    , t.name'

-- PRINT @sql

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_translations
-- =============================================

CREATE PROCEDURE [xls25].[usp_translations_change]
    @company_id int = NULL
    , @table_id int = NULL
    , @member_id int = NULL
    , @column_name nvarchar(128) = NULL
    , @cell_value nvarchar(400) = NULL
    , @cell_number_value int = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id1 int = @company_id
DECLARE @current_name nvarchar(400) = NULL

IF @table_id = 1
    SELECT TOP 1 @company_id = u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @member_id AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 2
    SELECT TOP 1 @company_id = u.company_id FROM dbo25.companies t INNER JOIN dbo25.users u ON u.company_id = t.id  WHERE t.id = @member_id AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 3
    SELECT TOP 1 @company_id = u.company_id FROM dbo25.units t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @member_id AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 4
    SELECT TOP 1 @company_id = u.company_id FROM dbo25.tax_rates t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @member_id AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 5
    SELECT TOP 1 @company_id = u.company_id FROM dbo25.companies t INNER JOIN dbo25.users u ON u.company_id = t.id  WHERE t.id = @company_id AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 6
    SELECT TOP 1 @company_id = u.company_id FROM dbo25.companies t INNER JOIN dbo25.users u ON u.company_id = t.id  WHERE t.id = @company_id AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 7
    SELECT TOP 1 @company_id = u.company_id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @member_id AND u.username = USER_NAME() AND u.is_active = 1
ELSE IF @table_id = 8
    SELECT TOP 1 @company_id = u.company_id FROM dbo25.form_rows r INNER JOIN dbo25.members t ON t.id = r.rowset_id INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE r.id = @member_id AND u.username = USER_NAME() AND u.is_active = 1
ELSE
    RETURN

IF @company_id IS NULL OR @company_id1 IS NOT NULL AND NOT @company_id1 = @company_id
    RETURN

IF @column_name = 'is_translatable'
    BEGIN
    IF NOT @cell_number_value IN (0, 1) AND NOT @cell_number_value IS NULL
        RETURN

    IF @table_id = 1
        UPDATE dbo25.members SET is_translatable = @cell_number_value WHERE id = @member_id
    ELSE IF @table_id = 3
        UPDATE dbo25.units SET is_translatable = @cell_number_value WHERE id = @member_id
    ELSE IF @table_id = 4
        UPDATE dbo25.tax_rates SET is_translatable = @cell_number_value WHERE id = @member_id
    RETURN
    END

IF LEN(@column_name) <> 2
    RETURN

IF @cell_value IS NULL
    BEGIN
    DELETE dbo25.translations
    WHERE
        company_id = @company_id AND table_id = @table_id AND member_id = @member_id AND language = @column_name
    RETURN
    END

UPDATE dbo25.translations
SET
    name = @cell_value
WHERE
    company_id = @company_id AND table_id = @table_id AND member_id = @member_id AND language = @column_name

IF @@ROWCOUNT > 0
    RETURN

IF @cell_value IS NULL
    RETURN

INSERT INTO dbo25.translations
    (company_id, table_id, member_id, language, name)
VALUES
    (@company_id, @table_id, @member_id, @column_name, @cell_value)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects common elements with translations from xls.translations
-- =============================================

CREATE PROCEDURE [xls25].[usp_translations_common]
    @is_complete bit = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON;

DECLARE @sql nvarchar(max)
DECLARE @languages nvarchar(max)
DECLARE @complete nvarchar(max)

SELECT @languages = STUFF((
    SELECT
        t.name
    FROM
        (
        SELECT
            DISTINCT
            ', [' + t.LANGUAGE_NAME + ']' AS name
            , CASE
                WHEN t.LANGUAGE_NAME = 'en' THEN '1'
                WHEN t.LANGUAGE_NAME = 'fr' THEN '2'
                WHEN t.LANGUAGE_NAME = 'it' THEN '3'
                WHEN t.LANGUAGE_NAME = 'es' THEN '4'
                WHEN t.LANGUAGE_NAME = 'pt' THEN '5'
                WHEN t.LANGUAGE_NAME = 'de' THEN '6'
                WHEN t.LANGUAGE_NAME = 'ru' THEN '7'
                ELSE t.LANGUAGE_NAME
                END AS sort_order
        FROM
            xls.translations t
        ) t
    ORDER BY
        t.sort_order
    FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 2, '')

IF @languages IS NULL SET @languages = '[en]'

SELECT @complete = STUFF((
    SELECT
        t.name
    FROM
        (
        SELECT
            DISTINCT
            ' OR [' + t.LANGUAGE_NAME + '] IS NULL' AS name
            , CASE
                WHEN t.LANGUAGE_NAME = 'en' THEN '1'
                WHEN t.LANGUAGE_NAME = 'fr' THEN '2'
                WHEN t.LANGUAGE_NAME = 'it' THEN '3'
                WHEN t.LANGUAGE_NAME = 'es' THEN '4'
                WHEN t.LANGUAGE_NAME = 'pt' THEN '5'
                WHEN t.LANGUAGE_NAME = 'de' THEN '6'
                WHEN t.LANGUAGE_NAME = 'ru' THEN '7'
                ELSE t.LANGUAGE_NAME
                END AS sort_order
        FROM
            xls.translations t
        ) t
    ORDER BY
        t.sort_order
    FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 4, '')

IF @complete IS NULL SET @complete = '[en] IS NULL'

SET @sql = 'SELECT
    t.TABLE_SCHEMA
    , t.TABLE_NAME
    , CASE WHEN ' + @complete + ' THEN 0 ELSE 1 END AS is_complete
    , t.COLUMN_NAME AS [COLUMN]
    , ' + @languages + '
FROM
    (
        SELECT
            t.sort_order
            , t.TABLE_SCHEMA
            , t.TABLE_NAME
            , t.COLUMN_NAME
            , tr.LANGUAGE_NAME
            , tr.TRANSLATED_NAME AS name
        FROM
            (
                SELECT 1 AS sort_order, name AS COLUMN_NAME, ''dbo25'' AS TABLE_SCHEMA, ''axis_types_name'' AS TABLE_NAME FROM dbo25.axis_types
                UNION ALL SELECT 2 AS sort_order, name AS COLUMN_NAME, ''dbo25'' AS TABLE_SCHEMA, ''calc_types_name'' AS TABLE_NAME FROM dbo25.calc_types
                UNION ALL SELECT 3 AS sort_order, name AS COLUMN_NAME, ''dbo25'' AS TABLE_SCHEMA, ''name_formats_name'' AS TABLE_NAME FROM dbo25.name_formats
                UNION ALL SELECT * FROM (
                    SELECT
                        5 AS sort_order
                        , NULL AS COLUMN_NAME
                        , c.TABLE_SCHEMA
                        , c.TABLE_NAME
                    FROM
                        INFORMATION_SCHEMA.TABLES c
                    WHERE
                        c.TABLE_SCHEMA = ''xls25''
                    UNION ALL
                    SELECT
                        5 AS sort_order
                        , NULL AS COLUMN_NAME
                        , p.SPECIFIC_SCHEMA AS TABLE_SCHEMA
                        , p.SPECIFIC_NAME AS TABLE_NAME
                    FROM
                        INFORMATION_SCHEMA.ROUTINES p
                    WHERE
                        p.SPECIFIC_SCHEMA = ''xls25''
                        AND p.ROUTINE_TYPE = ''PROCEDURE''
                        AND NOT p.SPECIFIC_NAME LIKE ''%_insert''
                        AND NOT p.SPECIFIC_NAME LIKE ''%_update''
                        AND NOT p.SPECIFIC_NAME LIKE ''%_delete''
                        AND NOT p.SPECIFIC_NAME LIKE ''%_change''
                        AND NOT p.SPECIFIC_NAME LIKE ''%_merge''
                        AND NOT p.SPECIFIC_NAME LIKE ''usp_import_%''
                        AND NOT p.SPECIFIC_NAME LIKE ''usp_export_%''
                        AND NOT p.SPECIFIC_NAME LIKE ''xl_parameter_values_%''
                        AND NOT p.SPECIFIC_NAME LIKE ''xl_validation_list_%''
                        AND NOT p.SPECIFIC_NAME LIKE ''usp_update_table_format''
                    ) c ' + '
                UNION ALL SELECT * FROM (
                    SELECT
                        6 AS sort_order
                        , c.COLUMN_NAME
                        , ''xls25'' AS TABLE_SCHEMA
                        , NULL AS TABLE_NAME
                    FROM
                        INFORMATION_SCHEMA.COLUMNS c
                    WHERE
                        c.TABLE_SCHEMA = ''dbo25''
                        AND NOT c.TABLE_NAME IN (''formats'', ''view_formats'', ''view_handlers'', ''view_translations'')
                        AND NOT c.TABLE_NAME LIKE ''view_query_list%''
                    UNION
                    SELECT
                        6 AS sort_order
                        , SUBSTRING(p.PARAMETER_NAME, 2, 127) AS COLUMN_NAME
                        , p.SPECIFIC_SCHEMA AS TABLE_SCHEMA
                        , NULL AS TABLE_NAME
                    FROM
                        INFORMATION_SCHEMA.PARAMETERS p
                        INNER JOIN INFORMATION_SCHEMA.ROUTINES r ON r.SPECIFIC_SCHEMA = p.SPECIFIC_SCHEMA AND r.SPECIFIC_NAME = p.SPECIFIC_NAME
                    WHERE
                        p.SPECIFIC_SCHEMA = ''xls25''
                        AND r.ROUTINE_TYPE = ''PROCEDURE''
                        AND p.ORDINAL_POSITION > 0
                        AND NOT p.SPECIFIC_NAME LIKE ''%_insert''
                        AND NOT p.SPECIFIC_NAME LIKE ''%_update''
                        AND NOT p.SPECIFIC_NAME LIKE ''%_delete''
                        AND NOT p.SPECIFIC_NAME LIKE ''%_change''
                        AND NOT p.SPECIFIC_NAME LIKE ''%_merge''
                        AND NOT p.SPECIFIC_NAME LIKE ''usp_import_%''
                        AND NOT p.SPECIFIC_NAME LIKE ''usp_export_%''
                        AND NOT p.SPECIFIC_NAME LIKE ''xl_parameter_values_%''
                        AND NOT p.SPECIFIC_NAME LIKE ''xl_validation_list_%''
                        AND NOT p.SPECIFIC_NAME LIKE ''usp_run%''
                        AND NOT p.SPECIFIC_NAME LIKE ''xl_actions_run_form_cell_data''
                        AND NOT p.SPECIFIC_NAME LIKE ''usp_update_table_format''
                        AND NOT p.PARAMETER_NAME IN (''@data_language'')
                    ) c
            ) t
            LEFT OUTER JOIN xls.translations tr ON
                tr.TABLE_SCHEMA = t.TABLE_SCHEMA
                AND (tr.TABLE_NAME = t.TABLE_NAME OR tr.TABLE_NAME IS NULL AND t.TABLE_NAME IS NULL)
                AND (tr.COLUMN_NAME = t.COLUMN_NAME OR tr.COLUMN_NAME IS NULL AND t.COLUMN_NAME IS NULL)
    ) s PIVOT (
        MAX(name) FOR LANGUAGE_NAME IN (' + @languages + ')
    ) t'
    + CASE WHEN @is_complete IS NULL THEN '' WHEN @is_complete = 1 THEN '
WHERE
    NOT (' + @complete + ')'
    ELSE '
WHERE
    ' + @complete END + '
ORDER BY
    t.sort_order
    , t.TABLE_SCHEMA
    , t.TABLE_NAME
    , t.COLUMN_NAME'

-- PRINT @sql

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects translations from xls.translations
-- =============================================

CREATE PROCEDURE [xls25].[usp_translations_xls]
    @field nvarchar(128) = NULL
    , @schema nvarchar(128) = NULL
    , @is_complete bit = NULL
AS
BEGIN

SET NOCOUNT ON;

IF @field IS NULL
    SET @field = 'TRANSLATED_NAME'
ELSE IF @field NOT IN ('TRANSLATED_NAME', 'TRANSLATED_DESC', 'TRANSLATED_COMMENT')
    BEGIN
    DECLARE @message nvarchar(max) = N'Invalid column name: %s' + CHAR(13) + CHAR(10)
         + 'Use TRANSLATED_NAME, TRANSLATED_DESC, or TRANSLATED_COMMENT'
    RAISERROR(@message, 11, 0, @field);
    RETURN
    END

DECLARE @sql nvarchar(max)
DECLARE @languages nvarchar(max)
DECLARE @complete nvarchar(max)

SELECT @languages = STUFF((
    SELECT
        t.name
    FROM
        (
        SELECT
            DISTINCT
            ', [' + t.LANGUAGE_NAME + ']' AS name
            , CASE
                WHEN t.LANGUAGE_NAME = 'en' THEN '1'
                WHEN t.LANGUAGE_NAME = 'fr' THEN '2'
                WHEN t.LANGUAGE_NAME = 'it' THEN '3'
                WHEN t.LANGUAGE_NAME = 'es' THEN '4'
                WHEN t.LANGUAGE_NAME = 'pt' THEN '5'
                WHEN t.LANGUAGE_NAME = 'de' THEN '6'
                WHEN t.LANGUAGE_NAME = 'ru' THEN '7'
                ELSE t.LANGUAGE_NAME
                END AS sort_order
        FROM
            xls.translations t
        ) t
    ORDER BY
        t.sort_order
    FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 2, '')

IF @languages IS NULL SET @languages = '[en]'

SELECT @complete = STUFF((
    SELECT
        t.name
    FROM
        (
        SELECT
            DISTINCT
            ' OR [' + t.LANGUAGE_NAME + '] IS NULL' AS name
            , CASE
                WHEN t.LANGUAGE_NAME = 'en' THEN '1'
                WHEN t.LANGUAGE_NAME = 'fr' THEN '2'
                WHEN t.LANGUAGE_NAME = 'it' THEN '3'
                WHEN t.LANGUAGE_NAME = 'es' THEN '4'
                WHEN t.LANGUAGE_NAME = 'pt' THEN '5'
                WHEN t.LANGUAGE_NAME = 'de' THEN '6'
                WHEN t.LANGUAGE_NAME = 'ru' THEN '7'
                ELSE t.LANGUAGE_NAME
                END AS sort_order
        FROM
            xls.translations t
        ) t
    ORDER BY
        t.sort_order
    FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)'), 1, 4, '')

IF @complete IS NULL SET @complete = '[en] IS NULL'

SET @sql = 'SELECT
    t.TABLE_SCHEMA
    , t.TABLE_NAME
    , CASE WHEN ' + @complete + ' THEN 0 ELSE 1 END AS is_complete
    , t.COLUMN_NAME AS [COLUMN]
    , ' + @languages + '
FROM
    (
        SELECT
            t.TABLE_SCHEMA
            , t.TABLE_NAME
            , t.COLUMN_NAME
            , t.LANGUAGE_NAME
            , t.' + @field + ' AS name
        FROM
            xls.translations t'
        + CASE WHEN @schema IS NULL THEN '' ELSE '
        WHERE
            ' + CASE WHEN LEFT(@schema, 1) = '(' AND RIGHT(@schema, 1) = ')' THEN 't.TABLE_SCHEMA IN ' + @schema
                ELSE 'COALESCE(t.TABLE_SCHEMA, '''') = COALESCE(' + COALESCE('''' + @schema + '''', 'NULL') + ', t.TABLE_SCHEMA, '''')'
                END
            END + '
    ) s PIVOT (
        MAX(name) FOR LANGUAGE_NAME IN (' + @languages + ')
    ) t'
    + CASE WHEN @is_complete IS NULL THEN '' WHEN @is_complete = 1 THEN '
WHERE
    NOT (' + @complete + ')'
    ELSE '
WHERE
    ' + @complete END + '
ORDER BY
    t.TABLE_SCHEMA
    , t.TABLE_NAME
    , t.COLUMN_NAME'

-- PRINT @sql

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_translations
--
-- @column_name is a column name of the edited cell
-- @cell_value is a new cell value
-- @field is a value of the field parameter of the usp_translations
-- @TABLE_SCHEMA, @TABLE_NAME, and @COLUMN are values of the Excel table columns
-- =============================================

CREATE PROCEDURE [xls25].[usp_translations_xls_change]
    @column_name nvarchar(128) = NULL
    , @cell_value nvarchar(max) = NULL
    , @TABLE_SCHEMA nvarchar(128) = NULL
    , @TABLE_NAME nvarchar(128) = NULL
    , @COLUMN nvarchar(128) = NULL
    , @field nvarchar(128) = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

IF NOT EXISTS(SELECT TOP 1 ID FROM xls.translations WHERE LANGUAGE_NAME = @column_name)
    RETURN

IF @field IS NULL SET @field = 'TRANSLATED_NAME'

IF @field = 'TRANSLATED_NAME'
    UPDATE xls.translations SET TRANSLATED_NAME = @cell_value
        WHERE COALESCE(TABLE_SCHEMA, '') = COALESCE(@TABLE_SCHEMA, '') AND COALESCE(TABLE_NAME, '') = COALESCE(@TABLE_NAME, '')
            AND COALESCE(COLUMN_NAME, '') = COALESCE(@COLUMN, '') AND LANGUAGE_NAME = @column_name
ELSE IF @field = 'TRANSLATED_DESC'
    UPDATE xls.translations SET TRANSLATED_DESC = @cell_value
        WHERE COALESCE(TABLE_SCHEMA, '') = COALESCE(@TABLE_SCHEMA, '') AND COALESCE(TABLE_NAME, '') = COALESCE(@TABLE_NAME, '')
            AND COALESCE(COLUMN_NAME, '') = COALESCE(@COLUMN, '') AND LANGUAGE_NAME = @column_name
ELSE IF @field = 'TRANSLATED_COMMENT'
    UPDATE xls.translations SET TRANSLATED_COMMENT = @cell_value
        WHERE COALESCE(TABLE_SCHEMA, '') = COALESCE(@TABLE_SCHEMA, '') AND COALESCE(TABLE_NAME, '') = COALESCE(@TABLE_NAME, '')
            AND COALESCE(COLUMN_NAME, '') = COALESCE(@COLUMN, '') AND LANGUAGE_NAME = @column_name
ELSE
    BEGIN
    SET @message = dbo25.get_translated_string('Invalid column name: %s', @data_language)
        + CHAR(13) + CHAR(10)
        + dbo25.get_translated_string('Use TRANSLATED_NAME, TRANSLATED_DESC, or TRANSLATED_COMMENT', @data_language)
    RAISERROR(@message, 11, 0, @field);
    RETURN
    END

IF @@ROWCOUNT > 0 RETURN

IF @cell_value IS NULL RETURN

IF @field = 'TRANSLATED_NAME'
    INSERT INTO xls.translations (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME)
        VALUES (@TABLE_SCHEMA, @TABLE_NAME, @COLUMN, @column_name, @cell_value)
ELSE IF @field = 'TRANSLATED_DESC'
    INSERT INTO xls.translations (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_DESC)
        VALUES (@TABLE_SCHEMA, @TABLE_NAME, @COLUMN, @column_name, @cell_value)
ELSE IF @field = 'TRANSLATED_COMMENT'
    INSERT INTO xls.translations (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_COMMENT)
        VALUES (@TABLE_SCHEMA, @TABLE_NAME, @COLUMN, @column_name, @cell_value)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects units
-- =============================================

CREATE PROCEDURE [xls25].[usp_units]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    u.id
    , u.code
    , u.name
    , tr.name AS translated_name
    , u.sort_order
    , u.factor
    , u.is_currency
    , u.is_functional_currency
    , is_active
FROM
    dbo25.units u
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = u.company_id AND tr.table_id = 3 AND tr.member_id = u.id AND tr.language = @data_language
WHERE
    u.company_id = @company_id
ORDER BY
    u.sort_order
    , u.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_units
-- =============================================

CREATE PROCEDURE [xls25].[usp_units_delete]
    @id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF NOT EXISTS (SELECT TOP 1 u.company_id FROM dbo25.units t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1)
    RETURN

SET NOCOUNT OFF

DELETE dbo25.units WHERE id = @id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_units
-- =============================================

CREATE PROCEDURE [xls25].[usp_units_insert]
    @company_id int = NULL
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @translated_name nvarchar(100) = NULL
    , @sort_order int = NULL
    , @factor float = NULL
    , @is_currency bit = NULL
    , @is_functional_currency bit = NULL
    , @is_active bit = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

SET NOCOUNT OFF

INSERT INTO dbo25.units
    ( company_id
    , code
    , name
    , sort_order
    , factor
    , is_currency
    , is_functional_currency
    , is_active
    )
VALUES
    ( @company_id
    , @code
    , @name
    , COALESCE(@sort_order, 0)
    , COALESCE(@factor, 1)
    , COALESCE(@is_currency, 1)
    , COALESCE(@is_functional_currency, 1)
    , COALESCE(@is_active, 1)
    )

DECLARE @id int = SCOPE_IDENTITY()

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    INSERT INTO dbo25.translations
        (company_id, table_id, member_id, language, name)
    VALUES
        (@company_id, 3, @id, @data_language, @translated_name)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_units
-- =============================================

CREATE PROCEDURE [xls25].[usp_units_update]
    @id int = NULL
    , @code nvarchar(50)
    , @name nvarchar(100)
    , @translated_name nvarchar(100) = NULL
    , @sort_order int = NULL
    , @factor float = NULL
    , @is_currency bit = NULL
    , @is_functional_currency bit = NULL
    , @is_active bit = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int
DECLARE @current_name nvarchar(100)

SELECT TOP 1 @company_id = u.company_id, @current_name = t.name FROM dbo25.units t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

SET @code = LTRIM(RTRIM(@code))
SET @name = LTRIM(RTRIM(@name))

SET NOCOUNT OFF

UPDATE dbo25.units
SET
    code = @code
    , name = @name
    , sort_order = COALESCE(@sort_order, 0)
    , factor = COALESCE(@factor, 1)
    , is_currency = COALESCE(@is_currency, 1)
    , is_functional_currency = COALESCE(@is_functional_currency, 0)
    , is_active = COALESCE(@is_active, 1)
WHERE
    id = @id
    AND NOT (
        COALESCE(code, '-1') = COALESCE(@code, '-2')
        AND COALESCE(name, '-1') = COALESCE(@name, '-2')
        AND COALESCE(sort_order, -1) = COALESCE(@sort_order, -2)
        AND COALESCE(factor, -1) = COALESCE(@factor, -2)
        AND COALESCE(is_currency, -1) = COALESCE(@is_currency, -2)
        AND COALESCE(is_functional_currency, -1) = COALESCE(@is_functional_currency, -2)
        AND COALESCE(is_active, -1) = COALESCE(@is_active, -2)
        )

IF @data_language IS NOT NULL AND @translated_name IS NOT NULL
    BEGIN
    MERGE dbo25.translations t
    USING
        (
        SELECT company_id = @company_id, 3 AS table_id, @id AS member_id, @data_language AS language, @translated_name AS name
        ) s ON s.company_id = t.company_id AND s.table_id = t.table_id AND s.member_id = t.member_id AND s.language = t.language
    WHEN MATCHED AND NOT COALESCE(t.name, '') = COALESCE(s.name, '') THEN
        UPDATE SET name = s.name
    WHEN NOT MATCHED THEN
        INSERT (company_id, table_id, member_id, language, name) VALUES (s.company_id, s.table_id, s.member_id, s.language, s.name);
    END


END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects users
-- =============================================

CREATE PROCEDURE [xls25].[usp_users]
    @company_id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    c.username
    , c.name
    , c.is_admin
    , c.is_active
    , c.is_default
FROM
    dbo25.users c
WHERE
    c.company_id = @company_id
ORDER BY
    c.username

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Delete procedure for usp_users
-- =============================================

CREATE PROCEDURE [xls25].[usp_users_delete]
    @company_id int = NULL
    , @username nvarchar(128) = NULL
AS
BEGIN

SET NOCOUNT ON

IF NOT EXISTS (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1 AND is_admin = 1)
    RETURN

SET NOCOUNT OFF

DELETE dbo25.users WHERE company_id = @company_id AND username = @username

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Insert procedure for usp_users
-- =============================================

CREATE PROCEDURE [xls25].[usp_users_insert]
    @company_id int = NULL
    , @username nvarchar(128) = NULL
    , @name nvarchar(100) = NULL
    , @is_admin bit = NULL
    , @is_active bit = NULL
    , @is_default bit = NULL
AS
BEGIN

SET NOCOUNT ON

IF NOT EXISTS (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1 AND is_admin = 1)
    RETURN

SET @username = LTRIM(RTRIM(@username))
SET @name = LTRIM(RTRIM(@name))

IF @is_default IS NULL
    SET @is_default = CASE WHEN EXISTS(SELECT company_id FROM dbo25.users WHERE username = @username) THEN 0 ELSE 1 END

SET NOCOUNT OFF

INSERT INTO dbo25.users (company_id, username, name, is_admin, is_active, is_default) VALUES (@company_id, @username, @name, COALESCE(@is_admin, 0), COALESCE(@is_active, 1), @is_default)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Update procedure for usp_users
-- =============================================

CREATE PROCEDURE [xls25].[usp_users_update]
    @company_id int = NULL
    , @username nvarchar(128) = NULL
    , @name nvarchar(100) = NULL
    , @is_admin bit = NULL
    , @is_active bit = NULL
    , @is_default bit = NULL
AS
BEGIN

SET NOCOUNT ON

IF NOT EXISTS (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1 AND is_admin = 1)
    RETURN

SET @username = LTRIM(RTRIM(@username))
SET @name = LTRIM(RTRIM(@name))

SET NOCOUNT OFF

UPDATE dbo25.users
SET
    name = @name
    , is_admin = @is_admin
    , is_active = @is_active
    , is_default = @is_default
FROM
    dbo25.users c
WHERE
    c.company_id = @company_id
    AND c.username = @username
    AND NOT (
        COALESCE(name, '-1') = COALESCE(@name, '-2')
        AND COALESCE(is_admin, -1) = COALESCE(@is_admin, -2)
        AND COALESCE(is_active, -1) = COALESCE(@is_active, -2)
        AND COALESCE(is_default, -1) = COALESCE(@is_default, -2)
        )

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Adds time members for a year
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_add_language]
    @company_id int
    , @language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT TOP 1 @company_id = company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1

IF @company_id IS NULL
    RETURN

IF EXISTS (SELECT TOP 1 t.company_id FROM dbo25.translations t WHERE t.company_id = @company_id AND t.language = @language)
    RETURN

SET NOCOUNT OFF

INSERT INTO dbo25.translations (company_id, table_id, member_id, language, name)
SELECT
    c.id, 2, c.id, @language, c.name
FROM
    dbo25.companies c
WHERE
    c.id = @company_id


END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Adds quarter members for a year
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_add_quarters]
    @company_id int
    , @year int
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

-- DECLARE @year int = 2020

DECLARE @message nvarchar(max)

IF @year IS NULL OR LEN(CAST(@year AS nvarchar)) <> 4
    BEGIN
    DECLARE @current_year int = DATEPART(Y, GETDATE())
    SET @message = dbo25.get_translated_string('Specify 4-digit year like %i', @data_language)
    RAISERROR (@message, 11, 0, @current_year)
    RETURN
    END

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

-- creating absent members first

INSERT INTO dbo25.members (company_id, code, name, dimension_id, sort_order, calc_type_id, row_color)
SELECT
    @company_id, c.code, c.name, c.dimension_id, c.sort_order, c.calc_type_id, c.row_color
FROM
    dbo25.get_quarter_members(@year) c
    LEFT OUTER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
WHERE
    m.code IS NULL
ORDER BY
    c.sort_order

-- updating member properties after creating members

UPDATE dbo25.members
SET
    previous_period_id = p.id
    , same_period_id = s.id
FROM
    dbo25.get_quarter_members(@year) c
    INNER JOIN dbo25.members ON dbo25.members.company_id = @company_id AND dbo25.members.code = c.code
    LEFT OUTER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.previous_period
    LEFT OUTER JOIN dbo25.members s ON s.company_id = @company_id AND s.code = c.same_period
WHERE
    NOT COALESCE(dbo25.members.previous_period_id, 0) = COALESCE(p.id, 0)
    OR NOT COALESCE(dbo25.members.same_period_id, 0) = COALESCE(s.id, 0)

-- inserting parents by parent1

INSERT INTO dbo25.parents (member_id, parent_id, factor)
SELECT
    m.id, p.id, 1
FROM
    dbo25.get_quarter_members(@year) c
    INNER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
    INNER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.parent1
    LEFT OUTER JOIN dbo25.parents mp ON mp.member_id = m.id AND mp.parent_id = p.id
WHERE
    mp.member_id IS NULL

-- inserting parents by parent2

INSERT INTO dbo25.parents (member_id, parent_id, factor)
SELECT
    m.id, p.id, 1
FROM
    dbo25.get_quarter_members(@year) c
    INNER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
    INNER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.parent2
    LEFT OUTER JOIN dbo25.parents mp ON mp.member_id = m.id AND mp.parent_id = p.id
WHERE
    mp.member_id IS NULL

-- inserting parents by parent3

INSERT INTO dbo25.parents (member_id, parent_id, factor)
SELECT
    m.id, p.id, 1
FROM
    dbo25.get_quarter_members(@year) c
    INNER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
    INNER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.parent3
    LEFT OUTER JOIN dbo25.parents mp ON mp.member_id = m.id AND mp.parent_id = p.id
WHERE
    mp.member_id IS NULL

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_closed_periods
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_copy_data]
    @company_id int = NULL
    , @code nvarchar(50) = NULL
    , @source_category_code nvarchar(50) = NULL
    , @target_category_code nvarchar(50) = NULL
    , @set_1_to_copy tinyint = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

-- DECLARE @code nvarchar(128) = '2018.01'
-- DECLARE @source_category_code nvarchar(128) = 'BUDGET'
-- DECLARE @target_category_code nvarchar(128) = 'ACTUAL'
-- DECLARE @set_1_to_copy tinyint = 1

SET NOCOUNT ON

IF @company_id IS NULL OR @code IS NULL OR @source_category_code IS NULL OR @target_category_code IS NULL
    RETURN

EXECUTE AS CALLER

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

DECLARE @message nvarchar(max)

DECLARE @time_code nvarchar(50)
DECLARE @time_id int

SELECT @time_id = id, @time_code = code FROM dbo25.members WHERE company_id = @company_id AND code = @code AND dimension_id = 2

IF @time_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('%s is not a member of times', @data_language)
    RAISERROR (@message, 11, 0, @code)
    RETURN
    END

DECLARE @source_category_id int, @target_category_id int

SELECT @source_category_id = id FROM dbo25.members WHERE company_id = @company_id AND code = @source_category_code AND dimension_id = 3
SELECT @target_category_id = id FROM dbo25.members WHERE company_id = @company_id AND code = @target_category_code AND dimension_id = 3

IF @source_category_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('%s is not a member of categories', @data_language)
    RAISERROR (@message, 11, 0, @source_category_code)
    RETURN
    END

IF @target_category_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('%s is not a member of categories', @data_language)
    RAISERROR (@message, 11, 0, @target_category_code)
    RETURN
    END

IF NOT COALESCE(@set_1_to_copy, 0) = 1
    BEGIN
    SET @message = dbo25.get_translated_string('Set 1 in the parameter to copy data', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

IF EXISTS (
    SELECT
        TOP 1 f.id
    FROM
        dbo25.facts f
    WHERE
        f.id2 = @time_id
        AND f.id3 = @target_category_id
    )
    BEGIN
    SET @message = dbo25.get_translated_string('The target period contains data.', @data_language) + CHAR(13) + CHAR(10) + dbo25.get_translated_string('Delete the data before copying.', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

DECLARE @source TABLE (id int PRIMARY KEY)
DECLARE @target TABLE (id int PRIMARY KEY)

BEGIN TRANSACTION

BEGIN TRY

    INSERT INTO dbo25.facts
        (source_value, value, id1, id2, id3, id4, id5, id6, id7)
    SELECT
        f.source_value, f.value, id1, id2, @target_category_id, id4, id5, id6, id7
    FROM
        dbo25.facts f
    WHERE
        f.id2 = @time_id
        AND f.id3 = @source_category_id

END TRY

BEGIN CATCH
    SELECT @message = ERROR_MESSAGE()

    IF @@TRANCOUNT > 0
        ROLLBACK TRANSACTION

    RAISERROR (@message, 11, 0, @message)
    RETURN

END CATCH

IF @@TRANCOUNT > 0
    COMMIT TRANSACTION

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_closed_periods
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_delete_data]
    @company_id int = NULL
    , @code nvarchar(50) = NULL
    , @category_code nvarchar(50) = NULL
    , @set_1_to_delete tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL OR @code IS NULL OR @category_code IS NULL
    RETURN

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

DECLARE @message nvarchar(max)

DECLARE @time_code nvarchar(50)
DECLARE @time_id int

SELECT @time_id = id, @time_code = code FROM dbo25.members WHERE company_id = @company_id AND code = @code AND dimension_id = 2

IF @time_code IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find member_id %s', @data_language)
    RAISERROR (@message, 11, 0, @code)
    RETURN
    END
ELSE IF @time_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('%s is not a member of times', @data_language)
    RAISERROR (@message, 11, 0, @time_code)
    RETURN
    END

DECLARE @category_id int

SELECT @category_id = id FROM dbo25.members WHERE company_id = @company_id AND code = @category_code AND dimension_id = 3

IF @category_id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('%s is not a member of categories', @data_language)
    RAISERROR (@message, 11, 0, @category_code)
    RETURN
    END

IF NOT COALESCE(@set_1_to_delete, 0) = 1
    BEGIN
    SET @message = dbo25.get_translated_string('Set 1 in the parameter to delete data in %s %s', @data_language)
    RAISERROR (@message, 11, 0, @category_code, @time_code)
    RETURN
    END

DELETE dbo25.facts
FROM
    dbo25.facts f
WHERE
    f.id2 = @time_id
    AND f.id3 = @category_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects source values for of the cell for usp_run_form
--
-- This procedure requires @table_name or @form_id.
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_run_form_cell_data]
    @column_name nvarchar(255) = NULL
    , @table_name nvarchar(255) = NULL
    , @member_id int = NULL
    , @member_id2 int = NULL
    , @member_id3 int = NULL
    , @form_id int = NULL
    , @form_p1 int = NULL
    , @form_p2 int = NULL
    , @form_p3 int = NULL
    , @form_p4 int = NULL
    , @form_p5 int = NULL
    , @form_p6 int = NULL
    , @form_p7 int = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER--, ENCRYPTION
AS
BEGIN

SET NOCOUNT ON

IF @column_name IN (
    'id', 'id2', 'id3', 'sort_oder', 'sort_oder2', 'sort_oder3',
    'member_id', 'member_id2', 'member_id3',
    'decimal_places', 'is_percent', 'row_color', 'row_bold', 'row_indent',
    'code', 'name', 'code1', 'code2', 'name2', 'code3', 'name3',
    'calc_type', 'tax_rate', 'unit', 'data_format'
    )
    RETURN

EXECUTE AS CALLER

DECLARE @company_id int

-- Getting form_id from name

IF @form_id IS NULL
    BEGIN
    DECLARE @form_code nvarchar(128)

    IF LEFT(@table_name, 1) = '['
        SET @form_code = REPLACE(REPLACE(LEFT(@table_name, LEN(@table_name) - 1),'[xls25a].[', ''), '[xls25b].[', '')
    ELSE
        SET @form_code = REPLACE(REPLACE(@table_name, 'xls25a.', ''), 'xls25b.', '')

    SELECT
        @company_id = f.company_id, @form_id = f.id
    FROM
        dbo25.forms f
        INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
        INNER JOIN dbo25.companies c ON c.id = f.company_id
    WHERE
        c.code + '_' + f.code = @form_code

    END
ELSE
    BEGIN
    SELECT TOP 1 @company_id = t.company_id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @form_id AND u.username = USER_NAME() AND u.is_active = 1
    END

IF @company_id IS NULL
    RETURN

DECLARE @functional_currency_id int = (SELECT TOP 1 id FROM dbo25.units WHERE company_id = @company_id AND is_functional_currency = 1 ORDER BY id)
DECLARE @functional_currency_str nvarchar(10) = COALESCE(CAST(@functional_currency_id AS nvarchar(10)), 'NULL')

DECLARE @code1 nvarchar(128)
DECLARE @code2 nvarchar(128)
DECLARE @code3 nvarchar(128)

DECLARE @index int

DECLARE @str nvarchar(255) = LTRIM(RTRIM(@column_name))

IF RIGHT(@str, 1) = ''''
    SET @str = LEFT(@str, LEN(@str) - 1)

SET @index = CHARINDEX(' ', @str, 1)

IF @index = 0
    SET @code1 = @str
ELSE
    BEGIN
    SET @code1 = SUBSTRING(@str, 1, @index - 1)
    SET @str = LTRIM(SUBSTRING(@str, @index + 1, LEN(@str) - @index))

    SET @index = CHARINDEX(' ', @str, 1)
    IF @index = 0
        SET @code2 = @str
    ELSE
        BEGIN
        SET @code2 = SUBSTRING(@str, 1, @index - 1)
        SET @str = LTRIM(SUBSTRING(@str, @index + 1, LEN(@str) - @index))

        SET @code3 = @str
        END
    END

DECLARE @parameters TABLE (dimension_id int PRIMARY KEY, member_id int, default_select_permission bit, d nvarchar(3), m nvarchar(10))

INSERT INTO @parameters (dimension_id, member_id, default_select_permission)
SELECT
    d.id
    , CASE fd.axis_type_id
        WHEN 1 THEN COALESCE(p.member_id, fd.root_member_id, d.default_member_id, d.root_member_id)
        WHEN 2 THEN
            CASE fd.header_order
            WHEN 1 THEN @member_id
            WHEN 2 THEN @member_id2
            WHEN 3 THEN @member_id3
            END
        WHEN 3 THEN CASE fd.header_order
            WHEN 1 THEN (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @code1)
            WHEN 2 THEN (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @code2)
            WHEN 3 THEN (SELECT id FROM dbo25.members WHERE company_id = @company_id AND code = @code3)
            END
        END AS member_id
    , d.default_select_permission
FROM
    dbo25.dimension_properties d
    LEFT OUTER JOIN dbo25.form_dimensions fd ON fd.dimension_id = d.id AND fd.form_id = @form_id
    LEFT OUTER JOIN (
            SELECT 1 AS id, @form_p1 AS member_id
            UNION ALL SELECT 2, @form_p2
            UNION ALL SELECT 3, @form_p3
            UNION ALL SELECT 4, @form_p4
            UNION ALL SELECT 5, @form_p5
            UNION ALL SELECT 6, @form_p6
            UNION ALL SELECT 7, @form_p7
        ) p ON p.id = fd.parameter_index
WHERE
    d.company_id = @company_id
    AND d.is_active = 1
ORDER BY
    d.id

UPDATE @parameters SET d = CAST(dimension_id AS nvarchar), m = CAST(member_id AS nvarchar)

-- SELECT * FROM @parameters

DECLARE @message nvarchar(max)
DECLARE @code nvarchar(max)

IF IS_MEMBER('planning_app_users') = 1
    BEGIN

    SELECT @code = STUFF((
            SELECT
                ', ' + m.code
            FROM
                @parameters p
                LEFT OUTER JOIN dbo25.member_permissions mp ON mp.member_id = p.member_id AND mp.username = USER_NAME()
                INNER JOIN dbo25.members m ON m.id = p.member_id
            WHERE
                m.company_id = @company_id
                AND COALESCE(mp.select_permission, p.default_select_permission, 0) = 0
            ORDER BY
                p.dimension_id
            FOR XML PATH(''), TYPE
        ).value('.', 'nvarchar(MAX)'), 1, 2, '')

    IF @code IS NOT NULL
        BEGIN
        SET @message = dbo25.get_translated_string('You have no permission to select members: %s', @data_language)
        RAISERROR (@message, 11, 0, @code)
        RETURN
        END

    END

DECLARE @sql nvarchar(max)

SET @sql = 'SELECT
    f.id'
    + (SELECT  '
    , '   + CASE d.name_format_id
                WHEN 0 THEN 'm' + d + '.code'
                WHEN 1 THEN 'm' + d + '.name'
                WHEN 2 THEN 'CAST(m' + d + '.id AS nvarchar)'
                WHEN 3 THEN 'CAST(m' + d + '.id AS nvarchar) + '' - '' + m' + d + '.name'
                ELSE 'm' + d + '.code + '' - '' + m'+ d + '.name' END
            + ' AS dim_' + d
        FROM @parameters p INNER JOIN dbo25.dimension_properties d ON d.company_id = @company_id AND d.id = p.dimension_id ORDER BY p.dimension_id
        FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')
    + '
    , u.name AS unit
    , CAST(f.source_value AS numeric(18,2)) AS source_value
    , COALESCE(r1.currency_rate, 1 / r2.currency_rate) AS currency_rate
    , f1.factor
    , CAST(f.value * f1.factor AS numeric(18,2)) AS value
FROM
    dbo25.facts f
    ' + (SELECT '
    INNER JOIN dbo25.factors f' + d + ' ON f' + d + '.member_id = f.id' + d + ' AND f' + d + '.parent_id = ' + m
            FROM @parameters p ORDER BY p.dimension_id
            FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)') + '
    ' + (SELECT '
    INNER JOIN dbo25.members m' + d + ' ON m' + d + '.id = f.id' + d
            FROM @parameters p ORDER BY p.dimension_id
            FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)') + '

    LEFT OUTER JOIN dbo25.units u ON u.id = m1.unit_id
    LEFT OUTER JOIN dbo25.currency_rates r1 ON r1.base_currency_id = m1.unit_id AND r1.quote_currency_id = ' + @functional_currency_str + ' AND r1.time_id = m2.id AND r1.category_id = m3.id AND NOT r1.currency_rate = 0
    LEFT OUTER JOIN dbo25.currency_rates r2 ON r2.base_currency_id = ' + @functional_currency_str + ' AND r2.quote_currency_id = m1.unit_id AND r2.time_id = m2.id AND r2.category_id = m3.id AND NOT r2.currency_rate = 0

WHERE
    f.value IS NOT NULL
    AND (f1.calc_type_id = 1 AND f2.calc_type_id = 1
        OR f1.calc_type_id = 3 AND f2.calc_type_id = 3
        )

ORDER BY
    m1.sort_order
    , m1.code'
    + (SELECT '
    , m' + d + '.sort_order
    , m' + d + '.code'
        FROM @parameters p WHERE p.dimension_id > 1 ORDER BY p.dimension_id
        FOR XML PATH(''), TYPE).value('.', 'nvarchar(MAX)')

-- PRINT @sql

REVERT

EXEC (@sql)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Sets a unit as the system currency
--
-- The application calculates cube values as
-- dbo25.facts.source_value = dbo25.facts.value * currency_rate where the currency_rate is
--
-- 1 where account unit is_functional_currency = 1
-- currency_rates.currency_rate where currency_rates.base_currency = account unit_id AND currency_rates.quote_currency = 0
-- 1 / currency_rates.currency_rate where currency_rates.base_currency = 0 AND currency_rates.quote_currency = account unit_id
--
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_set_functional_currency]
    @id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

IF @id IS NULL
    BEGIN
    SET @message = dbo25.get_translated_string('Specify a currency id', @data_language)

    RAISERROR (@message, 11, 0)
    RETURN
    END

DECLARE @company_id int

SELECT TOP 1 @company_id = t.company_id FROM dbo25.units t INNER JOIN dbo25.users u ON u.company_id = t.company_id  WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1

IF @company_id IS NULL
    RETURN

UPDATE dbo25.units
SET
    is_functional_currency = CASE WHEN id = @id THEN 1 ELSE 0 END
WHERE
    company_id = @company_id AND (id = @id OR is_functional_currency = 1)

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Updates fact values after currency changes
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_update_facts]
    @company_id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

DECLARE @functional_currency_id int = (SELECT TOP 1 id FROM dbo25.units WHERE company_id = @company_id AND is_functional_currency = 1 ORDER BY id)

-- Updating values for currencies like EUR/USD, GBP/USD, or AUD/USD if USD is the system currency

UPDATE dbo25.facts
SET
    value = f.source_value * r.currency_rate
FROM
    dbo25.facts f
    INNER JOIN dbo25.members m1 ON m1.id = f.id1
    INNER JOIN dbo25.units u ON u.id = m1.unit_id

    INNER JOIN dbo25.currency_rates r ON r.base_currency_id = m1.unit_id AND r.quote_currency_id = @functional_currency_id AND r.time_id = f.id2 AND r.category_id = f.id3
WHERE
    m1.company_id = @company_id
    AND NOT m1.unit_id = @functional_currency_id
    AND u.is_currency = 1
    AND NOT f.source_value IS NULL
    AND NOT r.currency_rate = 0
    AND NOT f.value = f.source_value * r.currency_rate

-- Updating values for currencies like USD/JPY, USD/CAD, or USD/CHF if USD is the system currency

UPDATE dbo25.facts
SET
    value = f.source_value / r.currency_rate
FROM
    dbo25.facts f
    INNER JOIN dbo25.members m1 ON m1.id = f.id1
    INNER JOIN dbo25.units u ON u.id = m1.unit_id

    INNER JOIN dbo25.currency_rates r ON r.base_currency_id = @functional_currency_id AND r.quote_currency_id = m1.unit_id AND r.time_id = f.id2 AND r.category_id = f.id3
WHERE
    m1.company_id = @company_id
    AND NOT m1.unit_id = @functional_currency_id
    AND u.is_currency = 1
    AND NOT f.source_value IS NULL
    AND NOT r.currency_rate = 0
    AND NOT f.value = f.source_value / r.currency_rate

-- Fixing issues with different values for the base currency

UPDATE dbo25.facts
SET
    value = f.source_value
FROM
    dbo25.facts f
    INNER JOIN dbo25.members m1 ON m1.id = f.id1
WHERE
    m1.company_id = @company_id
    AND m1.unit_id = @functional_currency_id
    AND NOT f.value = f.source_value

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Updates the hierarchies and factors tables
--
-- Execute the procedure after adding members or changing member parents and factors
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_update_hierarchies]
    @company_id int = NULL
AS
BEGIN

SET NOCOUNT ON

-- DELETE dbo25.hierarchies;
-- DELETE dbo25.factors;

BEGIN -- Parents --

DELETE FROM dbo25.parents
FROM
    dbo25.parents mp
    INNER JOIN dbo25.members m ON m.id = mp.member_id
    INNER JOIN dbo25.members p ON p.id = mp.parent_id
WHERE
    m.company_id = @company_id
    AND m.calc_type_id = 9
    OR p.calc_type_id = 9

END

BEGIN -- Hierarchies --

    WITH cte (member_id, parent_id, [level], calc_type_id)
    AS (
        SELECT
            m.id
            , m.id
            , 0
            , m.calc_type_id
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND NOT m.calc_type_id IN (9)  -- rowset
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
            , m.calc_type_id
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
        WHERE
            cte.[level] < 20
            AND NOT (cte.calc_type_id = 5 AND m.calc_type_id = 5)   -- No hierarchies for YTD + YTD
    )

    INSERT INTO dbo25.hierarchies
        (member_id, parent_id, [level])
    SELECT
        t.member_id
        , t.parent_id
        , t.[level]
    FROM
        (
        SELECT
            cte.member_id
            , cte.parent_id
            , MIN(cte.[level]) AS [level]
        FROM
            cte
        GROUP BY
            cte.member_id
            , cte.parent_id
        ) t
        LEFT OUTER JOIN dbo25.hierarchies f ON f.member_id = t.member_id AND f.parent_id = t.parent_id
    WHERE
        f.member_id IS NULL;

    WITH cte (member_id, parent_id, [level], calc_type_id)
    AS (
        SELECT
            m.id
            , m.id
            , 0
            , m.calc_type_id
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND NOT m.calc_type_id IN (9)  -- rowset
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
            , m.calc_type_id
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
        WHERE
            cte.[level] < 20
            AND NOT (cte.calc_type_id = 5 AND m.calc_type_id = 5)   -- No hierarchies for YTD + YTD
    )

    UPDATE dbo25.hierarchies
    SET
        [level] = t.[level]
    FROM
        (
        SELECT
            cte.member_id
            , cte.parent_id
            , MIN(cte.[level]) AS [level]
        FROM
            cte
        GROUP BY
            cte.member_id
            , cte.parent_id
        ) t
        INNER JOIN dbo25.hierarchies f ON f.member_id = t.member_id AND f.parent_id = t.parent_id
    WHERE
        NOT f.[level] = t.[level];

    WITH cte (member_id, parent_id, [level], calc_type_id)
    AS (
        SELECT
            m.id
            , m.id
            , 0
            , m.calc_type_id
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND NOT m.calc_type_id IN (9)  -- rowset
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
            , m.calc_type_id
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
        WHERE
            cte.[level] < 20
            AND NOT (cte.calc_type_id = 5 AND m.calc_type_id = 5)   -- No hierarchies for YTD + YTD
    )

    DELETE dbo25.hierarchies
    FROM
        dbo25.hierarchies f
        LEFT OUTER JOIN (
            SELECT
                cte.member_id
                , cte.parent_id
                , MIN(cte.[level]) AS [level]
            FROM
                cte
            GROUP BY
                cte.member_id
                , cte.parent_id
        ) t ON t.member_id = f.member_id AND t.parent_id = f.parent_id
        INNER JOIN dbo25.members m ON m.id = f.member_id
    WHERE
        m.company_id = @company_id
        AND t.member_id IS NULL
        AND NOT (f.member_id = f.parent_id AND NOT m.calc_type_id IN (9));

    -- Insert absent members as self-parent

    INSERT INTO dbo25.hierarchies
        (member_id, parent_id, [level])
    SELECT
        m.id
        , m.id
        , 0
    FROM
        dbo25.members m
        LEFT OUTER JOIN dbo25.hierarchies h ON h.member_id = m.id AND h.parent_id = m.id
    WHERE
        m.company_id = @company_id
        AND NOT m.calc_type_id IN (9)   -- rowset
        AND h.member_id IS NULL;

END

BEGIN -- Common Factors --

    WITH cte (member_id, parent_id, [level], factor, calc_type_id)
    AS (
        SELECT
            m.id
            , m.id
            , 0
            , CAST(1.0 AS float) AS factor
            , m.calc_type_id
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND m.id NOT IN (SELECT parent_id FROM dbo25.parents)
            AND m.dimension_id NOT IN (1, 2)
            AND m.calc_type_id IN (1)           -- total
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
            , cte.factor * p.factor
            , m.calc_type_id
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
        WHERE
            cte.[level] < 20
            AND m.calc_type_id IN (1)           -- total
            AND NOT p.factor = 0
    )

    INSERT INTO dbo25.factors
        (member_id, parent_id, factor, calc_type_id)
    SELECT
        t.member_id
        , t.parent_id
        , t.factor
        , t.calc_type_id
    FROM
        (
        SELECT
            cte.member_id
            , cte.parent_id
            , SUM(cte.factor) AS factor
            , cte.calc_type_id
        FROM
            cte
        GROUP BY
            cte.member_id
            , cte.parent_id
            , cte.calc_type_id
        ) t
        LEFT OUTER JOIN dbo25.factors f ON f.member_id = t.member_id AND f.parent_id = t.parent_id AND f.calc_type_id = t.calc_type_id
    WHERE
        f.member_id IS NULL;

    WITH cte (member_id, parent_id, [level], factor, calc_type_id)
    AS (
        SELECT
            m.id
            , m.id
            , 0
            , CAST(1.0 AS float) AS factor
            , m.calc_type_id
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND m.id NOT IN (SELECT parent_id FROM dbo25.parents)
            AND m.dimension_id NOT IN (1, 2)
            AND m.calc_type_id IN (1)           -- total
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
            , cte.factor * p.factor
            , cte.calc_type_id
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
        WHERE
            cte.[level] < 20
            AND m.calc_type_id IN (1)           -- total
            AND NOT p.factor = 0
    )

    UPDATE dbo25.factors
    SET
        factor = t.factor
    FROM
        (
        SELECT
            cte.member_id
            , cte.parent_id
            , SUM(cte.factor) AS factor
            , cte.calc_type_id
        FROM
            cte
        GROUP BY
            cte.member_id
            , cte.parent_id
            , cte.calc_type_id
        ) t
        INNER JOIN dbo25.factors f ON f.member_id = t.member_id AND f.parent_id = t.parent_id AND f.calc_type_id = t.calc_type_id
    WHERE
        NOT t.factor = f.factor;

    WITH cte (member_id, parent_id, [level], factor, calc_type_id)
    AS (
        SELECT
            m.id
            , m.id
            , 0
            , CAST(1.0 AS float) AS factor
            , m.calc_type_id
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND m.id NOT IN (SELECT parent_id FROM dbo25.parents)
            AND m.dimension_id NOT IN (1, 2)
            AND m.calc_type_id IN (1)           -- total
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
            , cte.factor * p.factor
            , cte.calc_type_id
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
        WHERE
            cte.[level] < 20
            AND m.calc_type_id IN (1)           -- total
            AND NOT p.factor = 0
    )

    DELETE dbo25.factors
    FROM
        dbo25.factors f
        INNER JOIN dbo25.members m ON m.id = f.member_id
        LEFT OUTER JOIN (
            SELECT
                cte.member_id
                , cte.parent_id
                , SUM(cte.factor) AS factor
                , cte.calc_type_id
            FROM
                cte
            GROUP BY
                cte.member_id
                , cte.parent_id
                , cte.calc_type_id
        ) t ON t.member_id = f.member_id AND t.parent_id = f.parent_id AND t.calc_type_id = f.calc_type_id
    WHERE
        m.company_id = @company_id
        AND m.dimension_id NOT IN (1, 2)
        AND t.member_id IS NULL

END

BEGIN -- Account Factors --

    DECLARE @calc_types TABLE (source_id int, member_id int, parent_id int, PRIMARY KEY (source_id, member_id, parent_id))

    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (1, 1, 1)   -- Total->Total                    -> Total
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (3, 1, 3)   -- OpeningBalance->Total           -> OpeningBalance
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (1, 2, 2)   -- Total->Average                    -> Average
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (2, 2, 2)   -- Average->Average                -> Average
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (1, 3, 3)   -- Total->OpeningBalance            -> OpeningBalance
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (3, 3, 3)   -- OpeningBalance->OpeningBalance    -> OpeningBalance
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (1, 4, 1)   -- Total->ClosingBalance            -> Total
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (1, 4, 3)   -- Total->ClosingBalance            -> OpeningBalance
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (4, 4, 1)   -- ClosingBalance->ClosingBalance  -> Total
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (4, 4, 3)   -- ClosingBalance->ClosingBalance  -> OpeningBalance
    INSERT INTO @calc_types (source_id, member_id, parent_id) VALUES (1, 11, 1)  -- Total->Revaluation                -> Total

    ;WITH cte (member_id, parent_id, [level], factor, child_calc_type_id, calc_type_id)
    AS (
        SELECT
            m.id
            , m.id
            , 0
            , CAST(1.0 AS decimal(30, 15)) AS factor
            , NULL
            , c.parent_id
        FROM
            dbo25.members m
            INNER JOIN @calc_types c ON c.source_id = 1 AND c.member_id = m.calc_type_id
        WHERE
            m.company_id = @company_id
            AND m.id NOT IN (SELECT parent_id FROM dbo25.parents)
            AND m.dimension_id = 1
            AND m.calc_type_id IN (1, 2, 3, 4, 11)  -- total, average, open, close, revaluation
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
            , CAST(cte.factor * p.factor AS decimal(30, 15)) AS factor
            , cte.calc_type_id
            , c.parent_id
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
            INNER JOIN @calc_types c ON c.source_id = cte.calc_type_id AND c.member_id = m.calc_type_id
        WHERE
            cte.[level] < 20
            AND m.calc_type_id IN (1, 2, 3, 4, 11)  -- total, average, open, close, revaluation
            AND NOT p.factor = 0
    )

    INSERT INTO dbo25.factors
        (member_id, parent_id, factor, calc_type_id)
    SELECT
        t.member_id
        , t.parent_id
        , CAST(t.factor AS float) AS factor
        , t.calc_type_id
    FROM
        (
        SELECT
            cte.member_id
            , cte.parent_id
            , SUM(cte.factor) AS factor
            , cte.calc_type_id
        FROM
            cte
        GROUP BY
            cte.member_id
            , cte.parent_id
            , cte.calc_type_id
        ) t
        LEFT OUTER JOIN dbo25.factors f ON f.member_id = t.member_id AND f.parent_id = t.parent_id AND f.calc_type_id = t.calc_type_id
    WHERE
        f.member_id IS NULL;

    ;WITH cte (member_id, parent_id, [level], factor, child_calc_type_id, calc_type_id)
    AS (
        SELECT
            m.id
            , m.id
            , 0
            , CAST(1.0 AS decimal(30, 15)) AS factor
            , NULL
            , c.parent_id
        FROM
            dbo25.members m
            INNER JOIN @calc_types c ON c.source_id = 1 AND c.member_id = m.calc_type_id
        WHERE
            m.company_id = @company_id
            AND m.id NOT IN (SELECT parent_id FROM dbo25.parents)
            AND m.dimension_id = 1
            AND m.calc_type_id IN (1, 2, 3, 4, 11)  -- total, average, open, close, revaluation
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
            , CAST(cte.factor * p.factor AS decimal(30, 15)) AS factor
            , cte.calc_type_id
            , c.parent_id
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
            INNER JOIN @calc_types c ON c.source_id = cte.calc_type_id AND c.member_id = m.calc_type_id
        WHERE
            cte.[level] < 20
            AND m.calc_type_id IN (1, 2, 3, 4, 11)  -- total, average, open, close, revaluation
            AND NOT p.factor = 0
    )

    UPDATE dbo25.factors
    SET
        factor = CAST(t.factor AS float)
    FROM
        (
        SELECT
            cte.member_id
            , cte.parent_id
            , SUM(cte.factor) AS factor
            , cte.calc_type_id
        FROM
            cte
        GROUP BY
            cte.member_id
            , cte.parent_id
            , cte.calc_type_id
        ) t
        INNER JOIN dbo25.factors f ON f.member_id = t.member_id AND f.parent_id = t.parent_id AND f.calc_type_id = t.calc_type_id
    WHERE
        NOT t.factor = f.factor;

    ;WITH cte (member_id, parent_id, [level], factor, child_calc_type_id, calc_type_id)
    AS (
        SELECT
            m.id
            , m.id
            , 0
            , CAST(1.0 AS decimal(30, 15)) AS factor
            , NULL
            , c.parent_id
        FROM
            dbo25.members m
            INNER JOIN @calc_types c ON c.source_id = 1 AND c.member_id = m.calc_type_id
        WHERE
            m.company_id = @company_id
            AND m.id NOT IN (SELECT parent_id FROM dbo25.parents)
            AND m.dimension_id = 1
            AND m.calc_type_id IN (1, 2, 3, 4, 11)  -- total, average, open, close, revaluation
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
            , CAST(cte.factor * p.factor AS decimal(30, 15)) AS factor
            , cte.calc_type_id
            , c.parent_id
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
            INNER JOIN @calc_types c ON c.source_id = cte.calc_type_id AND c.member_id = m.calc_type_id
        WHERE
            cte.[level] < 20
            AND m.calc_type_id IN (1, 2, 3, 4, 11)  -- total, average, open, close, revaluation
            AND NOT p.factor = 0
    )

    DELETE dbo25.factors
    FROM
        dbo25.factors f
        INNER JOIN dbo25.members m ON m.id = f.member_id
        LEFT OUTER JOIN (
            SELECT
                cte.member_id
                , cte.parent_id
                , SUM(cte.factor) AS factor
                , cte.calc_type_id
            FROM
                cte
            GROUP BY
                cte.member_id
                , cte.parent_id
                , cte.calc_type_id
        ) t ON t.member_id = f.member_id AND t.parent_id = f.parent_id AND t.calc_type_id = f.calc_type_id
    WHERE
        m.company_id = @company_id
        AND m.dimension_id = 1
        AND t.member_id IS NULL;

END

BEGIN -- Time Factors --

    WITH cte (member_id, parent_id, [level])
    AS (
        SELECT
            m.id
            , m.id
            , 0
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND m.id NOT IN (SELECT parent_id FROM dbo25.parents)
            AND m.dimension_id = 2
            AND m.calc_type_id IN (1, 5)    -- total, YTD
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
        WHERE
            cte.[level] < 20
            AND m.calc_type_id IN (1, 5)    -- total, YTD
            AND NOT p.factor = 0
    )

    INSERT INTO dbo25.factors
        (member_id, parent_id, calc_type_id)
    SELECT
        t.member_id
        , t.parent_id
        , 1            -- total
    FROM
        (
        SELECT
            cte.member_id
            , cte.parent_id
        FROM
            cte
        GROUP BY
            cte.member_id
            , cte.parent_id
        ) t
        LEFT OUTER JOIN dbo25.factors f ON f.member_id = t.member_id AND f.parent_id = t.parent_id AND f.calc_type_id = 1  -- total
    WHERE
        f.member_id IS NULL;

    WITH cte (member_id, parent_id, [level])
    AS (
        SELECT
            m.id
            , m.id
            , 0
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
            AND m.id NOT IN (SELECT parent_id FROM dbo25.parents)
            AND m.dimension_id = 2
            AND m.calc_type_id IN (1, 5)    -- total, YTD
        UNION ALL
        SELECT
            cte.member_id
            , p.parent_id
            , cte.[level] + 1
        FROM
            cte
            INNER JOIN dbo25.parents p ON p.member_id = cte.parent_id
            INNER JOIN dbo25.members m ON m.id = p.parent_id
        WHERE
            cte.[level] < 20
            AND m.calc_type_id IN (1, 5)    -- total, YTD
            AND NOT p.factor = 0
    )

    DELETE dbo25.factors
    FROM
        dbo25.factors f
        INNER JOIN dbo25.members m ON m.id = f.member_id
        LEFT OUTER JOIN (
            SELECT
                cte.member_id
                , cte.parent_id
            FROM
                cte
            GROUP BY
                cte.member_id
                , cte.parent_id
        ) t ON t.member_id = f.member_id AND t.parent_id = f.parent_id
    WHERE
        m.company_id = @company_id
        AND m.dimension_id = 2
        AND f.calc_type_id = 1     -- total
        AND t.member_id IS NULL;

END

BEGIN -- Time Factors - Opening Balance --

    WITH cte (member_id, parent_id, [level])
    AS (
        SELECT
            f.member_id
            , m.id
            , 0 AS [level]
        FROM
            dbo25.members m
            INNER JOIN dbo25.factors f ON f.parent_id = m.previous_period_id AND f.calc_type_id = 1    -- total
        WHERE
            m.company_id = @company_id
            AND m.dimension_id = 2
            AND m.calc_type_id IN (1, 5)    -- total, YTD
            AND NOT m.previous_period_id IS NULL
        UNION ALL
        SELECT
            cte.member_id
            , m.id
            , cte.[level] + 1 AS [level]
        FROM
            cte
            INNER JOIN dbo25.members m ON m.previous_period_id = cte.parent_id
        WHERE
            cte.[level] < 100
            AND m.calc_type_id IN (1, 5)    -- total, YTD
    )

    INSERT INTO dbo25.factors
        (member_id, parent_id, calc_type_id)
    SELECT
        t.member_id
        , t.parent_id
        , 3 AS calc_type_id        -- opening balance
    FROM
        (
        SELECT
            cte.member_id
            , cte.parent_id
        FROM
            cte
        GROUP BY
            cte.member_id
            , cte.parent_id
        ) t
        LEFT OUTER JOIN dbo25.factors f ON f.member_id = t.member_id AND f.parent_id = t.parent_id AND f.calc_type_id = 3
    WHERE
        f.member_id IS NULL;

    WITH cte (member_id, parent_id, [level])
    AS (
        SELECT
            f.member_id
            , m.id
            , 0 AS [level]
        FROM
            dbo25.members m
            INNER JOIN dbo25.factors f ON f.parent_id = m.previous_period_id AND f.calc_type_id = 1    -- total
        WHERE
            m.company_id = @company_id
            AND m.dimension_id = 2
            AND m.calc_type_id IN (1, 5)    -- total, YTD
            AND NOT m.previous_period_id IS NULL
        UNION ALL
        SELECT
            cte.member_id
            , m.id
            , cte.[level] + 1 AS [level]
        FROM
            cte
            INNER JOIN dbo25.members m ON m.previous_period_id = cte.parent_id
        WHERE
            cte.[level] < 100
            AND m.calc_type_id IN (1, 5)    -- total, YTD
    )

    DELETE dbo25.factors
    FROM
        dbo25.factors f
        INNER JOIN dbo25.members m ON m.id = f.member_id
        LEFT OUTER JOIN (
        SELECT
            cte.member_id
            , cte.parent_id
        FROM
            cte
        GROUP BY
            cte.member_id
            , cte.parent_id
        ) t ON f.member_id = t.member_id AND f.parent_id = t.parent_id AND f.calc_type_id = 3  -- opening balance
    WHERE
        m.company_id = @company_id
        AND m.dimension_id = 2
        AND f.calc_type_id = 3      -- opening balance
        AND t.member_id IS NULL;

END

BEGIN -- Cube formulas and is_calculated --

UPDATE dbo25.members
SET
    cube_formula = f.cube_formula
FROM
    dbo25.members m
    INNER JOIN (
        SELECT
            m.id
            , REPLACE(REPLACE('+' + (
                    SELECT
                        CASE
                            WHEN f.factor = 1  THEN '+'
                            WHEN f.factor = 0  THEN '+'
                            WHEN f.factor = -1 THEN '-'
                            WHEN f.factor > 0  THEN '+' + CAST(f.factor AS nvarchar)
                            ELSE                          CAST(f.factor AS nvarchar)
                            END
                        + '[' + a.code + ']'
                    FROM
                        dbo25.factors f
                        INNER JOIN dbo25.members a ON a.id = f.member_id
                    WHERE
                        f.parent_id = m.id
                        AND f.calc_type_id IN (1, 2, 3)                     -- total, average, open
                        AND NOT (f.calc_type_id = 3 AND m.calc_type_id = 4) -- except open for closing balances
                        AND NOT (f.calc_type_id = 3 AND m.calc_type_id = 1 AND m.dimension_id = 2) -- except open for times as totals
                    ORDER BY
                        a.sort_order
                        , a.id
                    FOR XML PATH(''), TYPE
                ).value('.', 'nvarchar(MAX)'), '++', ''), '+-', '-') AS cube_formula
        FROM
            dbo25.members m
        WHERE
            m.company_id = @company_id
    ) f ON f.id = m.id
WHERE
    NOT COALESCE(m.cube_formula, '') = COALESCE(f.cube_formula, '')

UPDATE dbo25.members
SET
    is_calculated = CASE WHEN p.parent_id IS NULL AND m.calc_type_id IN (1, 2, 3, 8) THEN 0 ELSE 1 END
FROM
    dbo25.members m
    LEFT OUTER JOIN (
        SELECT DISTINCT parent_id FROM dbo25.parents
    ) p ON p.parent_id = m.id
WHERE
    m.company_id = @company_id
    AND m.is_calculated IS NULL
    OR NOT m.is_calculated = CASE WHEN p.parent_id IS NULL AND m.calc_type_id IN (1, 2, 3, 8) THEN 0 ELSE 1 END

END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Adds time members for a year
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_add_year]
    @company_id int
    , @year int
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

-- DECLARE @year int = 2018

DECLARE @message nvarchar(max)

IF @year IS NULL OR LEN(CAST(@year AS nvarchar)) <> 4
    BEGIN
    DECLARE @current_year int = DATEPART(Y, GETDATE())
    SET @message = dbo25.get_translated_string('Specify 4-digit year like %i', @data_language)
    RAISERROR (@message, 11, 0, @current_year)
    RETURN
    END

SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

-- creating absent members first

INSERT INTO dbo25.members (company_id, code, name, dimension_id, sort_order, calc_type_id, row_color)
SELECT
    @company_id, c.code, c.name, c.dimension_id, c.sort_order, c.calc_type_id, c.row_color
FROM
    dbo25.get_year_members(@year) c
    LEFT OUTER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
WHERE
    m.code IS NULL
ORDER BY
    c.sort_order

-- updating member properties after creating members

UPDATE dbo25.members
SET
    previous_period_id = p.id
    , same_period_id = s.id
FROM
    dbo25.get_year_members(@year) c
    INNER JOIN dbo25.members ON dbo25.members.company_id = @company_id AND dbo25.members.code = c.code
    LEFT OUTER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.previous_period
    LEFT OUTER JOIN dbo25.members s ON s.company_id = @company_id AND s.code = c.same_period
WHERE
    NOT COALESCE(dbo25.members.previous_period_id, 0) = COALESCE(p.id, 0)
    OR NOT COALESCE(dbo25.members.same_period_id, 0) = COALESCE(s.id, 0)

-- inserting parents by parent1

INSERT INTO dbo25.parents (member_id, parent_id, factor)
SELECT
    m.id, p.id, 1
FROM
    dbo25.get_year_members(@year) c
    INNER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
    INNER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.parent1
    LEFT OUTER JOIN dbo25.parents mp ON mp.member_id = m.id AND mp.parent_id = p.id
WHERE
    mp.member_id IS NULL

-- inserting parents by parent2

INSERT INTO dbo25.parents (member_id, parent_id, factor)
SELECT
    m.id, p.id, 1
FROM
    dbo25.get_year_members(@year) c
    INNER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
    INNER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.parent2
    LEFT OUTER JOIN dbo25.parents mp ON mp.member_id = m.id AND mp.parent_id = p.id
WHERE
    mp.member_id IS NULL

-- inserting parents by parent3

INSERT INTO dbo25.parents (member_id, parent_id, factor)
SELECT
    m.id, p.id, 1
FROM
    dbo25.get_year_members(@year) c
    INNER JOIN dbo25.members m ON m.company_id = @company_id AND m.code = c.code
    INNER JOIN dbo25.members p ON p.company_id = @company_id AND p.code = c.parent3
    LEFT OUTER JOIN dbo25.parents mp ON mp.member_id = m.id AND mp.parent_id = p.id
WHERE
    mp.member_id IS NULL

-- updating hierarchies

EXEC xls25.xl_actions_update_hierarchies @company_id

-- inserting a year into the years* rowset

INSERT INTO dbo25.form_rows (rowset_id, sort_order, member_id)
SELECT
    r.id, @year, m.id
FROM
    dbo25.members m
    CROSS JOIN dbo25.members r
    LEFT OUTER JOIN dbo25.form_rows f ON f.rowset_id = r.id AND f.member_id = m.id
WHERE
    m.company_id = @company_id
    AND r.company_id = @company_id
    AND m.code = CAST(@year AS nvarchar)
    AND r.code = N'years*'
    AND f.id IS NULL

-- inserting quarters into the quarters* rowset

INSERT INTO dbo25.form_rows (rowset_id, sort_order, member_id)
SELECT
    r.id, 4000000 + @year * 100 + q.n, m.id
FROM
    (VALUES (1), (2), (3), (4)) q(n)
    INNER JOIN dbo25.members m ON m.code = CAST(@year AS nvarchar) + '.Q' + CAST(q.n AS nvarchar)
    CROSS JOIN dbo25.members r
    LEFT OUTER JOIN dbo25.form_rows f ON f.rowset_id = r.id AND f.member_id = m.id
WHERE
    m.company_id = @company_id
    AND r.company_id = @company_id
    AND r.code = N'quarters*'
    AND f.id IS NULL

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Deletes time members for a year
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_delete_year]
    @company_id int = NULL
    , @year int = NULL
    , @set_1_to_delete tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

-- DECLARE @year int = 2020
-- DECLARE @set_1_to_delete tinyint = 1

DECLARE @message nvarchar(max)

IF @year IS NULL OR LEN(CAST(@year AS nvarchar)) <> 4
    BEGIN
    DECLARE @current_year int = DATEPART(Y, GETDATE())
    SET @message = dbo25.get_translated_string('Specify 4-digit year like %i', @data_language)
    RAISERROR (@message, 11, 0, @current_year)
    RETURN
    END

IF NOT COALESCE(@set_1_to_delete, 0) = 1
    BEGIN
    SET @message = dbo25.get_translated_string('Set 1 in the set_1_to_delete parameter to delete members', @data_language)
    RAISERROR (@message, 11, 0)
    RETURN
    END

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

IF @company_id IS NULL
    RETURN

DELETE FROM dbo25.members
FROM
    dbo25.get_year_members(@year) c
    INNER JOIN dbo25.members ON dbo25.members.company_id = @company_id AND dbo25.members.code = c.code
WHERE
    dbo25.members.dimension_id = 2

-- updating hierarchies

EXEC xls25.xl_actions_update_hierarchies @company_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Updates inherited member permissions
--
-- The procedure is called directly from usp_form_permissions_change and xl_actions_update_hierarchies
-- =============================================

CREATE PROCEDURE [xls25].[xl_actions_update_member_permissions]
    @company_id int = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @COUNT int

-- Deleting inherited rows without parents

SET @COUNT = 1

WHILE @COUNT > 0
    BEGIN

    DELETE FROM dbo25.member_permissions
    FROM
        dbo25.member_permissions mp
        INNER JOIN dbo25.members m ON m.id = mp.member_id
        LEFT OUTER JOIN (
            SELECT
                h.member_id
                , mp.username
            FROM
                dbo25.member_permissions mp
                INNER JOIN dbo25.members m ON m.id = mp.member_id
                INNER JOIN dbo25.hierarchies h ON h.parent_id = mp.member_id
                INNER JOIN dbo25.member_permissions mp2 ON mp2.member_id = h.member_id AND mp2.username = mp.username
            WHERE
                m.company_id = @company_id
                AND NOT h.member_id = h.parent_id
            GROUP BY
                h.member_id
                , mp.username
        ) p ON p.member_id = mp.member_id AND p.username = mp.username
    WHERE
        m.company_id = @company_id
        AND mp.is_inherited = 1
        AND p.member_id IS NULL

    SET @COUNT = @@ROWCOUNT
    END

-- Updating inherited permissions

SET @COUNT = 1

WHILE @COUNT > 0
    BEGIN

    UPDATE dbo25.member_permissions
    SET
        select_permission = p.select_permission
        , update_permission = p.update_permission
    FROM
        dbo25.member_permissions mp
        INNER JOIN dbo25.members m ON m.id = mp.member_id
        INNER JOIN (
            SELECT
                h.member_id
                , mp.username
                , MIN(CAST(mp.select_permission AS tinyint)) AS select_permission
                , MIN(CAST(mp.update_permission AS tinyint)) AS update_permission
                , 1 AS is_inherited
            FROM
                dbo25.member_permissions mp
                INNER JOIN dbo25.members m ON m.id = mp.member_id
                INNER JOIN dbo25.hierarchies h ON h.parent_id = mp.member_id
                INNER JOIN dbo25.member_permissions mp2 ON mp2.member_id = h.member_id AND mp2.username = mp.username
            WHERE
                m.company_id = @company_id
                AND NOT h.member_id = h.parent_id
                AND h.[level] = 1
            GROUP BY
                h.member_id
                , mp.username
        ) p ON p.member_id = mp.member_id AND p.username = mp.username
    WHERE
        m.company_id = @company_id
        AND mp.is_inherited = 1
        AND (NOT mp.select_permission = p.select_permission
            OR NOT mp.update_permission = p.update_permission
            )

    SET @COUNT = @@ROWCOUNT
    END

-- Inserting new inherited permissions

SET @COUNT = 1

WHILE @COUNT > 0
    BEGIN

    INSERT INTO dbo25.member_permissions
        (member_id, username, select_permission, update_permission, is_inherited)
    SELECT
        h.member_id
        , mp.username
        , MIN(CAST(mp.select_permission AS tinyint)) AS select_permission
        , MIN(CAST(mp.update_permission AS tinyint)) AS update_permission
        , 1 AS is_inherited
    FROM
        dbo25.member_permissions mp
        INNER JOIN dbo25.members m ON m.id = mp.member_id
        INNER JOIN dbo25.hierarchies h ON h.parent_id = mp.member_id
        LEFT OUTER JOIN dbo25.member_permissions mp2 ON mp2.member_id = h.member_id AND mp2.username = mp.username
    WHERE
        m.company_id = @company_id
        AND NOT h.member_id = h.parent_id
        AND mp2.member_id IS NULL
        AND h.[level] = 1
    GROUP BY
        h.member_id
        , mp.username

    SET @COUNT = @@ROWCOUNT
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Cell change event handler for usp_member_permissions
-- =============================================

CREATE PROCEDURE [xls25].[usp_member_permissions_change]
    @column_name nvarchar(128) = NULL
    , @cell_value nvarchar(255) = NULL
    , @id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @message nvarchar(max)

-- Protecting non-data columns

IF @column_name IN ('member_id', 'sort_order', 'row_color', 'row_bold', 'row_indent', 'member', 'data_format')
    BEGIN
    SET @message = dbo25.get_translated_string('The column ''%s'' is not updatable', @data_language)
    RAISERROR (@message, 11, 0, @column_name)
    RETURN
    END

DECLARE @company_id int

SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.members t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @id AND u.username = USER_NAME() AND u.is_active = 1)

IF @company_id IS NULL
    RETURN

IF NOT EXISTS (SELECT id FROM dbo25.members WHERE id = @id)
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find member_id %s', @data_language)
    RAISERROR (@message, 11, 0, @id)
    RETURN
    END

IF @cell_value IS NULL
    BEGIN

    DELETE FROM dbo25.member_permissions WHERE member_id = @id AND username = @column_name

    EXEC xls25.xl_actions_update_member_permissions @company_id

    RETURN
    END

DECLARE @username nvarchar(128) = @column_name

IF NOT EXISTS (SELECT u.username FROM dbo25.users u WHERE u.company_id = @company_id AND u.username = @username)
    BEGIN
    SET @message = dbo25.get_translated_string('Cannot find the principal ''%s'', because it does not exist or you do not have permission.', @data_language)
    RAISERROR (@message, 11, 0, @username)
    RETURN
    END

-- Getting permissions from the cell value

DECLARE @select_permission bit = NULL
DECLARE @update_permission bit = NULL

IF UPPER(@cell_value) = 'D'
    BEGIN
    SET @select_permission = 0
    SET @update_permission = 0
    END
ELSE IF UPPER(@cell_value) = 'R'
    BEGIN
    SET @select_permission = 1
    SET @update_permission = 0
    END
ELSE IF UPPER(@cell_value) = 'W'
    BEGIN
    SET @select_permission = 1
    SET @update_permission = 1
    END
ELSE
    BEGIN
    SET @message = dbo25.get_translated_string('Unknown option: %s (Use R, W, D)', @data_language)
    RAISERROR (@message, 11, 0, @cell_value)
    RETURN
    END

UPDATE dbo25.member_permissions
SET
    select_permission = @select_permission
    , update_permission = @update_permission
    , is_inherited = 0
WHERE
    member_id = @id
    AND username = @column_name

IF @@ROWCOUNT = 0
    INSERT INTO dbo25.member_permissions (member_id, username, select_permission, update_permission, is_inherited)
    VALUES (@id, @username, @select_permission, @update_permission, 0)

EXEC xls25.xl_actions_update_member_permissions @company_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects usp_members aliases
-- =============================================

CREATE PROCEDURE [xls25].[xl_aliases_members]
    @dimension_id tinyint = NULL
    , @data_language varchar(10) = NULL
WITH EXECUTE AS OWNER
AS
BEGIN

SET NOCOUNT ON

SELECT
    t.COLUMN_NAME AS column_name
    , COALESCE(tr1.TRANSLATED_NAME, tr2.TRANSLATED_NAME, t.COLUMN_NAME) AS alias
    , CASE
        WHEN t.COLUMN_NAME IN ('id', 'company_id', 'dimension_id') THEN NULL
        WHEN t.COLUMN_NAME IN ('sort_order', 'code', 'name', 'calc_type_id', 'is_active') THEN 1
        WHEN t.COLUMN_NAME IN ('comment') THEN CASE @dimension_id WHEN 1 THEN NULL ELSE 0 END
        WHEN t.COLUMN_NAME IN ('decimal_places', 'is_percent', 'row_color', 'row_bold', 'row_indent', 'is_editable') THEN CASE @dimension_id WHEN 1 THEN NULL ELSE 0 END
        WHEN t.COLUMN_NAME IN ('show_line', 'show_line_before', 'show_line_after') THEN CASE @dimension_id WHEN 1 THEN NULL ELSE 0 END
        WHEN t.COLUMN_NAME IN ('tax_rate_id', 'unit_id') THEN CASE @dimension_id WHEN 1 THEN 1 ELSE 0 END
        WHEN t.COLUMN_NAME IN ('previous_period_id','same_period_id') THEN CASE @dimension_id WHEN 2 THEN 1 WHEN 3 THEN 1 ELSE 0 END
        WHEN t.COLUMN_NAME IN ('parent1', 'factor1', 'parent2', 'factor2') THEN 1
        WHEN t.COLUMN_NAME IN ('parent3', 'factor3', 'parent4', 'factor4') THEN CASE @dimension_id WHEN 1 THEN 1 WHEN 2 THEN 1 ELSE 0 END
        WHEN t.COLUMN_NAME IN ('parent5', 'factor5', 'parent6', 'factor6', 'parent7', 'factor7') THEN CASE @dimension_id WHEN 1 THEN NULL ELSE 0 END
        WHEN t.COLUMN_NAME IN ('excel_formula') THEN CASE @dimension_id WHEN 1 THEN NULL ELSE 0 END
        ELSE NULL END AS is_active
FROM
    (
        SELECT
            c.ORDINAL_POSITION
            , c.COLUMN_NAME
        FROM
            INFORMATION_SCHEMA.COLUMNS c
        WHERE
            c.TABLE_SCHEMA = 'dbo25' AND c.TABLE_NAME = 'members'
        UNION ALL
        SELECT
            c.ORDINAL_POSITION
            , c.COLUMN_NAME
        FROM
            (VALUES
                (29, 'is_editable')
                , (30, 'parent1'), (31, 'factor1')
                , (32, 'parent2'), (32, 'factor2')
                , (34, 'parent3'), (35, 'factor3')
                , (36, 'parent4'), (37, 'factor4')
                , (38, 'parent5'), (39, 'factor5')
                , (40, 'parent6'), (41, 'factor6')
                , (42, 'parent7'), (43, 'factor7')
            ) c(ORDINAL_POSITION, COLUMN_NAME)
    ) t
    LEFT OUTER JOIN xls.translations tr1 ON tr1.TABLE_SCHEMA = 'xls25' AND tr1.TABLE_NAME = 'usp_members' AND tr1.COLUMN_NAME = t.COLUMN_NAME AND tr1.LANGUAGE_NAME = @data_language
    LEFT OUTER JOIN xls.translations tr2 ON tr2.TABLE_SCHEMA = 'xls25' AND tr2.TABLE_NAME IS NULL AND tr2.COLUMN_NAME = t.COLUMN_NAME AND tr2.LANGUAGE_NAME = @data_language
ORDER BY
    t.ORDINAL_POSITION

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values as 0 or 1
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_0_or_1]
AS
BEGIN

SET NOCOUNT ON

SELECT 0 AS id UNION ALL SELECT 1 AS id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for calc_type_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_calc_type_id]
    @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    t.id
    , COALESCE(tr.TRANSLATED_NAME, t.name) AS name
FROM
    dbo25.calc_types t
    LEFT OUTER JOIN xls.translations tr ON tr.TABLE_SCHEMA = 'dbo25' AND tr.TABLE_NAME = 'calc_types_name' AND tr.COLUMN_NAME = t.name AND tr.LANGUAGE_NAME = @data_language
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for calc_type_id including NULL
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_calc_type_id_or_null]
    @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, -1 AS sort_order UNION ALL
    SELECT
        t.id
        , COALESCE(tr.TRANSLATED_NAME, t.name) AS name
        , t.sort_order
    FROM
        dbo25.calc_types t
        LEFT OUTER JOIN xls.translations tr ON tr.TABLE_SCHEMA = 'dbo25' AND tr.TABLE_NAME = 'calc_types_name' AND tr.COLUMN_NAME = t.name AND tr.LANGUAGE_NAME = @data_language
    ) t
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Parameter values of company_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_company_id]
    @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    t.id
    , COALESCE(tr.name, t.name) AS name
FROM
    dbo25.companies t
    INNER JOIN dbo25.users u ON u.company_id = t.id AND u.username = USER_NAME() AND u.is_active = 1
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.id AND tr.table_id = 2 AND tr.member_id = t.id AND tr.language = @data_language
ORDER BY
    t.sort_order
    , t.name

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for currency_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_currency_id]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    t.id
    , COALESCE(tr.name, t.name) AS name
FROM
    dbo25.units t
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 3 AND tr.member_id = t.id AND tr.language = @data_language
WHERE
    t.company_id = @company_id
    AND t.is_currency = 1
    AND t.is_active = 1
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_dimension_id]
    @company_id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    t.id
    , t.name
FROM
    dbo25.dimension_properties t
WHERE
    t.company_id = @company_id
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension_id including NULL
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_dimension_id_or_null]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS sort_order UNION ALL
    SELECT
        t.id
        , COALESCE(tr.name, t.name) AS name
        , t.sort_order
    FROM
        dbo25.dimension_properties t
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 5 AND tr.member_id = t.id AND tr.language = @data_language
    WHERE
        t.company_id = @company_id
) t
ORDER BY
    sort_order
    , id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for form_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_form_id]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    t.id
    , COALESCE(tr.name, t.name) AS name
FROM
    dbo25.forms t
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 7 AND tr.member_id = t.id AND tr.language = @data_language
WHERE
    t.company_id = @company_id
ORDER BY
    t.sort_order
    , t.code

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for form_id including NULL
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_form_id_or_null]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS sort_order UNION ALL
    SELECT
        t.id
        , COALESCE(tr.name, t.name) AS name
        , t.sort_order
    FROM
        dbo25.forms t
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 7 AND tr.member_id = t.id AND tr.language = @data_language
    WHERE
        t.company_id = @company_id
) t
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension member parameters
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_member_id_by_dimension_id]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS sort_order UNION ALL
    SELECT
        m.id
        , CASE d.name_format_id WHEN 0 THEN m.code WHEN 1 THEN COALESCE(tr.name, m.name) WHEN 2 THEN CAST(m.id AS nvarchar) WHEN 3 THEN CAST(m.id AS nvarchar) + ' - ' + COALESCE(tr.name, m.name) ELSE m.code + ' - ' + COALESCE(tr.name, m.name) END AS name
        , m.sort_order
    FROM
        dbo25.members m
        INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
        LEFT OUTER JOIN dbo25.member_permissions mp ON mp.member_id = m.id AND mp.username = USER_NAME()
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
    WHERE
        m.company_id = @company_id
        AND NOT m.calc_type_id = 9                                    -- not rowset
        AND m.dimension_id = @dimension_id
        AND (
            d.is_protected = 0                                      -- unprotected dimension
            OR COALESCE(IS_MEMBER('planning_app_users'), 0) = 0     -- analyst or developer
            OR COALESCE(mp.select_permission,                       -- user's select permission
                d.default_select_permission, 0) = 1                 -- default select permission
            )
    ) m
ORDER BY
    sort_order
    , name

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension member parameters
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_member_id_dim1]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_member_id_by_dimension_id @company_id, 1, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension member parameters
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_member_id_dim2]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_member_id_by_dimension_id @company_id, 2, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension member parameters
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_member_id_dim3]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_member_id_by_dimension_id @company_id, 3, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension member parameters
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_member_id_dim4]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_member_id_by_dimension_id @company_id, 4, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension member parameters
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_member_id_dim5]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_member_id_by_dimension_id @company_id, 5, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension member parameters
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_member_id_dim6]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_member_id_by_dimension_id @company_id, 6, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension member parameters
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_member_id_dim7]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_member_id_by_dimension_id @company_id, 7, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_relation_dimension_id]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    t.id
    , COALESCE(tr.name, t.name) AS name
FROM
    dbo25.dimension_properties t
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 5 AND tr.member_id = t.id AND tr.language = @data_language
WHERE
    t.company_id = @company_id
    AND t.id IN (1)
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for dimension_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_relation_partner_id]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    t.id
    , COALESCE(tr.name, t.name) AS name
FROM
    dbo25.dimension_properties t
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 5 AND tr.member_id = t.id AND tr.language = @data_language
WHERE
    t.company_id = @company_id
    AND t.id IN (4, 5, 6, 7)
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for root_member_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_root_member_id_code]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS dimension_sort_order, NULL AS dimension_id, NULL AS sort_order
    UNION ALL
    SELECT
        m.id
        , m.code
        , d.sort_order AS dimension_sort_order
        , m.dimension_id
        , m.sort_order
    FROM
        dbo25.members r
        LEFT OUTER JOIN dbo25.parents p ON p.member_id = r.id
        LEFT OUTER JOIN dbo25.hierarchies h ON h.parent_id = r.id
        LEFT OUTER JOIN dbo25.members m ON m.id = h.member_id
        LEFT OUTER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
    WHERE
        r.company_id = @company_id
        AND p.member_id IS NULL
        AND h.[level] IN (0, 1)
        AND NOT m.calc_type_id = 9     -- rowset
        AND r.dimension_id = COALESCE(@dimension_id, r.dimension_id)
) t
ORDER BY
    dimension_sort_order
    , dimension_id
    , sort_order
    , id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for rowset_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_rowset_id]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    m.id
    , COALESCE(tr.name, m.name) AS name
FROM
    dbo25.members m
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
WHERE
    m.company_id = @company_id
    AND m.calc_type_id = 9     -- rowset
ORDER BY
    dimension_id
    , sort_order
    , name

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for rowset_id including NULL
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_rowset_id_or_null]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS dimension_id, NULL AS sort_order UNION ALL
    SELECT
        m.id
        , COALESCE(tr.name, m.name) AS name
        , m.dimension_id
        , m.sort_order
    FROM
        dbo25.members m
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
    WHERE
        m.company_id = @company_id
        AND m.calc_type_id = 9     -- rowset
    ) m
ORDER BY
    dimension_id
    , sort_order
    , name

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for parameters of usp_run_form
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_run_form_p]
    @form_id int = NULL
    , @parameter_index int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

DECLARE @company_id int = NULL

SET @company_id = (SELECT TOP 1 u.company_id FROM dbo25.forms t INNER JOIN dbo25.users u ON u.company_id = t.company_id WHERE t.id = @form_id AND u.username = USER_NAME() AND u.is_active = 1)

DECLARE @dimension_id tinyint
DECLARE @name_format_id tinyint
DECLARE @is_protected bit
DECLARE @dim_root_member_id int
DECLARE @form_root_member_id int
DECLARE @axis_type_id int
DECLARE @start_level int
DECLARE @end_level int
DECLARE @default_select_permission bit

-- The end level by default is 1 for accounts and 255 for others

SELECT
    @dimension_id = fd.dimension_id
    , @name_format_id = d.name_format_id
    , @is_protected = d.is_protected
    , @default_select_permission = d.default_select_permission
    , @dim_root_member_id = d.root_member_id
    , @form_root_member_id = fd.root_member_id
    , @axis_type_id = fd.axis_type_id
    , @start_level = COALESCE(fd.parameter_start_level, 0)
    , @end_level = COALESCE(fd.parameter_end_level, CASE WHEN d.id = 1 THEN 1 ELSE 255 END)
FROM
    dbo25.form_dimensions fd
    INNER JOIN dbo25.forms f ON f.id = fd.form_id
    INNER JOIN dbo25.dimension_properties d ON d.company_id = f.company_id AND d.id = fd.dimension_id
WHERE
    f.company_id = @company_id
    AND fd.form_id = @form_id
    AND fd.parameter_index = @parameter_index
    AND d.is_active = 1

-- DECLARE @form_id int = 1
-- DECLARE @parameter_index int = 1

SELECT id, name FROM (

    -- show empty members if the dimension contains more than one data member
    SELECT
        NULL AS id, NULL AS name, NULL AS sort_order, 1 AS part
    WHERE
        (SELECT COUNT(*) FROM dbo25.members m2 WHERE m2.company_id = @company_id AND m2.dimension_id = @dimension_id AND m2.calc_type_id IN (1)) > 1

    -- rowsets
    UNION ALL
    SELECT
        m.id
        , COALESCE(tr.name, m.name) AS name    -- name only
        , m.sort_order
        , 2 AS part                        -- before other members
    FROM
        dbo25.members m
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
    WHERE
        m.company_id = @company_id
        AND m.dimension_id = @dimension_id
        AND m.calc_type_id = 9            -- rowsets
        AND m.id IN (

            -- specified single rowset
            SELECT
                m.id
            FROM
                dbo25.members m
            WHERE
                m.id = @form_root_member_id            -- rowset member itself

            -- rowsets with permitted rows
            UNION ALL
            SELECT
                r.rowset_id
            FROM
                dbo25.members m
                INNER JOIN dbo25.form_rows r ON r.member_id = m.id
                LEFT OUTER JOIN dbo25.member_permissions mp ON mp.member_id = r.member_id AND mp.username = USER_NAME()
            WHERE
                m.company_id = @company_id
                AND m.dimension_id = @dimension_id
                AND @axis_type_id IN (2, 3)                                    -- row and column axis only
                AND @form_root_member_id IS NULL                            -- no root
                AND (
                    @is_protected = 0                                        -- unprotected dimension
                    OR COALESCE(IS_MEMBER('planning_app_users'), 0) = 0     -- analyst or developer
                    OR COALESCE(mp.select_permission,                       -- user's select permission
                        @default_select_permission, 0) = 1                    -- default select permission
                    )
        )

    UNION ALL
    -- hierarchy members
    SELECT
        m.id
        , CASE @name_format_id WHEN 0 THEN m.code WHEN 1 THEN COALESCE(tr.name, m.name) WHEN 2 THEN CAST(m.id AS nvarchar) WHEN 3 THEN CAST(m.id AS nvarchar) + ' - ' + COALESCE(tr.name, m.name) ELSE m.code + ' - ' + COALESCE(tr.name, m.name) END AS name
        , m.sort_order
        , 3 AS part
    FROM
        dbo25.members m
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
    WHERE
        m.company_id = @company_id
        AND m.dimension_id = @dimension_id
        AND NOT m.calc_type_id = 9                -- not rowset
        AND (@form_root_member_id IS NULL AND @dim_root_member_id IS NULL    -- all members if there are no root members
            OR m.id IN (

            -- hierarchy members of the specified root
            SELECT
                h.member_id
            FROM
                dbo25.hierarchies h
                LEFT OUTER JOIN dbo25.member_permissions mp ON mp.member_id = h.member_id AND mp.username = USER_NAME()
                INNER JOIN dbo25.members m ON m.id = h.member_id
            WHERE
                h.parent_id = COALESCE(@form_root_member_id, @dim_root_member_id)
                AND h.[level] BETWEEN @start_level AND @end_level            -- 1 level for accounts by default, 255 for others
                AND (m.calc_type_id IN (1, 2, 5)                            -- total, average, YTD (for any axis)
                    OR @axis_type_id IN (2, 3) AND m.calc_type_id IN (0))    -- none  (for row and column axis)
                AND (
                    @is_protected = 0                                        -- unprotected dimension
                    OR COALESCE(IS_MEMBER('planning_app_users'), 0) = 0     -- analyst or developer
                    OR COALESCE(mp.select_permission,                       -- user's select permission
                        @default_select_permission, 0) = 1                    -- default select permission
                    )

            -- hierarchy-only parents for row and column axis
            UNION ALL
            SELECT
                h.member_id
            FROM
                dbo25.hierarchies h
                INNER JOIN dbo25.hierarchies h2 ON h2.parent_id = h.member_id
                LEFT OUTER JOIN dbo25.member_permissions mp ON mp.member_id = h2.member_id AND mp.username = USER_NAME()
                INNER JOIN dbo25.members m ON m.id = h.member_id
            WHERE
                h.parent_id = COALESCE(@form_root_member_id, @dim_root_member_id)
                AND @axis_type_id IN (2, 3)                                    -- row and column axis only
                AND m.calc_type_id = 0                                      -- none (parents with no aggregation)
                AND h.[level] BETWEEN @start_level AND @end_level            -- 1 level for accounts by default, 255 for others
                AND (
                    @is_protected = 0                                        -- unprotected dimension
                    OR COALESCE(IS_MEMBER('planning_app_users'), 0) = 0     -- analyst or developer
                    OR COALESCE(mp.select_permission,                       -- user's select permission
                        @default_select_permission, 0) = 1                    -- default select permission
                    )
        ))
    ) m
ORDER BY
    part
    , sort_order
    , name

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for @form_p1 of usp_run_form
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_run_form_p1]
    @form_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_run_form_p @form_id, 1, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for @form_p2 of usp_run_form
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_run_form_p2]
    @form_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_run_form_p @form_id, 2, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for @form_p3 of usp_run_form
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_run_form_p3]
    @form_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_run_form_p @form_id, 3, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for @form_p4 of usp_run_form
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_run_form_p4]
    @form_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_run_form_p @form_id, 4, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for @form_p5 of usp_run_form
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_run_form_p5]
    @form_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_run_form_p @form_id, 5, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for @form_p6 of usp_run_form
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_run_form_p6]
    @form_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_run_form_p @form_id, 6, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for @form_p6 of usp_run_form
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_run_form_p7]
    @form_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

EXEC xls25.xl_parameter_values_run_form_p @form_id, 7, @data_language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for unit_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_unit_id]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS sort_order UNION ALL
    SELECT
        t.id
        , COALESCE(tr.name, t.name) AS name
        , t.sort_order
    FROM
        dbo25.units t
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 3 AND tr.member_id = t.id AND tr.language = @data_language
    WHERE
        t.company_id = @company_id
        AND t.is_active = 1
    ) t
ORDER BY
    t.sort_order
    , t.name

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Selects parameter values for username
-- =============================================

CREATE PROCEDURE [xls25].[xl_parameter_values_username]
    @company_id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT NULL AS name UNION ALL
SELECT
    u.username AS name
FROM
    dbo25.users u
WHERE
    u.company_id = @company_id
    AND u.is_active = 1
    AND u.is_admin = 0
ORDER BY
    name

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of axis_type_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_axis_type_id]
    @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    t.id
    , COALESCE(tr.TRANSLATED_NAME, t.name) AS name
FROM
    dbo25.axis_types t
    LEFT OUTER JOIN xls.translations tr ON tr.TABLE_SCHEMA = 'dbo25' AND tr.TABLE_NAME = 'axis_types_name' AND tr.COLUMN_NAME = t.name AND tr.LANGUAGE_NAME = @data_language
ORDER BY
    id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of calc_type_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_calc_type_id]
    @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    t.id
    , COALESCE(tr.TRANSLATED_NAME, t.name) AS name
FROM
    dbo25.calc_types t
    LEFT OUTER JOIN xls.translations tr ON tr.TABLE_SCHEMA = 'dbo25' AND tr.TABLE_NAME = 'calc_types_name' AND tr.COLUMN_NAME = t.name AND tr.LANGUAGE_NAME = @data_language
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of calc_type_id
--
-- Accounts:    none, total, average, opening balance, closing balance, string
-- Times:       none, total, YTD
-- Categories:  none, total, same period, excel formula
-- Entities:    none, total
-- Others:      none, total, string (if activated)
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_calc_type_id_by_dimension_id]
    @dimension_id tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    t.id
    , COALESCE(tr.TRANSLATED_NAME, t.name) AS name
FROM
    dbo25.calc_types t
    LEFT OUTER JOIN dbo25.dimension_calc_types dc ON dc.calc_type_id = t.id AND dc.dimension_id = @dimension_id
    LEFT OUTER JOIN xls.translations tr ON tr.TABLE_SCHEMA = 'dbo25' AND tr.TABLE_NAME = 'calc_types_name' AND tr.COLUMN_NAME = t.name AND tr.LANGUAGE_NAME = @data_language
WHERE
    dc.is_active = 1
    OR @dimension_id IS NULL
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of default_member_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_default_member_id_code]
    @company_id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS dimension_sort_order, NULL AS dimension_id, NULL AS sort_order, NULL AS type_order
    UNION ALL
    SELECT
        m.id
        , m.code
        , d.sort_order AS dimension_sort_order
        , m.dimension_id
        , m.sort_order
        , 1 AS type_order
    FROM
        dbo25.members m
        LEFT OUTER JOIN dbo25.parents p ON p.parent_id = m.id
        INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
    WHERE
        m.company_id = @company_id
        AND p.member_id IS NULL
        AND NOT m.calc_type_id = 9     -- rowset
        AND NOT m.dimension_id IN (1, 2, 3, 4)
) t
ORDER BY
    dimension_sort_order
    , dimension_id
    , type_order
    , sort_order
    , id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of dimension_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_dimension_id]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    t.id
    , COALESCE(tr.name, t.name) AS name
FROM
    dbo25.dimension_properties t
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 5 AND tr.member_id = t.id AND tr.language = @data_language
WHERE
    t.company_id = @company_id
    AND t.is_active = 1
ORDER BY
    t.sort_order
    , t.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of member_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_member_id]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS dimension_sort_order, NULL AS dimension_id, NULL AS sort_order
    UNION ALL
    SELECT
        m.id
        , CASE d.name_format_id WHEN 0 THEN m.code WHEN 1 THEN m.name WHEN 2 THEN CAST(m.id AS nvarchar) WHEN 3 THEN CAST(m.id AS nvarchar) + ' - ' + m.name ELSE m.code + ' - ' + m.name END AS name
        , d.sort_order AS dimension_sort_order
        , m.dimension_id
        , m.sort_order
    FROM
        dbo25.members m
        INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
    WHERE
        m.company_id = @company_id
        AND m.is_active = 1
) t
ORDER BY
    dimension_sort_order
    , dimension_id
    , sort_order
    , id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of member_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_member_id_code_by_dimension_id]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS sort_order UNION ALL
    SELECT
        m.id
        , m.code AS name
        , m.sort_order
    FROM
        dbo25.members m
    WHERE
        m.company_id = @company_id
        AND m.dimension_id = @dimension_id
        AND m.is_active = 1
        AND NOT m.calc_type_id IN (9)
) m
ORDER BY
    m.sort_order
    , m.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of member_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_member_id_code_by_rowset_id]
    @company_id int = NULL
    , @rowset_id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS sort_order UNION ALL
    SELECT
        m.id
        , m.code AS name
        , m.sort_order
    FROM
        dbo25.members m
        INNER JOIN dbo25.members rs ON rs.dimension_id = m.dimension_id
        INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
    WHERE
        m.company_id = @company_id
        AND rs.id = @rowset_id
        AND m.is_active = 1
        AND NOT m.calc_type_id IN (9)   -- rowset
) m
ORDER BY
    m.sort_order
    , m.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of name_format_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_name_format_id]
    @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

SELECT
    t.id
    , COALESCE(tr.TRANSLATED_NAME, t.name) AS name
FROM
    dbo25.name_formats t
    LEFT OUTER JOIN xls.translations tr ON tr.TABLE_SCHEMA = 'dbo25' AND tr.TABLE_NAME = 'name_formats_name' AND tr.COLUMN_NAME = t.name AND tr.LANGUAGE_NAME = @data_language
ORDER BY
    id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of previous_period_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_previous_period_id_by_dimension_id]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS sort_order UNION ALL
    SELECT
        m.id
        , CASE d.name_format_id WHEN 0 THEN m.code WHEN 1 THEN COALESCE(tr.name, m.name) WHEN 2 THEN CAST(m.id AS nvarchar) WHEN 3 THEN CAST(m.id AS nvarchar) + ' - ' + COALESCE(tr.name, m.name) ELSE m.code + ' - ' + COALESCE(tr.name, m.name) END AS name
        , m.sort_order
    FROM
        dbo25.members m
        INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
    WHERE
        m.company_id = @company_id
        AND m.dimension_id = @dimension_id
        AND m.calc_type_id IN (1, 6)        -- total, same period (categories)
        AND d.id IN (2, 3)                  -- Times or Categories
) m
ORDER BY
    m.sort_order
    , m.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of root_member_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_root_member_id_code]
    @company_id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS dimension_sort_order, NULL AS dimension_id, NULL AS sort_order, NULL AS type_order
    UNION ALL
    SELECT
        m.id
        , m.code
        , d.sort_order AS dimension_sort_order
        , m.dimension_id
        , m.sort_order
        , 1 AS type_order
    FROM
        dbo25.members r
        LEFT OUTER JOIN dbo25.parents p ON p.member_id = r.id
        LEFT OUTER JOIN dbo25.hierarchies h ON h.parent_id = r.id
        LEFT OUTER JOIN dbo25.members m ON m.id = h.member_id
        LEFT OUTER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
    WHERE
        r.company_id = @company_id
        AND p.member_id IS NULL
        AND h.[level] IN (0, 1)
        AND m.is_active = 1
        AND NOT m.calc_type_id = 9     -- rowset
) t
ORDER BY
    dimension_sort_order
    , dimension_id
    , type_order
    , sort_order
    , id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of root_member_id or rowset_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_root_member_or_rowset_id_code]
    @company_id int = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name, dimension_id FROM (
    SELECT NULL AS id, NULL AS name, NULL AS dimension_sort_order, NULL AS dimension_id, NULL AS sort_order, NULL AS type_order
    UNION ALL
    SELECT
        m.id
        , m.code
        , d.sort_order AS dimension_sort_order
        , m.dimension_id
        , m.sort_order
        , 1 AS type_order
    FROM
        dbo25.members r
        LEFT OUTER JOIN dbo25.parents p ON p.member_id = r.id
        LEFT OUTER JOIN dbo25.hierarchies h ON h.parent_id = r.id
        LEFT OUTER JOIN dbo25.members m ON m.id = h.member_id
        LEFT OUTER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
    WHERE
        r.company_id = @company_id
        AND p.member_id IS NULL
        AND h.[level] IN (0, 1, 2, 3)
        AND NOT m.calc_type_id = 9      -- rowset
    UNION ALL
    SELECT
        m.id
        , m.code
        , d.sort_order AS dimension_sort_order
        , m.dimension_id
        , m.sort_order
        , 2 AS type_order
    FROM
        dbo25.members m
        LEFT OUTER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
    WHERE
        m.company_id = @company_id
        AND (m.calc_type_id = 9                -- rowset
            OR m.id = d.default_member_id)  -- default_member_id
) t
ORDER BY
    dimension_sort_order
    , dimension_id
    , type_order
    , sort_order
    , id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of rowset_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_rowset_id]
    @company_id int = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    t.id
    , COALESCE(tr.name, t.name) AS name
FROM
    dbo25.members t
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 1 AND tr.member_id = t.id AND tr.language = @data_language
WHERE
    t.company_id = @company_id
    AND t.calc_type_id = 9     -- rowset
ORDER BY
    id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of same_period_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_same_period_id_by_dimension_id]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT id, name FROM (
    SELECT NULL AS id, NULL AS name, NULL AS sort_order UNION ALL
    SELECT
        m.id
        , CASE d.name_format_id WHEN 0 THEN m.code WHEN 1 THEN COALESCE(tr.name, m.name) WHEN 2 THEN CAST(m.id AS nvarchar) WHEN 3 THEN CAST(m.id AS nvarchar) + ' - ' + COALESCE(tr.name, m.name) ELSE m.code + ' - ' + COALESCE(tr.name, m.name) END AS name
        , m.sort_order
    FROM
        dbo25.members m
        INNER JOIN dbo25.dimension_properties d ON d.company_id = m.company_id AND d.id = m.dimension_id
        LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = m.company_id AND tr.table_id = 1 AND tr.member_id = m.id AND tr.language = @data_language
    WHERE
        m.company_id = @company_id
        AND m.dimension_id = @dimension_id
        AND d.id IN (2, 3)                              -- Time or Category dimension
        AND NOT m.calc_type_id IN (0, 7, 9, 12, 13)     -- none, excel, rowset, difference, percent
    ) m
ORDER BY
    m.sort_order
    , m.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of tax_rate_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_tax_rate_id_by_dimension_id]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    t.id
    , COALESCE(tr.name, t.name) AS name
FROM
    dbo25.tax_rates t
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 4 AND tr.member_id = t.id AND tr.language = @data_language
WHERE
    t.company_id = @company_id
    AND t.is_active = 1
    AND @dimension_id = 1
ORDER BY
    t.sort_order
    , t.name

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: Validation list of unit_id
-- =============================================

CREATE PROCEDURE [xls25].[xl_validation_list_unit_id_by_dimension_id]
    @company_id int = NULL
    , @dimension_id tinyint = NULL
    , @data_language varchar(10) = NULL
AS
BEGIN

SET NOCOUNT ON

IF @company_id IS NULL
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE username = USER_NAME() AND is_active = 1 AND is_default = 1)
ELSE
    SET @company_id = (SELECT TOP 1 company_id FROM dbo25.users WHERE company_id = @company_id AND username = USER_NAME() AND is_active = 1)

SELECT
    t.id
    , COALESCE(tr.name, t.name) AS name
FROM
    dbo25.units t
    LEFT OUTER JOIN dbo25.translations tr ON tr.company_id = t.company_id AND tr.table_id = 3 AND tr.member_id = t.id AND tr.language = @data_language
WHERE
    t.company_id = @company_id
    AND t.is_active = 1
    AND @dimension_id = 1
ORDER BY
    t.sort_order
    , t.name

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: The procedure updates Excel table formats
-- =============================================

CREATE PROCEDURE [xls25a].[usp_update_table_format]
    @schema nvarchar(128) = NULL
    , @name nvarchar(128) = NULL
    , @ExcelFormatXML xml = NULL
AS
BEGIN

SET NOCOUNT ON

IF @schema IS NULL AND @name IS NULL
    RETURN

IF @schema IN ('xls25a', 'xls25b')
    BEGIN

    DECLARE @company_id int
    DECLARE @code nvarchar(128)

    SELECT
        @company_id = f.company_id, @code = f.code
    FROM
        dbo25.forms f
        INNER JOIN dbo25.users u ON u.company_id = f.company_id AND u.username = USER_NAME() AND u.is_active = 1
        INNER JOIN dbo25.companies c ON c.id = f.company_id
    WHERE
        c.code + '_' + f.code = @name

    IF @company_id IS NULL OR @code IS NULL
        RETURN

    SET NOCOUNT OFF

    IF @ExcelFormatXML IS NULL
        BEGIN
        DELETE dbo25.formats WHERE company_id = @company_id AND TABLE_SCHEMA = @schema AND TABLE_NAME = @code
        END
    ELSE
        BEGIN
        UPDATE dbo25.formats SET TABLE_EXCEL_FORMAT_XML = @ExcelFormatXML WHERE company_id = @company_id AND TABLE_SCHEMA = @schema AND TABLE_NAME = @code

        IF @@ROWCOUNT = 0
            INSERT dbo25.formats (company_id, TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (@company_id, @schema, @code, @ExcelFormatXML)
        END
    END
ELSE
    BEGIN

    SET NOCOUNT OFF

    IF @ExcelFormatXML IS NULL
        BEGIN
        DELETE xls.formats WHERE TABLE_SCHEMA = @schema AND TABLE_NAME = @name
        END
    ELSE
        BEGIN
        UPDATE xls.formats SET TABLE_EXCEL_FORMAT_XML = @ExcelFormatXML WHERE TABLE_SCHEMA = @schema AND TABLE_NAME = @name

        IF @@ROWCOUNT = 0
            INSERT xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (@schema, @name, @ExcelFormatXML)
        END
    END

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: The procedure updates Excel table formats
-- =============================================

CREATE PROCEDURE [xls25b].[usp_update_table_format]
    @schema nvarchar(128) = NULL
    , @name nvarchar(128) = NULL
    , @ExcelFormatXML xml = NULL
AS
BEGIN

SET NOCOUNT ON

EXEC xls25a.usp_update_table_format @schema, @name, @ExcelFormatXML

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.axis_types
-- =============================================

CREATE TRIGGER [dbo25].[tr_axis_types_insert]
   ON  [dbo25].[axis_types]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.axis_types
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.axis_types t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.axis_types
-- =============================================

CREATE TRIGGER [dbo25].[tr_axis_types_update]
   ON  [dbo25].[axis_types]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.axis_types
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.axis_types t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.calc_types
-- =============================================

CREATE TRIGGER [dbo25].[tr_calc_types_insert]
   ON  [dbo25].[calc_types]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.calc_types
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.calc_types t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.calc_types
-- =============================================

CREATE TRIGGER [dbo25].[tr_calc_types_update]
   ON  [dbo25].[calc_types]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.calc_types
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.calc_types t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.companies
-- =============================================

CREATE TRIGGER [dbo25].[tr_companies_insert]
   ON  [dbo25].[companies]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.companies
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.companies t INNER JOIN inserted ON t.id = inserted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on update of dbo25.companies
-- =============================================

CREATE TRIGGER [dbo25].[tr_companies_update]
   ON  [dbo25].[companies]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.companies
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.companies t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.currency_rates
-- =============================================

CREATE TRIGGER [dbo25].[tr_currency_rates_insert]
   ON  [dbo25].[currency_rates]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.currency_rates
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.currency_rates t INNER JOIN inserted ON t.base_currency_id = inserted.base_currency_id AND t.quote_currency_id = inserted.base_currency_id AND t.time_id = inserted.time_id AND t.category_id = inserted.category_id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.currency_rates
-- =============================================

CREATE TRIGGER [dbo25].[tr_currency_rates_update]
   ON  [dbo25].[currency_rates]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.currency_rates
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.currency_rates t INNER JOIN deleted ON t.base_currency_id = deleted.base_currency_id AND t.quote_currency_id = deleted.base_currency_id AND t.time_id = deleted.time_id AND t.category_id = deleted.category_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.dimension_calc_types
-- =============================================

CREATE TRIGGER [dbo25].[tr_dimension_calc_types_insert]
   ON  [dbo25].[dimension_calc_types]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.dimension_calc_types
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.dimension_calc_types t INNER JOIN inserted ON t.dimension_id = inserted.dimension_id AND t.calc_type_id = inserted.calc_type_id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.dimension_calc_types
-- =============================================

CREATE TRIGGER [dbo25].[tr_dimension_calc_types_update]
   ON  [dbo25].[dimension_calc_types]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.dimension_calc_types
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.dimension_calc_types t INNER JOIN deleted ON t.dimension_id = deleted.dimension_id AND t.calc_type_id = deleted.calc_type_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on delete from dbo25.dimension_properties
-- =============================================

CREATE TRIGGER [dbo25].[tr_dimension_properties_delete]
   ON  [dbo25].[dimension_properties]
   AFTER DELETE
AS
BEGIN

SET NOCOUNT ON;

DELETE dbo25.translations FROM deleted INNER JOIN dbo25.translations t ON t.company_id = deleted.company_id AND t.table_id IN (5, 6) AND t.member_id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.dimension_properties
-- =============================================

CREATE TRIGGER [dbo25].[tr_dimension_properties_insert]
   ON  [dbo25].[dimension_properties]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.dimension_properties
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.dimension_properties t INNER JOIN inserted ON t.company_id = inserted.company_id AND t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.dimension_properties
-- =============================================

CREATE TRIGGER [dbo25].[tr_dimension_properties_update]
   ON  [dbo25].[dimension_properties]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.dimension_properties
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.dimension_properties t INNER JOIN deleted ON t.company_id = deleted.company_id AND t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.dimensions
-- =============================================

CREATE TRIGGER [dbo25].[tr_dimensions_insert]
   ON  [dbo25].[dimensions]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.dimensions
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.dimensions t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.dimensions
-- =============================================

CREATE TRIGGER [dbo25].[tr_dimensions_update]
   ON  [dbo25].[dimensions]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.dimensions
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.dimensions t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.factors
-- =============================================

CREATE TRIGGER [dbo25].[tr_factors_insert]
   ON  [dbo25].[factors]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.factors
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.factors t INNER JOIN inserted ON t.member_id = inserted.member_id AND t.parent_id = inserted.parent_id AND t.calc_type_id = inserted.calc_type_id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.factors
-- =============================================

CREATE TRIGGER [dbo25].[tr_factors_update]
   ON  [dbo25].[factors]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.factors
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.factors t INNER JOIN deleted ON t.member_id = deleted.member_id AND t.parent_id = deleted.parent_id AND t.calc_type_id = deleted.calc_type_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.form_dimensions
-- =============================================

CREATE TRIGGER [dbo25].[tr_form_dimensions_insert]
   ON  [dbo25].[form_dimensions]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.form_dimensions
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.form_dimensions t INNER JOIN inserted ON t.dimension_id = inserted.dimension_id AND t.form_id = inserted.form_id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.form_dimensions
-- =============================================

CREATE TRIGGER [dbo25].[tr_form_dimensions_update]
   ON  [dbo25].[form_dimensions]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.form_dimensions
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.form_dimensions t INNER JOIN deleted ON t.dimension_id = deleted.dimension_id AND t.form_id = deleted.form_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.form_permissions
-- =============================================

CREATE TRIGGER [dbo25].[tr_form_permissions_insert]
   ON  [dbo25].[form_permissions]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.form_permissions
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.form_permissions t INNER JOIN inserted ON t.form_id = inserted.form_id AND t.username = inserted.username

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.form_permissions
-- =============================================

CREATE TRIGGER [dbo25].[tr_form_permissions_update]
   ON  [dbo25].[form_permissions]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.form_permissions
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.form_permissions t INNER JOIN deleted ON t.form_id = deleted.form_id AND t.username = deleted.username

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on delete from dbo25.form_rows
-- =============================================

CREATE TRIGGER [dbo25].[tr_form_rows_delete]
   ON  [dbo25].[form_rows]
   AFTER DELETE
AS
BEGIN

SET NOCOUNT ON;

DELETE dbo25.translations FROM deleted INNER JOIN dbo25.members m ON m.id = deleted.rowset_id INNER JOIN dbo25.translations t ON t.company_id = m.company_id AND t.table_id = 8 AND t.member_id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.form_rows
-- =============================================

CREATE TRIGGER [dbo25].[tr_form_rows_insert]
   ON  [dbo25].[form_rows]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.form_rows
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.form_rows t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on update of dbo25.form_rows
-- =============================================

CREATE TRIGGER [dbo25].[tr_form_rows_update]
   ON  [dbo25].[form_rows]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.form_rows
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.form_rows t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.formats
-- =============================================

CREATE TRIGGER [dbo25].[tr_formats_insert]
   ON  [dbo25].[formats]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.formats
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.formats t INNER JOIN inserted ON t.ID = inserted.ID

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.formats
-- =============================================

CREATE TRIGGER [dbo25].[tr_formats_update]
   ON  [dbo25].[formats]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.formats
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.formats t INNER JOIN deleted ON t.ID = deleted.ID

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on delete from dbo25.forms
-- =============================================

CREATE TRIGGER [dbo25].[tr_forms_delete]
   ON  [dbo25].[forms]
   AFTER DELETE
AS
BEGIN

SET NOCOUNT ON;

DELETE dbo25.translations FROM deleted INNER JOIN dbo25.translations t ON t.company_id = deleted.company_id AND t.table_id = 7 AND t.member_id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.forms
-- =============================================

CREATE TRIGGER [dbo25].[tr_forms_insert]
   ON  [dbo25].[forms]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.forms
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.forms t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on update of dbo25.forms
-- =============================================

CREATE TRIGGER [dbo25].[tr_forms_update]
   ON  [dbo25].[forms]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.forms
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.forms t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.hierarchies
-- =============================================

CREATE TRIGGER [dbo25].[tr_hierarchies_insert]
   ON  [dbo25].[hierarchies]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.hierarchies
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.hierarchies t INNER JOIN inserted ON t.member_id = inserted.member_id AND t.parent_id = inserted.parent_id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.hierarchies
-- =============================================

CREATE TRIGGER [dbo25].[tr_hierarchies_update]
   ON  [dbo25].[hierarchies]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.hierarchies
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.hierarchies t INNER JOIN deleted ON t.member_id = deleted.member_id AND t.parent_id = deleted.parent_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.member_permissions
-- =============================================

CREATE TRIGGER [dbo25].[tr_member_permissions_insert]
   ON  [dbo25].[member_permissions]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.member_permissions
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.member_permissions t INNER JOIN inserted ON t.member_id = inserted.member_id AND t.username = inserted.username

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.member_permissions
-- =============================================

CREATE TRIGGER [dbo25].[tr_member_permissions_update]
   ON  [dbo25].[member_permissions]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.member_permissions
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.member_permissions t INNER JOIN deleted ON t.member_id = deleted.member_id AND t.username = deleted.username

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.member_relations
-- =============================================

CREATE TRIGGER [dbo25].[tr_member_relations_insert]
   ON  [dbo25].[member_relations]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.member_relations
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.member_relations t INNER JOIN inserted ON t.member_id = inserted.member_id AND t.partner_id = inserted.partner_id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.member_relations
-- =============================================

CREATE TRIGGER [dbo25].[tr_member_relations_update]
   ON  [dbo25].[member_relations]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.member_relations
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.member_relations t INNER JOIN deleted ON t.member_id = deleted.member_id AND t.partner_id = deleted.partner_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on delete from dbo25.members
-- =============================================

CREATE TRIGGER [dbo25].[tr_members_delete]
   ON  [dbo25].[members]
   AFTER DELETE
AS
BEGIN

SET NOCOUNT ON;

DELETE dbo25.translations FROM deleted INNER JOIN dbo25.translations t ON t.company_id = deleted.company_id AND t.table_id = 1 AND t.member_id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.members
-- =============================================

CREATE TRIGGER [dbo25].[tr_members_insert]
   ON  [dbo25].[members]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.members
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.members t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on update of dbo25.members
-- =============================================

CREATE TRIGGER [dbo25].[tr_members_update]
   ON  [dbo25].[members]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.members
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.members t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.name_formats
-- =============================================

CREATE TRIGGER [dbo25].[tr_name_formats_insert]
   ON  [dbo25].[name_formats]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.name_formats
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.name_formats t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.name_formats
-- =============================================

CREATE TRIGGER [dbo25].[tr_name_formats_update]
   ON  [dbo25].[name_formats]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.name_formats
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.name_formats t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.parents
-- =============================================

CREATE TRIGGER [dbo25].[tr_parents_insert]
   ON  [dbo25].[parents]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.parents
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.parents t INNER JOIN inserted ON t.member_id = inserted.member_id AND t.parent_id = inserted.parent_id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.parents
-- =============================================

CREATE TRIGGER [dbo25].[tr_parents_update]
   ON  [dbo25].[parents]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.parents
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.parents t INNER JOIN deleted ON t.member_id = deleted.member_id AND t.parent_id = deleted.parent_id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on delete from dbo25.tax_rates
-- =============================================

CREATE TRIGGER [dbo25].[tr_tax_rates_delete]
   ON  [dbo25].[tax_rates]
   AFTER DELETE
AS
BEGIN

SET NOCOUNT ON;

DELETE dbo25.translations FROM deleted INNER JOIN dbo25.translations t ON t.company_id = deleted.company_id AND t.table_id = 4 AND t.member_id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.tax_rates
-- =============================================

CREATE TRIGGER [dbo25].[tr_tax_rates_insert]
   ON  [dbo25].[tax_rates]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.tax_rates
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.tax_rates t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on update of dbo25.tax_rates
-- =============================================

CREATE TRIGGER [dbo25].[tr_tax_rates_update]
   ON  [dbo25].[tax_rates]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.tax_rates
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.tax_rates t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.translated_tables
-- =============================================

CREATE TRIGGER [dbo25].[tr_translated_tables_insert]
   ON  [dbo25].[translated_tables]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.translated_tables
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.translated_tables t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.translated_tables
-- =============================================

CREATE TRIGGER [dbo25].[tr_translated_tables_update]
   ON  [dbo25].[translated_tables]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.translated_tables
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.translated_tables t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.translations
-- =============================================

CREATE TRIGGER [dbo25].[tr_translations_insert]
   ON  [dbo25].[translations]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.translations
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.translations t INNER JOIN inserted ON t.member_id = inserted.member_id AND t.language = inserted.language

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.translations
-- =============================================

CREATE TRIGGER [dbo25].[tr_translations_update]
   ON  [dbo25].[translations]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.translations
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.translations t INNER JOIN deleted ON t.member_id = deleted.member_id AND t.language = deleted.language

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on delete from dbo25.units
-- =============================================

CREATE TRIGGER [dbo25].[tr_units_delete]
   ON  [dbo25].[units]
   AFTER DELETE
AS
BEGIN

SET NOCOUNT ON;

DELETE dbo25.translations FROM deleted INNER JOIN dbo25.translations t ON t.company_id = deleted.company_id AND t.table_id = 3 AND t.member_id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.units
-- =============================================

CREATE TRIGGER [dbo25].[tr_units_insert]
   ON  [dbo25].[units]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.units
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.units t INNER JOIN inserted ON t.id = inserted.id

END

SET ANSI_NULLS ON


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.1, 2022-10-14
-- Description: This trigger fires on update of dbo25.units
-- =============================================

CREATE TRIGGER [dbo25].[tr_units_update]
   ON  [dbo25].[units]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.units
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.units t INNER JOIN deleted ON t.id = deleted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on insert into dbo25.users
-- =============================================

CREATE TRIGGER [dbo25].[tr_users_insert]
   ON  [dbo25].[users]
   AFTER INSERT
AS
BEGIN

SET NOCOUNT ON;

UPDATE dbo25.users
SET
    created_by = USER_NAME()
    , created_on = CAST(GETDATE() AS datetime2(0))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.users t INNER JOIN inserted ON t.id = inserted.id

END


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     5.0, 2022-07-05
-- Description: This trigger fires on update of dbo25.users
-- =============================================

CREATE TRIGGER [dbo25].[tr_users_update]
   ON  [dbo25].[users]
   AFTER UPDATE
AS
BEGIN

SET NOCOUNT ON;

IF TRIGGER_NESTLEVEL() > 1
    RETURN

UPDATE dbo25.users
SET
    created_by = COALESCE(deleted.created_by, t.created_by, USER_NAME())
    , created_on = COALESCE(deleted.created_on, t.created_on, CAST(GETDATE() AS datetime2(0)))
    , modified_by = USER_NAME()
    , modified_on = CAST(GETDATE() AS datetime2(0))
FROM
    dbo25.users t INNER JOIN deleted ON t.id = deleted.id

END


GO

INSERT INTO [dbo25].[axis_types] ([id], [code], [name]) VALUES (1, N'page', N'Page');
INSERT INTO [dbo25].[axis_types] ([id], [code], [name]) VALUES (2, N'rows', N'Rows');
INSERT INTO [dbo25].[axis_types] ([id], [code], [name]) VALUES (3, N'columns', N'Columns');
GO

INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (0, N'none', N'None', 0);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (1, N'total', N'Total', 1);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (2, N'average', N'Average', 2);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (3, N'open', N'Opening Balance', 3);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (4, N'close', N'Closing Balance', 4);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (5, N'ytd', N'YTD', 5);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (6, N'same_period', N'Same Period', 6);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (7, N'excel', N'Excel Formula', 7);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (8, N'string', N'String', 8);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (9, N'rowset', N'Rowset', 9);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (10, N'rate', N'Rate', 10);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (11, N'revaluation', N'Revaluation', 11);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (12, N'difference', N'Difference', 12);
INSERT INTO [dbo25].[calc_types] ([id], [code], [name], [sort_order]) VALUES (13, N'percent', N'Percent', 13);
GO

INSERT INTO [dbo25].[name_formats] ([id], [code], [name]) VALUES (0, N'code', N'code');
INSERT INTO [dbo25].[name_formats] ([id], [code], [name]) VALUES (1, N'name', N'name');
INSERT INTO [dbo25].[name_formats] ([id], [code], [name]) VALUES (2, N'id', N'id');
INSERT INTO [dbo25].[name_formats] ([id], [code], [name]) VALUES (3, N'id_name', N'id - name');
INSERT INTO [dbo25].[name_formats] ([id], [code], [name]) VALUES (4, N'code_name', N'code - name');
GO

INSERT INTO [dbo25].[translated_tables] ([id], [code], [name]) VALUES (1, N'members', N'members');
INSERT INTO [dbo25].[translated_tables] ([id], [code], [name]) VALUES (2, N'companies', N'companies');
INSERT INTO [dbo25].[translated_tables] ([id], [code], [name]) VALUES (3, N'units', N'units');
INSERT INTO [dbo25].[translated_tables] ([id], [code], [name]) VALUES (4, N'tax_rates', N'tax_rates');
INSERT INTO [dbo25].[translated_tables] ([id], [code], [name]) VALUES (5, N'dimensions', N'dimensions');
INSERT INTO [dbo25].[translated_tables] ([id], [code], [name]) VALUES (6, N'dimension_parameters', N'dimension_parameters');
INSERT INTO [dbo25].[translated_tables] ([id], [code], [name]) VALUES (7, N'forms', N'forms');
INSERT INTO [dbo25].[translated_tables] ([id], [code], [name]) VALUES (8, N'form_rows', N'form_rows');
GO

INSERT INTO [dbo25].[dimensions] ([id], [code], [name], [parameter_name], [sort_order], [join_order], [name_format_id], [is_protected], [default_select_permission], [default_update_permission], [is_active]) VALUES (1, N'accounts', N'Accounts', N'Account', 1, 5, 4, 1, 0, 0, 1);
INSERT INTO [dbo25].[dimensions] ([id], [code], [name], [parameter_name], [sort_order], [join_order], [name_format_id], [is_protected], [default_select_permission], [default_update_permission], [is_active]) VALUES (2, N'times', N'Times', N'Period', 2, 6, 1, 1, 1, 1, 1);
INSERT INTO [dbo25].[dimensions] ([id], [code], [name], [parameter_name], [sort_order], [join_order], [name_format_id], [is_protected], [default_select_permission], [default_update_permission], [is_active]) VALUES (3, N'categories', N'Categories', N'Category', 3, 3, 1, 1, 1, 1, 1);
INSERT INTO [dbo25].[dimensions] ([id], [code], [name], [parameter_name], [sort_order], [join_order], [name_format_id], [is_protected], [default_select_permission], [default_update_permission], [is_active]) VALUES (4, N'entities', N'Entities', N'Entity', 4, 4, 1, 1, 0, 0, 1);
INSERT INTO [dbo25].[dimensions] ([id], [code], [name], [parameter_name], [sort_order], [join_order], [name_format_id], [is_protected], [default_select_permission], [default_update_permission], [is_active]) VALUES (5, N'dim5', N'Regions', N'Region', 5, 1, 1, 0, 1, 1, 0);
INSERT INTO [dbo25].[dimensions] ([id], [code], [name], [parameter_name], [sort_order], [join_order], [name_format_id], [is_protected], [default_select_permission], [default_update_permission], [is_active]) VALUES (6, N'dim6', N'Products', N'Product', 6, 2, 1, 0, 1, 1, 0);
INSERT INTO [dbo25].[dimensions] ([id], [code], [name], [parameter_name], [sort_order], [join_order], [name_format_id], [is_protected], [default_select_permission], [default_update_permission], [is_active]) VALUES (7, N'dim7', N'Sub_Accounts', N'Sub.Accounts', 7, 7, 1, 0, 1, 1, 0);
GO

INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (1, 0, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (1, 1, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (1, 2, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (1, 3, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (1, 4, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (1, 8, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (1, 9, 0);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (1, 10, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (1, 11, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (2, 0, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (2, 1, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (2, 5, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (2, 9, 0);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (3, 0, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (3, 1, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (3, 6, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (3, 7, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (3, 9, 0);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (3, 12, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (3, 13, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (4, 0, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (4, 1, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (4, 9, 0);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (5, 0, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (5, 1, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (5, 9, 0);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (6, 0, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (6, 1, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (6, 9, 0);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (7, 0, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (7, 1, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (7, 8, 1);
INSERT INTO [dbo25].[dimension_calc_types] ([dimension_id], [calc_type_id], [is_active]) VALUES (7, 9, 0);
GO

INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_axis_types', N'<table name="xls25.usp_axis_types"><columnFormats><column name="" property="ListObjectName" value="axis_types" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="ColumnWidth" value="4.43" type="Double" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$D$4" type="String" /><column name="code" property="ColumnWidth" value="13.57" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$E$4" type="String" /><column name="name" property="ColumnWidth" value="13.57" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="" property="Tab.Color" value="0" type="Double" /><column name="" property="Tab.ThemeColor" value="2" type="Double" /><column name="" property="Tab.TintAndShade" value="0" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="9" type="Double" /></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_calc_types', N'<table name="xls25.usp_calc_types"><columnFormats><column name="" property="ListObjectName" value="calc_types" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="ColumnWidth" value="4.43" type="Double" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$D$4" type="String" /><column name="code" property="ColumnWidth" value="11.86" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$E$4" type="String" /><column name="name" property="ColumnWidth" value="15.29" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="Address" value="$F$4" type="String" /><column name="sort_order" property="ColumnWidth" value="11.86" type="Double" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="accounts" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="accounts" property="Address" value="$G$4" type="String" /><column name="accounts" property="ColumnWidth" value="13.57" type="Double" /><column name="accounts" property="NumberFormat" value="General" type="String" /><column name="accounts" property="HorizontalAlignment" value="-4108" type="Double" /><column name="accounts" property="Font.Size" value="10" type="Double" /><column name="times" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="times" property="Address" value="$H$4" type="String" /><column name="times" property="ColumnWidth" value="13.57" type="Double" /><column name="times" property="NumberFormat" value="General" type="String" /><column name="times" property="HorizontalAlignment" value="-4108" type="Double" /><column name="times" property="Font.Size" value="10" type="Double" /><column name="categories" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="categories" property="Address" value="$I$4" type="String" /><column name="categories" property="ColumnWidth" value="13.57" type="Double" /><column name="categories" property="NumberFormat" value="General" type="String" /><column name="categories" property="HorizontalAlignment" value="-4108" type="Double" /><column name="categories" property="Font.Size" value="10" type="Double" /><column name="entities" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="entities" property="Address" value="$J$4" type="String" /><column name="entities" property="ColumnWidth" value="13.57" type="Double" /><column name="entities" property="NumberFormat" value="General" type="String" /><column name="entities" property="HorizontalAlignment" value="-4108" type="Double" /><column name="entities" property="Font.Size" value="10" type="Double" /><column name="dim5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dim5" property="Address" value="$K$4" type="String" /><column name="dim5" property="ColumnWidth" value="13.57" type="Double" /><column name="dim5" property="NumberFormat" value="General" type="String" /><column name="dim5" property="HorizontalAlignment" value="-4108" type="Double" /><column name="dim5" property="Font.Size" value="10" type="Double" /><column name="dim6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dim6" property="Address" value="$L$4" type="String" /><column name="dim6" property="ColumnWidth" value="13.57" type="Double" /><column name="dim6" property="NumberFormat" value="General" type="String" /><column name="dim6" property="HorizontalAlignment" value="-4108" type="Double" /><column name="dim6" property="Font.Size" value="10" type="Double" /><column name="dim7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dim7" property="Address" value="$M$4" type="String" /><column name="dim7" property="ColumnWidth" value="13.57" type="Double" /><column name="dim7" property="NumberFormat" value="General" type="String" /><column name="dim7" property="HorizontalAlignment" value="-4108" type="Double" /><column name="dim7" property="Font.Size" value="10" type="Double" /><column name="accounts" property="FormatConditions(1).AppliesTo.Address" value="$G$4:$G$17" type="String" /><column name="accounts" property="FormatConditions(1).Type" value="6" type="Double" /><column name="accounts" property="FormatConditions(1).Priority" value="7" type="Double" /><column name="accounts" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="accounts" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="accounts" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="accounts" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="accounts" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="accounts" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="accounts" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="accounts" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="accounts" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="accounts" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="times" property="FormatConditions(1).AppliesTo.Address" value="$H$4:$H$17" type="String" /><column name="times" property="FormatConditions(1).Type" value="6" type="Double" /><column name="times" property="FormatConditions(1).Priority" value="6" type="Double" /><column name="times" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="times" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="times" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="times" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="times" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="times" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="times" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="times" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="times" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="times" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="categories" property="FormatConditions(1).AppliesTo.Address" value="$I$4:$I$17" type="String" /><column name="categories" property="FormatConditions(1).Type" value="6" type="Double" /><column name="categories" property="FormatConditions(1).Priority" value="5" type="Double" /><column name="categories" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="categories" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="categories" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="categories" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="categories" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="categories" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="categories" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="categories" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="categories" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="categories" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="entities" property="FormatConditions(1).AppliesTo.Address" value="$J$4:$J$17" type="String" /><column name="entities" property="FormatConditions(1).Type" value="6" type="Double" /><column name="entities" property="FormatConditions(1).Priority" value="4" type="Double" /><column name="entities" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="entities" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="entities" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="entities" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="entities" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="entities" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="entities" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="entities" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="entities" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="entities" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="dim5" property="FormatConditions(1).AppliesTo.Address" value="$K$4:$K$17" type="String" /><column name="dim5" property="FormatConditions(1).Type" value="6" type="Double" /><column name="dim5" property="FormatConditions(1).Priority" value="3" type="Double" /><column name="dim5" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="dim5" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="dim5" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="dim5" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="dim5" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="dim5" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="dim5" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="dim5" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="dim5" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="dim5" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="dim6" property="FormatConditions(1).AppliesTo.Address" value="$L$4:$L$17" type="String" /><column name="dim6" property="FormatConditions(1).Type" value="6" type="Double" /><column name="dim6" property="FormatConditions(1).Priority" value="2" type="Double" /><column name="dim6" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="dim6" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="dim6" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="dim6" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="dim6" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="dim6" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="dim6" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="dim6" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="dim6" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="dim6" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="dim7" property="FormatConditions(1).AppliesTo.Address" value="$M$4:$M$17" type="String" /><column name="dim7" property="FormatConditions(1).Type" value="6" type="Double" /><column name="dim7" property="FormatConditions(1).Priority" value="1" type="Double" /><column name="dim7" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="dim7" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="dim7" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="dim7" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="dim7" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="dim7" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="dim7" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="dim7" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="dim7" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="dim7" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="0" type="Double" /><column name="" property="Tab.ThemeColor" value="2" type="Double" /><column name="" property="Tab.TintAndShade" value="0" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="9" type="Double" /></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_closed_periods', N'<table name="xls25.usp_closed_periods"><columnFormats><column name="" property="ListObjectName" value="closes_periods" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$D$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$E$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="Address" value="$F$4" type="String" /><column name="decimal_places" property="NumberFormat" value="General" type="String" /><column name="decimal_places" property="HorizontalAlignment" value="-4108" type="Double" /><column name="decimal_places" property="Font.Size" value="10" type="Double" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="Address" value="$G$4" type="String" /><column name="is_percent" property="NumberFormat" value="General" type="String" /><column name="is_percent" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_percent" property="Font.Size" value="10" type="Double" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="Address" value="$H$4" type="String" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_color" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="Address" value="$I$4" type="String" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="Address" value="$J$4" type="String" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="row_indent" property="HorizontalAlignment" value="-4108" type="Double" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$K$4" type="String" /><column name="code" property="ColumnWidth" value="6.86" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name" property="Address" value="$L$4" type="String" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="Address" value="$M$4" type="String" /><column name="calc_type" property="NumberFormat" value="General" type="String" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="Address" value="$N$4" type="String" /><column name="data_format" property="NumberFormat" value="General" type="String" /><column name="data_format" property="HorizontalAlignment" value="-4108" type="Double" /><column name="data_format" property="Font.Size" value="10" type="Double" /><column name="BUDGET" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="BUDGET" property="Address" value="$O$4" type="String" /><column name="BUDGET" property="ColumnWidth" value="13.43" type="Double" /><column name="BUDGET" property="NumberFormat" value="General" type="String" /><column name="BUDGET" property="HorizontalAlignment" value="-4108" type="Double" /><column name="BUDGET" property="Font.Size" value="10" type="Double" /><column name="FORECAST" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FORECAST" property="Address" value="$P$4" type="String" /><column name="FORECAST" property="ColumnWidth" value="13.43" type="Double" /><column name="FORECAST" property="NumberFormat" value="General" type="String" /><column name="FORECAST" property="HorizontalAlignment" value="-4108" type="Double" /><column name="FORECAST" property="Font.Size" value="10" type="Double" /><column name="ACTUAL" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ACTUAL" property="Address" value="$Q$4" type="String" /><column name="ACTUAL" property="ColumnWidth" value="13.43" type="Double" /><column name="ACTUAL" property="NumberFormat" value="General" type="String" /><column name="ACTUAL" property="HorizontalAlignment" value="-4108" type="Double" /><column name="ACTUAL" property="Font.Size" value="10" type="Double" /><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$Q$16" type="String" /><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(1).Priority" value="5" type="Double" /><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$I4=1" type="String" /><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$Q$16" type="String" /><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(2).Priority" value="6" type="Double" /><column name="_RowNum" property="FormatConditions(2).Formula1" value="=$H4=12" type="String" /><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="15853276" type="Double" /><column name="_RowNum" property="FormatConditions(2).Interior.ThemeColor" value="5" type="Double" /><column name="_RowNum" property="FormatConditions(2).Interior.TintAndShade" value="0.799920651875362" type="Double" /><column name="_RowNum" property="FormatConditions(3).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).AppliesTo.Address" value="$B$4:$Q$16" type="String" /><column name="_RowNum" property="FormatConditions(3).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(3).Priority" value="7" type="Double" /><column name="_RowNum" property="FormatConditions(3).Formula1" value="=$H4=11" type="String" /><column name="_RowNum" property="FormatConditions(3).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).Font.Color" value="16777215" type="Double" /><column name="_RowNum" property="FormatConditions(3).Font.ThemeColor" value="1" type="Double" /><column name="_RowNum" property="FormatConditions(3).Font.TintAndShade" value="0" type="Double" /><column name="_RowNum" property="FormatConditions(3).Interior.Color" value="6299648" type="Double" /><column name="data_format" property="FormatConditions(1).ColumnsCount" value="4" type="Double" /><column name="data_format" property="FormatConditions(1).AppliesTo.Address" value="$N$4:$Q$16" type="String" /><column name="data_format" property="FormatConditions(1).Type" value="6" type="Double" /><column name="data_format" property="FormatConditions(1).Priority" value="102162" type="Double" /><column name="data_format" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="data_format" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="10498160" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="Sheet1_Table158" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="number_format" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="PLAN" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FACT" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="Sheet1_Table158" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="number_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="PLAN" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FACT" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_companies', N'<table name="xls25.usp_companies"><columnFormats><column name="" property="ListObjectName" value="companies" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="ColumnWidth" value="4.29" type="Double" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$D$4" type="String" /><column name="code" property="ColumnWidth" value="10.71" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$E$4" type="String" /><column name="name" property="ColumnWidth" value="27.86" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="translated_name" property="Address" value="$F$4" type="String" /><column name="translated_name" property="ColumnWidth" value="19.43" type="Double" /><column name="translated_name" property="NumberFormat" value="General" type="String" /><column name="default_language" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_language" property="Address" value="$G$4" type="String" /><column name="default_language" property="ColumnWidth" value="18.14" type="Double" /><column name="default_language" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="Address" value="$H$4" type="String" /><column name="sort_order" property="ColumnWidth" value="11.71" type="Double" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="" property="Tab.Color" value="6299649" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_currency_rates', N'<table name="xls25.usp_currency_rates"><columnFormats><column name="" property="ListObjectName" value="currency_rates" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$D$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$E$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="Address" value="$F$4" type="String" /><column name="decimal_places" property="NumberFormat" value="General" type="String" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="Address" value="$G$4" type="String" /><column name="is_percent" property="NumberFormat" value="General" type="String" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="Address" value="$H$4" type="String" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="Address" value="$I$4" type="String" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="Address" value="$J$4" type="String" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$K$4" type="String" /><column name="code" property="ColumnWidth" value="8.43" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name" property="Address" value="$L$4" type="String" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="Address" value="$M$4" type="String" /><column name="calc_type" property="NumberFormat" value="General" type="String" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="Address" value="$N$4" type="String" /><column name="data_format" property="NumberFormat" value="General" type="String" /><column name="BUDGET" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="BUDGET" property="Address" value="$O$4" type="String" /><column name="BUDGET" property="ColumnWidth" value="13.43" type="Double" /><column name="BUDGET" property="NumberFormat" value="General" type="String" /><column name="FORECAST" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FORECAST" property="Address" value="$P$4" type="String" /><column name="FORECAST" property="ColumnWidth" value="13.43" type="Double" /><column name="FORECAST" property="NumberFormat" value="General" type="String" /><column name="ACTUAL" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ACTUAL" property="Address" value="$Q$4" type="String" /><column name="ACTUAL" property="ColumnWidth" value="13.43" type="Double" /><column name="ACTUAL" property="NumberFormat" value="General" type="String" /><column name="" property="Tab.Color" value="10498160" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="rates" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="BUDGET" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FORECAST" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ACTUAL" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="rates" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="BUDGET" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FORECAST" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ACTUAL" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_data_management', N'<table name="xls25.usp_data_management"><columnFormats><column name="" property="ListObjectName" value="data_management" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$D$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$E$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="Address" value="$F$4" type="String" /><column name="decimal_places" property="NumberFormat" value="General" type="String" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="Address" value="$G$4" type="String" /><column name="is_percent" property="NumberFormat" value="General" type="String" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="Address" value="$H$4" type="String" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="Address" value="$I$4" type="String" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="Address" value="$J$4" type="String" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$K$4" type="String" /><column name="code" property="ColumnWidth" value="6.86" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name" property="Address" value="$L$4" type="String" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="Address" value="$M$4" type="String" /><column name="calc_type" property="NumberFormat" value="General" type="String" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="Address" value="$N$4" type="String" /><column name="data_format" property="NumberFormat" value="General" type="String" /><column name="BUDGET" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="BUDGET" property="Address" value="$O$4" type="String" /><column name="BUDGET" property="ColumnWidth" value="13.57" type="Double" /><column name="BUDGET" property="NumberFormat" value="General" type="String" /><column name="FORECAST" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FORECAST" property="Address" value="$P$4" type="String" /><column name="FORECAST" property="ColumnWidth" value="13.57" type="Double" /><column name="FORECAST" property="NumberFormat" value="General" type="String" /><column name="ACTUAL" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ACTUAL" property="Address" value="$Q$4" type="String" /><column name="ACTUAL" property="ColumnWidth" value="13.57" type="Double" /><column name="ACTUAL" property="NumberFormat" value="General" type="String" /><column name="" property="Tab.Color" value="10498160" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="data_management" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="BUDGET" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FORECAST" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ACTUAL" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="data_management" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="BUDGET" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FORECAST" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ACTUAL" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_dimensions', N'<table name="xls25.usp_dimensions"><columnFormats><column name="" property="ListObjectName" value="dimensions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="ColumnWidth" value="4.29" type="Double" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$D$4" type="String" /><column name="code" property="ColumnWidth" value="9.86" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$E$4" type="String" /><column name="name" property="ColumnWidth" value="12.57" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="translated_name" property="Address" value="$F$4" type="String" /><column name="translated_name" property="ColumnWidth" value="17.71" type="Double" /><column name="translated_name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="HorizontalAlignment" value="-4108" type="Double" /><column name="translated_name" property="Font.Size" value="10" type="Double" /><column name="parameter_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_name" property="Address" value="$G$4" type="String" /><column name="parameter_name" property="ColumnWidth" value="18" type="Double" /><column name="parameter_name" property="NumberFormat" value="General" type="String" /><column name="translated_parameter_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="translated_parameter_name" property="Address" value="$H$4" type="String" /><column name="translated_parameter_name" property="ColumnWidth" value="27.86" type="Double" /><column name="translated_parameter_name" property="NumberFormat" value="General" type="String" /><column name="translated_parameter_name" property="HorizontalAlignment" value="-4108" type="Double" /><column name="translated_parameter_name" property="Font.Size" value="10" type="Double" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="Address" value="$I$4" type="String" /><column name="sort_order" property="ColumnWidth" value="11.86" type="Double" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="name_format_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name_format_id" property="Address" value="$J$4" type="String" /><column name="name_format_id" property="ColumnWidth" value="17.43" type="Double" /><column name="name_format_id" property="NumberFormat" value="General" type="String" /><column name="root_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="root_member_id" property="Address" value="$K$4" type="String" /><column name="root_member_id" property="ColumnWidth" value="17.71" type="Double" /><column name="root_member_id" property="NumberFormat" value="General" type="String" /><column name="default_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_member_id" property="Address" value="$L$4" type="String" /><column name="default_member_id" property="ColumnWidth" value="17.43" type="Double" /><column name="default_member_id" property="NumberFormat" value="General" type="String" /><column name="is_protected" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_protected" property="Address" value="$M$4" type="String" /><column name="is_protected" property="ColumnWidth" value="13.71" type="Double" /><column name="is_protected" property="NumberFormat" value="General" type="String" /><column name="is_protected" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_protected" property="Font.Size" value="10" type="Double" /><column name="default_select_permission" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_select_permission" property="Address" value="$N$4" type="String" /><column name="default_select_permission" property="ColumnWidth" value="26.57" type="Double" /><column name="default_select_permission" property="NumberFormat" value="General" type="String" /><column name="default_select_permission" property="HorizontalAlignment" value="-4108" type="Double" /><column name="default_select_permission" property="Font.Size" value="10" type="Double" /><column name="default_update_permission" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_update_permission" property="Address" value="$O$4" type="String" /><column name="default_update_permission" property="ColumnWidth" value="27.71" type="Double" /><column name="default_update_permission" property="NumberFormat" value="General" type="String" /><column name="default_update_permission" property="HorizontalAlignment" value="-4108" type="Double" /><column name="default_update_permission" property="Font.Size" value="10" type="Double" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="Address" value="$P$4" type="String" /><column name="is_active" property="ColumnWidth" value="8.14" type="Double" /><column name="is_active" property="NumberFormat" value="General" type="String" /><column name="is_active" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_active" property="Font.Size" value="10" type="Double" /><column name="is_protected" property="FormatConditions(1).AppliesTo.Address" value="$M$4:$M$10" type="String" /><column name="is_protected" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_protected" property="FormatConditions(1).Priority" value="4" type="Double" /><column name="is_protected" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_protected" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_protected" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_protected" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_protected" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_protected" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_protected" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_protected" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_protected" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_protected" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="default_select_permission" property="FormatConditions(1).AppliesTo.Address" value="$N$4:$N$10" type="String" /><column name="default_select_permission" property="FormatConditions(1).Type" value="6" type="Double" /><column name="default_select_permission" property="FormatConditions(1).Priority" value="1" type="Double" /><column name="default_select_permission" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="default_select_permission" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="default_select_permission" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="default_select_permission" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="default_select_permission" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="default_select_permission" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="default_select_permission" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="default_select_permission" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="default_select_permission" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="default_select_permission" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="default_update_permission" property="FormatConditions(1).AppliesTo.Address" value="$O$4:$O$10" type="String" /><column name="default_update_permission" property="FormatConditions(1).Type" value="6" type="Double" /><column name="default_update_permission" property="FormatConditions(1).Priority" value="2" type="Double" /><column name="default_update_permission" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="default_update_permission" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="default_update_permission" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="default_update_permission" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="default_update_permission" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="default_update_permission" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="default_update_permission" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="default_update_permission" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="default_update_permission" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="default_update_permission" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="is_active" property="FormatConditions(1).ColumnsCount" value="3" type="Double" /><column name="is_active" property="FormatConditions(1).AppliesTo.Address" value="$P$4:$R$10" type="String" /><column name="is_active" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_active" property="FormatConditions(1).Priority" value="3" type="Double" /><column name="is_active" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_active" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="1" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="dimensions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name_format_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="root_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_protected" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_select_permission" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_update_permission" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="dimensions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name_format_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="root_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_protected" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_select_permission" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="default_update_permission" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_facts', N'<table name="xls25.usp_facts"><columnFormats><column name="" property="ListObjectName" value="facts" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="company_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="company_id" property="Address" value="$C$4" type="String" /><column name="company_id" property="NumberFormat" value="General" type="String" /><column name="id1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id1" property="Address" value="$D$4" type="String" /><column name="id1" property="NumberFormat" value="General" type="String" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="Address" value="$E$4" type="String" /><column name="id2" property="NumberFormat" value="General" type="String" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="Address" value="$F$4" type="String" /><column name="id3" property="NumberFormat" value="General" type="String" /><column name="id4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id4" property="Address" value="$G$4" type="String" /><column name="id4" property="NumberFormat" value="General" type="String" /><column name="id5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id5" property="Address" value="$H$4" type="String" /><column name="id5" property="NumberFormat" value="General" type="String" /><column name="id6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id6" property="Address" value="$I$4" type="String" /><column name="id6" property="NumberFormat" value="General" type="String" /><column name="id7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id7" property="Address" value="$J$4" type="String" /><column name="id7" property="NumberFormat" value="General" type="String" /><column name="sort_order1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order1" property="Address" value="$K$4" type="String" /><column name="sort_order1" property="NumberFormat" value="General" type="String" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="Address" value="$L$4" type="String" /><column name="sort_order2" property="NumberFormat" value="General" type="String" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="Address" value="$M$4" type="String" /><column name="sort_order3" property="NumberFormat" value="General" type="String" /><column name="sort_order4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order4" property="Address" value="$N$4" type="String" /><column name="sort_order4" property="NumberFormat" value="General" type="String" /><column name="sort_order5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order5" property="Address" value="$O$4" type="String" /><column name="sort_order5" property="NumberFormat" value="General" type="String" /><column name="sort_order6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order6" property="Address" value="$P$4" type="String" /><column name="sort_order6" property="NumberFormat" value="General" type="String" /><column name="sort_order7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order7" property="Address" value="$Q$4" type="String" /><column name="sort_order7" property="NumberFormat" value="General" type="String" /><column name="unit_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_id" property="Address" value="$R$4" type="String" /><column name="unit_id" property="NumberFormat" value="General" type="String" /><column name="calc_type_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_id" property="Address" value="$S$4" type="String" /><column name="calc_type_id" property="NumberFormat" value="General" type="String" /><column name="company_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_code" property="Address" value="$T$4" type="String" /><column name="company_code" property="ColumnWidth" value="16" type="Double" /><column name="company_code" property="NumberFormat" value="General" type="String" /><column name="company_name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="company_name" property="Address" value="$U$4" type="String" /><column name="company_name" property="NumberFormat" value="General" type="String" /><column name="code1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code1" property="Address" value="$V$4" type="String" /><column name="code1" property="ColumnWidth" value="14.86" type="Double" /><column name="code1" property="NumberFormat" value="General" type="String" /><column name="name1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name1" property="Address" value="$W$4" type="String" /><column name="name1" property="ColumnWidth" value="44.14" type="Double" /><column name="name1" property="NumberFormat" value="General" type="String" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="Address" value="$X$4" type="String" /><column name="code2" property="NumberFormat" value="General" type="String" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name2" property="Address" value="$Y$4" type="String" /><column name="name2" property="ColumnWidth" value="14.14" type="Double" /><column name="name2" property="NumberFormat" value="General" type="String" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code3" property="Address" value="$Z$4" type="String" /><column name="code3" property="NumberFormat" value="General" type="String" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name3" property="Address" value="$AA$4" type="String" /><column name="name3" property="ColumnWidth" value="16.29" type="Double" /><column name="name3" property="NumberFormat" value="General" type="String" /><column name="code4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code4" property="Address" value="$AB$4" type="String" /><column name="code4" property="NumberFormat" value="General" type="String" /><column name="name4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name4" property="Address" value="$AC$4" type="String" /><column name="name4" property="ColumnWidth" value="13.43" type="Double" /><column name="name4" property="NumberFormat" value="General" type="String" /><column name="code5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code5" property="Address" value="$AD$4" type="String" /><column name="code5" property="NumberFormat" value="General" type="String" /><column name="name5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name5" property="Address" value="$AE$4" type="String" /><column name="name5" property="ColumnWidth" value="13.29" type="Double" /><column name="name5" property="NumberFormat" value="General" type="String" /><column name="code6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code6" property="Address" value="$AF$4" type="String" /><column name="code6" property="NumberFormat" value="General" type="String" /><column name="name6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name6" property="Address" value="$AG$4" type="String" /><column name="name6" property="ColumnWidth" value="13.29" type="Double" /><column name="name6" property="NumberFormat" value="General" type="String" /><column name="code7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code7" property="Address" value="$AH$4" type="String" /><column name="code7" property="NumberFormat" value="General" type="String" /><column name="name7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name7" property="Address" value="$AI$4" type="String" /><column name="name7" property="ColumnWidth" value="13.29" type="Double" /><column name="name7" property="NumberFormat" value="General" type="String" /><column name="unit_code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_code" property="Address" value="$AJ$4" type="String" /><column name="unit_code" property="NumberFormat" value="General" type="String" /><column name="unit_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_name" property="Address" value="$AK$4" type="String" /><column name="unit_name" property="ColumnWidth" value="12.14" type="Double" /><column name="unit_name" property="NumberFormat" value="General" type="String" /><column name="calc_type_code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_code" property="Address" value="$AL$4" type="String" /><column name="calc_type_code" property="NumberFormat" value="General" type="String" /><column name="calc_type_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_name" property="Address" value="$AM$4" type="String" /><column name="calc_type_name" property="ColumnWidth" value="15.57" type="Double" /><column name="calc_type_name" property="NumberFormat" value="General" type="String" /><column name="source_value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="source_value" property="Address" value="$AN$4" type="String" /><column name="source_value" property="ColumnWidth" value="14.29" type="Double" /><column name="source_value" property="NumberFormat" value="#,##0" type="String" /><column name="value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="value" property="Address" value="$AO$4" type="String" /><column name="value" property="ColumnWidth" value="11.86" type="Double" /><column name="value" property="NumberFormat" value="#,##0" type="String" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order1" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order3" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="0" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="sort_order2" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="0" type="Double" /><column name="SortFields(4)" property="KeyfieldName" value="sort_order4" type="String" /><column name="SortFields(4)" property="SortOn" value="0" type="Double" /><column name="SortFields(4)" property="Order" value="1" type="Double" /><column name="SortFields(4)" property="DataOption" value="0" type="Double" /><column name="SortFields(5)" property="KeyfieldName" value="sort_order5" type="String" /><column name="SortFields(5)" property="SortOn" value="0" type="Double" /><column name="SortFields(5)" property="Order" value="1" type="Double" /><column name="SortFields(5)" property="DataOption" value="0" type="Double" /><column name="SortFields(6)" property="KeyfieldName" value="sort_order6" type="String" /><column name="SortFields(6)" property="SortOn" value="0" type="Double" /><column name="SortFields(6)" property="Order" value="1" type="Double" /><column name="SortFields(6)" property="DataOption" value="0" type="Double" /><column name="SortFields(7)" property="KeyfieldName" value="sort_order7" type="String" /><column name="SortFields(7)" property="SortOn" value="0" type="Double" /><column name="SortFields(7)" property="Order" value="1" type="Double" /><column name="SortFields(7)" property="DataOption" value="0" type="Double" /><column name="" property="Tab.Color" value="10498161" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="facts" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="source_value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order1" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="2" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order3" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="2" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="sort_order2" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="2" type="Double" /><column name="SortFields(4)" property="KeyfieldName" value="sort_order4" type="String" /><column name="SortFields(4)" property="SortOn" value="0" type="Double" /><column name="SortFields(4)" property="Order" value="1" type="Double" /><column name="SortFields(4)" property="DataOption" value="2" type="Double" /><column name="SortFields(5)" property="KeyfieldName" value="sort_order5" type="String" /><column name="SortFields(5)" property="SortOn" value="0" type="Double" /><column name="SortFields(5)" property="Order" value="1" type="Double" /><column name="SortFields(5)" property="DataOption" value="2" type="Double" /><column name="SortFields(6)" property="KeyfieldName" value="sort_order6" type="String" /><column name="SortFields(6)" property="SortOn" value="0" type="Double" /><column name="SortFields(6)" property="Order" value="1" type="Double" /><column name="SortFields(6)" property="DataOption" value="2" type="Double" /><column name="SortFields(7)" property="KeyfieldName" value="sort_order7" type="String" /><column name="SortFields(7)" property="SortOn" value="0" type="Double" /><column name="SortFields(7)" property="Order" value="1" type="Double" /><column name="SortFields(7)" property="DataOption" value="2" type="Double" /></view><view name="Names"><column name="" property="ListObjectName" value="facts" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="company_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="company_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="source_value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order1" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="2" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order3" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="2" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="sort_order2" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="2" type="Double" /><column name="SortFields(4)" property="KeyfieldName" value="sort_order4" type="String" /><column name="SortFields(4)" property="SortOn" value="0" type="Double" /><column name="SortFields(4)" property="Order" value="1" type="Double" /><column name="SortFields(4)" property="DataOption" value="2" type="Double" /><column name="SortFields(5)" property="KeyfieldName" value="sort_order5" type="String" /><column name="SortFields(5)" property="SortOn" value="0" type="Double" /><column name="SortFields(5)" property="Order" value="1" type="Double" /><column name="SortFields(5)" property="DataOption" value="2" type="Double" /><column name="SortFields(6)" property="KeyfieldName" value="sort_order6" type="String" /><column name="SortFields(6)" property="SortOn" value="0" type="Double" /><column name="SortFields(6)" property="Order" value="1" type="Double" /><column name="SortFields(6)" property="DataOption" value="2" type="Double" /><column name="SortFields(7)" property="KeyfieldName" value="sort_order7" type="String" /><column name="SortFields(7)" property="SortOn" value="0" type="Double" /><column name="SortFields(7)" property="Order" value="1" type="Double" /><column name="SortFields(7)" property="DataOption" value="2" type="Double" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_facts_data', N'<table name="xls25.usp_facts_data"><columnFormats><column name="" property="ListObjectName" value="data" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="company_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="company_id" property="Address" value="$C$4" type="String" /><column name="company_id" property="NumberFormat" value="General" type="String" /><column name="id1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id1" property="Address" value="$D$4" type="String" /><column name="id1" property="NumberFormat" value="General" type="String" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="Address" value="$E$4" type="String" /><column name="id2" property="NumberFormat" value="General" type="String" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="Address" value="$F$4" type="String" /><column name="id3" property="NumberFormat" value="General" type="String" /><column name="id4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id4" property="Address" value="$G$4" type="String" /><column name="id4" property="NumberFormat" value="General" type="String" /><column name="id5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id5" property="Address" value="$H$4" type="String" /><column name="id5" property="NumberFormat" value="General" type="String" /><column name="id6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id6" property="Address" value="$I$4" type="String" /><column name="id6" property="NumberFormat" value="General" type="String" /><column name="id7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id7" property="Address" value="$J$4" type="String" /><column name="id7" property="NumberFormat" value="General" type="String" /><column name="sort_order1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order1" property="Address" value="$K$4" type="String" /><column name="sort_order1" property="NumberFormat" value="General" type="String" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="Address" value="$L$4" type="String" /><column name="sort_order2" property="NumberFormat" value="General" type="String" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="Address" value="$M$4" type="String" /><column name="sort_order3" property="NumberFormat" value="General" type="String" /><column name="sort_order4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order4" property="Address" value="$N$4" type="String" /><column name="sort_order4" property="NumberFormat" value="General" type="String" /><column name="sort_order5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order5" property="Address" value="$O$4" type="String" /><column name="sort_order5" property="NumberFormat" value="General" type="String" /><column name="sort_order6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order6" property="Address" value="$P$4" type="String" /><column name="sort_order6" property="NumberFormat" value="General" type="String" /><column name="sort_order7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order7" property="Address" value="$Q$4" type="String" /><column name="sort_order7" property="NumberFormat" value="General" type="String" /><column name="unit_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_id" property="Address" value="$R$4" type="String" /><column name="unit_id" property="NumberFormat" value="General" type="String" /><column name="calc_type_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_id" property="Address" value="$S$4" type="String" /><column name="calc_type_id" property="NumberFormat" value="General" type="String" /><column name="company_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_code" property="Address" value="$T$4" type="String" /><column name="company_code" property="ColumnWidth" value="16" type="Double" /><column name="company_code" property="NumberFormat" value="General" type="String" /><column name="company_name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="company_name" property="Address" value="$U$4" type="String" /><column name="company_name" property="NumberFormat" value="General" type="String" /><column name="code1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code1" property="Address" value="$V$4" type="String" /><column name="code1" property="ColumnWidth" value="14.86" type="Double" /><column name="code1" property="NumberFormat" value="General" type="String" /><column name="name1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name1" property="Address" value="$W$4" type="String" /><column name="name1" property="ColumnWidth" value="43.14" type="Double" /><column name="name1" property="NumberFormat" value="General" type="String" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="Address" value="$X$4" type="String" /><column name="code2" property="NumberFormat" value="General" type="String" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name2" property="Address" value="$Y$4" type="String" /><column name="name2" property="ColumnWidth" value="14.14" type="Double" /><column name="name2" property="NumberFormat" value="General" type="String" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code3" property="Address" value="$Z$4" type="String" /><column name="code3" property="NumberFormat" value="General" type="String" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name3" property="Address" value="$AA$4" type="String" /><column name="name3" property="ColumnWidth" value="16.29" type="Double" /><column name="name3" property="NumberFormat" value="General" type="String" /><column name="code4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code4" property="Address" value="$AB$4" type="String" /><column name="code4" property="NumberFormat" value="General" type="String" /><column name="name4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name4" property="Address" value="$AC$4" type="String" /><column name="name4" property="ColumnWidth" value="13.43" type="Double" /><column name="name4" property="NumberFormat" value="General" type="String" /><column name="code5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code5" property="Address" value="$AD$4" type="String" /><column name="code5" property="NumberFormat" value="General" type="String" /><column name="name5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name5" property="Address" value="$AE$4" type="String" /><column name="name5" property="ColumnWidth" value="13.29" type="Double" /><column name="name5" property="NumberFormat" value="General" type="String" /><column name="code6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code6" property="Address" value="$AF$4" type="String" /><column name="code6" property="NumberFormat" value="General" type="String" /><column name="name6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name6" property="Address" value="$AG$4" type="String" /><column name="name6" property="ColumnWidth" value="13.29" type="Double" /><column name="name6" property="NumberFormat" value="General" type="String" /><column name="code7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code7" property="Address" value="$AH$4" type="String" /><column name="code7" property="NumberFormat" value="General" type="String" /><column name="name7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name7" property="Address" value="$AI$4" type="String" /><column name="name7" property="ColumnWidth" value="13.29" type="Double" /><column name="name7" property="NumberFormat" value="General" type="String" /><column name="unit_code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_code" property="Address" value="$AJ$4" type="String" /><column name="unit_code" property="NumberFormat" value="General" type="String" /><column name="unit_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_name" property="Address" value="$AK$4" type="String" /><column name="unit_name" property="ColumnWidth" value="12.14" type="Double" /><column name="unit_name" property="NumberFormat" value="General" type="String" /><column name="calc_type_code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_code" property="Address" value="$AL$4" type="String" /><column name="calc_type_code" property="NumberFormat" value="General" type="String" /><column name="calc_type_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_name" property="Address" value="$AM$4" type="String" /><column name="calc_type_name" property="ColumnWidth" value="12.43" type="Double" /><column name="calc_type_name" property="NumberFormat" value="General" type="String" /><column name="source_value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="source_value" property="Address" value="$AN$4" type="String" /><column name="source_value" property="ColumnWidth" value="14.43" type="Double" /><column name="source_value" property="NumberFormat" value="#,##0" type="String" /><column name="value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="value" property="Address" value="$AO$4" type="String" /><column name="value" property="ColumnWidth" value="13.14" type="Double" /><column name="value" property="NumberFormat" value="#,##0" type="String" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order1" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order3" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="0" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="sort_order2" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="0" type="Double" /><column name="SortFields(4)" property="KeyfieldName" value="sort_order4" type="String" /><column name="SortFields(4)" property="SortOn" value="0" type="Double" /><column name="SortFields(4)" property="Order" value="1" type="Double" /><column name="SortFields(4)" property="DataOption" value="0" type="Double" /><column name="SortFields(5)" property="KeyfieldName" value="sort_order5" type="String" /><column name="SortFields(5)" property="SortOn" value="0" type="Double" /><column name="SortFields(5)" property="Order" value="1" type="Double" /><column name="SortFields(5)" property="DataOption" value="0" type="Double" /><column name="SortFields(6)" property="KeyfieldName" value="sort_order6" type="String" /><column name="SortFields(6)" property="SortOn" value="0" type="Double" /><column name="SortFields(6)" property="Order" value="1" type="Double" /><column name="SortFields(6)" property="DataOption" value="0" type="Double" /><column name="SortFields(7)" property="KeyfieldName" value="sort_order7" type="String" /><column name="SortFields(7)" property="SortOn" value="0" type="Double" /><column name="SortFields(7)" property="Order" value="1" type="Double" /><column name="SortFields(7)" property="DataOption" value="0" type="Double" /><column name="" property="Tab.Color" value="10498161" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="data" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="source_value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order1" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order3" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="0" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="sort_order2" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="0" type="Double" /><column name="SortFields(4)" property="KeyfieldName" value="sort_order4" type="String" /><column name="SortFields(4)" property="SortOn" value="0" type="Double" /><column name="SortFields(4)" property="Order" value="1" type="Double" /><column name="SortFields(4)" property="DataOption" value="0" type="Double" /><column name="SortFields(5)" property="KeyfieldName" value="sort_order5" type="String" /><column name="SortFields(5)" property="SortOn" value="0" type="Double" /><column name="SortFields(5)" property="Order" value="1" type="Double" /><column name="SortFields(5)" property="DataOption" value="0" type="Double" /><column name="SortFields(6)" property="KeyfieldName" value="sort_order6" type="String" /><column name="SortFields(6)" property="SortOn" value="0" type="Double" /><column name="SortFields(6)" property="Order" value="1" type="Double" /><column name="SortFields(6)" property="DataOption" value="0" type="Double" /><column name="SortFields(7)" property="KeyfieldName" value="sort_order7" type="String" /><column name="SortFields(7)" property="SortOn" value="0" type="Double" /><column name="SortFields(7)" property="Order" value="1" type="Double" /><column name="SortFields(7)" property="DataOption" value="0" type="Double" /></view><view name="Names"><column name="" property="ListObjectName" value="data" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="company_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="company_code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="company_name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="source_value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="value" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order1" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="2" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order3" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="2" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="sort_order2" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="2" type="Double" /><column name="SortFields(4)" property="KeyfieldName" value="sort_order4" type="String" /><column name="SortFields(4)" property="SortOn" value="0" type="Double" /><column name="SortFields(4)" property="Order" value="1" type="Double" /><column name="SortFields(4)" property="DataOption" value="2" type="Double" /><column name="SortFields(5)" property="KeyfieldName" value="sort_order5" type="String" /><column name="SortFields(5)" property="SortOn" value="0" type="Double" /><column name="SortFields(5)" property="Order" value="1" type="Double" /><column name="SortFields(5)" property="DataOption" value="2" type="Double" /><column name="SortFields(6)" property="KeyfieldName" value="sort_order6" type="String" /><column name="SortFields(6)" property="SortOn" value="0" type="Double" /><column name="SortFields(6)" property="Order" value="1" type="Double" /><column name="SortFields(6)" property="DataOption" value="2" type="Double" /><column name="SortFields(7)" property="KeyfieldName" value="sort_order7" type="String" /><column name="SortFields(7)" property="SortOn" value="0" type="Double" /><column name="SortFields(7)" property="Order" value="1" type="Double" /><column name="SortFields(7)" property="DataOption" value="2" type="Double" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_form_dimensions', N'<table name="xls25.usp_form_dimensions"><columnFormats><column name="" property="ListObjectName" value="form_dimensions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="form_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_id" property="Address" value="$C$4" type="String" /><column name="form_id" property="ColumnWidth" value="8.43" type="Double" /><column name="form_id" property="NumberFormat" value="General" type="String" /><column name="dimension_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="dimension_id" property="Address" value="$D$4" type="String" /><column name="dimension_id" property="NumberFormat" value="General" type="String" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_name" property="Address" value="$E$4" type="String" /><column name="form_name" property="ColumnWidth" value="25.57" type="Double" /><column name="form_name" property="NumberFormat" value="General" type="String" /><column name="dimension_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_name" property="Address" value="$F$4" type="String" /><column name="dimension_name" property="ColumnWidth" value="18" type="Double" /><column name="dimension_name" property="NumberFormat" value="General" type="String" /><column name="axis_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="axis_type_id" property="Address" value="$G$4" type="String" /><column name="axis_type_id" property="ColumnWidth" value="11.43" type="Double" /><column name="axis_type_id" property="NumberFormat" value="General" type="String" /><column name="parameter_index" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_index" property="Address" value="$H$4" type="String" /><column name="parameter_index" property="ColumnWidth" value="18" type="Double" /><column name="parameter_index" property="NumberFormat" value="General" type="String" /><column name="header_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="header_order" property="Address" value="$I$4" type="String" /><column name="header_order" property="ColumnWidth" value="14.86" type="Double" /><column name="header_order" property="NumberFormat" value="General" type="String" /><column name="root_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="root_member_id" property="Address" value="$J$4" type="String" /><column name="root_member_id" property="ColumnWidth" value="17.71" type="Double" /><column name="root_member_id" property="NumberFormat" value="@" type="String" /><column name="parameter_start_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_start_level" property="Address" value="$K$4" type="String" /><column name="parameter_start_level" property="ColumnWidth" value="22.57" type="Double" /><column name="parameter_start_level" property="NumberFormat" value="General" type="String" /><column name="parameter_end_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_end_level" property="Address" value="$L$4" type="String" /><column name="parameter_end_level" property="ColumnWidth" value="21.43" type="Double" /><column name="parameter_end_level" property="NumberFormat" value="General" type="String" /><column name="form_start_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_start_level" property="Address" value="$M$4" type="String" /><column name="form_start_level" property="ColumnWidth" value="17.43" type="Double" /><column name="form_start_level" property="NumberFormat" value="General" type="String" /><column name="form_end_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_end_level" property="Address" value="$N$4" type="String" /><column name="form_end_level" property="ColumnWidth" value="16.86" type="Double" /><column name="form_end_level" property="NumberFormat" value="General" type="String" /><column name="axis_type_id" property="FormatConditions(1).AppliesTo.Address" value="$G$4:$G$73" type="String" /><column name="axis_type_id" property="FormatConditions(1).Type" value="10" type="Double" /><column name="axis_type_id" property="FormatConditions(1).Priority" value="15" type="Double" /><column name="axis_type_id" property="FormatConditions(1).StopIfTrue" value="True" type="Boolean" /><column name="axis_type_id" property="FormatConditions(1).Interior.Color" value="14477050" type="Double" /><column name="parameter_index" property="FormatConditions(1).AppliesTo.Address" value="$H$4:$H$73" type="String" /><column name="parameter_index" property="FormatConditions(1).Type" value="2" type="Double" /><column name="parameter_index" property="FormatConditions(1).Priority" value="7" type="Double" /><column name="parameter_index" property="FormatConditions(1).StopIfTrue" value="True" type="Boolean" /><column name="parameter_index" property="FormatConditions(1).Formula1" value="=$G4=&quot;Rows&quot;" type="String" /><column name="header_order" property="FormatConditions(1).AppliesTo.Address" value="$I$4:$I$73" type="String" /><column name="header_order" property="FormatConditions(1).Type" value="2" type="Double" /><column name="header_order" property="FormatConditions(1).Priority" value="1" type="Double" /><column name="header_order" property="FormatConditions(1).StopIfTrue" value="True" type="Boolean" /><column name="header_order" property="FormatConditions(1).Formula1" value="=OR($G4=&quot;Columns&quot;,$G4=&quot;Rows&quot;)" type="String" /><column name="header_order" property="FormatConditions(1).Interior.Color" value="14477050" type="Double" /><column name="header_order" property="FormatConditions(2).AppliesTo.Address" value="$I$4:$I$73" type="String" /><column name="header_order" property="FormatConditions(2).Type" value="2" type="Double" /><column name="header_order" property="FormatConditions(2).Priority" value="5" type="Double" /><column name="header_order" property="FormatConditions(2).StopIfTrue" value="True" type="Boolean" /><column name="header_order" property="FormatConditions(2).Formula1" value="=$G4&lt;&gt;&quot;Columns&quot;" type="String" /><column name="header_order" property="FormatConditions(2).Interior.Color" value="8224125" type="Double" /><column name="parameter_start_level" property="FormatConditions(1).ColumnsCount" value="2" type="Double" /><column name="parameter_start_level" property="FormatConditions(1).AppliesTo.Address" value="$K$4:$L$73" type="String" /><column name="parameter_start_level" property="FormatConditions(1).Type" value="2" type="Double" /><column name="parameter_start_level" property="FormatConditions(1).Priority" value="4" type="Double" /><column name="parameter_start_level" property="FormatConditions(1).StopIfTrue" value="True" type="Boolean" /><column name="parameter_start_level" property="FormatConditions(1).Formula1" value="=$G4=&quot;Rows&quot;" type="String" /><column name="parameter_start_level" property="FormatConditions(1).Interior.Color" value="8224125" type="Double" /><column name="parameter_start_level" property="FormatConditions(2).ColumnsCount" value="4" type="Double" /><column name="parameter_start_level" property="FormatConditions(2).AppliesTo.Address" value="$K$4:$N$73" type="String" /><column name="parameter_start_level" property="FormatConditions(2).Type" value="2" type="Double" /><column name="parameter_start_level" property="FormatConditions(2).Priority" value="14" type="Double" /><column name="parameter_start_level" property="FormatConditions(2).StopIfTrue" value="True" type="Boolean" /><column name="parameter_start_level" property="FormatConditions(2).Formula1" value="=$J4=&quot;&quot;" type="String" /><column name="parameter_start_level" property="FormatConditions(2).Interior.Color" value="8224125" type="Double" /><column name="form_start_level" property="FormatConditions(1).ColumnsCount" value="2" type="Double" /><column name="form_start_level" property="FormatConditions(1).AppliesTo.Address" value="$M$4:$N$73" type="String" /><column name="form_start_level" property="FormatConditions(1).Type" value="2" type="Double" /><column name="form_start_level" property="FormatConditions(1).Priority" value="6" type="Double" /><column name="form_start_level" property="FormatConditions(1).StopIfTrue" value="True" type="Boolean" /><column name="form_start_level" property="FormatConditions(1).Formula1" value="=$G4=&quot;Page&quot;" type="String" /><column name="form_start_level" property="FormatConditions(1).Interior.Color" value="8224125" type="Double" /><column name="" property="Tab.Color" value="15773696" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="form_dimensions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="axis_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_index" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="header_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="root_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_start_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_end_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_start_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_end_level" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="form_dimensions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="axis_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_index" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="header_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="root_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_start_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_end_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_start_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_end_level" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Layouts"><column name="" property="ListObjectName" value="form_dimensions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="axis_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_index" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="header_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="root_member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parameter_start_level" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parameter_end_level" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_start_level" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_end_level" property="EntireColumn.Hidden" value="True" type="Boolean" /></view><view name="Columns"><column name="" property="ListObjectName" value="form_dimensions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="axis_type_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parameter_index" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="header_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="root_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_start_level" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parameter_end_level" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_start_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_end_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="axis_type_id" property="AutoFilter.Criteria1" value="=Columns" type="String" /></view><view name="Parameters"><column name="" property="ListObjectName" value="form_dimensions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="axis_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_index" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="header_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="root_member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_start_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parameter_end_level" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="form_start_level" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="form_end_level" property="EntireColumn.Hidden" value="True" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_form_permissions', N'<table name="xls25.usp_form_permissions"><columnFormats><column name="" property="ListObjectName" value="form_permissions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$D$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$E$4" type="String" /><column name="code" property="ColumnWidth" value="25.57" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$F$4" type="String" /><column name="name" property="ColumnWidth" value="25.57" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="Address" value="$G$4" type="String" /><column name="data_format" property="NumberFormat" value="General" type="String" /><column name="data_format" property="HorizontalAlignment" value="-4108" type="Double" /><column name="data_format" property="Font.Size" value="10" type="Double" /><column name="pa_analyst_01" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_analyst_01" property="Address" value="$H$4" type="String" /><column name="pa_analyst_01" property="ColumnWidth" value="15.14" type="Double" /><column name="pa_analyst_01" property="NumberFormat" value="General" type="String" /><column name="pa_analyst_01" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_analyst_01" property="Font.Size" value="10" type="Double" /><column name="pa_user_101" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_101" property="Address" value="$I$4" type="String" /><column name="pa_user_101" property="ColumnWidth" value="13.57" type="Double" /><column name="pa_user_101" property="NumberFormat" value="General" type="String" /><column name="pa_user_101" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_user_101" property="Font.Size" value="10" type="Double" /><column name="pa_user_201" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_201" property="Address" value="$J$4" type="String" /><column name="pa_user_201" property="ColumnWidth" value="13.57" type="Double" /><column name="pa_user_201" property="NumberFormat" value="General" type="String" /><column name="pa_user_201" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_user_201" property="Font.Size" value="10" type="Double" /><column name="pa_user_301" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_301" property="Address" value="$K$4" type="String" /><column name="pa_user_301" property="ColumnWidth" value="13.57" type="Double" /><column name="pa_user_301" property="NumberFormat" value="General" type="String" /><column name="pa_user_301" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_user_301" property="Font.Size" value="10" type="Double" /><column name="pa_user_401" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_401" property="Address" value="$L$4" type="String" /><column name="pa_user_401" property="ColumnWidth" value="13.57" type="Double" /><column name="pa_user_401" property="NumberFormat" value="General" type="String" /><column name="pa_user_401" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_user_401" property="Font.Size" value="10" type="Double" /><column name="data_format" property="FormatConditions(1).ColumnsCount" value="6" type="Double" /><column name="data_format" property="FormatConditions(1).AppliesTo.Address" value="$G$4:$L$13" type="String" /><column name="data_format" property="FormatConditions(1).Type" value="6" type="Double" /><column name="data_format" property="FormatConditions(1).Priority" value="88" type="Double" /><column name="data_format" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="data_format" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="data_format" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="15773697" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="Sheet1_Table1" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_01" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="Sheet1_Table1" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="pa_user_01" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_form_rows', N'<table name="xls25.usp_form_rows"><columnFormats><column name="" property="ListObjectName" value="rows" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$D$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$E$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="HorizontalAlignment" value="-4108" type="Double" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="decimal_places" property="Address" value="$F$4" type="String" /><column name="decimal_places" property="ColumnWidth" value="15.86" type="Double" /><column name="decimal_places" property="NumberFormat" value="General" type="String" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="Address" value="$G$4" type="String" /><column name="is_percent" property="ColumnWidth" value="11.29" type="Double" /><column name="is_percent" property="NumberFormat" value="General" type="String" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="Address" value="$H$4" type="String" /><column name="row_color" property="ColumnWidth" value="11.57" type="Double" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_color" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="Address" value="$I$4" type="String" /><column name="row_bold" property="ColumnWidth" value="10.86" type="Double" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="Address" value="$J$4" type="String" /><column name="row_indent" property="ColumnWidth" value="12.71" type="Double" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="row_indent" property="HorizontalAlignment" value="-4108" type="Double" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$K$4" type="String" /><column name="code" property="ColumnWidth" value="7.14" type="Double" /><column name="code" property="NumberFormat" value="@" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$L$4" type="String" /><column name="name" property="ColumnWidth" value="37.86" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="translated_name" property="Address" value="$M$4" type="String" /><column name="translated_name" property="ColumnWidth" value="37.86" type="Double" /><column name="translated_name" property="NumberFormat" value="General" type="String" /><column name="member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id" property="Address" value="$N$4" type="String" /><column name="member_id" property="ColumnWidth" value="12.86" type="Double" /><column name="member_id" property="NumberFormat" value="General" type="String" /><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$N$15" type="String" /><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(1).Priority" value="5" type="Double" /><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$I4=1" type="String" /><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$N$15" type="String" /><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(2).Priority" value="6" type="Double" /><column name="_RowNum" property="FormatConditions(2).Formula1" value="=$H4=11" type="String" /><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).Font.Color" value="16777215" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.ThemeColor" value="1" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.TintAndShade" value="0" type="Double" /><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="6365184" type="Double" /><column name="_RowNum" property="FormatConditions(3).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).AppliesTo.Address" value="$B$4:$N$15" type="String" /><column name="_RowNum" property="FormatConditions(3).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(3).Priority" value="7" type="Double" /><column name="_RowNum" property="FormatConditions(3).Formula1" value="=$H4=12" type="String" /><column name="_RowNum" property="FormatConditions(3).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).Interior.Color" value="15787740" type="Double" /><column name="_RowNum" property="FormatConditions(4).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).AppliesTo.Address" value="$B$4:$N$15" type="String" /><column name="_RowNum" property="FormatConditions(4).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(4).Priority" value="8" type="Double" /><column name="_RowNum" property="FormatConditions(4).Formula1" value="=$H4=13" type="String" /><column name="_RowNum" property="FormatConditions(4).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).Interior.Color" value="15790320" type="Double" /><column name="" property="Tab.Color" value="5287936" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="rows" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="rows" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Members Only"><column name="" property="ListObjectName" value="rows" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_forms', N'<table name="xls25.usp_forms"><columnFormats><column name="" property="ListObjectName" value="forms" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="ColumnWidth" value="4.29" type="Double" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$D$4" type="String" /><column name="code" property="ColumnWidth" value="25.57" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$E$4" type="String" /><column name="name" property="ColumnWidth" value="25.57" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="translated_name" property="Address" value="$F$4" type="String" /><column name="translated_name" property="ColumnWidth" value="17.71" type="Double" /><column name="translated_name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="HorizontalAlignment" value="-4108" type="Double" /><column name="translated_name" property="Font.Size" value="10" type="Double" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="Address" value="$G$4" type="String" /><column name="sort_order" property="ColumnWidth" value="11.86" type="Double" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="Address" value="$H$4" type="String" /><column name="is_active" property="ColumnWidth" value="10.29" type="Double" /><column name="is_active" property="NumberFormat" value="General" type="String" /><column name="is_active" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_active" property="Font.Size" value="10" type="Double" /><column name="is_active" property="FormatConditions(1).ColumnsCount" value="2" type="Double" /><column name="is_active" property="FormatConditions(1).AppliesTo.Address" value="$H$4:$I$13" type="String" /><column name="is_active" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_active" property="FormatConditions(1).Priority" value="1" type="Double" /><column name="is_active" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_active" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="10498160" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="forms" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="forms" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_member_permissions', N'<table name="xls25.usp_member_permissions"><columnFormats><column name="" property="ListObjectName" value="member_permissions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="dimension_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="dimension_id" property="Address" value="$D$4" type="String" /><column name="dimension_id" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$E$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$F$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="Address" value="$G$4" type="String" /><column name="decimal_places" property="NumberFormat" value="General" type="String" /><column name="decimal_places" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="Address" value="$H$4" type="String" /><column name="is_percent" property="NumberFormat" value="General" type="String" /><column name="is_percent" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="Address" value="$I$4" type="String" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_color" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="Address" value="$J$4" type="String" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="Address" value="$K$4" type="String" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="row_indent" property="HorizontalAlignment" value="-4108" type="Double" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$L$4" type="String" /><column name="code" property="ColumnWidth" value="18.14" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$M$4" type="String" /><column name="name" property="ColumnWidth" value="50.57" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="Address" value="$N$4" type="String" /><column name="calc_type" property="NumberFormat" value="General" type="String" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="Address" value="$O$4" type="String" /><column name="data_format" property="NumberFormat" value="General" type="String" /><column name="data_format" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_analyst_01" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_analyst_01" property="Address" value="$P$4" type="String" /><column name="pa_analyst_01" property="ColumnWidth" value="15.14" type="Double" /><column name="pa_analyst_01" property="NumberFormat" value="General" type="String" /><column name="pa_analyst_01" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_user_101" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_101" property="Address" value="$Q$4" type="String" /><column name="pa_user_101" property="ColumnWidth" value="15.14" type="Double" /><column name="pa_user_101" property="NumberFormat" value="General" type="String" /><column name="pa_user_101" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_user_201" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_201" property="Address" value="$R$4" type="String" /><column name="pa_user_201" property="ColumnWidth" value="15.14" type="Double" /><column name="pa_user_201" property="NumberFormat" value="General" type="String" /><column name="pa_user_201" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_user_301" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_301" property="Address" value="$S$4" type="String" /><column name="pa_user_301" property="ColumnWidth" value="15.14" type="Double" /><column name="pa_user_301" property="NumberFormat" value="General" type="String" /><column name="pa_user_301" property="HorizontalAlignment" value="-4108" type="Double" /><column name="pa_user_401" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_401" property="Address" value="$T$4" type="String" /><column name="pa_user_401" property="ColumnWidth" value="15.14" type="Double" /><column name="pa_user_401" property="NumberFormat" value="General" type="String" /><column name="pa_user_401" property="HorizontalAlignment" value="-4108" type="Double" /><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$T$327" type="String" /><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(1).Priority" value="22" type="Double" /><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$J4=1" type="String" /><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$T$327" type="String" /><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(2).Priority" value="23" type="Double" /><column name="_RowNum" property="FormatConditions(2).Formula1" value="=$I4=12" type="String" /><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="15787740" type="Double" /><column name="_RowNum" property="FormatConditions(3).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).AppliesTo.Address" value="$B$4:$T$327" type="String" /><column name="_RowNum" property="FormatConditions(3).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(3).Priority" value="24" type="Double" /><column name="_RowNum" property="FormatConditions(3).Formula1" value="=$I4=11" type="String" /><column name="_RowNum" property="FormatConditions(3).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).Font.Color" value="16777215" type="Double" /><column name="_RowNum" property="FormatConditions(3).Font.ThemeColor" value="1" type="Double" /><column name="_RowNum" property="FormatConditions(3).Font.TintAndShade" value="0" type="Double" /><column name="_RowNum" property="FormatConditions(3).Interior.Color" value="6365184" type="Double" /><column name="pa_analyst_01" property="FormatConditions(1).ColumnsCount" value="5" type="Double" /><column name="pa_analyst_01" property="FormatConditions(1).AppliesTo.Address" value="$P$4:$T$327" type="String" /><column name="pa_analyst_01" property="FormatConditions(1).Type" value="1" type="Double" /><column name="pa_analyst_01" property="FormatConditions(1).Priority" value="4" type="Double" /><column name="pa_analyst_01" property="FormatConditions(1).StopIfTrue" value="True" type="Boolean" /><column name="pa_analyst_01" property="FormatConditions(1).Formula1" value="=&quot;R&quot;" type="String" /><column name="pa_analyst_01" property="FormatConditions(1).Operator" value="3" type="Double" /><column name="pa_analyst_01" property="FormatConditions(1).Font.Color" value="24832" type="Double" /><column name="pa_analyst_01" property="FormatConditions(1).Interior.Color" value="13561798" type="Double" /><column name="pa_analyst_01" property="FormatConditions(2).ColumnsCount" value="5" type="Double" /><column name="pa_analyst_01" property="FormatConditions(2).AppliesTo.Address" value="$P$4:$T$327" type="String" /><column name="pa_analyst_01" property="FormatConditions(2).Type" value="1" type="Double" /><column name="pa_analyst_01" property="FormatConditions(2).Priority" value="5" type="Double" /><column name="pa_analyst_01" property="FormatConditions(2).StopIfTrue" value="True" type="Boolean" /><column name="pa_analyst_01" property="FormatConditions(2).Formula1" value="=&quot;W&quot;" type="String" /><column name="pa_analyst_01" property="FormatConditions(2).Operator" value="3" type="Double" /><column name="pa_analyst_01" property="FormatConditions(2).Font.Color" value="22428" type="Double" /><column name="pa_analyst_01" property="FormatConditions(2).Interior.Color" value="10284031" type="Double" /><column name="pa_analyst_01" property="FormatConditions(3).ColumnsCount" value="5" type="Double" /><column name="pa_analyst_01" property="FormatConditions(3).AppliesTo.Address" value="$P$4:$T$327" type="String" /><column name="pa_analyst_01" property="FormatConditions(3).Type" value="1" type="Double" /><column name="pa_analyst_01" property="FormatConditions(3).Priority" value="6" type="Double" /><column name="pa_analyst_01" property="FormatConditions(3).StopIfTrue" value="True" type="Boolean" /><column name="pa_analyst_01" property="FormatConditions(3).Formula1" value="=&quot;D&quot;" type="String" /><column name="pa_analyst_01" property="FormatConditions(3).Operator" value="3" type="Double" /><column name="pa_analyst_01" property="FormatConditions(3).Font.Color" value="393372" type="Double" /><column name="pa_analyst_01" property="FormatConditions(3).Interior.Color" value="13551615" type="Double" /><column name="SortFields(1)" property="KeyfieldName" value="dimension_id" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="0" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="code" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="0" type="Double" /><column name="" property="Tab.Color" value="5287937" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="member_permissions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="pa_user_01" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="dimension_id" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="2" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="2" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="code" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="0" type="Double" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="member_permissions" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="dimension_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="pa_user_01" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="dimension_id" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="2" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="2" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="code" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="0" type="Double" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_member_relations', N'<table name="xls25.usp_member_relations"><columnFormats><column name="" property="ListObjectName" value="relations" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="dimension_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="dimension_id" property="Address" value="$D$4" type="String" /><column name="dimension_id" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$E$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$F$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="Address" value="$G$4" type="String" /><column name="decimal_places" property="NumberFormat" value="General" type="String" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="Address" value="$H$4" type="String" /><column name="is_percent" property="NumberFormat" value="General" type="String" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="Address" value="$I$4" type="String" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="Address" value="$J$4" type="String" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="Address" value="$K$4" type="String" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="row_indent" property="HorizontalAlignment" value="-4108" type="Double" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$L$4" type="String" /><column name="code" property="ColumnWidth" value="18.57" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$M$4" type="String" /><column name="name" property="ColumnWidth" value="50.57" type="Double" /><column name="name" property="NumberFormat" value="@" type="String" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="Address" value="$N$4" type="String" /><column name="calc_type" property="NumberFormat" value="General" type="String" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="Address" value="$O$4" type="String" /><column name="data_format" property="NumberFormat" value="General" type="String" /><column name="data_format" property="HorizontalAlignment" value="-4108" type="Double" /><column name="data_format" property="Font.Size" value="9" type="Double" /><column name="DIM6.NONE" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="DIM6.NONE" property="Address" value="$P$4" type="String" /><column name="DIM6.NONE" property="ColumnWidth" value="13.43" type="Double" /><column name="DIM6.NONE" property="NumberFormat" value="General" type="String" /><column name="DIM6.NONE" property="HorizontalAlignment" value="-4108" type="Double" /><column name="DIM6.NONE" property="Font.Size" value="9" type="Double" /><column name="OW" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="OW" property="Address" value="$Q$4" type="String" /><column name="OW" property="ColumnWidth" value="13.43" type="Double" /><column name="OW" property="NumberFormat" value="General" type="String" /><column name="OW" property="HorizontalAlignment" value="-4108" type="Double" /><column name="OW" property="Font.Size" value="9" type="Double" /><column name="FW" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="FW" property="Address" value="$R$4" type="String" /><column name="FW" property="ColumnWidth" value="13.43" type="Double" /><column name="FW" property="NumberFormat" value="General" type="String" /><column name="FW" property="HorizontalAlignment" value="-4108" type="Double" /><column name="FW" property="Font.Size" value="9" type="Double" /><column name="TW" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="TW" property="Address" value="$S$4" type="String" /><column name="TW" property="ColumnWidth" value="13.43" type="Double" /><column name="TW" property="NumberFormat" value="General" type="String" /><column name="TW" property="HorizontalAlignment" value="-4108" type="Double" /><column name="TW" property="Font.Size" value="9" type="Double" /><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$S$283" type="String" /><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(1).Priority" value="8" type="Double" /><column name="_RowNum" property="FormatConditions(1).StopIfTrue" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$I4=11" type="String" /><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).Font.Color" value="16777215" type="Double" /><column name="_RowNum" property="FormatConditions(1).Font.ThemeColor" value="1" type="Double" /><column name="_RowNum" property="FormatConditions(1).Font.TintAndShade" value="0" type="Double" /><column name="_RowNum" property="FormatConditions(1).Interior.Color" value="6365184" type="Double" /><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$S$283" type="String" /><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(2).Priority" value="9" type="Double" /><column name="_RowNum" property="FormatConditions(2).StopIfTrue" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).Formula1" value="=$I4=12" type="String" /><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="15787740" type="Double" /><column name="_RowNum" property="FormatConditions(3).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).AppliesTo.Address" value="$B$4:$S$283" type="String" /><column name="_RowNum" property="FormatConditions(3).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(3).Priority" value="10" type="Double" /><column name="_RowNum" property="FormatConditions(3).StopIfTrue" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).Formula1" value="=$I4=13" type="String" /><column name="_RowNum" property="FormatConditions(3).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).Interior.Color" value="15790320" type="Double" /><column name="_RowNum" property="FormatConditions(4).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).AppliesTo.Address" value="$B$4:$S$283" type="String" /><column name="_RowNum" property="FormatConditions(4).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(4).Priority" value="11" type="Double" /><column name="_RowNum" property="FormatConditions(4).StopIfTrue" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).Formula1" value="=$J4=1" type="String" /><column name="_RowNum" property="FormatConditions(4).Font.Bold" value="True" type="Boolean" /><column name="data_format" property="FormatConditions(1).ColumnsCount" value="5" type="Double" /><column name="data_format" property="FormatConditions(1).AppliesTo.Address" value="$O$4:$S$283" type="String" /><column name="data_format" property="FormatConditions(1).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(1).Priority" value="102148" type="Double" /><column name="data_format" property="FormatConditions(1).Formula1" value="=$F4=0" type="String" /><column name="data_format" property="FormatConditions(1).Interior.Color" value="15921906" type="Double" /><column name="data_format" property="FormatConditions(1).Interior.ThemeColor" value="1" type="Double" /><column name="data_format" property="FormatConditions(1).Interior.TintAndShade" value="-0.0499893185216834" type="Double" /><column name="data_format" property="FormatConditions(2).ColumnsCount" value="5" type="Double" /><column name="data_format" property="FormatConditions(2).AppliesTo.Address" value="$O$4:$S$283" type="String" /><column name="data_format" property="FormatConditions(2).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(2).Priority" value="102149" type="Double" /><column name="data_format" property="FormatConditions(2).Formula1" value="=$F4=1" type="String" /><column name="data_format" property="FormatConditions(2).Interior.Color" value="14477050" type="Double" /><column name="data_format" property="FormatConditions(3).ColumnsCount" value="5" type="Double" /><column name="data_format" property="FormatConditions(3).AppliesTo.Address" value="$O$4:$S$283" type="String" /><column name="data_format" property="FormatConditions(3).Type" value="6" type="Double" /><column name="data_format" property="FormatConditions(3).Priority" value="102150" type="Double" /><column name="data_format" property="FormatConditions(3).ShowIconOnly" value="True" type="Boolean" /><column name="data_format" property="FormatConditions(3).IconSet.ID" value="8" type="Double" /><column name="data_format" property="FormatConditions(3).IconCriteria(1).Type" value="3" type="Double" /><column name="data_format" property="FormatConditions(3).IconCriteria(1).Operator" value="7" type="Double" /><column name="data_format" property="FormatConditions(3).IconCriteria(2).Type" value="0" type="Double" /><column name="data_format" property="FormatConditions(3).IconCriteria(2).Value" value="0.5" type="Double" /><column name="data_format" property="FormatConditions(3).IconCriteria(2).Operator" value="7" type="Double" /><column name="data_format" property="FormatConditions(3).IconCriteria(3).Type" value="0" type="Double" /><column name="data_format" property="FormatConditions(3).IconCriteria(3).Value" value="1" type="Double" /><column name="data_format" property="FormatConditions(3).IconCriteria(3).Operator" value="7" type="Double" /><column name="SortFields(1)" property="KeyfieldName" value="dimension_id" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="0" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="code" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="0" type="Double" /><column name="" property="Tab.Color" value="5287936" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="12" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="relations" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dimension_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="dimension_id" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="0" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="code" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="0" type="Double" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="relations" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="dimension_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="dimension_id" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="2" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="2" type="Double" /><column name="SortFields(3)" property="KeyfieldName" value="code" type="String" /><column name="SortFields(3)" property="SortOn" value="0" type="Double" /><column name="SortFields(3)" property="Order" value="1" type="Double" /><column name="SortFields(3)" property="DataOption" value="0" type="Double" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_members', N'<table name="xls25.usp_members"><columnFormats><column name="" property="ListObjectName" value="accounts" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="Address" value="$D$4" type="String" /><column name="sort_order" property="ColumnWidth" value="11.71" type="Double" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$E$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="HorizontalAlignment" value="-4108" type="Double" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="Address" value="$F$4" type="String" /><column name="decimal_places" property="NumberFormat" value="General" type="String" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="Address" value="$G$4" type="String" /><column name="is_percent" property="NumberFormat" value="General" type="String" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="Address" value="$H$4" type="String" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_color" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="Address" value="$I$4" type="String" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="Address" value="$J$4" type="String" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="row_indent" property="HorizontalAlignment" value="-4108" type="Double" /><column name="show_line" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line" property="Address" value="$K$4" type="String" /><column name="show_line" property="NumberFormat" value="General" type="String" /><column name="show_line_before" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_before" property="Address" value="$L$4" type="String" /><column name="show_line_before" property="NumberFormat" value="General" type="String" /><column name="show_line_after" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_after" property="Address" value="$M$4" type="String" /><column name="show_line_after" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$N$4" type="String" /><column name="code" property="ColumnWidth" value="22.86" type="Double" /><column name="code" property="NumberFormat" value="@" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$O$4" type="String" /><column name="name" property="ColumnWidth" value="50.57" type="Double" /><column name="name" property="NumberFormat" value="@" type="String" /><column name="translated_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="translated_name" property="Address" value="$P$4" type="String" /><column name="translated_name" property="ColumnWidth" value="17.71" type="Double" /><column name="translated_name" property="NumberFormat" value="General" type="String" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="comment" property="Address" value="$Q$4" type="String" /><column name="comment" property="NumberFormat" value="General" type="String" /><column name="tax_rate_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="tax_rate_id" property="Address" value="$R$4" type="String" /><column name="tax_rate_id" property="ColumnWidth" value="4.71" type="Double" /><column name="tax_rate_id" property="NumberFormat" value="General" type="String" /><column name="unit_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_id" property="Address" value="$S$4" type="String" /><column name="unit_id" property="ColumnWidth" value="4.43" type="Double" /><column name="unit_id" property="NumberFormat" value="General" type="String" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_id" property="Address" value="$T$4" type="String" /><column name="calc_type_id" property="ColumnWidth" value="12.29" type="Double" /><column name="calc_type_id" property="NumberFormat" value="General" type="String" /><column name="previous_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="previous_period_id" property="Address" value="$U$4" type="String" /><column name="previous_period_id" property="NumberFormat" value="General" type="String" /><column name="same_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="same_period_id" property="Address" value="$V$4" type="String" /><column name="same_period_id" property="NumberFormat" value="General" type="String" /><column name="parent1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent1" property="Address" value="$W$4" type="String" /><column name="parent1" property="ColumnWidth" value="17.57" type="Double" /><column name="parent1" property="NumberFormat" value="@" type="String" /><column name="factor1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor1" property="Address" value="$X$4" type="String" /><column name="factor1" property="ColumnWidth" value="9.43" type="Double" /><column name="factor1" property="NumberFormat" value="General" type="String" /><column name="parent2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent2" property="Address" value="$Y$4" type="String" /><column name="parent2" property="ColumnWidth" value="9.86" type="Double" /><column name="parent2" property="NumberFormat" value="@" type="String" /><column name="factor2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor2" property="Address" value="$Z$4" type="String" /><column name="factor2" property="ColumnWidth" value="9.43" type="Double" /><column name="factor2" property="NumberFormat" value="General" type="String" /><column name="parent3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent3" property="Address" value="$AA$4" type="String" /><column name="parent3" property="ColumnWidth" value="9.86" type="Double" /><column name="parent3" property="NumberFormat" value="@" type="String" /><column name="factor3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor3" property="Address" value="$AB$4" type="String" /><column name="factor3" property="ColumnWidth" value="9.43" type="Double" /><column name="factor3" property="NumberFormat" value="General" type="String" /><column name="parent4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent4" property="Address" value="$AC$4" type="String" /><column name="parent4" property="ColumnWidth" value="9.86" type="Double" /><column name="parent4" property="NumberFormat" value="@" type="String" /><column name="factor4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor4" property="Address" value="$AD$4" type="String" /><column name="factor4" property="ColumnWidth" value="9.43" type="Double" /><column name="factor4" property="NumberFormat" value="General" type="String" /><column name="parent5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent5" property="Address" value="$AE$4" type="String" /><column name="parent5" property="NumberFormat" value="@" type="String" /><column name="factor5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor5" property="Address" value="$AF$4" type="String" /><column name="factor5" property="NumberFormat" value="General" type="String" /><column name="parent6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent6" property="Address" value="$AG$4" type="String" /><column name="parent6" property="NumberFormat" value="@" type="String" /><column name="factor6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor6" property="Address" value="$AH$4" type="String" /><column name="factor6" property="NumberFormat" value="General" type="String" /><column name="parent7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent7" property="Address" value="$AI$4" type="String" /><column name="parent7" property="NumberFormat" value="@" type="String" /><column name="factor7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor7" property="Address" value="$AJ$4" type="String" /><column name="factor7" property="NumberFormat" value="General" type="String" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="Address" value="$AK$4" type="String" /><column name="is_active" property="ColumnWidth" value="2.14" type="Double" /><column name="is_active" property="NumberFormat" value="General" type="String" /><column name="is_active" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_active" property="Font.Size" value="10" type="Double" /><column name="excel_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="excel_formula" property="Address" value="$AL$4" type="String" /><column name="excel_formula" property="NumberFormat" value="General" type="String" /><column name="cube_formula" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="cube_formula" property="Address" value="$AM$4" type="String" /><column name="cube_formula" property="ColumnWidth" value="56.43" type="Double" /><column name="cube_formula" property="NumberFormat" value="General" type="String" /><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$AM$286" type="String" /><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(1).Priority" value="6" type="Double" /><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$I4=1" type="String" /><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$AM$286" type="String" /><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(2).Priority" value="7" type="Double" /><column name="_RowNum" property="FormatConditions(2).Formula1" value="=$H4=11" type="String" /><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).Font.Color" value="16777215" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.ThemeColor" value="1" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.TintAndShade" value="0" type="Double" /><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="6365184" type="Double" /><column name="_RowNum" property="FormatConditions(3).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).AppliesTo.Address" value="$B$4:$AM$286" type="String" /><column name="_RowNum" property="FormatConditions(3).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(3).Priority" value="8" type="Double" /><column name="_RowNum" property="FormatConditions(3).Formula1" value="=$H4=12" type="String" /><column name="_RowNum" property="FormatConditions(3).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).Interior.Color" value="15787740" type="Double" /><column name="_RowNum" property="FormatConditions(4).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).AppliesTo.Address" value="$B$4:$AM$286" type="String" /><column name="_RowNum" property="FormatConditions(4).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(4).Priority" value="9" type="Double" /><column name="_RowNum" property="FormatConditions(4).Formula1" value="=$H4=13" type="String" /><column name="_RowNum" property="FormatConditions(4).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).Interior.Color" value="15790320" type="Double" /><column name="is_active" property="FormatConditions(1).AppliesTo.Address" value="$AK$4:$AK$286" type="String" /><column name="is_active" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_active" property="FormatConditions(1).Priority" value="5" type="Double" /><column name="is_active" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_active" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="" property="Tab.Color" value="5287937" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="members" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="show_line" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="show_line_before" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="show_line_after" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="tax_rate_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="previous_period_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="same_period_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="excel_formula" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="cube_formula" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="members" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="tax_rate_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="previous_period_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="same_period_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor5" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor6" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor7" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="excel_formula" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="cube_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /></view><view name="Accounts"><column name="" property="ListObjectName" value="members" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_before" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_after" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="previous_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="same_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="excel_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="cube_formula" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /></view><view name="Entities"><column name="" property="ListObjectName" value="members" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_before" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_after" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="previous_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="same_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="excel_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="cube_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /></view><view name="Categories"><column name="" property="ListObjectName" value="members" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_before" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_after" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="previous_period_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="same_period_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="excel_formula" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="cube_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /></view><view name="Times"><column name="" property="ListObjectName" value="members" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_before" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_after" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="previous_period_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="same_period_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor4" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="excel_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="cube_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /></view><view name="Dim"><column name="" property="ListObjectName" value="members" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_before" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_after" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="previous_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="same_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="parent3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="excel_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="cube_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /></view><view name="Cube Formulas"><column name="" property="ListObjectName" value="members" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_before" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="show_line_after" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="previous_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="same_period_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor4" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor5" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor6" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="parent7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="factor7" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_active" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="excel_formula" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="cube_formula" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="code" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="0" type="Double" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_role_members', N'<table name="xls25.usp_role_members"><columnFormats><column name="" property="ListObjectName" value="role_members" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="type" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="type" property="Address" value="$C$4" type="String" /><column name="type" property="ColumnWidth" value="13" type="Double" /><column name="type" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$D$4" type="String" /><column name="name" property="ColumnWidth" value="20" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="format_column" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="format_column" property="Address" value="$E$4" type="String" /><column name="format_column" property="NumberFormat" value="General" type="String" /><column name="format_column" property="HorizontalAlignment" value="-4108" type="Double" /><column name="format_column" property="Font.Size" value="10" type="Double" /><column name="log_users" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="log_users" property="Address" value="$F$4" type="String" /><column name="log_users" property="ColumnWidth" value="11" type="Double" /><column name="log_users" property="NumberFormat" value="General" type="String" /><column name="log_users" property="HorizontalAlignment" value="-4108" type="Double" /><column name="log_users" property="Font.Size" value="10" type="Double" /><column name="planning_app_admins" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="planning_app_admins" property="Address" value="$G$4" type="String" /><column name="planning_app_admins" property="ColumnWidth" value="22.29" type="Double" /><column name="planning_app_admins" property="NumberFormat" value="General" type="String" /><column name="planning_app_admins" property="HorizontalAlignment" value="-4108" type="Double" /><column name="planning_app_admins" property="Font.Size" value="10" type="Double" /><column name="planning_app_users" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="planning_app_users" property="Address" value="$H$4" type="String" /><column name="planning_app_users" property="ColumnWidth" value="20.57" type="Double" /><column name="planning_app_users" property="NumberFormat" value="General" type="String" /><column name="planning_app_users" property="HorizontalAlignment" value="-4108" type="Double" /><column name="planning_app_users" property="Font.Size" value="10" type="Double" /><column name="xls_admins" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="xls_admins" property="Address" value="$I$4" type="String" /><column name="xls_admins" property="ColumnWidth" value="12.43" type="Double" /><column name="xls_admins" property="NumberFormat" value="General" type="String" /><column name="xls_admins" property="HorizontalAlignment" value="-4108" type="Double" /><column name="xls_admins" property="Font.Size" value="10" type="Double" /><column name="xls_developers" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="xls_developers" property="Address" value="$J$4" type="String" /><column name="xls_developers" property="ColumnWidth" value="16.14" type="Double" /><column name="xls_developers" property="NumberFormat" value="General" type="String" /><column name="xls_developers" property="HorizontalAlignment" value="-4108" type="Double" /><column name="xls_developers" property="Font.Size" value="10" type="Double" /><column name="xls_formats" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="xls_formats" property="Address" value="$K$4" type="String" /><column name="xls_formats" property="ColumnWidth" value="12.86" type="Double" /><column name="xls_formats" property="NumberFormat" value="General" type="String" /><column name="xls_formats" property="HorizontalAlignment" value="-4108" type="Double" /><column name="xls_formats" property="Font.Size" value="10" type="Double" /><column name="xls_users" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="xls_users" property="Address" value="$L$4" type="String" /><column name="xls_users" property="ColumnWidth" value="10.71" type="Double" /><column name="xls_users" property="NumberFormat" value="General" type="String" /><column name="xls_users" property="HorizontalAlignment" value="-4108" type="Double" /><column name="xls_users" property="Font.Size" value="10" type="Double" /><column name="last_format_column" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="last_format_column" property="Address" value="$M$4" type="String" /><column name="last_format_column" property="NumberFormat" value="General" type="String" /><column name="last_format_column" property="HorizontalAlignment" value="-4108" type="Double" /><column name="last_format_column" property="Font.Size" value="10" type="Double" /><column name="format_column" property="FormatConditions(1).AppliesTo.Address" value="$E$4:$E$6" type="String" /><column name="format_column" property="FormatConditions(1).Type" value="6" type="Double" /><column name="format_column" property="FormatConditions(1).Priority" value="3" type="Double" /><column name="format_column" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="format_column" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="format_column" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="format_column" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="format_column" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="format_column" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="format_column" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="format_column" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="format_column" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="format_column" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="log_users" property="FormatConditions(1).ColumnsCount" value="7" type="Double" /><column name="log_users" property="FormatConditions(1).AppliesTo.Address" value="$F$4:$L$6" type="String" /><column name="log_users" property="FormatConditions(1).Type" value="6" type="Double" /><column name="log_users" property="FormatConditions(1).Priority" value="2" type="Double" /><column name="log_users" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="log_users" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="log_users" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="log_users" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="log_users" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="log_users" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="log_users" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="log_users" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="log_users" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="log_users" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="last_format_column" property="FormatConditions(1).AppliesTo.Address" value="$M$4:$M$6" type="String" /><column name="last_format_column" property="FormatConditions(1).Type" value="6" type="Double" /><column name="last_format_column" property="FormatConditions(1).Priority" value="1" type="Double" /><column name="last_format_column" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="last_format_column" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="last_format_column" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="last_format_column" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="last_format_column" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="last_format_column" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="last_format_column" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="last_format_column" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="last_format_column" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="last_format_column" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="5287937" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_rows', N'<table name="xls25.usp_rows"><columnFormats><column name="" property="ListObjectName" value="rows" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$D$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="dummy1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="dummy1" property="Address" value="$E$4" type="String" /><column name="dummy1" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$F$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="Address" value="$G$4" type="String" /><column name="row_color" property="ColumnWidth" value="11.43" type="Double" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_color" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="Address" value="$H$4" type="String" /><column name="row_bold" property="ColumnWidth" value="11" type="Double" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="Address" value="$I$4" type="String" /><column name="row_indent" property="ColumnWidth" value="12.86" type="Double" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="row_indent" property="HorizontalAlignment" value="-4108" type="Double" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$J$4" type="String" /><column name="code" property="ColumnWidth" value="6.86" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$K$4" type="String" /><column name="name" property="ColumnWidth" value="35.57" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id" property="Address" value="$L$4" type="String" /><column name="member_id" property="ColumnWidth" value="40.29" type="Double" /><column name="member_id" property="NumberFormat" value="General" type="String" /><column name="member_id" property="Validation.Type" value="3" type="Double" /><column name="member_id" property="Validation.AlertStyle" value="1" type="Double" /><column name="member_id" property="Validation.Operator" value="1" type="Double" /><column name="member_id" property="Validation.Formula1" value="=INDIRECT(&quot;vl_xls25_xl_validation_list_form_member_id_1[name]&quot;)" type="String" /><column name="member_id" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="member_id" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="member_id" property="Validation.ShowInput" value="True" type="Boolean" /><column name="member_id" property="Validation.ShowError" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$L$35" type="String" /><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(1).Priority" value="5" type="Double" /><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$H4=1" type="String" /><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" /><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$L$35" type="String" /><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(2).Priority" value="6" type="Double" /><column name="_RowNum" property="FormatConditions(2).Formula1" value="=$G4=11" type="String" /><column name="_RowNum" property="FormatConditions(2).Font.Color" value="16777215" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.ThemeColor" value="1" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="6299648" type="Double" /><column name="_RowNum" property="FormatConditions(2).Interior.PatternThemeColor" value="-1" type="Double" /><column name="_RowNum" property="FormatConditions(3).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).AppliesTo.Address" value="$B$4:$L$35" type="String" /><column name="_RowNum" property="FormatConditions(3).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(3).Priority" value="7" type="Double" /><column name="_RowNum" property="FormatConditions(3).Formula1" value="=$G4=12" type="String" /><column name="_RowNum" property="FormatConditions(3).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).Interior.Color" value="15853276" type="Double" /><column name="_RowNum" property="FormatConditions(3).Interior.PatternThemeColor" value="-1" type="Double" /><column name="_RowNum" property="FormatConditions(3).Interior.ThemeColor" value="5" type="Double" /><column name="_RowNum" property="FormatConditions(3).Interior.TintAndShade" value="0.799981688894314" type="Double" /><column name="_RowNum" property="FormatConditions(4).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).AppliesTo.Address" value="$B$4:$L$35" type="String" /><column name="_RowNum" property="FormatConditions(4).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(4).Priority" value="8" type="Double" /><column name="_RowNum" property="FormatConditions(4).Formula1" value="=$G4=13" type="String" /><column name="_RowNum" property="FormatConditions(4).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).Interior.Color" value="15921906" type="Double" /><column name="_RowNum" property="FormatConditions(4).Interior.PatternThemeColor" value="-1" type="Double" /><column name="_RowNum" property="FormatConditions(4).Interior.ThemeColor" value="1" type="Double" /><column name="_RowNum" property="FormatConditions(4).Interior.TintAndShade" value="-0.0499893185216834" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /></columnFormats><views><view name="All Fields"><column name="" property="ListObjectName" value="rows" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="dummy1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="Form Fields"><column name="" property="ListObjectName" value="rows" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="dummy1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_rowsets', N'<table name="xls25.usp_rowsets"><columnFormats><column name="" property="ListObjectName" value="rowsets" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$D$4" type="String" /><column name="code" property="ColumnWidth" value="31.14" type="Double" /><column name="code" property="NumberFormat" value="@" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$E$4" type="String" /><column name="name" property="ColumnWidth" value="23" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="translated_name" property="Address" value="$F$4" type="String" /><column name="translated_name" property="ColumnWidth" value="27.86" type="Double" /><column name="translated_name" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="Address" value="$G$4" type="String" /><column name="sort_order" property="ColumnWidth" value="16.86" type="Double" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="" property="Tab.Color" value="10498160" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="rowsets" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="id" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="0" type="Double" /></view><view name="Data Columns"><column name="" property="ListObjectName" value="rowsets" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="SortFields(1)" property="KeyfieldName" value="sort_order" type="String" /><column name="SortFields(1)" property="SortOn" value="0" type="Double" /><column name="SortFields(1)" property="Order" value="1" type="Double" /><column name="SortFields(1)" property="DataOption" value="0" type="Double" /><column name="SortFields(2)" property="KeyfieldName" value="id" type="String" /><column name="SortFields(2)" property="SortOn" value="0" type="Double" /><column name="SortFields(2)" property="Order" value="1" type="Double" /><column name="SortFields(2)" property="DataOption" value="0" type="Double" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_run_form', N'<table name="xls25.usp_run_form"><columnFormats><column name="" property="ListObjectName" value="Sheet1_Table1" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="Address" value="$D$4" type="String" /><column name="id2" property="NumberFormat" value="General" type="String" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="Address" value="$E$4" type="String" /><column name="id3" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$F$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="Address" value="$G$4" type="String" /><column name="sort_order2" property="NumberFormat" value="General" type="String" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="Address" value="$H$4" type="String" /><column name="sort_order3" property="NumberFormat" value="General" type="String" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="Address" value="$I$4" type="String" /><column name="member_id" property="NumberFormat" value="General" type="String" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="Address" value="$J$4" type="String" /><column name="member_id2" property="NumberFormat" value="General" type="String" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="Address" value="$K$4" type="String" /><column name="member_id3" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$L$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="HorizontalAlignment" value="-4108" type="Double" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="Address" value="$M$4" type="String" /><column name="decimal_places" property="NumberFormat" value="General" type="String" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="Address" value="$N$4" type="String" /><column name="is_percent" property="NumberFormat" value="General" type="String" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="Address" value="$O$4" type="String" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_color" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="Address" value="$P$4" type="String" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="Address" value="$Q$4" type="String" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$R$4" type="String" /><column name="code" property="ColumnWidth" value="7" type="Double" /><column name="code" property="NumberFormat" value="@" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$S$4" type="String" /><column name="name" property="ColumnWidth" value="42.14" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="comment" property="Address" value="$T$4" type="String" /><column name="comment" property="NumberFormat" value="General" type="String" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="Address" value="$U$4" type="String" /><column name="code1" property="NumberFormat" value="General" type="String" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="Address" value="$V$4" type="String" /><column name="code2" property="NumberFormat" value="General" type="String" /><column name="name2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="Address" value="$W$4" type="String" /><column name="name2" property="NumberFormat" value="General" type="String" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code3" property="Address" value="$X$4" type="String" /><column name="code3" property="NumberFormat" value="General" type="String" /><column name="name3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="Address" value="$Y$4" type="String" /><column name="name3" property="NumberFormat" value="General" type="String" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="Address" value="$Z$4" type="String" /><column name="calc_type" property="NumberFormat" value="General" type="String" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="Address" value="$AA$4" type="String" /><column name="tax_rate" property="NumberFormat" value="General" type="String" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit" property="Address" value="$AB$4" type="String" /><column name="unit" property="ColumnWidth" value="6.14" type="Double" /><column name="unit" property="NumberFormat" value="General" type="String" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="Address" value="$AC$4" type="String" /><column name="data_format" property="NumberFormat" value="#,##0_ ;[Red]-#,##0 " type="String" /><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(1).Priority" value="15" type="Double" /><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$P4=1" type="String" /><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(2).Priority" value="16" type="Double" /><column name="_RowNum" property="FormatConditions(2).Formula1" value="=$O4=11" type="String" /><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).Font.Color" value="16777215" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.ThemeColor" value="1" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.TintAndShade" value="0" type="Double" /><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="6365184" type="Double" /><column name="_RowNum" property="FormatConditions(3).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(3).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(3).Priority" value="17" type="Double" /><column name="_RowNum" property="FormatConditions(3).Formula1" value="=$O4=12" type="String" /><column name="_RowNum" property="FormatConditions(3).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).Interior.Color" value="15787740" type="Double" /><column name="_RowNum" property="FormatConditions(4).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(4).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(4).Priority" value="18" type="Double" /><column name="_RowNum" property="FormatConditions(4).Formula1" value="=$O4=13" type="String" /><column name="_RowNum" property="FormatConditions(4).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).Interior.Color" value="15790320" type="Double" /><column name="_RowNum" property="FormatConditions(5).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(5).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(5).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(5).Priority" value="19" type="Double" /><column name="_RowNum" property="FormatConditions(5).Formula1" value="=AND(RIGHT(B$3,1)=&quot;''&quot;,$L4=1)" type="String" /><column name="_RowNum" property="FormatConditions(5).Interior.Color" value="14477050" type="Double" /><column name="data_format" property="FormatConditions(1).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(1).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(1).Priority" value="1" type="Double" /><column name="data_format" property="FormatConditions(1).Formula1" value="=(RIGHT(AC$3,1)=&quot;%&quot;)" type="String" /><column name="data_format" property="FormatConditions(1).NumberFormat" value="0.0%;[Red]-0.0%;" type="String" /><column name="data_format" property="FormatConditions(2).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(2).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(2).Priority" value="2" type="Double" /><column name="data_format" property="FormatConditions(2).Formula1" value="=AND($M4=0,NOT(ISBLANK($M4)),NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(2).NumberFormat" value="# ##0_ ;[Red]-# ##0\ " type="String" /><column name="data_format" property="FormatConditions(3).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(3).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(3).Priority" value="3" type="Double" /><column name="data_format" property="FormatConditions(3).Formula1" value="=AND($M4=1,NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(3).NumberFormat" value="# ##0.0_ ;[Red]-# ##0.0\ " type="String" /><column name="data_format" property="FormatConditions(4).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(4).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(4).Priority" value="4" type="Double" /><column name="data_format" property="FormatConditions(4).Formula1" value="=AND($M4=2,NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(4).NumberFormat" value="# ##0.00_ ;[Red]-# ##0.00\ " type="String" /><column name="data_format" property="FormatConditions(5).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(5).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(5).Priority" value="5" type="Double" /><column name="data_format" property="FormatConditions(5).Formula1" value="=AND($M4=3,NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(5).NumberFormat" value="# ##0.000_ ;[Red]-# ##0.000\ " type="String" /><column name="data_format" property="FormatConditions(6).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(6).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(6).Priority" value="6" type="Double" /><column name="data_format" property="FormatConditions(6).Formula1" value="=AND($M4=4,NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(6).NumberFormat" value="# ##0.0000_ ;[Red]-# ##0.0000\ " type="String" /><column name="data_format" property="FormatConditions(7).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(7).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(7).Priority" value="7" type="Double" /><column name="data_format" property="FormatConditions(7).Formula1" value="=AND($M4=0,$N4=1)" type="String" /><column name="data_format" property="FormatConditions(7).NumberFormat" value="# ##0%;[Red]-# ##0%;" type="String" /><column name="data_format" property="FormatConditions(8).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(8).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(8).Priority" value="8" type="Double" /><column name="data_format" property="FormatConditions(8).Formula1" value="=AND($M4=1,$N4=1)" type="String" /><column name="data_format" property="FormatConditions(8).NumberFormat" value="# ##0.0%;[Red]-# ##0.0%;" type="String" /><column name="data_format" property="FormatConditions(9).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(9).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(9).Priority" value="9" type="Double" /><column name="data_format" property="FormatConditions(9).Formula1" value="=AND($M4=2,$N4=1)" type="String" /><column name="data_format" property="FormatConditions(9).NumberFormat" value="# ##0.00%;[Red]-# ##0.00%;" type="String" /><column name="" property="Tab.Color" value="5287936" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="1-Dimension Report"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /></view><view name="2-Dimension Report"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /></view><view name="3-Dimension Report"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /></view><view name="Data Format"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_run_offline_form', N'<table name="xls25.usp_run_form"><columnFormats><column name="" property="ListObjectName" value="Sheet1_Table1" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="Address" value="$D$4" type="String" /><column name="id2" property="NumberFormat" value="General" type="String" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="Address" value="$E$4" type="String" /><column name="id3" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="Address" value="$F$4" type="String" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="Address" value="$G$4" type="String" /><column name="sort_order2" property="NumberFormat" value="General" type="String" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="Address" value="$H$4" type="String" /><column name="sort_order3" property="NumberFormat" value="General" type="String" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="Address" value="$I$4" type="String" /><column name="member_id" property="NumberFormat" value="General" type="String" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="Address" value="$J$4" type="String" /><column name="member_id2" property="NumberFormat" value="General" type="String" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="Address" value="$K$4" type="String" /><column name="member_id3" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="Address" value="$L$4" type="String" /><column name="is_editable" property="NumberFormat" value="General" type="String" /><column name="is_editable" property="HorizontalAlignment" value="-4108" type="Double" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="Address" value="$M$4" type="String" /><column name="decimal_places" property="NumberFormat" value="General" type="String" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="Address" value="$N$4" type="String" /><column name="is_percent" property="NumberFormat" value="General" type="String" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="Address" value="$O$4" type="String" /><column name="row_color" property="NumberFormat" value="General" type="String" /><column name="row_color" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="Address" value="$P$4" type="String" /><column name="row_bold" property="NumberFormat" value="General" type="String" /><column name="row_bold" property="HorizontalAlignment" value="-4108" type="Double" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="Address" value="$Q$4" type="String" /><column name="row_indent" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$R$4" type="String" /><column name="code" property="ColumnWidth" value="7" type="Double" /><column name="code" property="NumberFormat" value="@" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$S$4" type="String" /><column name="name" property="ColumnWidth" value="42.14" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="comment" property="Address" value="$T$4" type="String" /><column name="comment" property="NumberFormat" value="General" type="String" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="Address" value="$U$4" type="String" /><column name="code1" property="NumberFormat" value="General" type="String" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="Address" value="$V$4" type="String" /><column name="code2" property="NumberFormat" value="General" type="String" /><column name="name2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="Address" value="$W$4" type="String" /><column name="name2" property="NumberFormat" value="General" type="String" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code3" property="Address" value="$X$4" type="String" /><column name="code3" property="NumberFormat" value="General" type="String" /><column name="name3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="Address" value="$Y$4" type="String" /><column name="name3" property="NumberFormat" value="General" type="String" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="Address" value="$Z$4" type="String" /><column name="calc_type" property="NumberFormat" value="General" type="String" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="Address" value="$AA$4" type="String" /><column name="tax_rate" property="NumberFormat" value="General" type="String" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit" property="Address" value="$AB$4" type="String" /><column name="unit" property="ColumnWidth" value="6.14" type="Double" /><column name="unit" property="NumberFormat" value="General" type="String" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="Address" value="$AC$4" type="String" /><column name="data_format" property="NumberFormat" value="#,##0_ ;[Red]-#,##0 " type="String" /><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(1).Priority" value="15" type="Double" /><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$P4=1" type="String" /><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(2).Priority" value="16" type="Double" /><column name="_RowNum" property="FormatConditions(2).Formula1" value="=$O4=11" type="String" /><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(2).Font.Color" value="16777215" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.ThemeColor" value="1" type="Double" /><column name="_RowNum" property="FormatConditions(2).Font.TintAndShade" value="0" type="Double" /><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="6365184" type="Double" /><column name="_RowNum" property="FormatConditions(3).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(3).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(3).Priority" value="17" type="Double" /><column name="_RowNum" property="FormatConditions(3).Formula1" value="=$O4=12" type="String" /><column name="_RowNum" property="FormatConditions(3).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(3).Interior.Color" value="15787740" type="Double" /><column name="_RowNum" property="FormatConditions(4).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(4).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(4).Priority" value="18" type="Double" /><column name="_RowNum" property="FormatConditions(4).Formula1" value="=$O4=13" type="String" /><column name="_RowNum" property="FormatConditions(4).Font.Bold" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(4).Interior.Color" value="15790320" type="Double" /><column name="_RowNum" property="FormatConditions(5).AppliesToTable" value="True" type="Boolean" /><column name="_RowNum" property="FormatConditions(5).AppliesTo.Address" value="$B$4:$AC$26" type="String" /><column name="_RowNum" property="FormatConditions(5).Type" value="2" type="Double" /><column name="_RowNum" property="FormatConditions(5).Priority" value="19" type="Double" /><column name="_RowNum" property="FormatConditions(5).Formula1" value="=AND(RIGHT(B$3,1)=&quot;''&quot;,$L4=1)" type="String" /><column name="_RowNum" property="FormatConditions(5).Interior.Color" value="14477050" type="Double" /><column name="data_format" property="FormatConditions(1).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(1).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(1).Priority" value="1" type="Double" /><column name="data_format" property="FormatConditions(1).Formula1" value="=(RIGHT(AC$3,1)=&quot;%&quot;)" type="String" /><column name="data_format" property="FormatConditions(1).NumberFormat" value="0.0%;[Red]-0.0%;" type="String" /><column name="data_format" property="FormatConditions(2).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(2).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(2).Priority" value="2" type="Double" /><column name="data_format" property="FormatConditions(2).Formula1" value="=AND($M4=0,NOT(ISBLANK($M4)),NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(2).NumberFormat" value="# ##0_ ;[Red]-# ##0\ " type="String" /><column name="data_format" property="FormatConditions(3).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(3).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(3).Priority" value="3" type="Double" /><column name="data_format" property="FormatConditions(3).Formula1" value="=AND($M4=1,NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(3).NumberFormat" value="# ##0.0_ ;[Red]-# ##0.0\ " type="String" /><column name="data_format" property="FormatConditions(4).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(4).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(4).Priority" value="4" type="Double" /><column name="data_format" property="FormatConditions(4).Formula1" value="=AND($M4=2,NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(4).NumberFormat" value="# ##0.00_ ;[Red]-# ##0.00\ " type="String" /><column name="data_format" property="FormatConditions(5).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(5).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(5).Priority" value="5" type="Double" /><column name="data_format" property="FormatConditions(5).Formula1" value="=AND($M4=3,NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(5).NumberFormat" value="# ##0.000_ ;[Red]-# ##0.000\ " type="String" /><column name="data_format" property="FormatConditions(6).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(6).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(6).Priority" value="6" type="Double" /><column name="data_format" property="FormatConditions(6).Formula1" value="=AND($M4=4,NOT($N4=1))" type="String" /><column name="data_format" property="FormatConditions(6).NumberFormat" value="# ##0.0000_ ;[Red]-# ##0.0000\ " type="String" /><column name="data_format" property="FormatConditions(7).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(7).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(7).Priority" value="7" type="Double" /><column name="data_format" property="FormatConditions(7).Formula1" value="=AND($M4=0,$N4=1)" type="String" /><column name="data_format" property="FormatConditions(7).NumberFormat" value="# ##0%;[Red]-# ##0%;" type="String" /><column name="data_format" property="FormatConditions(8).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(8).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(8).Priority" value="8" type="Double" /><column name="data_format" property="FormatConditions(8).Formula1" value="=AND($M4=1,$N4=1)" type="String" /><column name="data_format" property="FormatConditions(8).NumberFormat" value="# ##0.0%;[Red]-# ##0.0%;" type="String" /><column name="data_format" property="FormatConditions(9).AppliesTo.Address" value="$AC$4:$AC$26" type="String" /><column name="data_format" property="FormatConditions(9).Type" value="2" type="Double" /><column name="data_format" property="FormatConditions(9).Priority" value="9" type="Double" /><column name="data_format" property="FormatConditions(9).Formula1" value="=AND($M4=2,$N4=1)" type="String" /><column name="data_format" property="FormatConditions(9).NumberFormat" value="# ##0.00%;[Red]-# ##0.00%;" type="String" /><column name="" property="Tab.Color" value="5287936" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats><views><view name="All Columns"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="False" type="Boolean" /></view><view name="1-Dimension Report"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /></view><view name="2-Dimension Report"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /></view><view name="3-Dimension Report"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="True" type="Boolean" /></view><view name="Data Format"><column name="" property="ListObjectName" value="form" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="sort_order3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_editable" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="decimal_places" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="is_percent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_color" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_bold" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="row_indent" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="comment" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code1" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name2" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="code3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="name3" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="calc_type" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="tax_rate" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="unit" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="data_format" property="EntireColumn.Hidden" value="False" type="Boolean" /></view></views></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_tax_rates', N'<table name="xls25.usp_tax_rates"><columnFormats><column name="" property="ListObjectName" value="tax_rates" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="ColumnWidth" value="4.29" type="Double" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="id" property="Validation.Type" value="1" type="Double" /><column name="id" property="Validation.Operator" value="1" type="Double" /><column name="id" property="Validation.Formula1" value="0" type="String" /><column name="id" property="Validation.Formula2" value="255" type="String" /><column name="id" property="Validation.AlertStyle" value="1" type="Double" /><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="id" property="Validation.ShowInput" value="True" type="Boolean" /><column name="id" property="Validation.ShowError" value="True" type="Boolean" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$D$4" type="String" /><column name="code" property="ColumnWidth" value="13.57" type="Double" /><column name="code" property="NumberFormat" value="@" type="String" /><column name="code" property="Validation.Type" value="6" type="Double" /><column name="code" property="Validation.Operator" value="8" type="Double" /><column name="code" property="Validation.Formula1" value="50" type="String" /><column name="code" property="Validation.AlertStyle" value="1" type="Double" /><column name="code" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="code" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="code" property="Validation.ShowInput" value="True" type="Boolean" /><column name="code" property="Validation.ShowError" value="True" type="Boolean" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$E$4" type="String" /><column name="name" property="ColumnWidth" value="13.57" type="Double" /><column name="name" property="NumberFormat" value="@" type="String" /><column name="name" property="Validation.Type" value="6" type="Double" /><column name="name" property="Validation.Operator" value="8" type="Double" /><column name="name" property="Validation.Formula1" value="50" type="String" /><column name="name" property="Validation.AlertStyle" value="1" type="Double" /><column name="name" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="name" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="name" property="Validation.ShowInput" value="True" type="Boolean" /><column name="name" property="Validation.ShowError" value="True" type="Boolean" /><column name="translated_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="translated_name" property="Address" value="$F$4" type="String" /><column name="translated_name" property="ColumnWidth" value="17.71" type="Double" /><column name="translated_name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="HorizontalAlignment" value="-4108" type="Double" /><column name="translated_name" property="Font.Size" value="10" type="Double" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="Address" value="$G$4" type="String" /><column name="sort_order" property="ColumnWidth" value="11.86" type="Double" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="sort_order" property="Validation.Type" value="1" type="Double" /><column name="sort_order" property="Validation.Operator" value="1" type="Double" /><column name="sort_order" property="Validation.Formula1" value="0" type="String" /><column name="sort_order" property="Validation.Formula2" value="255" type="String" /><column name="sort_order" property="Validation.AlertStyle" value="1" type="Double" /><column name="sort_order" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="sort_order" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="sort_order" property="Validation.ShowInput" value="True" type="Boolean" /><column name="sort_order" property="Validation.ShowError" value="True" type="Boolean" /><column name="factor" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor" property="Address" value="$H$4" type="String" /><column name="factor" property="ColumnWidth" value="12" type="Double" /><column name="factor" property="NumberFormat" value="General" type="String" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="Address" value="$I$4" type="String" /><column name="is_active" property="ColumnWidth" value="10.29" type="Double" /><column name="is_active" property="NumberFormat" value="General" type="String" /><column name="is_active" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_active" property="Font.Size" value="10" type="Double" /><column name="is_active" property="FormatConditions(1).ColumnsCount" value="2" type="Double" /><column name="is_active" property="FormatConditions(1).AppliesTo.Address" value="$I$4:$J$5" type="String" /><column name="is_active" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_active" property="FormatConditions(1).Priority" value="2" type="Double" /><column name="is_active" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_active" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_active" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="1" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /><column name="" property="PageSetup.PaperSize" value="1" type="Double" /></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_translations', N'<table name="xls25.usp_translations"><columnFormats><column name="" property="ListObjectName" value="translations" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="table_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="table_id" property="Address" value="$C$4" type="String" /><column name="table_id" property="NumberFormat" value="General" type="String" /><column name="member_id" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="member_id" property="Address" value="$D$4" type="String" /><column name="member_id" property="NumberFormat" value="General" type="String" /><column name="table_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="table_name" property="Address" value="$E$4" type="String" /><column name="table_name" property="ColumnWidth" value="21.43" type="Double" /><column name="table_name" property="NumberFormat" value="General" type="String" /><column name="is_translatable" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_translatable" property="Address" value="$F$4" type="String" /><column name="is_translatable" property="ColumnWidth" value="2.86" type="Double" /><column name="is_translatable" property="NumberFormat" value="General" type="String" /><column name="is_translatable" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_translatable" property="Font.Size" value="10" type="Double" /><column name="is_complete" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_complete" property="Address" value="$G$4" type="String" /><column name="is_complete" property="ColumnWidth" value="2.86" type="Double" /><column name="is_complete" property="NumberFormat" value="General" type="String" /><column name="is_complete" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_complete" property="Font.Size" value="10" type="Double" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$H$4" type="String" /><column name="name" property="ColumnWidth" value="50.57" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="fr" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="fr" property="Address" value="$I$4" type="String" /><column name="fr" property="ColumnWidth" value="42.43" type="Double" /><column name="fr" property="NumberFormat" value="General" type="String" /><column name="it" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="it" property="Address" value="$J$4" type="String" /><column name="it" property="ColumnWidth" value="42.43" type="Double" /><column name="it" property="NumberFormat" value="General" type="String" /><column name="es" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="es" property="Address" value="$K$4" type="String" /><column name="es" property="ColumnWidth" value="48.14" type="Double" /><column name="es" property="NumberFormat" value="General" type="String" /><column name="de" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="de" property="Address" value="$L$4" type="String" /><column name="de" property="ColumnWidth" value="42.43" type="Double" /><column name="de" property="NumberFormat" value="General" type="String" /><column name="ru" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ru" property="Address" value="$M$4" type="String" /><column name="ru" property="ColumnWidth" value="42.43" type="Double" /><column name="ru" property="NumberFormat" value="General" type="String" /><column name="is_translatable" property="FormatConditions(1).AppliesTo.Address" value="$F$4:$F$416" type="String" /><column name="is_translatable" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_translatable" property="FormatConditions(1).Priority" value="102151" type="Double" /><column name="is_translatable" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_translatable" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_translatable" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_translatable" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_translatable" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_translatable" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_translatable" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_translatable" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_translatable" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_translatable" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="is_complete" property="FormatConditions(1).AppliesTo.Address" value="$G$4:$G$416" type="String" /><column name="is_complete" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_complete" property="FormatConditions(1).Priority" value="102152" type="Double" /><column name="is_complete" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_complete" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="5287936" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_translations_common', N'<table name="xls25.usp_translations_common"><columnFormats><column name="" property="ListObjectName" value="translations_common" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="TABLE_SCHEMA" property="Address" value="$C$4" type="String" /><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String" /><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="TABLE_NAME" property="Address" value="$D$4" type="String" /><column name="TABLE_NAME" property="ColumnWidth" value="34.86" type="Double" /><column name="TABLE_NAME" property="NumberFormat" value="General" type="String" /><column name="is_complete" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_complete" property="Address" value="$E$4" type="String" /><column name="is_complete" property="ColumnWidth" value="2.86" type="Double" /><column name="is_complete" property="NumberFormat" value="General" type="String" /><column name="is_complete" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_complete" property="Font.Size" value="10" type="Double" /><column name="COLUMN" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="COLUMN" property="Address" value="$F$4" type="String" /><column name="COLUMN" property="ColumnWidth" value="25.57" type="Double" /><column name="COLUMN" property="NumberFormat" value="General" type="String" /><column name="en" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="en" property="Address" value="$G$4" type="String" /><column name="en" property="ColumnWidth" value="27.86" type="Double" /><column name="en" property="NumberFormat" value="General" type="String" /><column name="fr" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="fr" property="Address" value="$H$4" type="String" /><column name="fr" property="ColumnWidth" value="27.86" type="Double" /><column name="fr" property="NumberFormat" value="General" type="String" /><column name="it" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="it" property="Address" value="$I$4" type="String" /><column name="it" property="ColumnWidth" value="27.86" type="Double" /><column name="it" property="NumberFormat" value="General" type="String" /><column name="es" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="es" property="Address" value="$J$4" type="String" /><column name="es" property="ColumnWidth" value="27.86" type="Double" /><column name="es" property="NumberFormat" value="General" type="String" /><column name="de" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="de" property="Address" value="$K$4" type="String" /><column name="de" property="ColumnWidth" value="27.86" type="Double" /><column name="de" property="NumberFormat" value="General" type="String" /><column name="ru" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ru" property="Address" value="$L$4" type="String" /><column name="ru" property="ColumnWidth" value="27.86" type="Double" /><column name="ru" property="NumberFormat" value="General" type="String" /><column name="is_complete" property="FormatConditions(1).AppliesTo.Address" value="$E$4:$E$179" type="String" /><column name="is_complete" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_complete" property="FormatConditions(1).Priority" value="167" type="Double" /><column name="is_complete" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_complete" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="6299649" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_translations_xls', N'<table name="xls25.usp_translations_xls"><columnFormats><column name="" property="ListObjectName" value="translations_xls" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="TABLE_SCHEMA" property="Address" value="$C$4" type="String" /><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double" /><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String" /><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="TABLE_NAME" property="Address" value="$D$4" type="String" /><column name="TABLE_NAME" property="ColumnWidth" value="30.57" type="Double" /><column name="TABLE_NAME" property="NumberFormat" value="General" type="String" /><column name="is_complete" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_complete" property="Address" value="$E$4" type="String" /><column name="is_complete" property="ColumnWidth" value="2.86" type="Double" /><column name="is_complete" property="NumberFormat" value="General" type="String" /><column name="is_complete" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_complete" property="Font.Size" value="10" type="Double" /><column name="COLUMN" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="COLUMN" property="Address" value="$F$4" type="String" /><column name="COLUMN" property="ColumnWidth" value="27.86" type="Double" /><column name="COLUMN" property="NumberFormat" value="General" type="String" /><column name="en" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="en" property="Address" value="$G$4" type="String" /><column name="en" property="ColumnWidth" value="27.86" type="Double" /><column name="en" property="NumberFormat" value="General" type="String" /><column name="fr" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="fr" property="Address" value="$H$4" type="String" /><column name="fr" property="ColumnWidth" value="27.86" type="Double" /><column name="fr" property="NumberFormat" value="General" type="String" /><column name="it" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="it" property="Address" value="$I$4" type="String" /><column name="it" property="ColumnWidth" value="27.86" type="Double" /><column name="it" property="NumberFormat" value="General" type="String" /><column name="es" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="es" property="Address" value="$J$4" type="String" /><column name="es" property="ColumnWidth" value="27.86" type="Double" /><column name="es" property="NumberFormat" value="General" type="String" /><column name="de" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="de" property="Address" value="$K$4" type="String" /><column name="de" property="ColumnWidth" value="27.86" type="Double" /><column name="de" property="NumberFormat" value="General" type="String" /><column name="ru" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ru" property="Address" value="$L$4" type="String" /><column name="ru" property="ColumnWidth" value="27.86" type="Double" /><column name="ru" property="NumberFormat" value="General" type="String" /><column name="is_complete" property="FormatConditions(1).AppliesTo.Address" value="$E$4" type="String" /><column name="is_complete" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_complete" property="FormatConditions(1).Priority" value="173" type="Double" /><column name="is_complete" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_complete" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_complete" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="" property="Tab.Color" value="6299649" type="Double" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls25', N'usp_units', N'<table name="xls25.usp_units"><columnFormats><column name="" property="ListObjectName" value="units" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="id" property="Address" value="$C$4" type="String" /><column name="id" property="ColumnWidth" value="4.29" type="Double" /><column name="id" property="NumberFormat" value="General" type="String" /><column name="code" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="code" property="Address" value="$D$4" type="String" /><column name="code" property="ColumnWidth" value="13.57" type="Double" /><column name="code" property="NumberFormat" value="General" type="String" /><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="name" property="Address" value="$E$4" type="String" /><column name="name" property="ColumnWidth" value="13.57" type="Double" /><column name="name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="translated_name" property="Address" value="$F$4" type="String" /><column name="translated_name" property="ColumnWidth" value="17.86" type="Double" /><column name="translated_name" property="NumberFormat" value="General" type="String" /><column name="translated_name" property="HorizontalAlignment" value="-4108" type="Double" /><column name="translated_name" property="Font.Size" value="10" type="Double" /><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="sort_order" property="Address" value="$G$4" type="String" /><column name="sort_order" property="ColumnWidth" value="11.86" type="Double" /><column name="sort_order" property="NumberFormat" value="General" type="String" /><column name="factor" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="factor" property="Address" value="$H$4" type="String" /><column name="factor" property="ColumnWidth" value="8" type="Double" /><column name="factor" property="NumberFormat" value="General" type="String" /><column name="is_currency" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_currency" property="Address" value="$I$4" type="String" /><column name="is_currency" property="ColumnWidth" value="12.29" type="Double" /><column name="is_currency" property="NumberFormat" value="General" type="String" /><column name="is_currency" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_currency" property="Font.Size" value="10" type="Double" /><column name="is_functional_currency" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_functional_currency" property="Address" value="$J$4" type="String" /><column name="is_functional_currency" property="ColumnWidth" value="23" type="Double" /><column name="is_functional_currency" property="NumberFormat" value="General" type="String" /><column name="is_functional_currency" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_functional_currency" property="Font.Size" value="10" type="Double" /><column name="is_active" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="is_active" property="Address" value="$K$4" type="String" /><column name="is_active" property="ColumnWidth" value="10" type="Double" /><column name="is_active" property="NumberFormat" value="General" type="String" /><column name="is_active" property="HorizontalAlignment" value="-4108" type="Double" /><column name="is_active" property="Font.Size" value="10" type="Double" /><column name="is_currency" property="FormatConditions(1).AppliesTo.Address" value="$I$4:$I$7" type="String" /><column name="is_currency" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_currency" property="FormatConditions(1).Priority" value="1" type="Double" /><column name="is_currency" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_currency" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_currency" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_currency" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_currency" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_currency" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_currency" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_currency" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_currency" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_currency" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).AppliesTo.Address" value="$J$4:$J$7" type="String" /><column name="is_functional_currency" property="FormatConditions(1).Type" value="6" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).Priority" value="3" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean" /><column name="is_functional_currency" property="FormatConditions(1).IconSet.ID" value="8" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double" /><column name="is_functional_currency" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double" /><column name="is_functional_curre