-- =============================================
-- SaveToDB Framework for Microsoft SQL Server
-- Version 10.8, January 9, 2023
--
-- Copyright 2011-2023 Gartle LLC
--
-- License: MIT
-- =============================================

SET NOCOUNT ON
GO

CREATE SCHEMA xls;
GO

CREATE TABLE xls.columns (
    ID int IDENTITY(1,1) NOT NULL
    , TABLE_SCHEMA nvarchar(128) NOT NULL
    , TABLE_NAME nvarchar(128) NOT NULL
    , COLUMN_NAME nvarchar(128) NOT NULL
    , ORDINAL_POSITION int NOT NULL
    , IS_PRIMARY_KEY bit NULL
    , IS_NULLABLE bit NULL
    , IS_IDENTITY bit NULL
    , IS_COMPUTED bit NULL
    , COLUMN_DEFAULT nvarchar(256) NULL
    , DATA_TYPE nvarchar(128) NULL
    , CHARACTER_MAXIMUM_LENGTH int NULL
    , PRECISION tinyint NULL
    , SCALE tinyint NULL
    , CONSTRAINT PK_columns PRIMARY KEY (ID)
    , CONSTRAINT IX_columns UNIQUE (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME)
);
GO

CREATE TABLE xls.formats (
    ID int IDENTITY(1,1) NOT NULL
    , TABLE_SCHEMA nvarchar(128) NOT NULL
    , TABLE_NAME nvarchar(128) NOT NULL
    , TABLE_EXCEL_FORMAT_XML xml NULL
    , APP nvarchar(50) NULL
    , CONSTRAINT PK_formats PRIMARY KEY (ID)
    , CONSTRAINT IX_formats UNIQUE (TABLE_SCHEMA, TABLE_NAME, APP)
);
GO

CREATE TABLE xls.handlers (
    ID int IDENTITY(1,1) NOT NULL
    , TABLE_SCHEMA nvarchar(20) NULL
    , TABLE_NAME nvarchar(128) NOT NULL
    , COLUMN_NAME nvarchar(128) NULL
    , EVENT_NAME varchar(25) NULL
    , HANDLER_SCHEMA nvarchar(20) NULL
    , HANDLER_NAME nvarchar(128) NULL
    , HANDLER_TYPE varchar(25) NULL
    , HANDLER_CODE nvarchar(max) NULL
    , TARGET_WORKSHEET nvarchar(128) NULL
    , MENU_ORDER int NULL
    , EDIT_PARAMETERS bit NULL
    , CONSTRAINT PK_handlers PRIMARY KEY (ID)
    , CONSTRAINT IX_handlers UNIQUE (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME)
);
GO

CREATE TABLE xls.objects (
    ID int IDENTITY(1,1) NOT NULL
    , TABLE_SCHEMA nvarchar(128) NOT NULL
    , TABLE_NAME nvarchar(128) NOT NULL
    , TABLE_TYPE nvarchar(128) NOT NULL
    , TABLE_CODE nvarchar(max) NULL
    , INSERT_OBJECT nvarchar(max) NULL
    , UPDATE_OBJECT nvarchar(max) NULL
    , DELETE_OBJECT nvarchar(max) NULL
    , CONSTRAINT PK_objects PRIMARY KEY (ID)
    , CONSTRAINT IX_objects UNIQUE (TABLE_SCHEMA, TABLE_NAME)
);
GO

CREATE TABLE xls.translations (
    ID int IDENTITY(1,1) NOT NULL
    , TABLE_SCHEMA nvarchar(128) NULL
    , TABLE_NAME nvarchar(128) NULL
    , COLUMN_NAME nvarchar(128) NULL
    , LANGUAGE_NAME varchar(10) NOT NULL
    , TRANSLATED_NAME nvarchar(128) NULL
    , TRANSLATED_DESC nvarchar(1024) NULL
    , TRANSLATED_COMMENT nvarchar(2000) NULL
    , CONSTRAINT PK_translations PRIMARY KEY (ID)
    , CONSTRAINT IX_translations UNIQUE (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME)
);
GO

CREATE TABLE xls.workbooks (
    ID int IDENTITY(1,1) NOT NULL
    , NAME nvarchar(128) NOT NULL
    , TEMPLATE nvarchar(255) NULL
    , DEFINITION nvarchar(max) NOT NULL
    , TABLE_SCHEMA nvarchar(128) NULL
    , CONSTRAINT PK_workbooks PRIMARY KEY (ID)
    , CONSTRAINT IX_workbooks UNIQUE (NAME)
);
GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     10.0, 2022-07-05
-- Description: SaveToDB Query List
-- =============================================

CREATE VIEW [xls].[queries]
AS
SELECT
    s.name AS TABLE_SCHEMA
    , o.name AS TABLE_NAME
    , CASE o.[type] WHEN 'U' THEN 'BASE TABLE' WHEN 'V' THEN 'VIEW' WHEN 'P' THEN 'PROCEDURE' WHEN 'IF' THEN 'FUNCTION' ELSE o.type_desc COLLATE Latin1_General_CI_AS END AS TABLE_TYPE
    , CAST(NULL AS nvarchar(max)) AS TABLE_CODE
    , CAST(NULL AS nvarchar(max)) AS INSERT_PROCEDURE
    , CAST(NULL AS nvarchar(max)) AS UPDATE_PROCEDURE
    , CAST(NULL AS nvarchar(max)) AS DELETE_PROCEDURE
    , CAST(NULL AS nvarchar(50)) AS PROCEDURE_TYPE
FROM
    sys.objects o
    INNER JOIN sys.schemas s ON s.[schema_id] = o.[schema_id]
WHERE
    o.[type] IN ('U', 'V', 'P', 'IF')
    AND (HAS_PERMS_BY_NAME(DB_NAME() + '.' + s.name + '.' + o.name, 'OBJECT', 'SELECT') = 1
        OR HAS_PERMS_BY_NAME(DB_NAME() + '.' + s.name + '.' + o.name, 'OBJECT', 'EXECUTE') = 1)
    AND NOT s.name IN ('sys', 'xls', 'etl01', 'dbo01', 'xls01', 'savetodb_dev', 'savetodb_xls', 'savetodb_etl', 'SAVETODB_DEV', 'SAVETODB_XLS', 'SAVETODB_ETL')
    AND NOT (s.name = 'dbo' AND (
           o.name LIKE 'sp_%'
        OR o.name LIKE 'fn_%'
        OR o.name LIKE 'sys%'
        ))
    AND NOT o.name LIKE 'xl_%'
    AND NOT (o.[type] = 'V' AND (
           o.name LIKE 'viewQueryList%'
        OR o.name LIKE 'viewParameterValues%'
        OR o.name LIKE 'viewValidationList%'
        OR o.name LIKE 'view_query_list%'
        OR o.name LIKE 'view_xl_%'
        OR o.name LIKE 'xl_%'
        ))
    AND NOT (o.[type] = 'P' AND (
           o.name LIKE '%_insert'
        OR o.name LIKE '%_update'
        OR o.name LIKE '%_delete'
        OR o.name LIKE '%_merge'
        OR o.name LIKE '%_change'
        OR o.name LIKE 'uspExcelEvent%'
        OR o.name LIKE 'uspParameterValues%'
        OR o.name LIKE 'uspValidationList%'
        OR o.name LIKE 'uspAdd%'
        OR o.name LIKE 'uspSet%'
        OR o.name LIKE 'uspInsert%'
        OR o.name LIKE 'uspUpdate%'
        OR o.name LIKE 'uspDelete%'
        OR o.name LIKE 'usp_insert_%'
        OR o.name LIKE 'usp_update_%'
        OR o.name LIKE 'usp_delete_%'
        OR o.name LIKE 'Add%'
        OR o.name LIKE 'Set%'
        OR o.name LIKE 'Insert%'
        OR o.name LIKE 'Update%'
        OR o.name LIKE 'Delete%'
        OR o.name LIKE 'usp_xl_%'
        OR o.name LIKE 'usp_import_%'
        OR o.name LIKE 'usp_export_%'
        ))
    AND NOT (o.[type] = 'IF' AND (
           o.name LIKE 'Has%'
        OR o.name LIKE 'ufnExcelEvent%'
        OR o.name LIKE 'ufnParameterValues%'
        OR o.name LIKE 'ufn_xl_%'
        ))
UNION ALL
SELECT
    o.TABLE_SCHEMA
    , o.TABLE_NAME
    , o.TABLE_TYPE COLLATE Latin1_General_CI_AS
    , o.TABLE_CODE
    , o.INSERT_OBJECT AS INSERT_PROCEDURE
    , o.UPDATE_OBJECT AS UPDATE_PROCEDURE
    , o.DELETE_OBJECT AS DELETE_PROCEDURE
    , CAST(NULL AS nvarchar(50)) AS PROCEDURE_TYPE
FROM
    xls.[objects] o
WHERE
    o.TABLE_TYPE IN ('CODE', 'HTTP', 'TEXT')
    AND o.TABLE_SCHEMA IS NOT NULL
    AND o.TABLE_NAME IS NOT NULL
    AND o.TABLE_CODE IS NOT NULL
    AND NOT o.TABLE_NAME LIKE 'xl_%'


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     10.0, 2022-07-05
-- Description: The view select users
-- =============================================

CREATE VIEW [xls].[users]
AS
SELECT
    m.name AS [user]
    , r.name AS [role]
FROM
    sys.database_principals m
    LEFT JOIN sys.database_role_members rm ON rm.member_principal_id = m.principal_id
    LEFT JOIN sys.database_principals r ON r.principal_id = rm.role_principal_id
WHERE
    m.[type] IN ('S', 'U', 'R')
    AND m.is_fixed_role = 0
    AND NOT m.name IN ('dbo', 'sys', 'guest', 'public', 'INFORMATION_SCHEMA', 'xls_users', 'xls_developers', 'xls_formats', 'xls_admins', 'doc_readers', 'doc_writers', 'log_app', 'log_admins', 'log_administrators', 'log_users')
    AND (r.name IN ('xls_users', 'xls_developers', 'xls_formats') OR r.name IS NULL AND m.type IN ('S', 'U'))


GO

-- =============================================
-- Author:      Gartle LLC
-- Release:     10.4, 2022-10-13
-- Description: The procedure sets permissions of the SaveToDB Framework roles
-- =============================================

CREATE PROCEDURE [xls].[xl_actions_set_role_permissions]
    @confirm bit = 0
AS
BEGIN

SET NOCOUNT ON;

IF COALESCE(@confirm, 0) = 0 RETURN;

REVOKE SELECT, EXECUTE, VIEW DEFINITION ON SCHEMA::xls TO xls_users;

GRANT SELECT, VIEW DEFINITION ON xls.columns        TO xls_users;
GRANT SELECT, VIEW DEFINITION ON xls.formats        TO xls_users;
GRANT SELECT, VIEW DEFINITION ON xls.handlers       TO xls_users;
GRANT SELECT, VIEW DEFINITION ON xls.objects        TO xls_users;
GRANT SELECT, VIEW DEFINITION ON xls.translations   TO xls_users;
GRANT SELECT, VIEW DEFINITION ON xls.workbooks      TO xls_users;
GRANT SELECT, VIEW DEFINITION ON xls.queries        TO xls_users;

GRANT SELECT, INSERT, UPDATE, DELETE, VIEW DEFINITION ON xls.formats TO xls_formats;

GRANT SELECT, EXECUTE, INSERT, UPDATE, DELETE, VIEW DEFINITION ON SCHEMA::xls   TO xls_developers;

END


GO

CREATE ROLE xls_developers;
GO
CREATE ROLE xls_users;
GO
CREATE ROLE xls_formats;
GO

EXEC xls.xl_actions_set_role_permissions 1;
GO

INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'columns', NULL, N'Actions', N'xls', N'Developer Guide', N'HTTP', N'https://www.savetodb.com/dev-guide/xls-columns.htm', NULL, 13, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'formats', NULL, N'Actions', N'xls', N'Developer Guide', N'HTTP', N'https://www.savetodb.com/dev-guide/xls-formats.htm', NULL, 13, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'handlers', NULL, N'Actions', N'xls', N'Developer Guide', N'HTTP', N'https://www.savetodb.com/dev-guide/xls-handlers.htm', NULL, 13, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'objects', NULL, N'Actions', N'xls', N'Developer Guide', N'HTTP', N'https://www.savetodb.com/dev-guide/xls-objects.htm', NULL, 13, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'queries', NULL, N'Actions', N'xls', N'Developer Guide', N'HTTP', N'https://www.savetodb.com/dev-guide/xls-queries.htm', NULL, 13, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'translations', NULL, N'Actions', N'xls', N'Developer Guide', N'HTTP', N'https://www.savetodb.com/dev-guide/xls-translations.htm', NULL, 13, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'users', NULL, N'Actions', N'xls', N'Developer Guide', N'HTTP', N'https://www.savetodb.com/dev-guide/xls-users.htm', NULL, 13, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'workbooks', NULL, N'Actions', N'xls', N'Developer Guide', N'HTTP', N'https://www.savetodb.com/dev-guide/xls-workbooks.htm', NULL, 13, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'users', NULL, N'ContextMenu', N'xls', N'Add to xls_users', N'CODE', N'ALTER ROLE xls_users ADD MEMBER @user', N'_Reload', 31, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'users', NULL, N'ContextMenu', N'xls', N'Add to xls_formats', N'CODE', N'ALTER ROLE xls_formats ADD MEMBER @user', N'_Reload', 32, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'users', NULL, N'ContextMenu', N'xls', N'Add to xls_developers', N'CODE', N'ALTER ROLE xls_developers ADD MEMBER @user', N'_Reload', 33, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'users', NULL, N'ContextMenu', N'xls', N'MenuSeparator40', N'MENUSEPARATOR', NULL, NULL, 40, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'users', NULL, N'ContextMenu', N'xls', N'Remove from xls_users', N'CODE', N'ALTER ROLE xls_users DROP MEMBER @user', N'_Reload', 41, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'users', NULL, N'ContextMenu', N'xls', N'Remove from xls_formats', N'CODE', N'ALTER ROLE xls_formats DROP MEMBER @user', N'_Reload', 42, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'users', NULL, N'ContextMenu', N'xls', N'Remove from xls_developers', N'CODE', N'ALTER ROLE xls_developers DROP MEMBER @user', N'_Reload', 43, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'handlers', N'HANDLER_CODE', N'DoNotConvertFormulas', NULL, NULL, N'ATTRIBUTE', NULL, NULL, NULL, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'savetodb_framework', N'version', N'Information', NULL, NULL, N'ATTRIBUTE', N'10.8', NULL, NULL, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'handlers', N'EVENT_NAME', N'ValidationList', NULL, NULL, N'VALUES', N'Actions, AddHyperlinks, AddStateColumn, Authentication, BitColumn, Change, ContextMenu, ConvertFormulas, DataTypeBit, DataTypeBoolean, DataTypeDate, DataTypeDateTime, DataTypeDateTimeOffset, DataTypeDouble, DataTypeInt, DataTypeGuid, DataTypeString, DataTypeTime, DataTypeTimeSpan, DefaultListObject, DefaultValue, DependsOn, DoNotAddChangeHandler, DoNotAddDependsOn, DoNotAddManyToMany, DoNotAddValidation, DoNotChange, DoNotConvertFormulas, DoNotKeepComments, DoNotKeepFormulas, DoNotSave, DoNotSelect, DoNotSort, DoNotTranslate, DoubleClick, DynamicColumns, Format, Formula, FormulaValue, Information, JsonForm, KeepFormulas, KeepComments, License, LoadFormat, ManyToMany, ParameterValues, ProtectRows, RegEx, SaveFormat, SaveWithoutTransaction, SelectionChange, SelectionList, SelectPeriod, SyncParameter, UpdateChangedCellsOnly, UpdateEntireRow, ValidationList', NULL, NULL, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'handlers', N'HANDLER_TYPE', N'ValidationList', NULL, NULL, N'VALUES', N'TABLE, VIEW, PROCEDURE, FUNCTION, CODE, HTTP, TEXT, MACRO, CMD, VALUES, RANGE, REFRESH, MENUSEPARATOR, PDF, REPORT, SHOWSHEETS, HIDESHEETS, SELECTSHEET, ATTRIBUTE', NULL, NULL, NULL);
INSERT INTO xls.handlers (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES (N'xls', N'objects', N'TABLE_TYPE', N'ValidationList', NULL, NULL, N'VALUES', N'TABLE, VIEW, PROCEDURE, FUNCTION, CODE, HTTP, TEXT, HIDDEN', NULL, NULL, NULL);
GO

INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls', N'columns', N'<table name="xls.columns"><columnFormats><column name="" property="ListObjectName" value="columns" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="ID" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="ColumnWidth" value="4.43" type="Double"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="Validation.Type" value="1" type="Double"/><column name="ID" property="Validation.Operator" value="1" type="Double"/><column name="ID" property="Validation.Formula1" value="-2147483648" type="String"/><column name="ID" property="Validation.Formula2" value="2147483647" type="String"/><column name="ID" property="Validation.AlertStyle" value="2" type="Double"/><column name="ID" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="ID" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="ID" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="ID" property="Validation.ErrorMessage" value="The column requires values of the int datatype." type="String"/><column name="ID" property="Validation.ShowInput" value="True" type="Boolean"/><column name="ID" property="Validation.ShowError" value="True" type="Boolean"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="Validation.Type" value="6" type="Double"/><column name="TABLE_SCHEMA" property="Validation.Operator" value="8" type="Double"/><column name="TABLE_SCHEMA" property="Validation.Formula1" value="128" type="String"/><column name="TABLE_SCHEMA" property="Validation.AlertStyle" value="2" type="Double"/><column name="TABLE_SCHEMA" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="TABLE_SCHEMA" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="TABLE_SCHEMA" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="TABLE_SCHEMA" property="Validation.ErrorMessage" value="The column requires values of the nvarchar(128) datatype." type="String"/><column name="TABLE_SCHEMA" property="Validation.ShowInput" value="True" type="Boolean"/><column name="TABLE_SCHEMA" property="Validation.ShowError" value="True" type="Boolean"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$E$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="15.43" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="Validation.Type" value="6" type="Double"/><column name="TABLE_NAME" property="Validation.Operator" value="8" type="Double"/><column name="TABLE_NAME" property="Validation.Formula1" value="128" type="String"/><column name="TABLE_NAME" property="Validation.AlertStyle" value="2" type="Double"/><column name="TABLE_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="TABLE_NAME" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="TABLE_NAME" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="TABLE_NAME" property="Validation.ErrorMessage" value="The column requires values of the nvarchar(128) datatype." type="String"/><column name="TABLE_NAME" property="Validation.ShowInput" value="True" type="Boolean"/><column name="TABLE_NAME" property="Validation.ShowError" value="True" type="Boolean"/><column name="COLUMN_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="COLUMN_NAME" property="Address" value="$F$4" type="String"/><column name="COLUMN_NAME" property="ColumnWidth" value="27.86" type="Double"/><column name="COLUMN_NAME" property="NumberFormat" value="General" type="String"/><column name="COLUMN_NAME" property="Validation.Type" value="6" type="Double"/><column name="COLUMN_NAME" property="Validation.Operator" value="8" type="Double"/><column name="COLUMN_NAME" property="Validation.Formula1" value="128" type="String"/><column name="COLUMN_NAME" property="Validation.AlertStyle" value="2" type="Double"/><column name="COLUMN_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="COLUMN_NAME" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="COLUMN_NAME" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="COLUMN_NAME" property="Validation.ErrorMessage" value="The column requires values of the nvarchar(128) datatype." type="String"/><column name="COLUMN_NAME" property="Validation.ShowInput" value="True" type="Boolean"/><column name="COLUMN_NAME" property="Validation.ShowError" value="True" type="Boolean"/><column name="ORDINAL_POSITION" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="ORDINAL_POSITION" property="Address" value="$G$4" type="String"/><column name="ORDINAL_POSITION" property="ColumnWidth" value="20.43" type="Double"/><column name="ORDINAL_POSITION" property="NumberFormat" value="General" type="String"/><column name="ORDINAL_POSITION" property="Validation.Type" value="1" type="Double"/><column name="ORDINAL_POSITION" property="Validation.Operator" value="1" type="Double"/><column name="ORDINAL_POSITION" property="Validation.Formula1" value="-2147483648" type="String"/><column name="ORDINAL_POSITION" property="Validation.Formula2" value="2147483647" type="String"/><column name="ORDINAL_POSITION" property="Validation.AlertStyle" value="2" type="Double"/><column name="ORDINAL_POSITION" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="ORDINAL_POSITION" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="ORDINAL_POSITION" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="ORDINAL_POSITION" property="Validation.ErrorMessage" value="The column requires values of the int datatype." type="String"/><column name="ORDINAL_POSITION" property="Validation.ShowInput" value="True" type="Boolean"/><column name="ORDINAL_POSITION" property="Validation.ShowError" value="True" type="Boolean"/><column name="IS_PRIMARY_KEY" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="IS_PRIMARY_KEY" property="Address" value="$H$4" type="String"/><column name="IS_PRIMARY_KEY" property="ColumnWidth" value="17.86" type="Double"/><column name="IS_PRIMARY_KEY" property="NumberFormat" value="General" type="String"/><column name="IS_PRIMARY_KEY" property="HorizontalAlignment" value="-4108" type="Double"/><column name="IS_PRIMARY_KEY" property="Font.Size" value="10" type="Double"/><column name="IS_NULLABLE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="IS_NULLABLE" property="Address" value="$I$4" type="String"/><column name="IS_NULLABLE" property="ColumnWidth" value="14" type="Double"/><column name="IS_NULLABLE" property="NumberFormat" value="General" type="String"/><column name="IS_NULLABLE" property="HorizontalAlignment" value="-4108" type="Double"/><column name="IS_NULLABLE" property="Font.Size" value="10" type="Double"/><column name="IS_IDENTITY" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="IS_IDENTITY" property="Address" value="$J$4" type="String"/><column name="IS_IDENTITY" property="ColumnWidth" value="13.14" type="Double"/><column name="IS_IDENTITY" property="NumberFormat" value="General" type="String"/><column name="IS_IDENTITY" property="HorizontalAlignment" value="-4108" type="Double"/><column name="IS_IDENTITY" property="Font.Size" value="10" type="Double"/><column name="IS_COMPUTED" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="IS_COMPUTED" property="Address" value="$K$4" type="String"/><column name="IS_COMPUTED" property="ColumnWidth" value="15.57" type="Double"/><column name="IS_COMPUTED" property="NumberFormat" value="General" type="String"/><column name="IS_COMPUTED" property="HorizontalAlignment" value="-4108" type="Double"/><column name="IS_COMPUTED" property="Font.Size" value="10" type="Double"/><column name="COLUMN_DEFAULT" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="COLUMN_DEFAULT" property="Address" value="$L$4" type="String"/><column name="COLUMN_DEFAULT" property="ColumnWidth" value="19.86" type="Double"/><column name="COLUMN_DEFAULT" property="NumberFormat" value="General" type="String"/><column name="COLUMN_DEFAULT" property="Validation.Type" value="6" type="Double"/><column name="COLUMN_DEFAULT" property="Validation.Operator" value="8" type="Double"/><column name="COLUMN_DEFAULT" property="Validation.Formula1" value="256" type="String"/><column name="COLUMN_DEFAULT" property="Validation.AlertStyle" value="2" type="Double"/><column name="COLUMN_DEFAULT" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="COLUMN_DEFAULT" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="COLUMN_DEFAULT" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="COLUMN_DEFAULT" property="Validation.ErrorMessage" value="The column requires values of the nvarchar(256) datatype." type="String"/><column name="COLUMN_DEFAULT" property="Validation.ShowInput" value="True" type="Boolean"/><column name="COLUMN_DEFAULT" property="Validation.ShowError" value="True" type="Boolean"/><column name="DATA_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="DATA_TYPE" property="Address" value="$M$4" type="String"/><column name="DATA_TYPE" property="ColumnWidth" value="12.71" type="Double"/><column name="DATA_TYPE" property="NumberFormat" value="General" type="String"/><column name="DATA_TYPE" property="Validation.Type" value="6" type="Double"/><column name="DATA_TYPE" property="Validation.Operator" value="8" type="Double"/><column name="DATA_TYPE" property="Validation.Formula1" value="128" type="String"/><column name="DATA_TYPE" property="Validation.AlertStyle" value="2" type="Double"/><column name="DATA_TYPE" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="DATA_TYPE" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="DATA_TYPE" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="DATA_TYPE" property="Validation.ErrorMessage" value="The column requires values of the nvarchar(128) datatype." type="String"/><column name="DATA_TYPE" property="Validation.ShowInput" value="True" type="Boolean"/><column name="DATA_TYPE" property="Validation.ShowError" value="True" type="Boolean"/><column name="CHARACTER_MAXIMUM_LENGTH" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Address" value="$N$4" type="String"/><column name="CHARACTER_MAXIMUM_LENGTH" property="ColumnWidth" value="32.71" type="Double"/><column name="CHARACTER_MAXIMUM_LENGTH" property="NumberFormat" value="General" type="String"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.Type" value="1" type="Double"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.Operator" value="1" type="Double"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.Formula1" value="-2147483648" type="String"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.Formula2" value="2147483647" type="String"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.AlertStyle" value="2" type="Double"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.ErrorMessage" value="The column requires values of the int datatype." type="String"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.ShowInput" value="True" type="Boolean"/><column name="CHARACTER_MAXIMUM_LENGTH" property="Validation.ShowError" value="True" type="Boolean"/><column name="PRECISION" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="PRECISION" property="Address" value="$O$4" type="String"/><column name="PRECISION" property="ColumnWidth" value="12" type="Double"/><column name="PRECISION" property="NumberFormat" value="General" type="String"/><column name="PRECISION" property="Validation.Type" value="1" type="Double"/><column name="PRECISION" property="Validation.Operator" value="1" type="Double"/><column name="PRECISION" property="Validation.Formula1" value="0" type="String"/><column name="PRECISION" property="Validation.Formula2" value="255" type="String"/><column name="PRECISION" property="Validation.AlertStyle" value="2" type="Double"/><column name="PRECISION" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="PRECISION" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="PRECISION" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="PRECISION" property="Validation.ErrorMessage" value="The column requires values of the tinyint datatype." type="String"/><column name="PRECISION" property="Validation.ShowInput" value="True" type="Boolean"/><column name="PRECISION" property="Validation.ShowError" value="True" type="Boolean"/><column name="SCALE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="SCALE" property="Address" value="$P$4" type="String"/><column name="SCALE" property="ColumnWidth" value="7.86" type="Double"/><column name="SCALE" property="NumberFormat" value="General" type="String"/><column name="SCALE" property="Validation.Type" value="1" type="Double"/><column name="SCALE" property="Validation.Operator" value="1" type="Double"/><column name="SCALE" property="Validation.Formula1" value="0" type="String"/><column name="SCALE" property="Validation.Formula2" value="255" type="String"/><column name="SCALE" property="Validation.AlertStyle" value="2" type="Double"/><column name="SCALE" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="SCALE" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="SCALE" property="Validation.ErrorTitle" value="Datatype Control" type="String"/><column name="SCALE" property="Validation.ErrorMessage" value="The column requires values of the tinyint datatype." type="String"/><column name="SCALE" property="Validation.ShowInput" value="True" type="Boolean"/><column name="SCALE" property="Validation.ShowError" value="True" type="Boolean"/><column name="TABLE_SCHEMA" property="FormatConditions(1).AppliesTo.Address" value="$D$4:$D$423" type="String"/><column name="TABLE_SCHEMA" property="FormatConditions(1).Type" value="2" type="Double"/><column name="TABLE_SCHEMA" property="FormatConditions(1).Priority" value="5" type="Double"/><column name="TABLE_SCHEMA" property="FormatConditions(1).Formula1" value="=ISBLANK(D4)" type="String"/><column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.Color" value="65535" type="Double"/><column name="TABLE_NAME" property="FormatConditions(1).AppliesTo.Address" value="$E$4:$E$423" type="String"/><column name="TABLE_NAME" property="FormatConditions(1).Type" value="2" type="Double"/><column name="TABLE_NAME" property="FormatConditions(1).Priority" value="6" type="Double"/><column name="TABLE_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(E4)" type="String"/><column name="TABLE_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double"/><column name="COLUMN_NAME" property="FormatConditions(1).AppliesTo.Address" value="$F$4:$F$423" type="String"/><column name="COLUMN_NAME" property="FormatConditions(1).Type" value="2" type="Double"/><column name="COLUMN_NAME" property="FormatConditions(1).Priority" value="7" type="Double"/><column name="COLUMN_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(F4)" type="String"/><column name="COLUMN_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double"/><column name="ORDINAL_POSITION" property="FormatConditions(1).AppliesTo.Address" value="$G$4:$G$423" type="String"/><column name="ORDINAL_POSITION" property="FormatConditions(1).Type" value="2" type="Double"/><column name="ORDINAL_POSITION" property="FormatConditions(1).Priority" value="8" type="Double"/><column name="ORDINAL_POSITION" property="FormatConditions(1).Formula1" value="=ISBLANK(G4)" type="String"/><column name="ORDINAL_POSITION" property="FormatConditions(1).Interior.Color" value="65535" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).AppliesTo.Address" value="$H$4:$H$423" type="String"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).Type" value="6" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).Priority" value="4" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).IconSet.ID" value="8" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double"/><column name="IS_PRIMARY_KEY" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).AppliesTo.Address" value="$I$4:$I$423" type="String"/><column name="IS_NULLABLE" property="FormatConditions(1).Type" value="6" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).Priority" value="3" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean"/><column name="IS_NULLABLE" property="FormatConditions(1).IconSet.ID" value="8" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double"/><column name="IS_NULLABLE" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).AppliesTo.Address" value="$J$4:$J$423" type="String"/><column name="IS_IDENTITY" property="FormatConditions(1).Type" value="6" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).Priority" value="2" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean"/><column name="IS_IDENTITY" property="FormatConditions(1).IconSet.ID" value="8" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double"/><column name="IS_IDENTITY" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).AppliesTo.Address" value="$K$4:$K$423" type="String"/><column name="IS_COMPUTED" property="FormatConditions(1).Type" value="6" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).Priority" value="1" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean"/><column name="IS_COMPUTED" property="FormatConditions(1).IconSet.ID" value="8" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double"/><column name="IS_COMPUTED" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double"/><column name="SortFields(1)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="2" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="2" type="Double"/><column name="SortFields(3)" property="KeyfieldName" value="ORDINAL_POSITION" type="String"/><column name="SortFields(3)" property="SortOn" value="0" type="Double"/><column name="SortFields(3)" property="Order" value="1" type="Double"/><column name="SortFields(3)" property="DataOption" value="2" type="Double"/><column name="SortFields(4)" property="KeyfieldName" value="COLUMN_NAME" type="String"/><column name="SortFields(4)" property="SortOn" value="0" type="Double"/><column name="SortFields(4)" property="Order" value="1" type="Double"/><column name="SortFields(4)" property="DataOption" value="2" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls', N'formats', N'<table name="xls.formats"><columnFormats><column name="" property="ListObjectName" value="formats" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$E$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_EXCEL_FORMAT_XML" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_EXCEL_FORMAT_XML" property="Address" value="$F$4" type="String"/><column name="TABLE_EXCEL_FORMAT_XML" property="ColumnWidth" value="42.29" type="Double"/><column name="TABLE_EXCEL_FORMAT_XML" property="NumberFormat" value="General" type="String"/><column name="SortFields(1)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="0" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="0" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls', N'handlers', N'<table name="xls.handlers"><columnFormats><column name="" property="ListObjectName" value="handlers" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$E$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="COLUMN_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="COLUMN_NAME" property="Address" value="$F$4" type="String"/><column name="COLUMN_NAME" property="ColumnWidth" value="17.43" type="Double"/><column name="COLUMN_NAME" property="NumberFormat" value="General" type="String"/><column name="COLUMN_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="EVENT_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="EVENT_NAME" property="Address" value="$G$4" type="String"/><column name="EVENT_NAME" property="ColumnWidth" value="21.57" type="Double"/><column name="EVENT_NAME" property="NumberFormat" value="General" type="String"/><column name="EVENT_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="HANDLER_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="HANDLER_SCHEMA" property="Address" value="$H$4" type="String"/><column name="HANDLER_SCHEMA" property="ColumnWidth" value="19.71" type="Double"/><column name="HANDLER_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="HANDLER_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="HANDLER_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="HANDLER_NAME" property="Address" value="$I$4" type="String"/><column name="HANDLER_NAME" property="ColumnWidth" value="31.14" type="Double"/><column name="HANDLER_NAME" property="NumberFormat" value="General" type="String"/><column name="HANDLER_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="HANDLER_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="HANDLER_TYPE" property="Address" value="$J$4" type="String"/><column name="HANDLER_TYPE" property="ColumnWidth" value="16.29" type="Double"/><column name="HANDLER_TYPE" property="NumberFormat" value="General" type="String"/><column name="HANDLER_TYPE" property="VerticalAlignment" value="-4160" type="Double"/><column name="HANDLER_CODE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="HANDLER_CODE" property="Address" value="$K$4" type="String"/><column name="HANDLER_CODE" property="ColumnWidth" value="70.71" type="Double"/><column name="HANDLER_CODE" property="NumberFormat" value="General" type="String"/><column name="HANDLER_CODE" property="VerticalAlignment" value="-4160" type="Double"/><column name="TARGET_WORKSHEET" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TARGET_WORKSHEET" property="Address" value="$L$4" type="String"/><column name="TARGET_WORKSHEET" property="ColumnWidth" value="21.71" type="Double"/><column name="TARGET_WORKSHEET" property="NumberFormat" value="General" type="String"/><column name="TARGET_WORKSHEET" property="VerticalAlignment" value="-4160" type="Double"/><column name="MENU_ORDER" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="MENU_ORDER" property="Address" value="$M$4" type="String"/><column name="MENU_ORDER" property="ColumnWidth" value="15.43" type="Double"/><column name="MENU_ORDER" property="NumberFormat" value="General" type="String"/><column name="MENU_ORDER" property="VerticalAlignment" value="-4160" type="Double"/><column name="EDIT_PARAMETERS" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="EDIT_PARAMETERS" property="Address" value="$N$4" type="String"/><column name="EDIT_PARAMETERS" property="ColumnWidth" value="19.57" type="Double"/><column name="EDIT_PARAMETERS" property="NumberFormat" value="General" type="String"/><column name="EDIT_PARAMETERS" property="HorizontalAlignment" value="-4108" type="Double"/><column name="EDIT_PARAMETERS" property="VerticalAlignment" value="-4160" type="Double"/><column name="EDIT_PARAMETERS" property="Font.Size" value="10" type="Double"/><column name="SortFields(1)" property="KeyfieldName" value="EVENT_NAME" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="0" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="0" type="Double"/><column name="SortFields(3)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(3)" property="SortOn" value="0" type="Double"/><column name="SortFields(3)" property="Order" value="1" type="Double"/><column name="SortFields(3)" property="DataOption" value="0" type="Double"/><column name="SortFields(4)" property="KeyfieldName" value="COLUMN_NAME" type="String"/><column name="SortFields(4)" property="SortOn" value="0" type="Double"/><column name="SortFields(4)" property="Order" value="1" type="Double"/><column name="SortFields(4)" property="DataOption" value="0" type="Double"/><column name="SortFields(5)" property="KeyfieldName" value="MENU_ORDER" type="String"/><column name="SortFields(5)" property="SortOn" value="0" type="Double"/><column name="SortFields(5)" property="Order" value="1" type="Double"/><column name="SortFields(5)" property="DataOption" value="0" type="Double"/><column name="SortFields(6)" property="KeyfieldName" value="HANDLER_SCHEMA" type="String"/><column name="SortFields(6)" property="SortOn" value="0" type="Double"/><column name="SortFields(6)" property="Order" value="1" type="Double"/><column name="SortFields(6)" property="DataOption" value="0" type="Double"/><column name="SortFields(7)" property="KeyfieldName" value="HANDLER_NAME" type="String"/><column name="SortFields(7)" property="SortOn" value="0" type="Double"/><column name="SortFields(7)" property="Order" value="1" type="Double"/><column name="SortFields(7)" property="DataOption" value="0" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls', N'objects', N'<table name="xls.objects"><columnFormats><column name="" property="ListObjectName" value="objects" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$E$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_TYPE" property="Address" value="$F$4" type="String"/><column name="TABLE_TYPE" property="ColumnWidth" value="13.14" type="Double"/><column name="TABLE_TYPE" property="NumberFormat" value="General" type="String"/><column name="TABLE_TYPE" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_TYPE" property="Validation.Type" value="3" type="Double"/><column name="TABLE_TYPE" property="Validation.Operator" value="1" type="Double"/><column name="TABLE_TYPE" property="Validation.Formula1" value="TABLE; VIEW; PROCEDURE; CODE; HTTP; TEXT; HIDDEN" type="String"/><column name="TABLE_TYPE" property="Validation.AlertStyle" value="1" type="Double"/><column name="TABLE_TYPE" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="TABLE_TYPE" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="TABLE_TYPE" property="Validation.ShowInput" value="True" type="Boolean"/><column name="TABLE_TYPE" property="Validation.ShowError" value="True" type="Boolean"/><column name="TABLE_CODE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_CODE" property="Address" value="$G$4" type="String"/><column name="TABLE_CODE" property="ColumnWidth" value="13.57" type="Double"/><column name="TABLE_CODE" property="NumberFormat" value="General" type="String"/><column name="TABLE_CODE" property="VerticalAlignment" value="-4160" type="Double"/><column name="INSERT_OBJECT" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="INSERT_OBJECT" property="Address" value="$H$4" type="String"/><column name="INSERT_OBJECT" property="ColumnWidth" value="27.86" type="Double"/><column name="INSERT_OBJECT" property="NumberFormat" value="General" type="String"/><column name="INSERT_OBJECT" property="VerticalAlignment" value="-4160" type="Double"/><column name="UPDATE_OBJECT" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="UPDATE_OBJECT" property="Address" value="$I$4" type="String"/><column name="UPDATE_OBJECT" property="ColumnWidth" value="27.86" type="Double"/><column name="UPDATE_OBJECT" property="NumberFormat" value="General" type="String"/><column name="UPDATE_OBJECT" property="VerticalAlignment" value="-4160" type="Double"/><column name="DELETE_OBJECT" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="DELETE_OBJECT" property="Address" value="$J$4" type="String"/><column name="DELETE_OBJECT" property="ColumnWidth" value="27.86" type="Double"/><column name="DELETE_OBJECT" property="NumberFormat" value="General" type="String"/><column name="DELETE_OBJECT" property="VerticalAlignment" value="-4160" type="Double"/><column name="SortFields(1)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="2" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="2" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls', N'queries', N'<table name="xls.queries"><columnFormats><column name="" property="ListObjectName" value="queries" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$C$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$D$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_TYPE" property="Address" value="$E$4" type="String"/><column name="TABLE_TYPE" property="ColumnWidth" value="13.14" type="Double"/><column name="TABLE_TYPE" property="NumberFormat" value="General" type="String"/><column name="TABLE_CODE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_CODE" property="Address" value="$F$4" type="String"/><column name="TABLE_CODE" property="ColumnWidth" value="13.57" type="Double"/><column name="TABLE_CODE" property="NumberFormat" value="General" type="String"/><column name="INSERT_PROCEDURE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="INSERT_PROCEDURE" property="Address" value="$G$4" type="String"/><column name="INSERT_PROCEDURE" property="ColumnWidth" value="27.86" type="Double"/><column name="INSERT_PROCEDURE" property="NumberFormat" value="General" type="String"/><column name="UPDATE_PROCEDURE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="UPDATE_PROCEDURE" property="Address" value="$H$4" type="String"/><column name="UPDATE_PROCEDURE" property="ColumnWidth" value="27.86" type="Double"/><column name="UPDATE_PROCEDURE" property="NumberFormat" value="General" type="String"/><column name="DELETE_PROCEDURE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="DELETE_PROCEDURE" property="Address" value="$I$4" type="String"/><column name="DELETE_PROCEDURE" property="ColumnWidth" value="27.86" type="Double"/><column name="DELETE_PROCEDURE" property="NumberFormat" value="General" type="String"/><column name="PROCEDURE_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="PROCEDURE_TYPE" property="Address" value="$J$4" type="String"/><column name="PROCEDURE_TYPE" property="ColumnWidth" value="18.86" type="Double"/><column name="PROCEDURE_TYPE" property="NumberFormat" value="General" type="String"/><column name="SortFields(1)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="2" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="2" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls', N'translations', N'<table name="xls.translations"><columnFormats><column name="" property="ListObjectName" value="translations" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$E$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="32.14" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="COLUMN_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="COLUMN_NAME" property="Address" value="$F$4" type="String"/><column name="COLUMN_NAME" property="ColumnWidth" value="20.71" type="Double"/><column name="COLUMN_NAME" property="NumberFormat" value="General" type="String"/><column name="COLUMN_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="LANGUAGE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="LANGUAGE_NAME" property="Address" value="$G$4" type="String"/><column name="LANGUAGE_NAME" property="ColumnWidth" value="19.57" type="Double"/><column name="LANGUAGE_NAME" property="NumberFormat" value="General" type="String"/><column name="LANGUAGE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TRANSLATED_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TRANSLATED_NAME" property="Address" value="$H$4" type="String"/><column name="TRANSLATED_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TRANSLATED_NAME" property="NumberFormat" value="General" type="String"/><column name="TRANSLATED_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TRANSLATED_DESC" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TRANSLATED_DESC" property="Address" value="$I$4" type="String"/><column name="TRANSLATED_DESC" property="ColumnWidth" value="19.57" type="Double"/><column name="TRANSLATED_DESC" property="NumberFormat" value="General" type="String"/><column name="TRANSLATED_DESC" property="VerticalAlignment" value="-4160" type="Double"/><column name="TRANSLATED_COMMENT" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TRANSLATED_COMMENT" property="Address" value="$J$4" type="String"/><column name="TRANSLATED_COMMENT" property="ColumnWidth" value="25" type="Double"/><column name="TRANSLATED_COMMENT" property="NumberFormat" value="General" type="String"/><column name="TRANSLATED_COMMENT" property="VerticalAlignment" value="-4160" type="Double"/><column name="SortFields(1)" property="KeyfieldName" value="LANGUAGE_NAME" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="2" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="2" type="Double"/><column name="SortFields(3)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(3)" property="SortOn" value="0" type="Double"/><column name="SortFields(3)" property="Order" value="1" type="Double"/><column name="SortFields(3)" property="DataOption" value="2" type="Double"/><column name="SortFields(4)" property="KeyfieldName" value="COLUMN_NAME" type="String"/><column name="SortFields(4)" property="SortOn" value="0" type="Double"/><column name="SortFields(4)" property="Order" value="1" type="Double"/><column name="SortFields(4)" property="DataOption" value="2" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls', N'users', N'<table name="xls.users"><columnFormats><column name="" property="ListObjectName" value="users" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="user" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="user" property="Address" value="$C$4" type="String"/><column name="user" property="ColumnWidth" value="22.14" type="Double"/><column name="user" property="NumberFormat" value="General" type="String"/><column name="role" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="role" property="Address" value="$D$4" type="String"/><column name="role" property="ColumnWidth" value="20.71" type="Double"/><column name="role" property="NumberFormat" value="General" type="String"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO xls.formats (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES (N'xls', N'workbooks', N'<table name="xls.workbooks"><columnFormats><column name="" property="ListObjectName" value="workbooks" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="NAME" property="Address" value="$D$4" type="String"/><column name="NAME" property="ColumnWidth" value="42.14" type="Double"/><column name="NAME" property="NumberFormat" value="General" type="String"/><column name="NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TEMPLATE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TEMPLATE" property="Address" value="$E$4" type="String"/><column name="TEMPLATE" property="ColumnWidth" value="30" type="Double"/><column name="TEMPLATE" property="NumberFormat" value="General" type="String"/><column name="TEMPLATE" property="VerticalAlignment" value="-4160" type="Double"/><column name="DEFINITION" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="DEFINITION" property="Address" value="$F$4" type="String"/><column name="DEFINITION" property="ColumnWidth" value="70.71" type="Double"/><column name="DEFINITION" property="NumberFormat" value="General" type="String"/><column name="DEFINITION" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$G$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="SortFields(1)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="0" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="NAME" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="0" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
GO

INSERT INTO xls.workbooks (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES (N'savetodb_configuration.xlsx', NULL, N'objects=xls.objects,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":null,"TABLE_TYPE":null},"ListObjectName":"objects"}
handlers=xls.handlers,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":null,"EVENT_NAME":null,"HANDLER_TYPE":null},"ListObjectName":"handlers"}
translations=xls.translations,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":null,"LANGUAGE_NAME":null},"ListObjectName":"translations"}
workbooks=xls.workbooks,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":null},"ListObjectName":"workbooks"}
users=xls.users,(Default),False,$B$3,,{"Parameters":{},"ListObjectName":"users"}', N'xls');
GO

print 'SaveToDB Framework installed';
