-- =============================================
-- Application: Sample 02 - Advanced Features
-- Version 10.8, January 9, 2023
--
-- Copyright 2017-2023 Gartle LLC
--
-- License: MIT
-- =============================================

CREATE SCHEMA S02;

CREATE TABLE S02.ACCOUNTS (
    ID INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    NAME VARCHAR(50) NOT NULL,
    CONSTRAINT PK_ACCOUNTS PRIMARY KEY (ID),
    CONSTRAINT IX_ACCOUNTS_NAME UNIQUE (NAME)
);

CREATE TABLE S02.COMPANIES (
    ID INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    NAME VARCHAR(50) NOT NULL,
    CONSTRAINT PK_COMPANIES PRIMARY KEY (ID)
);

CREATE INDEX IX_COMPANIES_NAME ON S02.COMPANIES (NAME);

CREATE TABLE S02.ITEMS (
    ID INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    NAME VARCHAR(50) NOT NULL,
    CONSTRAINT PK_ITEMS PRIMARY KEY (ID),
    CONSTRAINT IX_ITEMS_NAME UNIQUE (NAME)
);

CREATE TABLE S02.ITEM_COMPANIES (
    ITEM_ID INTEGER,
    COMPANY_ID INTEGER,
    CONSTRAINT PK_ITEM_COMPANIES PRIMARY KEY (ITEM_ID, COMPANY_ID)
);

ALTER TABLE S02.ITEM_COMPANIES ADD CONSTRAINT FK_ITEM_COMPANIES_COMPANIES FOREIGN KEY (COMPANY_ID) REFERENCES S02.COMPANIES (ID);

ALTER TABLE S02.ITEM_COMPANIES ADD CONSTRAINT FK_ITEM_COMPANIES_ITEMS FOREIGN KEY (ITEM_ID) REFERENCES S02.ITEMS (ID);

CREATE TABLE S02.CASHBOOK(
    ID INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    DATE DATETIME NOT NULL,
    ACCOUNT_ID INTEGER NOT NULL,
    ITEM_ID INTEGER NOT NULL,
    COMPANY_ID INTEGER NULL,
    DEBIT DOUBLE DEFAULT NULL,
    CREDIT DOUBLE DEFAULT NULL,
    CHECKED BOOLEAN NULL,
    CONSTRAINT PK_CASHBOOK PRIMARY KEY (ID)
);

ALTER TABLE S02.CASHBOOK ADD CONSTRAINT FK_CASHBOOK_ACCOUNTS FOREIGN KEY (ACCOUNT_ID) REFERENCES S02.ACCOUNTS (ID);

ALTER TABLE S02.CASHBOOK ADD CONSTRAINT FK_CASHBOOK_COMPANIES FOREIGN KEY (COMPANY_ID) REFERENCES S02.COMPANIES (ID);

ALTER TABLE S02.CASHBOOK ADD CONSTRAINT FK_CASHBOOK_ITEMS FOREIGN KEY (ITEM_ID) REFERENCES S02.ITEMS (ID);

CREATE OR REPLACE VIEW S02.VIEW_CASHBOOK
AS
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT_ID
    , p.ITEM_ID
    , p.COMPANY_ID
    , p.DEBIT
    , p.CREDIT
    , p.CHECKED
FROM
    S02.CASHBOOK p
;

CREATE OR REPLACE VIEW S02.VIEW_CASHBOOK2
AS
SELECT
    *
FROM
    S02.CASHBOOK p
;

CREATE OR REPLACE VIEW S02.VIEW_CASHBOOK3
AS
SELECT
    p.*
FROM
    S02.CASHBOOK p
;

CREATE OR REPLACE VIEW S02.XL_ACTIONS_ONLINE_HELP
AS
SELECT
    t.SCHEMA AS TABLE_SCHEMA
    , t.TABLENAME AS TABLE_NAME
    , CAST(NULL AS STRING) COLUMN_NAME
    , 'Actions' AS EVENT_NAME
    , t.SCHEMA AS HANDLER_SCHEMA
    , 'See Online Help' AS HANDLER_NAME
    , 'HTTP' AS HANDLER_TYPE
    , 'https://www.savetodb.com/samples/sample' || SUBSTRING(t.SCHEMA, 2, 2) || '-' || LCASE(t.TABLENAME) || CASE WHEN CURRENT_USER() LIKE 'sample%' THEN '_' || CURRENT_USER() ELSE '' END AS HANDLER_CODE
    , CAST(NULL AS STRING) AS TARGET_WORKSHEET
    , 1 AS MENU_ORDER
    , 0 AS EDIT_PARAMETERS
FROM
    SYSTEM.TABLES t
WHERE
    t.SCHEMA = 'S02'
    AND NOT t.TABLENAME LIKE 'XL_%'
UNION ALL
SELECT
    t.SCHEMA AS TABLE_SCHEMA
    , t.PROCEDURENAME AS TABLE_NAME
    , CAST(NULL AS STRING) AS COLUMN_NAME
    , 'Actions' AS EVENT_NAME
    , t.SCHEMA AS HANDLER_SCHEMA
    , 'See Online Help' AS HANDLER_NAME
    , 'HTTP' AS HANDLER_TYPE
    , 'https://www.savetodb.com/samples/sample' || SUBSTRING(t.SCHEMA, 2, 2) || '-' || LCASE(t.PROCEDURENAME) || CASE WHEN CURRENT_USER() LIKE 'sample%' THEN '_' || CURRENT_USER() ELSE '' END AS HANDLER_CODE
    , CAST(NULL AS STRING) AS TARGET_WORKSHEET
    , 1 AS MENU_ORDER
    , 0 AS EDIT_PARAMETERS
FROM
    SYSTEM.PROCEDURES t
WHERE
    t.SCHEMA = 'S02'
    AND NOT t.PROCEDURENAME LIKE 'XL_%'
    AND NOT T.PROCEDURENAME LIKE '%_INSERT'
    AND NOT T.PROCEDURENAME LIKE '%_UPDATE'
    AND NOT T.PROCEDURENAME LIKE '%_DELETE'
    AND NOT T.PROCEDURENAME LIKE '%_CHANGE'
    AND NOT T.PROCEDURENAME LIKE '%_MERGE'
;

SET DELIMITER @@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK (
    IN ACCOUNT_ID INTEGER
    , IN ITEM_ID INTEGER
    , IN COMPANY_ID INTEGER
    )
RETURNS tmp_tab (
    ID INTEGER,
    DATE DATETIME,
    ACCOUNT_ID INTEGER,
    ITEM_ID INTEGER,
    COMPANY_ID INTEGER,
    DEBIT DOUBLE,
    CREDIT DOUBLE,
    CHECKED BOOLEAN
    )
AS
VAR ACCOUNT_ID1 INTEGER = ACCOUNT_ID;
VAR ITEM_ID1 INTEGER = ITEM_ID;
VAR COMPANY_ID1 INTEGER = COMPANY_ID;

INSERT INTO tmp_tab
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT_ID
    , p.ITEM_ID
    , p.COMPANY_ID
    , p.DEBIT
    , p.CREDIT
    , p.CHECKED
FROM
    S02.CASHBOOK p
WHERE
    COALESCE(p.ACCOUNT_ID, -1) = COALESCE(ACCOUNT_ID1, p.ACCOUNT_ID, -1)
    AND COALESCE(p.ITEM_ID, -1) = COALESCE(ITEM_ID1, p.ITEM_ID, -1)
    AND COALESCE(p.COMPANY_ID, -1) = COALESCE(COMPANY_ID1, p.COMPANY_ID, -1);

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK2 (
    IN ACCOUNT_ID INTEGER
    , IN ITEM_ID INTEGER
    , IN COMPANY_ID INTEGER
    , IN START_DATE DATE
    , IN END_DATE DATE
    , IN CHECKED BOOLEAN
    )
RETURNS tmp_tab (
    ID INTEGER,
    DATE DATETIME,
    ACCOUNT_ID INTEGER,
    ITEM_ID INTEGER,
    COMPANY_ID INTEGER,
    DEBIT DOUBLE,
    CREDIT DOUBLE,
    CHECKED BOOLEAN
    )
AS
VAR ACCOUNT_ID1 INTEGER = ACCOUNT_ID;
VAR ITEM_ID1 INTEGER = ITEM_ID;
VAR COMPANY_ID1 INTEGER = COMPANY_ID;
VAR CHECKED1 BOOLEAN = CHECKED;

INSERT INTO tmp_tab
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT_ID
    , p.ITEM_ID
    , p.COMPANY_ID
    , p.DEBIT
    , p.CREDIT
    , p.CHECKED
FROM
    S02.CASHBOOK p
WHERE
    COALESCE(p.ACCOUNT_ID, -1) = COALESCE(ACCOUNT_ID1, p.ACCOUNT_ID, -1)
    AND COALESCE(p.ITEM_ID, -1) = COALESCE(ITEM_ID1, p.ITEM_ID, -1)
    AND COALESCE(p.COMPANY_ID, -1) = COALESCE(COMPANY_ID1, p.COMPANY_ID, -1)
    AND p.DATE BETWEEN COALESCE(START_DATE, '2010-01-01') AND COALESCE(END_DATE, '2038-01-19')
    AND (CHECKED1 IS NULL OR p.CHECKED = CHECKED1);

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK3 (
    IN ACCOUNT_ID INTEGER
    , IN ITEM_ID INTEGER
    , IN COMPANY_ID INTEGER
    )
RETURNS tmp_tab (
    ID INTEGER,
    DATE DATETIME,
    ACCOUNT_ID INTEGER,
    ITEM_ID INTEGER,
    COMPANY_ID INTEGER,
    DEBIT DOUBLE,
    CREDIT DOUBLE,
    CHECKED BOOLEAN
    )
AS
VAR ACCOUNT_ID1 INTEGER = ACCOUNT_ID;
VAR ITEM_ID1 INTEGER = ITEM_ID;
VAR COMPANY_ID1 INTEGER = COMPANY_ID;

INSERT INTO tmp_tab
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT_ID
    , p.ITEM_ID
    , p.COMPANY_ID
    , p.DEBIT
    , p.CREDIT
    , p.CHECKED
FROM
    S02.CASHBOOK p
WHERE
    COALESCE(p.ACCOUNT_ID, -1) = COALESCE(ACCOUNT_ID1, p.ACCOUNT_ID, -1)
    AND COALESCE(p.ITEM_ID, -1) = COALESCE(ITEM_ID1, p.ITEM_ID, -1)
    AND COALESCE(p.COMPANY_ID, -1) = COALESCE(COMPANY_ID1, p.COMPANY_ID, -1);

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK4 (
    IN ACCOUNT_ID INTEGER
    , IN ITEM_ID INTEGER
    , IN COMPANY_ID INTEGER
    )
RETURNS tmp_tab (
    ID INTEGER,
    DATE DATETIME,
    ACCOUNT_ID INTEGER,
    ITEM_ID INTEGER,
    COMPANY_ID INTEGER,
    DEBIT DOUBLE,
    CREDIT DOUBLE,
    CHECKED BOOLEAN
    )
AS
VAR ACCOUNT_ID1 INTEGER = ACCOUNT_ID;
VAR ITEM_ID1 INTEGER = ITEM_ID;
VAR COMPANY_ID1 INTEGER = COMPANY_ID;

INSERT INTO tmp_tab
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT_ID
    , p.ITEM_ID
    , p.COMPANY_ID
    , p.DEBIT
    , p.CREDIT
    , p.CHECKED
FROM
    S02.CASHBOOK p
WHERE
    COALESCE(p.ACCOUNT_ID, -1) = COALESCE(ACCOUNT_ID1, p.ACCOUNT_ID, -1)
    AND COALESCE(p.ITEM_ID, -1) = COALESCE(ITEM_ID1, p.ITEM_ID, -1)
    AND COALESCE(p.COMPANY_ID, -1) = COALESCE(COMPANY_ID1, p.COMPANY_ID, -1);

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK5 (
    IN ACCOUNT_ID INTEGER
    , IN ITEM_ID INTEGER
    , IN COMPANY_ID INTEGER
    )
RETURNS tmp_tab (
    ID INTEGER,
    DATE DATETIME,
    ACCOUNT_ID INTEGER,
    ITEM_ID INTEGER,
    COMPANY_ID INTEGER,
    DEBIT DOUBLE,
    CREDIT DOUBLE,
    CHECKED BOOLEAN
    )
AS
VAR ACCOUNT_ID1 INTEGER = ACCOUNT_ID;
VAR ITEM_ID1 INTEGER = ITEM_ID;
VAR COMPANY_ID1 INTEGER = COMPANY_ID;

INSERT INTO tmp_tab
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT_ID
    , p.ITEM_ID
    , p.COMPANY_ID
    , p.DEBIT
    , p.CREDIT
    , p.CHECKED
FROM
    S02.CASHBOOK p
WHERE
    COALESCE(p.ACCOUNT_ID, -1) = COALESCE(ACCOUNT_ID1, p.ACCOUNT_ID, -1)
    AND COALESCE(p.ITEM_ID, -1) = COALESCE(ITEM_ID1, p.ITEM_ID, -1)
    AND COALESCE(p.COMPANY_ID, -1) = COALESCE(COMPANY_ID1, p.COMPANY_ID, -1);

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK2_INSERT (
    ID INTEGER,
    DATE DATE,
    ACCOUNT_ID INTEGER,
    ITEM_ID INTEGER,
    COMPANY_ID INTEGER,
    DEBIT DOUBLE,
    CREDIT DOUBLE,
    CHECKED BOOLEAN
    )
AS

INSERT INTO S02.CASHBOOK
    ( DATE
    , ACCOUNT_ID
    , COMPANY_ID
    , ITEM_ID
    , DEBIT
    , CREDIT
    , CHECKED
    )
VALUES
    ( DATE
    , ACCOUNT_ID
    , COMPANY_ID
    , ITEM_ID
    , DEBIT
    , CREDIT
    , CHECKED
    );

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK2_UPDATE (
    ID INTEGER,
    DATE DATE,
    ACCOUNT_ID INTEGER,
    ITEM_ID INTEGER,
    COMPANY_ID INTEGER,
    DEBIT DOUBLE,
    CREDIT DOUBLE,
    CHECKED BOOLEAN
    )
AS

VAR ID1 INTEGER = ID;
VAR DATE1 DATE = DATE;
VAR ACCOUNT_ID1 INTEGER = ACCOUNT_ID;
VAR ITEM_ID1 INTEGER = ITEM_ID;
VAR COMPANY_ID1 INTEGER = COMPANY_ID;
VAR DEBIT1 DOUBLE = DEBIT;
VAR CREDIT1 DOUBLE = CREDIT;
VAR CHECKED1 BOOLEAN = CHECKED;

UPDATE S02.CASHBOOK P
SET
    DATE = DATE1
    , ACCOUNT_ID = ACCOUNT_ID1
    , COMPANY_ID = COMPANY_ID1
    , ITEM_ID = ITEM_ID1
    , DEBIT = DEBIT1
    , CREDIT = CREDIT1
    , CHECKED = CHECKED1
WHERE
    ID = ID1;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK2_DELETE (
    ID INTEGER
    )
AS

VAR ID1 INTEGER = ID;

DELETE FROM S02.CASHBOOK
WHERE
    ID = ID1;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK3_CHANGE (
    COLUMN_NAME VARCHAR(255)
    , CELL_VALUE VARCHAR(255)
    , CELL_NUMBER_VALUE DOUBLE
    , CELL_DATETIME_VALUE DATE
    , ID INTEGER
    )
AS

VAR ID1 INTEGER = ID;

IF (COLUMN_NAME = 'DATE')
    UPDATE S02.CASHBOOK SET DATE = CELL_DATETIME_VALUE WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'DEBIT')
    UPDATE S02.CASHBOOK SET DEBIT = CELL_NUMBER_VALUE WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'CREDIT')
    UPDATE S02.CASHBOOK SET CREDIT = CELL_NUMBER_VALUE WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'ACCOUNT_ID')
    UPDATE S02.CASHBOOK SET ACCOUNT_ID = CAST(CELL_NUMBER_VALUE AS INTEGER) WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'COMPANY_ID')
    UPDATE S02.CASHBOOK SET COMPANY_ID = CAST(CELL_NUMBER_VALUE AS INTEGER) WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'ITEM_ID')
    UPDATE S02.CASHBOOK SET ITEM_ID = CAST(CELL_NUMBER_VALUE AS INTEGER) WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'CHECKED')
    UPDATE S02.CASHBOOK SET CHECKED = CAST(CELL_NUMBER_VALUE AS BOOLEAN) WHERE ID = ID1;
    RETURN;
END_IF;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASHBOOK4_MERGE (
    ID INTEGER,
    DATE DATE,
    ACCOUNT_ID INTEGER,
    ITEM_ID INTEGER,
    COMPANY_ID INTEGER,
    DEBIT DOUBLE,
    CREDIT DOUBLE,
    CHECKED BOOLEAN
    )
AS

VAR ID1 INTEGER = ID;
VAR DATE1 DATE = DATE;
VAR ACCOUNT_ID1 INTEGER = ACCOUNT_ID;
VAR ITEM_ID1 INTEGER = ITEM_ID;
VAR COMPANY_ID1 INTEGER = COMPANY_ID;
VAR DEBIT1 DOUBLE = DEBIT;
VAR CREDIT1 DOUBLE = CREDIT;
VAR CHECKED1 BOOLEAN = CHECKED;

UPDATE S02.CASHBOOK P
SET
    DATE = DATE1
    , ACCOUNT_ID = ACCOUNT_ID1
    , COMPANY_ID = COMPANY_ID1
    , ITEM_ID = ITEM_ID1
    , DEBIT = DEBIT1
    , CREDIT = CREDIT1
    , CHECKED = CHECKED1
WHERE
    ID = ID1;

IF ((SELECT GETUPDATECOUNT() FROM DUAL) = 0)
    INSERT INTO S02.CASHBOOK
        ( DATE
        , ACCOUNT_ID
        , COMPANY_ID
        , ITEM_ID
        , DEBIT
        , CREDIT
        , CHECKED
        )
    VALUES
        ( DATE1
        , ACCOUNT_ID1
        , COMPANY_ID1
        , ITEM_ID1
        , DEBIT1
        , CREDIT1
        , CHECKED1
        );
END_IF;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.XL_LIST_ACCOUNT_ID (
    DATA_LANGUAGE VARCHAR(10)
    )
RETURNS tmp_tab (
    ID INTEGER,
    NAME VARCHAR(50)
    )
AS

INSERT INTO tmp_tab
SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ACCOUNTS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = 'S02' AND t.TABLE_NAME = 'strings'
            AND t.LANGUAGE_NAME = DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.XL_LIST_COMPANY_ID (
    DATA_LANGUAGE VARCHAR(10)
    )
RETURNS tmp_tab (
    ID INTEGER,
    NAME VARCHAR(50)
    )
AS

INSERT INTO tmp_tab
SELECT
    c.ID
    , COALESCE(t.TRANSLATED_NAME, c.NAME) AS NAME
FROM
    S02.COMPANIES c
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = 'S02' AND t.TABLE_NAME = 'strings'
            AND t.LANGUAGE_NAME = DATA_LANGUAGE AND t.COLUMN_NAME = c.NAME
ORDER BY
    NAME
    , ID;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.XL_LIST_COMPANY_ID_FOR_ITEM_ID (
    ITEM_ID int
    , DATA_LANGUAGE VARCHAR(10)
    )
RETURNS tmp_tab (
    ID INTEGER,
    NAME VARCHAR(50)
    )
AS
VAR ITEM_ID1 INTEGER = ITEM_ID;

IF (ITEM_ID1 IS NULL)
    INSERT INTO tmp_tab
    SELECT
        m.ID
        , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
    FROM
        S02.COMPANIES m
        LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = 'S02' AND t.TABLE_NAME = 'strings'
                AND t.LANGUAGE_NAME = DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
    ORDER BY
        NAME;
ELSE
    INSERT INTO tmp_tab
    SELECT
        c.ID
        , COALESCE(t.TRANSLATED_NAME, c.NAME) AS NAME
    FROM
        S02.ITEM_COMPANIES ic
        INNER JOIN S02.COMPANIES c ON c.ID = ic.COMPANY_ID
        LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = 'S02' AND t.TABLE_NAME = 'strings'
                AND t.LANGUAGE_NAME = DATA_LANGUAGE AND t.COLUMN_NAME = c.NAME
    WHERE
        ic.ITEM_ID = ITEM_ID1
    ORDER BY
        NAME;
END_IF;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.XL_LIST_COMPANY_ID_WITH_ITEM_ID (
    DATA_LANGUAGE VARCHAR(10)
    )
RETURNS tmp_tab (
    ID INTEGER,
    NAME VARCHAR(50),
    ITEM_ID INTEGER
    )
AS

INSERT INTO tmp_tab
SELECT
    c.ID
    , COALESCE(t.TRANSLATED_NAME, c.NAME) AS NAME
    , ic.ITEM_ID
FROM
    S02.ITEM_COMPANIES ic
    INNER JOIN S02.COMPANIES c ON c.ID = ic.COMPANY_ID
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = 'S02' AND t.TABLE_NAME = 'strings'
            AND t.LANGUAGE_NAME = DATA_LANGUAGE AND t.COLUMN_NAME = c.NAME
ORDER BY
    ic.ITEM_ID
    , NAME;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.XL_LIST_ITEM_ID (
    DATA_LANGUAGE VARCHAR(10)
    )
RETURNS tmp_tab (
    ID INTEGER,
    NAME VARCHAR(50)
    )
AS

INSERT INTO tmp_tab
SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ITEMS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = 'S02' AND t.TABLE_NAME = 'strings'
            AND t.LANGUAGE_NAME = DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASH_BY_MONTHS (
    IN "Year" SMALLINT
    , IN DATA_LANGUAGE VARCHAR(10)
    )
RETURNS tmp_tab (
    SORT_ORDER BIGINT
    , SECTION INTEGER
    , LEVEL INTEGER
    , ITEM_ID INTEGER
    , COMPANY_ID INTEGER
    , "Name" STRING
    , "Total" DOUBLE
    , "Jan" DOUBLE
    , "Feb" DOUBLE
    , "Mar" DOUBLE
    , "Apr" DOUBLE
    , "May" DOUBLE
    , "Jun" DOUBLE
    , "Jul" DOUBLE
    , "Aug" DOUBLE
    , "Sep" DOUBLE
    , "Oct" DOUBLE
    , "Nov" DOUBLE
    , "Dec" DOUBLE
    )
AS
INSERT INTO tmp_tab
SELECT
    ROW_NUMBER() OVER(ORDER BY p.SECTION, p.ITEM_ID, p.COMPANY_ID) AS SORT_ORDER
    , p.SECTION
    , MAX(p.Level) AS LEVEL
    , p.ITEM_ID
    , p.COMPANY_ID
    , CASE WHEN P.COMPANY_ID IS NOT NULL THEN CONCAT('  ', MAX(COALESCE(t1.TRANSLATED_NAME, p.NAME))) ELSE MAX(COALESCE(t1.TRANSLATED_NAME, p.NAME)) END AS "Name"
    , CASE WHEN p.SECTION = 1 THEN SUM(COALESCE(p."Jan", 0)) WHEN p.SECTION = 5 THEN SUM(COALESCE(p."Dec", 0)) ELSE SUM(COALESCE(p.TOTAL, 0)) END AS "Total"
    , SUM(p."Jan") AS "Jan"
    , SUM(p."Feb") AS "Feb"
    , SUM(p."Mar") AS "Mar"
    , SUM(p."Apr") AS "Apr"
    , SUM(p."May") AS "May"
    , SUM(p."Jun") AS "Jun"
    , SUM(p."Jul") AS "Jul"
    , SUM(p."Aug") AS "Aug"
    , SUM(p."Sep") AS "Sep"
    , SUM(p."Oct") AS "Oct"
    , SUM(p."Nov") AS "Nov"
    , SUM(p."Dec") AS "Dec"
FROM
    (
    -- Opening balance
    SELECT
        1 AS SECTION
        , 0 AS LEVEL
        , NULL AS ITEM_ID
        , NULL AS COMPANY_ID
        , 'Opening Balance' AS NAME
        , p.PERIOD
        , NULL AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            p.PERIOD
            , p.AMOUNT
        FROM
            (
            SELECT
                d.PERIOD
                , p."DATE" AS DATE1
                , COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) AS AMOUNT
            FROM
                (
                    SELECT 1 AS PERIOD FROM DUAL
                    UNION ALL SELECT 2 FROM DUAL
                    UNION ALL SELECT 3 FROM DUAL
                    UNION ALL SELECT 4 FROM DUAL
                    UNION ALL SELECT 5 FROM DUAL
                    UNION ALL SELECT 6 FROM DUAL
                    UNION ALL SELECT 7 FROM DUAL
                    UNION ALL SELECT 8 FROM DUAL
                    UNION ALL SELECT 9 FROM DUAL
                    UNION ALL SELECT 10 FROM DUAL
                    UNION ALL SELECT 11 FROM DUAL
                    UNION ALL SELECT 12 FROM DUAL
                ) d
                , S02.CASHBOOK p
            ) p
        WHERE
            YEAR(p.DATE1) < "Year"
            OR (YEAR(p.DATE1) = "Year" AND MONTH(p.DATE1) < p.PERIOD)
        ) p

    -- Closing balance
    UNION ALL
    SELECT
        5 AS SECTION
        , 0 AS LEVEL
        , NULL AS ITEM_ID
        , NULL AS COMPANY_ID
        , 'Closing Balance' AS NAME
        , p.PERIOD
        , NULL AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            p.PERIOD
            , p.AMOUNT
        FROM
            (
            SELECT
                d.PERIOD
                , p."DATE" AS DATE1
                , COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) AS AMOUNT
            FROM
                (
                    SELECT 1 AS PERIOD FROM DUAL
                    UNION ALL SELECT 2 FROM DUAL
                    UNION ALL SELECT 3 FROM DUAL
                    UNION ALL SELECT 4 FROM DUAL
                    UNION ALL SELECT 5 FROM DUAL
                    UNION ALL SELECT 6 FROM DUAL
                    UNION ALL SELECT 7 FROM DUAL
                    UNION ALL SELECT 8 FROM DUAL
                    UNION ALL SELECT 9 FROM DUAL
                    UNION ALL SELECT 10 FROM DUAL
                    UNION ALL SELECT 11 FROM DUAL
                    UNION ALL SELECT 12 FROM DUAL
                ) d
                , S02.CASHBOOK p
            ) p
        WHERE
            YEAR(p.DATE1) < "Year"
            OR (YEAR(p.DATE1) = "Year" AND MONTH(p.DATE1) <= p.PERIOD)
        ) p

    -- Companies
    UNION ALL
    SELECT
        p.SECTION
        , 2 AS LEVEL
        , p.ITEM_ID
        , p.COMPANY_ID
        , p.NAME
        , p.PERIOD
        , p.AMOUNT AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            CAST(CASE WHEN p.CREDIT IS NOT NULL THEN 3 ELSE 2 END AS INTEGER) AS SECTION
            , p.ITEM_ID
            , p.COMPANY_ID
            , c.NAME
            , MONTH(p."DATE") AS PERIOD
            , CASE WHEN p.CREDIT IS NOT NULL THEN COALESCE(p.CREDIT, 0) - COALESCE(p.DEBIT, 0) ELSE COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) END AS AMOUNT
        FROM
            S02.CASHBOOK p
            LEFT OUTER JOIN S02.COMPANIES c ON c.ID = p.COMPANY_ID
            LEFT OUTER JOIN S02.ITEMS i ON i.ID = p.ITEM_ID
        WHERE
            p.COMPANY_ID IS NOT NULL
            AND YEAR(p."DATE") = "Year"
        ) p

    -- Total Items
    UNION ALL
    SELECT
        p.SECTION
        , 1 AS LEVEL
        , p.ITEM_ID
        , NULL AS COMPANY_ID
        , p.NAME
        , p.PERIOD
        , p.AMOUNT AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            CAST(CASE WHEN p.CREDIT IS NOT NULL THEN 3 ELSE 2 END AS INTEGER) AS SECTION
            , p.ITEM_ID
            , i.NAME
            , MONTH(p."DATE") AS PERIOD
            , CASE WHEN p.CREDIT IS NOT NULL THEN COALESCE(p.CREDIT, 0) - COALESCE(p.DEBIT, 0) ELSE COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) END AS AMOUNT
        FROM
            S02.CASHBOOK p
            LEFT OUTER JOIN S02.ITEMS i ON i.ID = p.ITEM_ID
        WHERE
            p.ITEM_ID IS NOT NULL
            AND YEAR(p."DATE") = "Year"
        ) p

    -- Total Income/Expenses
    UNION ALL
    SELECT
        p.SECTION
        , 0 AS LEVEL
        , NULL AS ITEM_ID
        , NULL AS COMPANY_ID
        , p.NAME
        , p.PERIOD
        , p.AMOUNT AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            CAST(CASE WHEN p.CREDIT IS NOT NULL THEN 3 ELSE 2 END AS INTEGER) AS SECTION
            , CASE WHEN p.CREDIT IS NOT NULL THEN 'Total Expenses' ELSE 'Total Income' END AS NAME
            , MONTH(p."DATE") AS PERIOD
            , CASE WHEN p.CREDIT IS NOT NULL THEN COALESCE(p.CREDIT, 0) - COALESCE(p.DEBIT, 0) ELSE COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) END AS AMOUNT
        FROM
            S02.CASHBOOK p
        WHERE
            YEAR(p."DATE") = "Year"
        ) p

    -- Net Chanhge
    UNION ALL
    SELECT
        4 AS SECTION
        , 0 AS LEVEL
        , NULL AS ITEM_ID
        , NULL AS COMPANY_ID
        , 'Net Change' AS NAME
        , p.PERIOD
        , p.AMOUNT AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            MONTH(p."DATE") AS PERIOD
            , COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) AS AMOUNT
        FROM
            S02.CASHBOOK p
        WHERE
            YEAR(p."DATE") = "Year"
        ) p
    ) p
    LEFT OUTER JOIN XLS.TRANSLATIONS t1 ON t1.TABLE_SCHEMA = 'S02' AND t1.TABLE_NAME = 'strings' AND t1.COLUMN_NAME = p.name AND t1.LANGUAGE_NAME = data_language
GROUP BY
    p.SECTION
    , p.ITEM_ID
    , p.COMPANY_ID
ORDER BY
    SORT_ORDER;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S02.USP_CASH_BY_MONTHS_CHANGE (
    COLUMN_NAME VARCHAR(255)
    , CELL_NUMBER_VALUE DOUBLE
    , SECTION INTEGER
    , ITEM_ID INTEGER
    , COMPANY_ID INTEGER
    , YEAR INTEGER
    )
AS

VAR ITEM_ID1 INTEGER = ITEM_ID;
VAR COMPANY_ID1 INTEGER = COMPANY_ID;

VAR month INTEGER;
VAR start_date DATE;
VAR end_date DATE;
VAR id1 INTEGER;
VAR count1 INTEGER;
VAR date1 DATE;
VAR account_id1 INTEGER;

month = POSITION(' ' || COLUMN_NAME || ' ' IN '    Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec ') / 4;

IF (month < 1) RETURN; END_IF;

IF (year = NULL) year = YEAR(CURRENT DATE); END_IF;

IF (year = NULL) year = (SELECT YEAR(MAX(DATE)) FROM S02.CASHBOOK); END_IF;

start_date = DATE_ADD(CAST(year AS string) || '-01-01', interval (month - 1) MONTH);

end_date = DATE_ADD(DATE_ADD(CAST(year AS string) || '-01-01', interval month MONTH), interval - 1 DAY);

count1 = (SELECT COUNT(*) FROM S02.CASHBOOK WHERE ITEM_ID = ITEM_ID1 AND COALESCE(COMPANY_ID, -1) = COALESCE(COMPANY_ID1, -1) AND DATE BETWEEN start_date AND end_date);

IF (count1 = 0)

    IF (ITEM_ID1 IS NULL)
        RETURN;
    END_IF;

    id1 = (SELECT MAX(ID) FROM S02.CASHBOOK WHERE ITEM_ID = ITEM_ID1 AND COALESCE(COMPANY_ID, -1) = COALESCE(COMPANY_ID1, -1) AND DATE < end_date);

    IF (id1 IS NOT NULL)

        date1 = (SELECT DATE FROM S02.CASHBOOK WHERE ID = id1);
        account_id1 = (SELECT ACCOUNT_ID FROM S02.CASHBOOK WHERE ID = id1);

        IF (DAY(date1) > DAY(end_date))
            date1 = end_date;
        ELSE
            date1 = DATE_ADD(DATE_ADD(date1, interval -MONTH(date1) MONTH), interval month MONTH);
        END_IF;
    ELSE
        date1 = end_date;
    END_IF;

    INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT)
        VALUES (date1, account_id1, ITEM_ID, COMPANY_ID,
            CASE WHEN SECTION = 3 THEN NULL ELSE CELL_NUMBER_VALUE END,
            CASE WHEN SECTION = 3 THEN CELL_NUMBER_VALUE ELSE NULL END);

    RETURN;
END_IF;

IF (count1 > 1)
    RETURN;
END_IF;

id1 = (SELECT MAX(id) FROM S02.CASHBOOK t WHERE t.ITEM_ID = ITEM_ID1 AND COALESCE(t.COMPANY_ID, -1) = COALESCE(COMPANY_ID1, -1) AND t."DATE" BETWEEN start_date AND end_date);

UPDATE S02.CASHBOOK
SET
    DEBIT = CASE WHEN SECTION = 3 THEN NULL ELSE CELL_NUMBER_VALUE END
    , CREDIT = CASE WHEN SECTION = 3 THEN CELL_NUMBER_VALUE ELSE NULL END
WHERE
    ID = id1;

END_PROCEDURE;
@@

CREATE OR REPLACE PROCEDURE S02.XL_CONTEXT_MENU_CASH_BY_MONTHS (
    COLUMN_NAME VARCHAR(255)
    , ITEM_ID INTEGER
    , COMPANY_ID INTEGER
    , SECTION INTEGER
    , YEAR INTEGER
    )
RETURNS tmp_tab (
    ID integer,
    DATE datetime,
    ACCOUNT string,
    ITEM string,
    COMPANY string,
    DEBIT double,
    CREDIT double
    )
AS

VAR ITEM_ID1 INTEGER = ITEM_ID;
VAR COMPANY_ID1 INTEGER = COMPANY_ID;

VAR month INTEGER;
VAR start_date DATE;
VAR end_date DATE;

month = POSITION(' ' || COLUMN_NAME || ' ' IN '    Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec ') / 4;

IF (month < 1) month = NULL; END_IF;

IF (year = NULL) year = YEAR(CURRENT DATE); END_IF;

IF (year = NULL) year = (SELECT YEAR(MAX(DATE)) FROM S02.CASHBOOK); END_IF;

start_date = DATE_ADD(CAST(year AS string) || '-01-01', interval (COALESCE(month, 1) - 1) MONTH);

end_date = DATE_ADD(DATE_ADD(CAST(year AS string) || '-01-01', interval COALESCE(month, 12) MONTH), interval - 1 DAY);

INSERT INTO tmp_tab
SELECT
    t.ID
    , t.DATE
    , a.NAME AS ACCOUNT
    , i.NAME AS ITEM
    , c.NAME AS COMPANY
    , CAST(t.DEBIT AS DECIMAL(15,2)) AS DEBIT
    , CAST(t.CREDIT AS DECIMAL(15,2)) AS CREDIT
FROM
    S02.CASHBOOK t
    LEFT OUTER JOIN S02.ACCOUNTS a ON a.ID = t.ACCOUNT_ID
    LEFT OUTER JOIN S02.ITEMS i ON i.ID = t.ITEM_ID
    LEFT OUTER JOIN S02.COMPANIES c ON c.ID = t.COMPANY_ID
WHERE
    COALESCE(t.ITEM_ID, 0) = COALESCE(ITEM_ID1, t.ITEM_ID, 0)
    AND COALESCE(t.COMPANY_ID, 0) = COALESCE(COMPANY_ID1, t.COMPANY_ID, 0)
    AND t.DATE BETWEEN start_date AND end_date
    AND ((SECTION = 2 AND t.DEBIT IS NOT NULL)
      OR (SECTION = 3 AND t.CREDIT IS NOT NULL)
      OR (SECTION = 4));

END_PROCEDURE
@@

SET DELIMITER ;

INSERT INTO S02.ACCOUNTS (ID, NAME) VALUES (1, 'Bank');

INSERT INTO S02.ITEMS (ID, NAME) VALUES (1, 'Revenue');
INSERT INTO S02.ITEMS (ID, NAME) VALUES (2, 'Expenses');
INSERT INTO S02.ITEMS (ID, NAME) VALUES (3, 'Payroll');
INSERT INTO S02.ITEMS (ID, NAME) VALUES (4, 'Taxes');

INSERT INTO S02.COMPANIES (ID, NAME) VALUES (1, 'Customer C1');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (2, 'Customer C2');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (3, 'Customer C3');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (4, 'Customer C4');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (5, 'Customer C5');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (6, 'Customer C6');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (7, 'Customer C7');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (8, 'Supplier S1');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (9, 'Supplier S2');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (10, 'Supplier S3');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (11, 'Supplier S4');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (12, 'Supplier S5');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (13, 'Supplier S6');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (14, 'Supplier S7');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (15, 'Corporate Income Tax');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (16, 'Individual Income Tax');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (17, 'Payroll Taxes');

INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 1);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 2);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 3);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 4);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 5);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 6);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 7);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 8);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 9);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 10);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 11);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 12);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 13);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 14);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (4, 15);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (4, 16);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (4, 17);

INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-10', 1, 1, 1, 200000, NULL,1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-10', 1, 2, 8, NULL, 50000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-31', 1, 3, NULL, NULL, 85000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-31', 1, 4, 16, NULL, 15000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-31', 1, 4, 17, NULL, 15000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-10', 1, 1, 1, 300000, NULL, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-10', 1, 1, 2, 100000, NULL, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-10', 1, 2, 9, NULL, 50000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-10', 1, 2, 8, NULL, 100000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-28', 1, 3, NULL, NULL, 85000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-28', 1, 4, 16, NULL, 15000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-28', 1, 4, 17, NULL, 15000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-10', 1, 1, 1, 300000, NULL, 0);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-10', 1, 1, 2, 200000, NULL, 0);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-10', 1, 1, 3, 100000, NULL, 0);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-15', 1, 4, 15, NULL, 100000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 3, NULL, NULL, 170000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 4, 16, NULL, 30000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 4, 17, NULL, 30000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 2, 9, NULL, 50000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 2, 8, NULL, 100000, NULL);

INSERT INTO XLS.OBJECTS (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_OBJECT, UPDATE_OBJECT, DELETE_OBJECT) VALUES ('S02', 'VIEW_CASHBOOK', 'VIEW', NULL, 'S02.VIEW_CASHBOOK', 'S02.VIEW_CASHBOOK', 'S02.VIEW_CASHBOOK');
INSERT INTO XLS.OBJECTS (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_OBJECT, UPDATE_OBJECT, DELETE_OBJECT) VALUES ('S02', 'USP_CASHBOOK', 'PROCEDURE', NULL, 'S02.CASHBOOK', 'S02.CASHBOOK', 'S02.CASHBOOK');
INSERT INTO XLS.OBJECTS (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_OBJECT, UPDATE_OBJECT, DELETE_OBJECT) VALUES ('S02', 'USP_CASHBOOK5', 'PROCEDURE', NULL, 'S02.USP_CASHBOOK2_INSERT', 'S02.USP_CASHBOOK2_UPDATE', 'S02.USP_CASHBOOK2_DELETE');
INSERT INTO XLS.OBJECTS (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_OBJECT, UPDATE_OBJECT, DELETE_OBJECT) VALUES ('S02', 'CODE_CASHBOOK', 'CODE', 'SELECT
    t.ID
    , t.DATE
    , t.ACCOUNT_ID
    , t.ITEM_ID
    , t.COMPANY_ID
    , t.DEBIT
    , t.CREDIT
    , t.CHECKED
FROM
    S02.CASHBOOK t
WHERE
    COALESCE(:ACCOUNT_ID, t.ACCOUNT_ID, -1) = COALESCE(t.ACCOUNT_ID, -1)
    AND COALESCE(:ITEM_ID, t.ITEM_ID, -1) = COALESCE(t.ITEM_ID, -1)
    AND COALESCE(:COMPANY_ID, t.COMPANY_ID, -1) = COALESCE(t.COMPANY_ID, -1)
    AND t.DATE BETWEEN COALESCE(:START_DATE, ''2010-01-01'') AND COALESCE(:END_DATE, ''2038-01-19'')
    AND (:CHECKED IS NULL OR t.CHECKED = :CHECKED)', 'INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES (:DATE, :ACCOUNT_ID, :ITEM_ID, :COMPANY_ID, :DEBIT, :CREDIT, :CHECKED)', 'UPDATE S02.CASHBOOK SET DATE = :DATE, ACCOUNT_ID = :ACCOUNT_ID, ITEM_ID = :ITEM_ID, COMPANY_ID = :COMPANY_ID, DEBIT = :DEBIT, CREDIT = :CREDIT, CHECKED = :CHECKED WHERE ID = :ID', 'DELETE FROM S02.CASHBOOK WHERE ID = :ID');

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CASHBOOK', 'DATE', 'SelectPeriod', NULL, NULL, 'ATTRIBUTE', NULL, 'HideWeeks HideYears', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'START_DATE', 'SelectPeriod', NULL, NULL, 'ATTRIBUTE', NULL, 'END_DATE HideWeeks HideYears', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK2', 'START_DATE', 'SelectPeriod', NULL, NULL, 'ATTRIBUTE', NULL, 'END_DATE HideWeeks HideYears', NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK', 'ACCOUNT_ID', 'ValidationList', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK', 'COMPANY_ID', 'ValidationList', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK', 'ITEM_ID', 'ValidationList', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', 'ACCOUNT_ID', 'ValidationList', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', 'COMPANY_ID', 'ValidationList', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', 'ITEM_ID', 'ValidationList', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', NULL, 'Change', 'S02', 'CASHBOOK', 'TABLE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', NULL, 'DoNotSave', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', NULL, 'ProtectRows', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'ACCOUNT_ID', 'ValidationList', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'COMPANY_ID', 'ValidationList', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'ITEM_ID', 'ValidationList', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'DEBIT', 'Change', 'S02', 'VIEW_CASHBOOK3_DEBIT_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET DEBIT = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'CREDIT', 'Change', 'S02', 'VIEW_CASHBOOK3_CREDIT_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET CREDIT = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'ACCOUNT_ID', 'Change', 'S02', 'VIEW_CASHBOOK3_ACCOUNT_ID_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET ACCOUNT_ID = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'COMPANY_ID', 'Change', 'S02', 'VIEW_CASHBOOK3_COMPANY_ID_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET COMPANY_ID = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'ITEM_ID', 'Change', 'S02', 'VIEW_CASHBOOK3_ITEM_ID_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET ITEM_ID = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'DATE', 'Change', 'S02', 'VIEW_CASHBOOK3_DATE_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET DATE = :cell_datetime_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', NULL, 'DoNotSave', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', NULL, 'ProtectRows', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASH_BY_MONTHS', NULL, 'ContextMenu', 'S02', 'XL_CONTEXT_MENU_CASH_BY_MONTHS', 'PROCEDURE', NULL, '_TaskPane', 11, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASH_BY_MONTHS', NULL, 'ContextMenu', 'S02', 'MENUSEPARATOR12', 'MENUSEPARATOR', NULL, NULL, 12, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASH_BY_MONTHS', NULL, 'ContextMenu', 'S02', 'USP_CASHBOOK2', 'PROCEDURE', 'CALL S02.USP_CASHBOOK2 (1, :ITEM_ID, :COMPANY_ID)', '_New', 13, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASH_BY_MONTHS', 'year', 'ParameterValues', 'S02', 'XL_PARAMETER_VALUES_YEAR', 'PROCEDURE', NULL, NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'ACCOUNT_ID', 'ValidationList', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'COMPANY_ID', 'ValidationList', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'ITEM_ID', 'ValidationList', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'ACCOUNT_ID', 'ParameterValues', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'COMPANY_ID', 'ParameterValues', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'ITEM_ID', 'ParameterValues', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK2', 'ACCOUNT_ID', 'ValidationList', 'S02', 'XL_LIST_ACCOUNT_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK2', 'COMPANY_ID', 'ValidationList', 'S02', 'XL_LIST_COMPANY_ID_WITH_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK2', 'ITEM_ID', 'ValidationList', 'S02', 'XL_LIST_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK2', 'ACCOUNT_ID', 'ParameterValues', 'S02', 'XL_LIST_ACCOUNT_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK2', 'COMPANY_ID', 'ParameterValues', 'S02', 'XL_LIST_COMPANY_ID_FOR_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK2', 'ITEM_ID', 'ParameterValues', 'S02', 'XL_LIST_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK3', 'ACCOUNT_ID', 'ValidationList', 'S02', 'XL_LIST_ACCOUNT_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK3', 'COMPANY_ID', 'ValidationList', 'S02', 'XL_LIST_COMPANY_ID_WITH_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK3', 'ITEM_ID', 'ValidationList', 'S02', 'XL_LIST_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK3', 'ACCOUNT_ID', 'ParameterValues', 'S02', 'XL_LIST_ACCOUNT_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK3', 'COMPANY_ID', 'ParameterValues', 'S02', 'XL_LIST_COMPANY_ID_FOR_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK3', 'ITEM_ID', 'ParameterValues', 'S02', 'XL_LIST_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK3', NULL, 'Change', 'S02', 'USP_CASHBOOK3_CHANGE', 'PROCEDURE', NULL, '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK3', NULL, 'DoNotSave', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK3', NULL, 'ProtectRows', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK4', 'ACCOUNT_ID', 'ValidationList', 'S02', 'XL_LIST_ACCOUNT_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK4', 'COMPANY_ID', 'ValidationList', 'S02', 'XL_LIST_COMPANY_ID_WITH_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK4', 'ITEM_ID', 'ValidationList', 'S02', 'XL_LIST_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK4', 'ACCOUNT_ID', 'ParameterValues', 'S02', 'XL_LIST_ACCOUNT_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK4', 'COMPANY_ID', 'ParameterValues', 'S02', 'XL_LIST_COMPANY_ID_FOR_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK4', 'ITEM_ID', 'ParameterValues', 'S02', 'XL_LIST_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK5', 'ACCOUNT_ID', 'ValidationList', 'S02', 'XL_LIST_ACCOUNT_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK5', 'COMPANY_ID', 'ValidationList', 'S02', 'XL_LIST_COMPANY_ID_WITH_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK5', 'ITEM_ID', 'ValidationList', 'S02', 'XL_LIST_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK5', 'ACCOUNT_ID', 'ParameterValues', 'S02', 'XL_LIST_ACCOUNT_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK5', 'COMPANY_ID', 'ParameterValues', 'S02', 'XL_LIST_COMPANY_ID_FOR_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK5', 'ITEM_ID', 'ParameterValues', 'S02', 'XL_LIST_ITEM_ID', 'PROCEDURE', NULL, NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', NULL, 'Actions', 'S02', 'See Online Help', 'HTTP', 'https://www.savetodb.com/samples/sample02-code_cashbook', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'ACCOUNT_ID', 'ParameterValues', 'S02', 'XL_LIST_ACCOUNT_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ACCOUNTS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'COMPANY_ID', 'ParameterValues', 'S02', 'XL_LIST_COMPANY_ID_FOR_ITEM_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.COMPANIES m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
WHERE
    :ITEM_ID IS NULL
UNION ALL
SELECT
    c.ID
    , COALESCE(t.TRANSLATED_NAME, c.NAME) AS NAME
FROM
    S02.ITEM_COMPANIES ic
    INNER JOIN S02.COMPANIES c ON c.ID = ic.COMPANY_ID
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = c.NAME
WHERE
    ic.ITEM_ID = :ITEM_ID
ORDER BY
    NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'ITEM_ID', 'ParameterValues', 'S02', 'XL_LIST_ITEM_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ITEMS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'ACCOUNT_ID', 'ValidationList', 'S02', 'XL_LIST_ACCOUNT_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ACCOUNTS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'COMPANY_ID', 'ValidationList', 'S02', 'XL_LIST_COMPANY_ID_WITH_ITEM_ID_CODE', 'CODE', 'SELECT
    c.ID
    , COALESCE(t.TRANSLATED_NAME, c.NAME) AS NAME
    , ic.ITEM_ID
FROM
    S02.ITEM_COMPANIES ic
    INNER JOIN S02.COMPANIES c ON c.ID = ic.COMPANY_ID
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = c.NAME
ORDER BY
    ic.ITEM_ID
    , NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'ITEM_ID', 'ValidationList', 'S02', 'XL_LIST_ITEM_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ITEMS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME', NULL, NULL, NULL);

INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','de','Konto','','');
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','de','Konto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','de','Apr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','de','Aug.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','de','Balance', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','de','Überprüft', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','de','Unternehmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','de','Unternehmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','de','Kosten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','de','Datum', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','de','Tag', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','de','Einkommen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','de','Dez.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','de','Endtermin', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','de','Feb.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','de','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','de','Artikel', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','de','Artikel', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','de','Jan.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','de','Juli', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','de','Juni', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','de','Niveau', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','de','März', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','de','Mai', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','de','Monat', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','de','Name', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','de','Nov.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','de','Okt.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','de','Sektion', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','de','Sept.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','de','Sortierung', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','de','Startdatum', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','de','Jahr', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'de','Konten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'de','Kassenbuch', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'de','Kassenbuch (SQL-Code)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'de','Unternehmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'de','Artikel und Firmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'de','Artikel', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','de','Bank', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','de','Schlussbilanz', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','de','Körperschaftssteuer', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','de','Kunde C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','de','Kunde C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','de','Kunde C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','de','Kunde C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','de','Kunde C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','de','Kunde C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','de','Kunde C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','de','Kosten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','de','Lohnsteuer', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','de','Nettoveränderung', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','de','Anfangsbestand', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','de','Lohn-und Gehaltsabrechnung', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','de','Sozialabgaben', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','de','Einnahmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','de','Lieferant S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','de','Lieferant S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','de','Lieferant S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','de','Lieferant S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','de','Lieferant S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','de','Lieferant S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','de','Lieferant S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','de','Steuern', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','de','Gesamtausgaben', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','de','Gesamteinkommen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'de','Bargeld nach Monaten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','de','Firmen-ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','de','Artikel-ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'de','Kassenbuch (Prozedur)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'de','Kassenbuch (Prozedur, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'de','Kassenbuch (Prozedur, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'de','Kassenbuch (Prozedur, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'de','Kassenbuch (Formeln)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'de','Kassenbuch (Ansicht)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'de','Kassenbuch (Ansicht, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'de','Kassenbuch (Ansicht, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'de','Translationen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'de','Einzelheiten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','en','Account', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','en','Account', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','en','Apr', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','en','Aug', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','en','Balance', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','en','Checked', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','en','Company', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','en','Company', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','en','Expenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','en','Date', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','en','Day', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','en','Income', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','en','Dec', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','en','End Date', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','en','Feb', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','en','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','en','Item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','en','Item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','en','Jan', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','en','Jul', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','en','Jun', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','en','Level', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','en','Mar', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','en','May', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','en','Month', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','en','Name', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','en','Nov', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','en','Oct', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','en','Section', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','en','Sep', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','en','Sort Order', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','en','Start Date', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','en','Year', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'en','Accounts', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'en','Cashbook', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'en','Cashbook (SQL code)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'en','Companies', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'en','Item and Companies', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'en','Items', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','en','Bank', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','en','Closing Balance', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','en','Corporate Income Tax', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','en','Customer C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','en','Customer C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','en','Customer C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','en','Customer C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','en','Customer C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','en','Customer C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','en','Customer C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','en','Expenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','en','Individual Income Tax', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','en','Net Change', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','en','Opening Balance', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','en','Payroll', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','en','Payroll Taxes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','en','Revenue', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','en','Supplier S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','en','Supplier S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','en','Supplier S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','en','Supplier S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','en','Supplier S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','en','Supplier S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','en','Supplier S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','en','Taxes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','en','Total Expenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','en','Total Income', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'en','Cash by Months', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','en','Company Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','en','Item Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'en','Cashbook (procedure)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'en','Cashbook (procedure, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'en','Cashbook (procedure, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'en','Cashbook (procedure, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'en','Cashbook (formulas)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'en','Cashbook (view)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'en','Cashbook (view, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'en','Cashbook (view, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'en','Translations', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'en','Details', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','es','Cuenta', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','es','Cuenta', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','es','Abr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','es','Agosto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','es','Equilibrio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','es','Comprobado', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','es','Empresa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','es','Empresa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','es','Gasto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','es','Fecha', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','es','Día', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','es','Ingresos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','es','Dic.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','es','Fecha final', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','es','Feb.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','es','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','es','Artículo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','es','Artículo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','es','Enero', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','es','Jul.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','es','Jun.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','es','Nivel', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','es','Marzo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','es','Mayo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','es','Mes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','es','Nombre', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','es','Nov.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','es','Oct.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','es','Sección', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','es','Sept.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','es','Orden', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','es','Fecha de inicio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','es','Año', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'es','Cuentas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'es','Libro de caja', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'es','Libro de caja (código SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'es','Compañías', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'es','Artículo y empresas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'es','Artículos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','es','Banco', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','es','Balance de cierre', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','es','Impuesto sobre Sociedades', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','es','Cliente C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','es','Cliente C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','es','Cliente C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','es','Cliente C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','es','Cliente C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','es','Cliente C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','es','Cliente C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','es','Gasto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','es','IRPF', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','es','Cambio neto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','es','Saldo de apertura', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','es','Salario', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','es','Cargas sociales', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','es','Ingresos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','es','Abastecedor A1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','es','Abastecedor A2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','es','Abastecedor A3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','es','Abastecedor A4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','es','Abastecedor A5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','es','Abastecedor A6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','es','Abastecedor A7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','es','Impuestos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','es','Gasto total', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','es','Ingresos totales', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'es','Efectivo por meses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','es','ID de empresa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','es','ID del artículo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'es','Libro de caja (proc)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'es','Libro de caja (proc, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'es','Libro de caja (proc, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'es','Libro de caja (proc, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'es','Libro de caja (fórmulas)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'es','Libro de caja (ver)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'es','Libro de caja (ver, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'es','Libro de caja (ver, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'es','Traducciones', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'es','Detalles', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','fr','Compte', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','fr','Compte', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','fr','Avril', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','fr','Août', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','fr','Solde', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','fr','Vérifié', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','fr','Entreprise', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','fr','Entreprise', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','fr','Dépenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','fr','Date', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','fr','Journée', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','fr','Revenu', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','fr','Déc.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','fr','Date de fin', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','fr','Févr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','fr','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','fr','Article', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','fr','Article', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','fr','Janv.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','fr','Juil.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','fr','Juin', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','fr','Niveau', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','fr','Mars', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','fr','Mai', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','fr','Mois', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','fr','Prénom', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','fr','Nov.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','fr','Oct.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','fr','Section', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','fr','Sept.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','fr','Ordre de tri', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','fr','Date de début', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','fr','Année', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'fr','Comptes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'fr','Livre de caisse', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'fr','Livre de caisse (code SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'fr','Entreprises', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'fr','Article et sociétés', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'fr','Articles', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','fr','Banque', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','fr','Solde de clôture', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','fr','Impôt sur les sociétés', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','fr','Client 01', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','fr','Client 02', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','fr','Client 03', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','fr','Client 04', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','fr','Client 05', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','fr','Client 06', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','fr','Client 07', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','fr','Dépenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','fr','Impôt sur le revenu', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','fr','Changement net', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','fr','Solde d''ouverture', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','fr','Paie', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','fr','Charges sociales', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','fr','Revenu', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','fr','Fournisseur 01', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','fr','Fournisseur 02', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','fr','Fournisseur 03', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','fr','Fournisseur 04', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','fr','Fournisseur 05', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','fr','Fournisseur 06', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','fr','Fournisseur 07', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','fr','Taxes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','fr','Dépenses totales', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','fr','Revenu total', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'fr','Cash par mois', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','fr','ID de l''entreprise', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','fr','ID de l''article', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'fr','Livre de caisse (procédure)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'fr','Livre de caisse (procédure, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'fr','Livre de caisse (procédure, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'fr','Livre de caisse (procédure, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'fr','Livre de caisse (formules)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'fr','Livre de caisse (vue)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'fr','Livre de caisse (vue, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'fr','Livre de caisse (vue, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'fr','Traductions', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'fr','Détails', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','it','Conto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','it','Conto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','it','Apr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','it','Ag.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','it','Saldo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','it','Controllato', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','it','Azienda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','it','Azienda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','it','Credito', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','it','Data', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','it','Giorno', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','it','Debito', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','it','Dic.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','it','Data di fine', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','it','Febbr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','it','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','it','Articolo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','it','Articolo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','it','Genn.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','it','Luglio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','it','Giugno', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','it','Livello', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','it','Mar.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','it','Magg.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','it','Mese', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','it','Conome', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','it','Nov.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','it','Ott.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','it','Sezione', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','it','Sett.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','it','Ordinamento', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','it','Data d''inizio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','it','Anno', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'it','Conti', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'it','Cashbook', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'it','Cashbook (codice SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'it','Aziende', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'it','Articolo e società', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'it','Elementi', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','it','Banca', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','it','Saldo finale', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','it','Imposta sul reddito delle società', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','it','Cliente C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','it','Cliente C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','it','Cliente C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','it','Cliente C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','it','Cliente C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','it','Cliente C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','it','Cliente C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','it','Spese', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','it','IRPEF', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','it','Cambio netto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','it','Saldo iniziale', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','it','Paga', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','it','Imposte sui salari', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','it','Reddito', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','it','Fornitore F1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','it','Fornitore F2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','it','Fornitore F3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','it','Fornitore F4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','it','Fornitore F5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','it','Fornitore F6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','it','Fornitore F7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','it','Tasse', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','it','Spese totale', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','it','Reddito totale', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'it','Contanti per mesi', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','it','ID dell''azienda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','it','ID articolo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'it','Cashbook (procedura)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'it','Cashbook (procedura, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'it','Cashbook (procedura, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'it','Cashbook (procedura, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'it','Cashbook (formule)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'it','Cashbook (visualizza)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'it','Cashbook (visualizza, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'it','Cashbook (visualizza, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'it','Traduzioni', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'it','Dettagli', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','ja','アカウント', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','ja','アカウント', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','ja','4月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','ja','8月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','ja','バランス', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','ja','チェック済み', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','ja','会社', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','ja','会社', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','ja','経費', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','ja','日付', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','ja','日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','ja','所得', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','ja','12月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','ja','終了日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','ja','2月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','ja','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','ja','アイテム', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','ja','アイテム', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','ja','1月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','ja','7月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','ja','6月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','ja','レベル', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','ja','3月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','ja','5月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','ja','月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','ja','名', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','ja','11月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','ja','10月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','ja','セクション', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','ja','9月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','ja','並べ替え順序', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','ja','開始日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','ja','年', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'ja','アカウント', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'ja','キャッシュブック', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'ja','キャッシュブック（SQLコード）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'ja','会社', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'ja','アイテムと会社', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'ja','アイテム', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','ja','銀行', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','ja','クロージングバランス', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','ja','法人所得税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','ja','顧客C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','ja','顧客C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','ja','顧客C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','ja','顧客C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','ja','顧客C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','ja','顧客C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','ja','顧客C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','ja','経費', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','ja','個人所得税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','ja','正味変化', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','ja','オープニングバランス', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','ja','給与', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','ja','給与税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','ja','収益', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','ja','サプライヤーS1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','ja','サプライヤーS2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','ja','サプライヤーS3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','ja','サプライヤーS4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','ja','サプライヤーS5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','ja','サプライヤーS6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','ja','サプライヤーS7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','ja','税金', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','ja','総費用', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','ja','総収入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'ja','月ごとの現金', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','ja','会社ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','ja','アイテムID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'ja','キャッシュブック（手順）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'ja','キャッシュブック（手順、_編集）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'ja','キャッシュブック（手順、_変更）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'ja','キャッシュブック（手順、_マージ）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'ja','キャッシュブック（数式）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'ja','キャッシュブック（表示）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'ja','キャッシュブック（表示、_変更）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'ja','キャッシュブック（表示、_変更、SQL）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'ja','翻訳', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'ja','詳細', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','ko','계정', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','ko','계정', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','ko','4월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','ko','8월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','ko','균형', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','ko','확인됨', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','ko','회사', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','ko','회사', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','ko','소요 경비', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','ko','날짜', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','ko','하루', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','ko','소득', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','ko','12월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','ko','종료일', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','ko','2월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','ko','ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','ko','아이템', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','ko','아이템', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','ko','1월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','ko','7월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','ko','6월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','ko','수준', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','ko','3월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','ko','5월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','ko','월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','ko','이름', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','ko','11월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','ko','10월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','ko','섹션', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','ko','9월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','ko','정렬 순서', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','ko','시작일', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','ko','년', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'ko','계정', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'ko','캐쉬북', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'ko','캐쉬북(SQL 코드)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'ko','회사', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'ko','아이템 및 회사', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'ko','아이템', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','ko','은행', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','ko','폐쇄 균형', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','ko','법인 소득세', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','ko','고객 C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','ko','고객 C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','ko','고객 C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','ko','고객 C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','ko','고객 C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','ko','고객 C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','ko','고객 C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','ko','소요 경비', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','ko','개인소득세', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','ko','순 변화', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','ko','기초 잔액', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','ko','월급', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','ko','급여 세금', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','ko','수익', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','ko','공급 업체 S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','ko','공급 업체 S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','ko','공급 업체 S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','ko','공급 업체 S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','ko','공급 업체 S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','ko','공급 업체 S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','ko','공급 업체 S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','ko','세금', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','ko','총 경비', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','ko','총 수입', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'ko','월별 현금', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','ko','회사 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','ko','항목 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'ko','캐쉬북(절차)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'ko','캐쉬북(절차, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'ko','캐쉬북(절차, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'ko','캐쉬북(절차, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'ko','캐쉬북(공식)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'ko','캐쉬북(보기)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'ko','캐쉬북(보기, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'ko','캐쉬북(보기, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'ko','번역', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'ko','세부', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','pt','Conta', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','pt','Conta', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','pt','Abr', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','pt','Agosto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','pt','Saldo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','pt','Verificado', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','pt','Companhia', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','pt','Companhia', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','pt','Despesas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','pt','Encontro', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','pt','Dia', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','pt','Renda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','pt','Dez', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','pt','Data final', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','pt','Fev', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','pt','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','pt','Item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','pt','Item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','pt','Jan', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','pt','Julho', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','pt','Junho', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','pt','Nível', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','pt','Março', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','pt','Maio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','pt','Mês', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','pt','Nome', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','pt','Nov', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','pt','Out', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','pt','Seção', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','pt','Set', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','pt','Ordem de classificação', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','pt','Data de início', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','pt','Ano', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'pt','Contas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'pt','Livro caixa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'pt','Livro caixa (código SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'pt','Empresas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'pt','Item e Empresas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'pt','Itens', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','pt','Banco', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','pt','Saldo final', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','pt','Imposto de Renda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','pt','Cliente C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','pt','Cliente C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','pt','Cliente C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','pt','Cliente C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','pt','Cliente C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','pt','Cliente C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','pt','Cliente C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','pt','Despesas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','pt','Imposto de renda individual', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','pt','Mudança de rede', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','pt','Saldo inicial', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','pt','Folha de pagamento', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','pt','Impostos sobre os salários', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','pt','Receita', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','pt','Fornecedor S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','pt','Fornecedor S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','pt','Fornecedor S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','pt','Fornecedor S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','pt','Fornecedor S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','pt','Fornecedor S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','pt','Fornecedor S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','pt','Impostos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','pt','Despesas totais', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','pt','Renda total', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'pt','Dinheiro por meses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','pt','ID da empresa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','pt','ID do item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'pt','Livro caixa (proc)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'pt','Livro caixa (proc, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'pt','Livro caixa (proc, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'pt','Livro caixa (proc, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'pt','Livro caixa (fórmulas)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'pt','Livro caixa (ver)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'pt','Livro caixa (ver, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'pt','Livro caixa (ver, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'pt','Traduções', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'pt','Detalhes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','ru','Счет', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','ru','Счет', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','ru','Апр', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','ru','Авг', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','ru','Остаток', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','ru','Проверено', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','ru','Компания', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','ru','Компания', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','ru','Расходы', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','ru','Дата', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','ru','Число', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','ru','Доходы', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','ru','Дек', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','ru','Конечная дата', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','ru','Фев', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','ru','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','ru','Статья', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','ru','Статья', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','ru','Янв', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','ru','Июл', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','ru','Июн', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','ru','Уровень', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','ru','Мар', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','ru','Май', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','ru','Месяц', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','ru','Наименование', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','ru','Ноя', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','ru','Окт', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','ru','Секция', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','ru','Сен', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','ru','Порядок', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','ru','Начальная дата', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','ru','Год', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'ru','Счета', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'ru','Кассовая книга', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'ru','Кассовая книга (SQL код)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'ru','Компании', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'ru','Статьи и компании', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'ru','Статьи', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','ru','Банк', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','ru','Конечное сальдо', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','ru','Налог на прибыль', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','ru','Покупатель C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','ru','Покупатель C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','ru','Покупатель C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','ru','Покупатель C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','ru','Покупатель C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','ru','Покупатель C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','ru','Покупатель C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','ru','Расходы', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','ru','НДФЛ', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','ru','Изменение', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','ru','Начальное сальдо', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','ru','Заработная плата', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','ru','Страховые взносы', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','ru','Выручка', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','ru','Поставщик S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','ru','Поставщик S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','ru','Поставщик S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','ru','Поставщик S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','ru','Поставщик S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','ru','Поставщик S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','ru','Поставщик S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','ru','Налоги', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','ru','Всего выплаты', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','ru','Всего поступления', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'ru','ДДС по месяцам', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','ru','Id компании', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','ru','Id статьи', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'ru','Кассовая книга (процедура)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'ru','Кассовая книга (процедура, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'ru','Кассовая книга (процедура, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'ru','Кассовая книга (процедура, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'ru','Кассовая книга (с формулами)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'ru','Кассовая книга (view)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'ru','Кассовая книга (view, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'ru','Кассовая книга (view, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'ru','Исходные данные', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','zh-hans','帐户', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','zh-hans','帐户', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','zh-hans','四月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','zh-hans','八月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','zh-hans','平衡', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','zh-hans','已检查', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','zh-hans','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','zh-hans','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','zh-hans','花费', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','zh-hans','日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','zh-hans','日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','zh-hans','收入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','zh-hans','十二月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','zh-hans','结束日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','zh-hans','二月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','zh-hans','ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','zh-hans','项', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','zh-hans','项', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','zh-hans','一月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','zh-hans','七月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','zh-hans','六月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','zh-hans','水平', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','zh-hans','三月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','zh-hans','五月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','zh-hans','月份', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','zh-hans','名称', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','zh-hans','十一月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','zh-hans','十月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','zh-hans','部分', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','zh-hans','九月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','zh-hans','排序顺序', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','zh-hans','开始日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','zh-hans','年', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'zh-hans','账户', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'zh-hans','现金簿', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'zh-hans','现金簿（SQL 代码）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'zh-hans','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'zh-hans','物品和公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'zh-hans','物品', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','zh-hans','银行', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','zh-hans','期末余额', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','zh-hans','公司所得税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','zh-hans','顾客C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','zh-hans','顾客C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','zh-hans','顾客C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','zh-hans','顾客C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','zh-hans','顾客C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','zh-hans','顾客C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','zh-hans','顾客C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','zh-hans','花费', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','zh-hans','个人所得税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','zh-hans','净变化', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','zh-hans','期初余额', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','zh-hans','工资单', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','zh-hans','工资税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','zh-hans','营收', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','zh-hans','供应商 S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','zh-hans','供应商 S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','zh-hans','供应商 S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','zh-hans','供应商 S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','zh-hans','供应商 S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','zh-hans','供应商 S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','zh-hans','供应商 S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','zh-hans','税收', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','zh-hans','总费用', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','zh-hans','总收入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'zh-hans','每月现金', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','zh-hans','公司 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','zh-hans','项目 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'zh-hans','现金簿（程序）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'zh-hans','现金簿（程序，_edit）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'zh-hans','现金簿（程序，_change）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'zh-hans','现金簿（程序，_merge）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'zh-hans','现金簿（公式）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'zh-hans','现金簿（查看）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'zh-hans','现金簿（查看，_change）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'zh-hans','现金簿（视图、_change、SQL）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'zh-hans','翻译', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'zh-hans','详情', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','zh-hant','帳戶', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','zh-hant','帳戶', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','zh-hant','四月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','zh-hant','八月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','zh-hant','平衡', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','zh-hant','已檢查', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','zh-hant','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','zh-hant','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','zh-hant','花費', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','zh-hant','日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','zh-hant','日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','zh-hant','收入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','zh-hant','十二月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','zh-hant','結束日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','zh-hant','二月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','zh-hant','ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','zh-hant','項', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','zh-hant','項', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','zh-hant','一月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','zh-hant','七月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','zh-hant','六月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','zh-hant','水平', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','zh-hant','三月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','zh-hant','五月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','zh-hant','月份', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','zh-hant','名稱', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','zh-hant','十一月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','zh-hant','十月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','zh-hant','部分', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','zh-hant','九月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','zh-hant','排序順序', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','zh-hant','開始日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','zh-hant','年', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'zh-hant','賬戶', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'zh-hant','現金簿', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'zh-hant','現金簿（SQL 代碼）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'zh-hant','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'zh-hant','物品和公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'zh-hant','物品', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','zh-hant','銀行', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','zh-hant','期末餘額', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','zh-hant','公司所得稅', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','zh-hant','顧客C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','zh-hant','顧客C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','zh-hant','顧客C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','zh-hant','顧客C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','zh-hant','顧客C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','zh-hant','顧客C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','zh-hant','顧客C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','zh-hant','花費', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','zh-hant','個人所得稅', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','zh-hant','淨變化', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','zh-hant','期初餘額', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','zh-hant','工資單', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','zh-hant','工資稅', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','zh-hant','營收', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','zh-hant','供應商 S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','zh-hant','供應商 S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','zh-hant','供應商 S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','zh-hant','供應商 S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','zh-hant','供應商 S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','zh-hant','供應商 S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','zh-hant','供應商 S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','zh-hant','稅收', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','zh-hant','總費用', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','zh-hant','總收入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'zh-hant','每月現金', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','zh-hant','公司 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','zh-hant','項目 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'zh-hant','现金簿（程序）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'zh-hant','现金簿（程序，_edit）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'zh-hant','现金簿（程序，_change）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'zh-hant','现金簿（程序，_merge）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'zh-hant','现金簿（公式）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'zh-hant','现金簿（查看）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'zh-hant','现金簿（查看，_change）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'zh-hant','现金簿（视图、_change、SQL）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'zh-hant','翻译', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'zh-hant','詳情', NULL, NULL);

INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'ACCOUNTS', '<table name="s02.accounts"><columnFormats><column name="" property="ListObjectName" value="accounts" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="5" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="name" property="Address" value="$D$4" type="String"/><column name="name" property="ColumnWidth" value="27.86" type="Double"/><column name="name" property="NumberFormat" value="General" type="String"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/><column name="" property="PageSetup.PaperSize" value="1" type="Double"/></columnFormats></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'ITEMS', '<table name="s02.items"><columnFormats><column name="" property="ListObjectName" value="items" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="5" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="name" property="Address" value="$D$4" type="String"/><column name="name" property="ColumnWidth" value="27.86" type="Double"/><column name="name" property="NumberFormat" value="General" type="String"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/><column name="" property="PageSetup.PaperSize" value="1" type="Double"/></columnFormats></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'COMPANIES', '<table name="s02.companies"><columnFormats><column name="" property="ListObjectName" value="companies" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="5" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="name" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="name" property="Address" value="$D$4" type="String"/><column name="name" property="ColumnWidth" value="27.86" type="Double"/><column name="name" property="NumberFormat" value="General" type="String"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/><column name="" property="PageSetup.PaperSize" value="1" type="Double"/></columnFormats></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'ITEM_COMPANIES', '<table name="s02.item_companies"><columnFormats><column name="" property="ListObjectName" value="item_companies" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$C$4" type="String"/><column name="item_id" property="ColumnWidth" value="27.86" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$D$4" type="String"/><column name="company_id" property="ColumnWidth" value="27.86" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="_State_" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="_State_" property="Address" value="$E$4" type="String"/><column name="_State_" property="ColumnWidth" value="9.14" type="Double"/><column name="_State_" property="NumberFormat" value="General" type="String"/><column name="_State_" property="HorizontalAlignment" value="-4108" type="Double"/><column name="_State_" property="Font.Size" value="10" type="Double"/><column name="_State_" property="FormatConditions(1).AppliesTo.Address" value="$E$4:$E$20" type="String"/><column name="_State_" property="FormatConditions(1).Type" value="6" type="Double"/><column name="_State_" property="FormatConditions(1).Priority" value="1" type="Double"/><column name="_State_" property="FormatConditions(1).ShowIconOnly" value="True" type="Boolean"/><column name="_State_" property="FormatConditions(1).IconSet.ID" value="8" type="Double"/><column name="_State_" property="FormatConditions(1).IconCriteria(1).Type" value="3" type="Double"/><column name="_State_" property="FormatConditions(1).IconCriteria(1).Operator" value="7" type="Double"/><column name="_State_" property="FormatConditions(1).IconCriteria(2).Type" value="0" type="Double"/><column name="_State_" property="FormatConditions(1).IconCriteria(2).Value" value="0.5" type="Double"/><column name="_State_" property="FormatConditions(1).IconCriteria(2).Operator" value="7" type="Double"/><column name="_State_" property="FormatConditions(1).IconCriteria(3).Type" value="0" type="Double"/><column name="_State_" property="FormatConditions(1).IconCriteria(3).Value" value="1" type="Double"/><column name="_State_" property="FormatConditions(1).IconCriteria(3).Operator" value="7" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'USP_CASH_BY_MONTHS', '<table name="s02.usp_cash_by_months"><columnFormats><column name="" property="ListObjectName" value="Sheet1_Table16" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="sort_order" property="Address" value="$C$4" type="String"/><column name="sort_order" property="NumberFormat" value="General" type="String"/><column name="section" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="section" property="Address" value="$D$4" type="String"/><column name="section" property="NumberFormat" value="General" type="String"/><column name="level" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="level" property="Address" value="$E$4" type="String"/><column name="level" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="Name" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Name" property="Address" value="$H$4" type="String"/><column name="Name" property="ColumnWidth" value="21.43" type="Double"/><column name="Name" property="NumberFormat" value="General" type="String"/><column name="Total" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Total" property="Address" value="$I$4" type="String"/><column name="Total" property="ColumnWidth" value="8.43" type="Double"/><column name="Total" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Jan" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jan" property="Address" value="$J$4" type="String"/><column name="Jan" property="ColumnWidth" value="10" type="Double"/><column name="Jan" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Feb" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Feb" property="Address" value="$K$4" type="String"/><column name="Feb" property="ColumnWidth" value="10" type="Double"/><column name="Feb" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Mar" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Mar" property="Address" value="$L$4" type="String"/><column name="Mar" property="ColumnWidth" value="10" type="Double"/><column name="Mar" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Apr" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Apr" property="Address" value="$M$4" type="String"/><column name="Apr" property="ColumnWidth" value="10" type="Double"/><column name="Apr" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="May" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="May" property="Address" value="$N$4" type="String"/><column name="May" property="ColumnWidth" value="10" type="Double"/><column name="May" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Jun" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jun" property="Address" value="$O$4" type="String"/><column name="Jun" property="ColumnWidth" value="10" type="Double"/><column name="Jun" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Jul" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jul" property="Address" value="$P$4" type="String"/><column name="Jul" property="ColumnWidth" value="10" type="Double"/><column name="Jul" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Aug" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Aug" property="Address" value="$Q$4" type="String"/><column name="Aug" property="ColumnWidth" value="10" type="Double"/><column name="Aug" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Sep" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Sep" property="Address" value="$R$4" type="String"/><column name="Sep" property="ColumnWidth" value="10" type="Double"/><column name="Sep" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Oct" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Oct" property="Address" value="$S$4" type="String"/><column name="Oct" property="ColumnWidth" value="10" type="Double"/><column name="Oct" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Nov" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Nov" property="Address" value="$T$4" type="String"/><column name="Nov" property="ColumnWidth" value="10" type="Double"/><column name="Nov" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Dec" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Dec" property="Address" value="$U$4" type="String"/><column name="Dec" property="ColumnWidth" value="10" type="Double"/><column name="Dec" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean"/><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$U$20" type="String"/><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double"/><column name="_RowNum" property="FormatConditions(1).Priority" value="3" type="Double"/><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$E4&lt;2" type="String"/><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean"/><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean"/><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$U$20" type="String"/><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double"/><column name="_RowNum" property="FormatConditions(2).Priority" value="4" type="Double"/><column name="_RowNum" property="FormatConditions(2).Formula1" value="=AND($E4=0,$D4&gt;1,$D4&lt;5)" type="String"/><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean"/><column name="_RowNum" property="FormatConditions(2).Font.Color" value="16777215" type="Double"/><column name="_RowNum" property="FormatConditions(2).Font.ThemeColor" value="1" type="Double"/><column name="_RowNum" property="FormatConditions(2).Font.TintAndShade" value="0" type="Double"/><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="6773025" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All columns"><column name="" property="ListObjectName" value="cash_by_month" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="section" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="level" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Name" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jan" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Feb" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Mar" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Apr" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="May" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jun" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jul" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Aug" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Sep" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Oct" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Nov" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Dec" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Default"><column name="" property="ListObjectName" value="cash_by_month" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="section" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="level" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="Name" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jan" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Feb" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Mar" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Apr" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="May" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jun" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jul" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Aug" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Sep" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Oct" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Nov" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Dec" property="EntireColumn.Hidden" value="False" type="Boolean"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'CASHBOOK', '<table name="s02.cashbook"><columnFormats><column name="" property="ListObjectName" value="cashbook" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'VIEW_CASHBOOK', '<table name="s02.view_cashbook"><columnFormats><column name="" property="ListObjectName" value="view_cashbook" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'VIEW_CASHBOOK2', '<table name="s02.view_cashbook2"><columnFormats><column name="" property="ListObjectName" value="view_cashbook2" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'VIEW_CASHBOOK3', '<table name="s02.view_cashbook3"><columnFormats><column name="" property="ListObjectName" value="view_cashbook3" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'USP_CASHBOOK', '<table name="s02.usp_cashbook"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'USP_CASHBOOK2', '<table name="s02.usp_cashbook2"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook2" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'USP_CASHBOOK3', '<table name="s02.usp_cashbook3"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook3" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'USP_CASHBOOK4', '<table name="s02.usp_cashbook4"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook4" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'USP_CASHBOOK5', '<table name="s02.usp_cashbook5"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook5" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'CODE_CASHBOOK', '<table name="s02.code_cashbook"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="Address" value="$E$4" type="String"/><column name="account_id" property="ColumnWidth" value="12.14" type="Double"/><column name="account_id" property="NumberFormat" value="General" type="String"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="Address" value="$F$4" type="String"/><column name="item_id" property="ColumnWidth" value="20.71" type="Double"/><column name="item_id" property="NumberFormat" value="General" type="String"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="Address" value="$G$4" type="String"/><column name="company_id" property="ColumnWidth" value="20.71" type="Double"/><column name="company_id" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company_id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S02', 'VIEW_TRANSLATIONS', '<table name="s02.view_translations"><columnFormats><column name="" property="ListObjectName" value="view_translations" type="String" /><column name="" property="ShowTotals" value="False" type="Boolean" /><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String" /><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" /><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" /><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" /><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean" /><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="_RowNum" property="Address" value="$B$4" type="String" /><column name="_RowNum" property="ColumnWidth" value="0.08" type="Double" /><column name="_RowNum" property="NumberFormat" value="General" type="String" /><column name="ID" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="ID" property="Address" value="$C$4" type="String" /><column name="ID" property="ColumnWidth" value="4.86" type="Double" /><column name="ID" property="NumberFormat" value="#,##0" type="String" /><column name="ID" property="Validation.Type" value="1" type="Double" /><column name="ID" property="Validation.Operator" value="1" type="Double" /><column name="ID" property="Validation.Formula1" value="-2147483648" type="String" /><column name="ID" property="Validation.Formula2" value="2147483647" type="String" /><column name="ID" property="Validation.AlertStyle" value="2" type="Double" /><column name="ID" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="ID" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="ID" property="Validation.ErrorTitle" value="Data Type Control" type="String" /><column name="ID" property="Validation.ErrorMessage" value="The column requires values of the int data type." type="String" /><column name="ID" property="Validation.ShowInput" value="True" type="Boolean" /><column name="ID" property="Validation.ShowError" value="True" type="Boolean" /><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String" /><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double" /><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String" /><column name="TABLE_SCHEMA" property="Validation.Type" value="6" type="Double" /><column name="TABLE_SCHEMA" property="Validation.Operator" value="8" type="Double" /><column name="TABLE_SCHEMA" property="Validation.Formula1" value="128" type="String" /><column name="TABLE_SCHEMA" property="Validation.AlertStyle" value="2" type="Double" /><column name="TABLE_SCHEMA" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="TABLE_SCHEMA" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="TABLE_SCHEMA" property="Validation.ErrorTitle" value="Data Type Control" type="String" /><column name="TABLE_SCHEMA" property="Validation.ErrorMessage" value="The column requires values of the nvarchar(128) data type." type="String" /><column name="TABLE_SCHEMA" property="Validation.ShowInput" value="True" type="Boolean" /><column name="TABLE_SCHEMA" property="Validation.ShowError" value="True" type="Boolean" /><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="TABLE_NAME" property="Address" value="$E$4" type="String" /><column name="TABLE_NAME" property="ColumnWidth" value="25.14" type="Double" /><column name="TABLE_NAME" property="NumberFormat" value="General" type="String" /><column name="TABLE_NAME" property="Validation.Type" value="6" type="Double" /><column name="TABLE_NAME" property="Validation.Operator" value="8" type="Double" /><column name="TABLE_NAME" property="Validation.Formula1" value="128" type="String" /><column name="TABLE_NAME" property="Validation.AlertStyle" value="2" type="Double" /><column name="TABLE_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="TABLE_NAME" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="TABLE_NAME" property="Validation.ErrorTitle" value="Data Type Control" type="String" /><column name="TABLE_NAME" property="Validation.ErrorMessage" value="The column requires values of the nvarchar(128) data type." type="String" /><column name="TABLE_NAME" property="Validation.ShowInput" value="True" type="Boolean" /><column name="TABLE_NAME" property="Validation.ShowError" value="True" type="Boolean" /><column name="COLUMN_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="COLUMN_NAME" property="Address" value="$F$4" type="String" /><column name="COLUMN_NAME" property="ColumnWidth" value="19.86" type="Double" /><column name="COLUMN_NAME" property="NumberFormat" value="General" type="String" /><column name="COLUMN_NAME" property="Validation.Type" value="6" type="Double" /><column name="COLUMN_NAME" property="Validation.Operator" value="8" type="Double" /><column name="COLUMN_NAME" property="Validation.Formula1" value="128" type="String" /><column name="COLUMN_NAME" property="Validation.AlertStyle" value="2" type="Double" /><column name="COLUMN_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="COLUMN_NAME" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="COLUMN_NAME" property="Validation.ErrorTitle" value="Data Type Control" type="String" /><column name="COLUMN_NAME" property="Validation.ErrorMessage" value="The column requires values of the nvarchar(128) data type." type="String" /><column name="COLUMN_NAME" property="Validation.ShowInput" value="True" type="Boolean" /><column name="COLUMN_NAME" property="Validation.ShowError" value="True" type="Boolean" /><column name="LANGUAGE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="LANGUAGE_NAME" property="Address" value="$G$4" type="String" /><column name="LANGUAGE_NAME" property="ColumnWidth" value="19.57" type="Double" /><column name="LANGUAGE_NAME" property="NumberFormat" value="General" type="String" /><column name="LANGUAGE_NAME" property="Validation.Type" value="6" type="Double" /><column name="LANGUAGE_NAME" property="Validation.Operator" value="8" type="Double" /><column name="LANGUAGE_NAME" property="Validation.Formula1" value="10" type="String" /><column name="LANGUAGE_NAME" property="Validation.AlertStyle" value="2" type="Double" /><column name="LANGUAGE_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="LANGUAGE_NAME" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="LANGUAGE_NAME" property="Validation.ErrorTitle" value="Data Type Control" type="String" /><column name="LANGUAGE_NAME" property="Validation.ErrorMessage" value="The column requires values of the varchar(10) data type." type="String" /><column name="LANGUAGE_NAME" property="Validation.ShowInput" value="True" type="Boolean" /><column name="LANGUAGE_NAME" property="Validation.ShowError" value="True" type="Boolean" /><column name="TRANSLATED_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" /><column name="TRANSLATED_NAME" property="Address" value="$H$4" type="String" /><column name="TRANSLATED_NAME" property="ColumnWidth" value="31.14" type="Double" /><column name="TRANSLATED_NAME" property="NumberFormat" value="General" type="String" /><column name="TRANSLATED_NAME" property="Validation.Type" value="6" type="Double" /><column name="TRANSLATED_NAME" property="Validation.Operator" value="8" type="Double" /><column name="TRANSLATED_NAME" property="Validation.Formula1" value="128" type="String" /><column name="TRANSLATED_NAME" property="Validation.AlertStyle" value="2" type="Double" /><column name="TRANSLATED_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean" /><column name="TRANSLATED_NAME" property="Validation.InCellDropdown" value="True" type="Boolean" /><column name="TRANSLATED_NAME" property="Validation.ErrorTitle" value="Data Type Control" type="String" /><column name="TRANSLATED_NAME" property="Validation.ErrorMessage" value="The column requires values of the nvarchar(128) data type." type="String" /><column name="TRANSLATED_NAME" property="Validation.ShowInput" value="True" type="Boolean" /><column name="TRANSLATED_NAME" property="Validation.ShowError" value="True" type="Boolean" /><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" /><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" /><column name="" property="ActiveWindow.Split" value="True" type="Boolean" /><column name="" property="ActiveWindow.SplitRow" value="0" type="Double" /><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" /><column name="" property="PageSetup.Orientation" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" /><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" /></columnFormats></table>');

INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User1.xlsx', 'https://www.savetodb.com/downloads/v10/sample02-user1.xlsx','cashbook=S02.CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"cashbook"}
view_cashbook=S02.VIEW_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"view_cashbook"}
usp_cashbook=S02.USP_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook"}
usp_cashbook2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2"}
usp_cashbook3=S02.USP_CASHBOOK3,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook3"}
usp_cashbook4=S02.USP_CASHBOOK4,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook4"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months"}', 'S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User2 (Restricted).xlsx', 'https://www.savetodb.com/downloads/v10/sample02-user2.xlsx','cashbook=S02.CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"cashbook"}
view_cashbook=S02.VIEW_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"view_cashbook"}
usp_cashbook=S02.USP_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook"}
usp_cashbook2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2"}
usp_cashbook3=S02.USP_CASHBOOK3,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook3"}
usp_cashbook4=S02.USP_CASHBOOK4,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook4"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months"}', 'S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (SaveToDB Framework).xlsx', 'https://www.savetodb.com/downloads/v10/sample02-user3.xlsx','cashbook=S02.CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"cashbook"}
view_cashbook=S02.VIEW_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"view_cashbook"}
view_cashbook2=S02.VIEW_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"view_cashbook2"}
view_cashbook3=S02.VIEW_CASHBOOK3,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"view_cashbook3"}
usp_cashbook=S02.USP_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook"}
usp_cashbook2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2"}
usp_cashbook3=S02.USP_CASHBOOK3,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook3"}
usp_cashbook4=S02.USP_CASHBOOK4,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook4"}
usp_cashbook5=S02.USP_CASHBOOK5,(Default),False,$B$3,,{"Parameters":{"account_id":null,"item_id":null,"company_id":null},"ListObjectName":"usp_cashbook5"}
code_cashbook=S02.CODE_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null},"ListObjectName":"code_cashbook"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months"}
objects=XLS.OBJECTS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"s02","TABLE_NAME":null,"TABLE_TYPE":null},"ListObjectName":"objects"}
handlers=XLS.HANDLERS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"s02","EVENT_NAME":null,"HANDLER_TYPE":null},"ListObjectName":"handlers"}
translations=XLS.TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"s02","LANGUAGE_NAME":"en"},"ListObjectName":"translations"}
workbooks=XLS.WORKBOOKS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"s02"},"ListObjectName":"workbooks"}', 'S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-en.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"en"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"en"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"en"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"en"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, Chinese Simplified).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-zh-hans.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"zh-hans"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"zh-hans"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"zh-hans"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"zh-hans"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, Chinese Traditional).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-zh-hant.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"zh-hant"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"zh-hant"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"zh-hant"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"zh-hant"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, French).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-fr.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"fr"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"fr"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"fr"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"fr"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, German).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-de.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"de"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"de"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"de"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"de"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, Italian).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-it.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"it"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"it"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"it"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"it"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, Japanese).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-ja.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"ja"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"ja"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"ja"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"ja"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, Korean).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-ko.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"ko"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"ko"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"ko"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"ko"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, Portuguese).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-pt.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"pt"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"pt"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"pt"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"pt"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, Russian).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-ru.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"ru"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"ru"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"ru"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"ru"}','S02');
INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 02 - Advanced Features - User3 (Translation, Spanish).xlsx','https://www.savetodb.com/downloads/v10/sample02-user3-es.xlsx','USP_CASHBOOK2=S02.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account_id":1,"item_id":null,"company_id":null,"start_date":null,"end_date":null,"checked":null},"ListObjectName":"usp_cashbook2","UseTranslatedName":true,"WorkbookLanguage":"es"}
cash_by_months=S02.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2023},"ListObjectName":"cash_by_months","UseTranslatedName":true,"WorkbookLanguage":"es"}
translations=S02.VIEW_TRANSLATIONS,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":"S02","LANGUAGE_NAME":"es"},"ListObjectName":"translations","UseTranslatedName":true,"WorkbookLanguage":"es"}','S02');

-- print Application installed
