-- =============================================
-- SaveToDB Framework for NuoDB
-- Version 9.12, July 5, 2022
--
-- Copyright 2014-2022 Gartle LLC
-- =============================================

CREATE SCHEMA XLS;

CREATE TABLE IF NOT EXISTS XLS.OBJECTS (
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING NOT NULL,
    TABLE_NAME STRING NOT NULL,
    TABLE_TYPE VARCHAR(64) NOT NULL,
    TABLE_CODE STRING,
    INSERT_OBJECT STRING,
    UPDATE_OBJECT STRING,
    DELETE_OBJECT STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_OBJECTS ON XLS.OBJECTS (TABLE_SCHEMA, TABLE_NAME);

CREATE TABLE IF NOT EXISTS XLS.HANDLERS (
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING NOT NULL,
    TABLE_NAME STRING NOT NULL,
    COLUMN_NAME STRING,
    EVENT_NAME VARCHAR(50) NOT NULL,
    HANDLER_SCHEMA STRING,
    HANDLER_NAME STRING,
    HANDLER_TYPE VARCHAR(64),
    HANDLER_CODE STRING,
    TARGET_WORKSHEET VARCHAR(256),
    MENU_ORDER SMALLINT,
    EDIT_PARAMETERS BOOLEAN,
    PRIMARY KEY (ID)
);

CREATE TABLE IF NOT EXISTS XLS.TRANSLATIONS (
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING,
    TABLE_NAME STRING,
    COLUMN_NAME STRING,
    LANGUAGE_NAME STRING NOT NULL,
    TRANSLATED_NAME STRING,
    TRANSLATED_DESC STRING,
    TRANSLATED_COMMENT STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_TRANSLATIONS ON XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME);

CREATE TABLE IF NOT EXISTS XLS.FORMATS (
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING NOT NULL,
    TABLE_NAME STRING NOT NULL,
    TABLE_EXCEL_FORMAT_XML STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_FORMATS ON XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME);

CREATE TABLE XLS.WORKBOOKS (
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    NAME STRING NOT NULL,
    TEMPLATE STRING,
    DEFINITION STRING NOT NULL,
    TABLE_SCHEMA STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_WORKBOOKS ON XLS.WORKBOOKS (NAME);

CREATE VIEW XLS.QUERIES
AS
SELECT
    t.SCHEMA AS TABLE_SCHEMA
    , t.TABLENAME AS TABLE_NAME
    , t.TYPE AS TABLE_TYPE
    , NULL AS TABLE_CODE
    , NULL AS INSERT_PROCEDURE
    , NULL AS UPDATE_PROCEDURE
    , NULL AS DELETE_PROCEDURE
    , NULL AS PROCEDURE_TYPE
FROM
    SYSTEM.TABLES t
WHERE
    NOT t.SCHEMA IN ('SYSTEM', 'XLS', 'SAVETODB_DEV', 'SAVETODB_XLS')
    AND NOT t.TABLENAME LIKE 'XL_%'
UNION
SELECT
    r.SCHEMA AS TABLE_SCHEMA
    , r.PROCEDURENAME AS TABLE_NAME
    , 'PROCEDURE' AS TABLE_TYPE
    , NULL AS TABLE_CODE
    , NULL AS INSERT_PROCEDURE
    , NULL AS UPDATE_PROCEDURE
    , NULL AS DELETE_PROCEDURE
    , NULL AS PROCEDURE_TYPE
FROM
    SYSTEM.PROCEDURES r
WHERE
    NOT r.SCHEMA IN ('SYSTEM', 'XLS', 'SAVETODB_DEV', 'SAVETODB_XLS')
    AND NOT (
        r.PROCEDURENAME LIKE '%_INSERT'
        OR r.PROCEDURENAME LIKE '%_UPDATE'
        OR r.PROCEDURENAME LIKE '%_DELETE'
        OR r.PROCEDURENAME LIKE '%_MERGE'
        OR r.PROCEDURENAME LIKE '%_CHANGE'
        OR r.PROCEDURENAME LIKE 'XL_%'
    )
UNION
SELECT
    o.TABLE_SCHEMA
    , o.TABLE_NAME
    , o.TABLE_TYPE
    , o.TABLE_CODE
    , o.INSERT_OBJECT AS INSERT_PROCEDURE
    , o.UPDATE_OBJECT AS UPDATE_PROCEDURE
    , o.DELETE_OBJECT AS DELETE_PROCEDURE
    , NULL AS PROCEDURE_TYPE
FROM
    XLS.OBJECTS o
WHERE
    o.TABLE_TYPE IN ('CODE', 'HTTP', 'TEXT')
    AND o.TABLE_SCHEMA IS NOT NULL
    AND o.TABLE_NAME IS NOT NULL
    AND o.TABLE_CODE IS NOT NULL
    AND NOT o.TABLE_NAME LIKE 'XL_%';

CREATE ROLE XLS.XLS_USERS;
CREATE ROLE XLS.XLS_DEVELOPERS;

GRANT SELECT ON TABLE XLS.OBJECTS       TO ROLE XLS.XLS_USERS;
GRANT SELECT ON TABLE XLS.HANDLERS      TO ROLE XLS.XLS_USERS;
GRANT SELECT ON TABLE XLS.TRANSLATIONS  TO ROLE XLS.XLS_USERS;
GRANT SELECT ON TABLE XLS.FORMATS       TO ROLE XLS.XLS_USERS;
GRANT SELECT ON TABLE XLS.QUERIES       TO ROLE XLS.XLS_USERS;
GRANT SELECT ON TABLE XLS.WORKBOOKS     TO ROLE XLS.XLS_USERS;

GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE XLS.OBJECTS       TO ROLE XLS.XLS_DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE XLS.HANDLERS      TO ROLE XLS.XLS_DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE XLS.TRANSLATIONS  TO ROLE XLS.XLS_DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE XLS.FORMATS       TO ROLE XLS.XLS_DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE XLS.QUERIES       TO ROLE XLS.XLS_DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE XLS.WORKBOOKS     TO ROLE XLS.XLS_DEVELOPERS;

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE) VALUES ('XLS', 'SAVETODB_FRAMEWORK', 'VERSION', 'Information', NULL, NULL, 'ATTRIBUTE', '9.12');
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE) VALUES ('XLS', 'OBJECTS',  'TABLE_TYPE',     'ValidationList', NULL, NULL, 'VALUES', 'TABLE, VIEW, PROCEDURE, CODE, HTTP, TEXT, HIDDEN');
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE) VALUES ('XLS', 'HANDLERS', 'EVENT_NAME',     'ValidationList', NULL, NULL, 'VALUES', 'Actions, AddHyperlinks, AddStateColumn, BitColumn, Change, ContextMenu, ConvertFormulas, DataTypeBit, DataTypeBoolean, DataTypeDate, DataTypeDateTime, DataTypeDateTimeOffset, DataTypeDouble, DataTypeInt, DataTypeGuid, DataTypeString, DataTypeTime, DataTypeTimeSpan, DefaultListObject, DefaultValue, DependsOn, DoNotAddChangeHandler, DoNotAddDependsOn, DoNotAddManyToMany, DoNotAddValidation, DoNotChange, DoNotConvertFormulas, DoNotKeepComments, DoNotKeepFormulas, DoNotSave, DoNotSelect, DoNotSort, DoNotTranslate, DoubleClick, DynamicColumns, Format, Formula, FormulaValue, Information, JsonForm, KeepFormulas, KeepComments, License, ManyToMany, ParameterValues, ProtectRows, RegEx, SelectionChange, SelectionList, SelectPeriod, SyncParameter, UpdateChangedCellsOnly, UpdateEntireRow, ValidationList');
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE) VALUES ('XLS', 'HANDLERS', 'HANDLER_TYPE',   'ValidationList', NULL, NULL, 'VALUES', 'TABLE, VIEW, PROCEDURE, FUNCTION, CODE, HTTP, TEXT, MACRO, CMD, VALUES, RANGE, REFRESH, MENUSEPARATOR, PDF, REPORT, SHOWSHEETS, HIDESHEETS, SELECTSHEET, ATTRIBUTE');
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE) VALUES ('XLS', 'HANDLERS', 'HANDLER_CODE',   'DoNotConvertFormulas', NULL, NULL, 'ATTRIBUTE', NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('XLS', 'FORMATS', NULL, 'Actions', 'XLS', 'Developer Guide', 'HTTP', 'https://www.savetodb.com/dev-guide/xls-formats.htm', NULL, 13, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('XLS', 'HANDLERS', NULL, 'Actions', 'XLS', 'Developer Guide', 'HTTP', 'https://www.savetodb.com/dev-guide/xls-handlers.htm', NULL, 13, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('XLS', 'OBJECTS', NULL, 'Actions', 'XLS', 'Developer Guide', 'HTTP', 'https://www.savetodb.com/dev-guide/xls-objects.htm', NULL, 13, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('XLS', 'QUERIES', NULL, 'Actions', 'XLS', 'Developer Guide', 'HTTP', 'https://www.savetodb.com/dev-guide/xls-queries.htm', NULL, 13, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('XLS', 'TRANSLATIONS', NULL, 'Actions', 'XLS', 'Developer Guide', 'HTTP', 'https://www.savetodb.com/dev-guide/xls-translations.htm', NULL, 13, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('XLS', 'WORKBOOKS', NULL, 'Actions', 'XLS', 'Developer Guide', 'HTTP', 'https://www.savetodb.com/dev-guide/xls-workbooks.htm', NULL, 13, NULL);

INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('XLS', 'FORMATS', '<table name="XLS.FORMATS"><columnFormats><column name="" property="ListObjectName" value="formats" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$E$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_EXCEL_FORMAT_XML" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_EXCEL_FORMAT_XML" property="Address" value="$F$4" type="String"/><column name="TABLE_EXCEL_FORMAT_XML" property="ColumnWidth" value="42.29" type="Double"/><column name="TABLE_EXCEL_FORMAT_XML" property="NumberFormat" value="General" type="String"/><column name="SortFields(1)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="0" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="0" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('XLS', 'HANDLERS', '<table name="XLS.HANDLERS"><columnFormats><column name="" property="ListObjectName" value="handlers" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$E$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="COLUMN_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="COLUMN_NAME" property="Address" value="$F$4" type="String"/><column name="COLUMN_NAME" property="ColumnWidth" value="17.43" type="Double"/><column name="COLUMN_NAME" property="NumberFormat" value="General" type="String"/><column name="COLUMN_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="EVENT_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="EVENT_NAME" property="Address" value="$G$4" type="String"/><column name="EVENT_NAME" property="ColumnWidth" value="21.57" type="Double"/><column name="EVENT_NAME" property="NumberFormat" value="General" type="String"/><column name="EVENT_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="HANDLER_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="HANDLER_SCHEMA" property="Address" value="$H$4" type="String"/><column name="HANDLER_SCHEMA" property="ColumnWidth" value="19.71" type="Double"/><column name="HANDLER_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="HANDLER_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="HANDLER_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="HANDLER_NAME" property="Address" value="$I$4" type="String"/><column name="HANDLER_NAME" property="ColumnWidth" value="31.14" type="Double"/><column name="HANDLER_NAME" property="NumberFormat" value="General" type="String"/><column name="HANDLER_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="HANDLER_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="HANDLER_TYPE" property="Address" value="$J$4" type="String"/><column name="HANDLER_TYPE" property="ColumnWidth" value="16.29" type="Double"/><column name="HANDLER_TYPE" property="NumberFormat" value="General" type="String"/><column name="HANDLER_TYPE" property="VerticalAlignment" value="-4160" type="Double"/><column name="HANDLER_CODE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="HANDLER_CODE" property="Address" value="$K$4" type="String"/><column name="HANDLER_CODE" property="ColumnWidth" value="70.71" type="Double"/><column name="HANDLER_CODE" property="NumberFormat" value="General" type="String"/><column name="HANDLER_CODE" property="VerticalAlignment" value="-4160" type="Double"/><column name="TARGET_WORKSHEET" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TARGET_WORKSHEET" property="Address" value="$L$4" type="String"/><column name="TARGET_WORKSHEET" property="ColumnWidth" value="21.71" type="Double"/><column name="TARGET_WORKSHEET" property="NumberFormat" value="General" type="String"/><column name="TARGET_WORKSHEET" property="VerticalAlignment" value="-4160" type="Double"/><column name="MENU_ORDER" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="MENU_ORDER" property="Address" value="$M$4" type="String"/><column name="MENU_ORDER" property="ColumnWidth" value="15.43" type="Double"/><column name="MENU_ORDER" property="NumberFormat" value="General" type="String"/><column name="MENU_ORDER" property="VerticalAlignment" value="-4160" type="Double"/><column name="EDIT_PARAMETERS" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="EDIT_PARAMETERS" property="Address" value="$N$4" type="String"/><column name="EDIT_PARAMETERS" property="ColumnWidth" value="19.57" type="Double"/><column name="EDIT_PARAMETERS" property="NumberFormat" value="General" type="String"/><column name="EDIT_PARAMETERS" property="HorizontalAlignment" value="-4108" type="Double"/><column name="EDIT_PARAMETERS" property="VerticalAlignment" value="-4160" type="Double"/><column name="EDIT_PARAMETERS" property="Font.Size" value="10" type="Double"/><column name="SortFields(1)" property="KeyfieldName" value="EVENT_NAME" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="0" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="0" type="Double"/><column name="SortFields(3)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(3)" property="SortOn" value="0" type="Double"/><column name="SortFields(3)" property="Order" value="1" type="Double"/><column name="SortFields(3)" property="DataOption" value="0" type="Double"/><column name="SortFields(4)" property="KeyfieldName" value="COLUMN_NAME" type="String"/><column name="SortFields(4)" property="SortOn" value="0" type="Double"/><column name="SortFields(4)" property="Order" value="1" type="Double"/><column name="SortFields(4)" property="DataOption" value="0" type="Double"/><column name="SortFields(5)" property="KeyfieldName" value="MENU_ORDER" type="String"/><column name="SortFields(5)" property="SortOn" value="0" type="Double"/><column name="SortFields(5)" property="Order" value="1" type="Double"/><column name="SortFields(5)" property="DataOption" value="0" type="Double"/><column name="SortFields(6)" property="KeyfieldName" value="HANDLER_SCHEMA" type="String"/><column name="SortFields(6)" property="SortOn" value="0" type="Double"/><column name="SortFields(6)" property="Order" value="1" type="Double"/><column name="SortFields(6)" property="DataOption" value="0" type="Double"/><column name="SortFields(7)" property="KeyfieldName" value="HANDLER_NAME" type="String"/><column name="SortFields(7)" property="SortOn" value="0" type="Double"/><column name="SortFields(7)" property="Order" value="1" type="Double"/><column name="SortFields(7)" property="DataOption" value="0" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('XLS', 'OBJECTS', '<table name="XLS.OBJECTS"><columnFormats><column name="" property="ListObjectName" value="objects" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$E$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_TYPE" property="Address" value="$F$4" type="String"/><column name="TABLE_TYPE" property="ColumnWidth" value="13.14" type="Double"/><column name="TABLE_TYPE" property="NumberFormat" value="General" type="String"/><column name="TABLE_TYPE" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_TYPE" property="Validation.Type" value="3" type="Double"/><column name="TABLE_TYPE" property="Validation.Operator" value="1" type="Double"/><column name="TABLE_TYPE" property="Validation.Formula1" value="TABLE; VIEW; PROCEDURE; CODE; HTTP; TEXT; HIDDEN" type="String"/><column name="TABLE_TYPE" property="Validation.AlertStyle" value="1" type="Double"/><column name="TABLE_TYPE" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="TABLE_TYPE" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="TABLE_TYPE" property="Validation.ShowInput" value="True" type="Boolean"/><column name="TABLE_TYPE" property="Validation.ShowError" value="True" type="Boolean"/><column name="TABLE_CODE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_CODE" property="Address" value="$G$4" type="String"/><column name="TABLE_CODE" property="ColumnWidth" value="13.57" type="Double"/><column name="TABLE_CODE" property="NumberFormat" value="General" type="String"/><column name="TABLE_CODE" property="VerticalAlignment" value="-4160" type="Double"/><column name="INSERT_OBJECT" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="INSERT_OBJECT" property="Address" value="$H$4" type="String"/><column name="INSERT_OBJECT" property="ColumnWidth" value="27.86" type="Double"/><column name="INSERT_OBJECT" property="NumberFormat" value="General" type="String"/><column name="INSERT_OBJECT" property="VerticalAlignment" value="-4160" type="Double"/><column name="UPDATE_OBJECT" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="UPDATE_OBJECT" property="Address" value="$I$4" type="String"/><column name="UPDATE_OBJECT" property="ColumnWidth" value="27.86" type="Double"/><column name="UPDATE_OBJECT" property="NumberFormat" value="General" type="String"/><column name="UPDATE_OBJECT" property="VerticalAlignment" value="-4160" type="Double"/><column name="DELETE_OBJECT" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="DELETE_OBJECT" property="Address" value="$J$4" type="String"/><column name="DELETE_OBJECT" property="ColumnWidth" value="27.86" type="Double"/><column name="DELETE_OBJECT" property="NumberFormat" value="General" type="String"/><column name="DELETE_OBJECT" property="VerticalAlignment" value="-4160" type="Double"/><column name="SortFields(1)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="2" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="2" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('XLS', 'QUERIES', '<table name="XLS.QUERIES"><columnFormats><column name="" property="ListObjectName" value="queries" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$C$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$D$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_TYPE" property="Address" value="$E$4" type="String"/><column name="TABLE_TYPE" property="ColumnWidth" value="13.14" type="Double"/><column name="TABLE_TYPE" property="NumberFormat" value="General" type="String"/><column name="TABLE_CODE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_CODE" property="Address" value="$F$4" type="String"/><column name="TABLE_CODE" property="ColumnWidth" value="13.57" type="Double"/><column name="TABLE_CODE" property="NumberFormat" value="General" type="String"/><column name="INSERT_PROCEDURE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="INSERT_PROCEDURE" property="Address" value="$G$4" type="String"/><column name="INSERT_PROCEDURE" property="ColumnWidth" value="27.86" type="Double"/><column name="INSERT_PROCEDURE" property="NumberFormat" value="General" type="String"/><column name="UPDATE_PROCEDURE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="UPDATE_PROCEDURE" property="Address" value="$H$4" type="String"/><column name="UPDATE_PROCEDURE" property="ColumnWidth" value="27.86" type="Double"/><column name="UPDATE_PROCEDURE" property="NumberFormat" value="General" type="String"/><column name="DELETE_PROCEDURE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="DELETE_PROCEDURE" property="Address" value="$I$4" type="String"/><column name="DELETE_PROCEDURE" property="ColumnWidth" value="27.86" type="Double"/><column name="DELETE_PROCEDURE" property="NumberFormat" value="General" type="String"/><column name="PROCEDURE_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="PROCEDURE_TYPE" property="Address" value="$J$4" type="String"/><column name="PROCEDURE_TYPE" property="ColumnWidth" value="18.86" type="Double"/><column name="PROCEDURE_TYPE" property="NumberFormat" value="General" type="String"/><column name="SortFields(1)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="2" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="2" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('XLS', 'TRANSLATIONS', '<table name="XLS.TRANSLATIONS"><columnFormats><column name="" property="ListObjectName" value="translations" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$D$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_NAME" property="Address" value="$E$4" type="String"/><column name="TABLE_NAME" property="ColumnWidth" value="32.14" type="Double"/><column name="TABLE_NAME" property="NumberFormat" value="General" type="String"/><column name="TABLE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="COLUMN_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="COLUMN_NAME" property="Address" value="$F$4" type="String"/><column name="COLUMN_NAME" property="ColumnWidth" value="20.71" type="Double"/><column name="COLUMN_NAME" property="NumberFormat" value="General" type="String"/><column name="COLUMN_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="LANGUAGE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="LANGUAGE_NAME" property="Address" value="$G$4" type="String"/><column name="LANGUAGE_NAME" property="ColumnWidth" value="19.57" type="Double"/><column name="LANGUAGE_NAME" property="NumberFormat" value="General" type="String"/><column name="LANGUAGE_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TRANSLATED_NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TRANSLATED_NAME" property="Address" value="$H$4" type="String"/><column name="TRANSLATED_NAME" property="ColumnWidth" value="30" type="Double"/><column name="TRANSLATED_NAME" property="NumberFormat" value="General" type="String"/><column name="TRANSLATED_NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TRANSLATED_DESC" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TRANSLATED_DESC" property="Address" value="$I$4" type="String"/><column name="TRANSLATED_DESC" property="ColumnWidth" value="19.57" type="Double"/><column name="TRANSLATED_DESC" property="NumberFormat" value="General" type="String"/><column name="TRANSLATED_DESC" property="VerticalAlignment" value="-4160" type="Double"/><column name="TRANSLATED_COMMENT" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TRANSLATED_COMMENT" property="Address" value="$J$4" type="String"/><column name="TRANSLATED_COMMENT" property="ColumnWidth" value="25" type="Double"/><column name="TRANSLATED_COMMENT" property="NumberFormat" value="General" type="String"/><column name="TRANSLATED_COMMENT" property="VerticalAlignment" value="-4160" type="Double"/><column name="SortFields(1)" property="KeyfieldName" value="LANGUAGE_NAME" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="2" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="2" type="Double"/><column name="SortFields(3)" property="KeyfieldName" value="TABLE_NAME" type="String"/><column name="SortFields(3)" property="SortOn" value="0" type="Double"/><column name="SortFields(3)" property="Order" value="1" type="Double"/><column name="SortFields(3)" property="DataOption" value="2" type="Double"/><column name="SortFields(4)" property="KeyfieldName" value="COLUMN_NAME" type="String"/><column name="SortFields(4)" property="SortOn" value="0" type="Double"/><column name="SortFields(4)" property="Order" value="1" type="Double"/><column name="SortFields(4)" property="DataOption" value="2" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');
INSERT INTO XLS.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('XLS', 'WORKBOOKS', '<table name="XLS.WORKBOOKS"><columnFormats><column name="" property="ListObjectName" value="workbooks" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double"/><column name="ID" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="ID" property="Address" value="$C$4" type="String"/><column name="ID" property="NumberFormat" value="General" type="String"/><column name="ID" property="VerticalAlignment" value="-4160" type="Double"/><column name="NAME" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="NAME" property="Address" value="$D$4" type="String"/><column name="NAME" property="ColumnWidth" value="42.14" type="Double"/><column name="NAME" property="NumberFormat" value="General" type="String"/><column name="NAME" property="VerticalAlignment" value="-4160" type="Double"/><column name="TEMPLATE" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TEMPLATE" property="Address" value="$E$4" type="String"/><column name="TEMPLATE" property="ColumnWidth" value="30" type="Double"/><column name="TEMPLATE" property="NumberFormat" value="General" type="String"/><column name="TEMPLATE" property="VerticalAlignment" value="-4160" type="Double"/><column name="DEFINITION" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="DEFINITION" property="Address" value="$F$4" type="String"/><column name="DEFINITION" property="ColumnWidth" value="70.71" type="Double"/><column name="DEFINITION" property="NumberFormat" value="General" type="String"/><column name="DEFINITION" property="VerticalAlignment" value="-4160" type="Double"/><column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="TABLE_SCHEMA" property="Address" value="$G$4" type="String"/><column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double"/><column name="TABLE_SCHEMA" property="NumberFormat" value="General" type="String"/><column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double"/><column name="SortFields(1)" property="KeyfieldName" value="TABLE_SCHEMA" type="String"/><column name="SortFields(1)" property="SortOn" value="0" type="Double"/><column name="SortFields(1)" property="Order" value="1" type="Double"/><column name="SortFields(1)" property="DataOption" value="0" type="Double"/><column name="SortFields(2)" property="KeyfieldName" value="NAME" type="String"/><column name="SortFields(2)" property="SortOn" value="0" type="Double"/><column name="SortFields(2)" property="Order" value="1" type="Double"/><column name="SortFields(2)" property="DataOption" value="0" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats></table>');

INSERT INTO XLS.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('savetodb_configuration.xlsx', NULL,
'objects=xls.objects,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":null,"TABLE_TYPE":null},"ListObjectName":"objects"}
handlers=xls.handlers,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":null,"EVENT_NAME":null,"HANDLER_TYPE":null},"ListObjectName":"handlers"}
translations=xls.translations,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":null,"LANGUAGE_NAME":null},"ListObjectName":"translations"}
workbooks=xls.workbooks,(Default),False,$B$3,,{"Parameters":{"TABLE_SCHEMA":null},"ListObjectName":"workbooks"}', 'XLS');

-- print SaveToDB Framework installed
