-- =============================================
-- SaveToDB Framework for NuoDB
-- Version 7.19, March 12, 2018
--
-- Copyright 2014-2018 Gartle Technology Corporation
-- =============================================

CREATE SCHEMA SAVETODB_DEV;
CREATE SCHEMA SAVETODB_XLS;

-- print Schemas have been created

CREATE TABLE IF NOT EXISTS SAVETODB_DEV.QUERY_LIST (
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING NOT NULL,
    TABLE_NAME STRING NOT NULL,
    TABLE_TYPE VARCHAR(64) NOT NULL,
    TABLE_CODE STRING,
    INSERT_PROCEDURE STRING,
    UPDATE_PROCEDURE STRING,
    DELETE_PROCEDURE STRING,
    PROCEDURE_TYPE VARCHAR(64),
    PRIMARY KEY (ID)
);

CREATE VIEW SAVETODB_DEV.VIEW_QUERY_LIST
AS
SELECT
    t.SCHEMA AS TABLE_SCHEMA
    , t.TABLENAME AS TABLE_NAME
    , t.TYPE AS TABLE_TYPE
    , NULL AS TABLE_CODE
    , COALESCE(q.INSERT_PROCEDURE, CONCAT(i.SCHEMA, '.', i.PROCEDURENAME)) AS INSERT_PROCEDURE
    , COALESCE(q.UPDATE_PROCEDURE, CONCAT(u.SCHEMA, '.', u.PROCEDURENAME)) AS UPDATE_PROCEDURE
    , COALESCE(q.DELETE_PROCEDURE, CONCAT(d.SCHEMA, '.', d.PROCEDURENAME)) AS DELETE_PROCEDURE
    , COALESCE(q.PROCEDURE_TYPE, CASE WHEN i.PROCEDURENAME IS NOT NULL THEN 'PROCEDURE' ELSE NULL END) AS PROCEDURE_TYPE
FROM
    SYSTEM.TABLES t
    LEFT OUTER JOIN SYSTEM.PROCEDURES i ON i.SCHEMA = t.SCHEMA AND i.PROCEDURENAME = CONCAT(REPLACE(t.TABLENAME, '_SELECT', ''), '_INSERT')
    LEFT OUTER JOIN SYSTEM.PROCEDURES u ON u.SCHEMA = t.SCHEMA AND u.PROCEDURENAME = CONCAT(REPLACE(t.TABLENAME, '_SELECT', ''), '_UPDATE')
    LEFT OUTER JOIN SYSTEM.PROCEDURES d ON d.SCHEMA = t.SCHEMA AND d.PROCEDURENAME = CONCAT(REPLACE(t.TABLENAME, '_SELECT', ''), '_DELETE')
    LEFT OUTER JOIN SAVETODB_DEV.QUERY_LIST q ON q.TABLE_SCHEMA = t.SCHEMA AND q.TABLE_NAME = t.TABLENAME
WHERE
    NOT t.SCHEMA IN ('SYSTEM')
UNION
SELECT
    r.SCHEMA AS TABLE_SCHEMA
    , r.PROCEDURENAME AS TABLE_NAME
    , 'PROCEDURE' AS TABLE_TYPE
    , NULL AS TABLE_CODE
    , COALESCE(q.INSERT_PROCEDURE, CONCAT(i.SCHEMA, '.', i.PROCEDURENAME)) AS INSERT_PROCEDURE
    , COALESCE(q.UPDATE_PROCEDURE, CONCAT(u.SCHEMA, '.', u.PROCEDURENAME)) AS UPDATE_PROCEDURE
    , COALESCE(q.DELETE_PROCEDURE, CONCAT(d.SCHEMA, '.', d.PROCEDURENAME)) AS DELETE_PROCEDURE
    , COALESCE(q.PROCEDURE_TYPE, CASE WHEN i.PROCEDURENAME IS NOT NULL THEN 'PROCEDURE' ELSE NULL END) AS PROCEDURE_TYPE
FROM 
    SYSTEM.PROCEDURES r
    LEFT OUTER JOIN SYSTEM.PROCEDURES i ON i.SCHEMA = r.SCHEMA AND i.PROCEDURENAME = CONCAT(REPLACE(r.PROCEDURENAME, '_SELECT', ''), '_INSERT')
    LEFT OUTER JOIN SYSTEM.PROCEDURES u ON u.SCHEMA = r.SCHEMA AND u.PROCEDURENAME = CONCAT(REPLACE(r.PROCEDURENAME, '_SELECT', ''), '_UPDATE')
    LEFT OUTER JOIN SYSTEM.PROCEDURES d ON d.SCHEMA = r.SCHEMA AND d.PROCEDURENAME = CONCAT(REPLACE(r.PROCEDURENAME, '_SELECT', ''), '_DELETE')
    LEFT OUTER JOIN SAVETODB_DEV.QUERY_LIST q ON q.TABLE_SCHEMA = r.SCHEMA AND q.TABLE_NAME = r.PROCEDURENAME
WHERE
    NOT r.SCHEMA IN ('SYSTEM')
    AND NOT (
        r.PROCEDURENAME LIKE '%_INSERT'
        OR r.PROCEDURENAME LIKE '%_UPDATE'
        OR r.PROCEDURENAME LIKE '%_DELETE'
        OR r.PROCEDURENAME LIKE '%_MERGE'
        OR r.PROCEDURENAME LIKE '%_CHANGE'
    )
UNION
SELECT
    q.TABLE_SCHEMA
    , q.TABLE_NAME
    , q.TABLE_TYPE
    , q.TABLE_CODE
    , q.INSERT_PROCEDURE
    , q.UPDATE_PROCEDURE
    , q.DELETE_PROCEDURE
    , q.PROCEDURE_TYPE
FROM
    SAVETODB_DEV.QUERY_LIST q
    LEFT OUTER JOIN SYSTEM.TABLES t ON q.TABLE_SCHEMA = t.SCHEMA AND q.TABLE_NAME = t.TABLENAME
    LEFT OUTER JOIN SYSTEM.PROCEDURES r ON q.TABLE_SCHEMA = r.SCHEMA AND q.TABLE_NAME = r.PROCEDURENAME
WHERE
    q.TABLE_TYPE IN ('CODE', 'HTTP', 'TEXT')
    AND t.TABLENAME IS NULL
    AND r.PROCEDURENAME IS NULL;

CREATE VIEW SAVETODB_XLS.VIEW_QUERY_LIST
AS
SELECT
    t.TABLE_SCHEMA
    , t.TABLE_NAME
    , t.TABLE_TYPE
    , t.TABLE_CODE
    , t.INSERT_PROCEDURE
    , t.UPDATE_PROCEDURE
    , t.DELETE_PROCEDURE
    , t.PROCEDURE_TYPE
FROM
    SAVETODB_DEV.VIEW_QUERY_LIST t
WHERE
    NOT t.TABLE_SCHEMA IN ('SAVETODB_DEV')
    AND NOT t.TABLE_NAME IN (
        'VIEW_COLUMN_TRANSLATION'
        , 'VIEW_OBJECT_TRANSLATION'
        , 'VIEW_EVENT_HANDLERS'
        , 'VIEW_TABLE_FORMATS'
        , 'VIEW_PARAMETER_VALUES'
        , 'VIEW_QUERY_LIST'
        )
    AND NOT t.TABLE_NAME LIKE 'XL_%';

-- print QueryList objects have been created

CREATE TABLE IF NOT EXISTS SAVETODB_DEV.OBJECT_TRANSLATION (
    ID INTEGER NOT NULL NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING NOT NULL,
    TABLE_NAME STRING NOT NULL,
    LANGUAGE_NAME CHARACTER(2) NOT NULL,
    TRANSLATED_NAME STRING,
    TRANSLATED_DESC STRING,
    TRANSLATED_COMMENT STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_SCHEMA_NAME_LANGUAGE
    ON SAVETODB_DEV.OBJECT_TRANSLATION
    (
        TABLE_SCHEMA,
        TABLE_NAME,
        LANGUAGE_NAME
    );

CREATE TABLE IF NOT EXISTS SAVETODB_DEV.COLUMN_TRANSLATION (
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING,
    TABLE_NAME STRING,
    COLUMN_NAME STRING NOT NULL,
    LANGUAGE_NAME CHARACTER(2) NOT NULL,
    TRANSLATED_NAME STRING,
    TRANSLATED_DESC STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_SCHEMA_NAME_COLUMN_LANGUAGE
    ON SAVETODB_DEV.COLUMN_TRANSLATION
    (
        TABLE_SCHEMA,
        TABLE_NAME,
        COLUMN_NAME,
        LANGUAGE_NAME
    );
    
CREATE VIEW SAVETODB_XLS.VIEW_OBJECT_TRANSLATION
AS
SELECT
    t.ID,
    t.TABLE_SCHEMA,
    t.TABLE_NAME,
    t.LANGUAGE_NAME,
    t.TRANSLATED_NAME,
    t.TRANSLATED_DESC,
    t.TRANSLATED_COMMENT
FROM
    SAVETODB_DEV.OBJECT_TRANSLATION t
;

CREATE VIEW SAVETODB_XLS.VIEW_COLUMN_TRANSLATION
AS
SELECT
    t.ID,
    t.TABLE_SCHEMA,
    t.TABLE_NAME,
    t.COLUMN_NAME,
    t.LANGUAGE_NAME,
    t.TRANSLATED_NAME,
    t.TRANSLATED_DESC
FROM
    SAVETODB_DEV.COLUMN_TRANSLATION t
;

CREATE VIEW SAVETODB_XLS.VIEW_OBJECT_DESCRIPTION
AS
SELECT
    t.ID
    , t.LANGUAGE_NAME
    , t.TABLE_SCHEMA
    , t.TABLE_NAME
    , t.TRANSLATED_NAME
    , t.TRANSLATED_COMMENT
FROM
    SAVETODB_DEV.OBJECT_TRANSLATION t
WHERE
    t.TRANSLATED_NAME IS NOT NULL
    AND t.TRANSLATED_COMMENT IS NOT NULL
    AND NOT t.TABLE_SCHEMA IN ('SYSTEM', 'SAVETODB_DEV', 'SAVETODB_XLS')
    AND NOT t.TABLE_NAME LIKE '%_INSERT'
    AND NOT t.TABLE_NAME LIKE '%_UPDATE'
    AND NOT t.TABLE_NAME LIKE '%_DELETE'
    AND NOT t.TABLE_NAME LIKE '%_MERGE'
    AND NOT t.TABLE_NAME LIKE 'XL_%'
;

-- print Translation objects have been created

CREATE TABLE IF NOT EXISTS SAVETODB_DEV.EVENT_HANDLERS (
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING NOT NULL,
    TABLE_NAME STRING NOT NULL,
    COLUMN_NAME STRING,
    EVENT_NAME VARCHAR(50) NOT NULL,
    HANDLER_SCHEMA STRING,
    HANDLER_NAME STRING,
    HANDLER_TYPE VARCHAR(64),
    HANDLER_CODE STRING,
    TARGET_WORKSHEET VARCHAR(256),
    MENU_ORDER SMALLINT,
    EDIT_PARAMETERS BOOLEAN,
    PRIMARY KEY (ID)
);

CREATE VIEW SAVETODB_XLS.VIEW_EVENT_HANDLERS
AS
SELECT
    t.ID,
    t.TABLE_SCHEMA,
    t.TABLE_NAME,
    t.COLUMN_NAME,
    t.EVENT_NAME,
    t.HANDLER_SCHEMA,
    t.HANDLER_NAME,
    t.HANDLER_TYPE,
    t.HANDLER_CODE,
    t.TARGET_WORKSHEET,
    t.MENU_ORDER,
    t.EDIT_PARAMETERS
FROM
    SAVETODB_DEV.EVENT_HANDLERS t
;

-- print EventHandlers objects have been created

CREATE TABLE IF NOT EXISTS SAVETODB_DEV.PARAMETER_VALUES
(
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING NOT NULL,
    TABLE_NAME STRING NOT NULL,
    PARAMETER_NAME STRING NOT NULL,
    SELECT_SCHEMA STRING,
    SELECT_NAME STRING NOT NULL,
    SELECT_TYPE VARCHAR(64),
    SELECT_CODE STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_SCHEMA_NAME_PARAMETER
    ON SAVETODB_DEV.PARAMETER_VALUES
    (
        TABLE_SCHEMA,
        TABLE_NAME,
        PARAMETER_NAME
    );
    
CREATE VIEW SAVETODB_XLS.VIEW_PARAMETER_VALUES
AS
SELECT 
    t.ID,
    t.TABLE_SCHEMA,
    t.TABLE_NAME,
    t.PARAMETER_NAME,
    t.SELECT_SCHEMA,
    t.SELECT_NAME,
    t.SELECT_TYPE,
    t.SELECT_CODE
FROM
    SAVETODB_DEV.PARAMETER_VALUES t
;

-- print ParameterValues  objects have been created

CREATE TABLE IF NOT EXISTS SAVETODB_DEV.TABLE_FORMATS (
    ID INTEGER NOT NULL GENERATED ALWAYS AS IDENTITY,
    TABLE_SCHEMA STRING NOT NULL,
    TABLE_NAME STRING NOT NULL,
    TABLE_EXCEL_FORMAT_XML STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_TABLE_FORMATS_SCHEMA_NAME
    ON SAVETODB_DEV.TABLE_FORMATS
    (
        TABLE_SCHEMA, 
        TABLE_NAME
    );
    
CREATE VIEW SAVETODB_XLS.VIEW_TABLE_FORMATS
AS
SELECT
    t.ID,
    t.TABLE_SCHEMA,
    t.TABLE_NAME,
    t.TABLE_EXCEL_FORMAT_XML
FROM
    SAVETODB_DEV.TABLE_FORMATS t
;

SET DELIMITER @
CREATE PROCEDURE SAVETODB_DEV.TABLE_FORMAT_SAVE (
        IN Schema VARCHAR(64),
        IN Name VARCHAR(64),
        IN ExcelFormatXML STRING
    )
AS
VAR RowCount INTEGER;

IF (Schema IS NOT NULL AND Name IS NOT NULL)
    IF (ExcelFormatXML IS NULL)
        DELETE FROM SAVETODB_DEV.TABLE_FORMATS
        WHERE
            TABLE_SCHEMA = Schema AND TABLE_NAME = Name;
    ELSE
        RowCount = (SELECT COUNT(*)
                    FROM SAVETODB_DEV.TABLE_FORMATS
                    WHERE TABLE_SCHEMA = Schema AND TABLE_NAME = Name);

        IF (RowCount = 0)
            INSERT INTO SAVETODB_DEV.TABLE_FORMATS
                (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML)
            VALUES
                (Schema, Name, ExcelFormatXML);
        ELSE
            UPDATE SAVETODB_DEV.TABLE_FORMATS
            SET
                TABLE_SCHEMA = Schema,
                TABLE_NAME = Name,
                TABLE_EXCEL_FORMAT_XML = ExcelFormatXML
            WHERE
                TABLE_SCHEMA = Schema AND TABLE_NAME = Name;
        END_IF;
    END_IF;
END_IF;
END_PROCEDURE
@
SET DELIMITER ;

-- print TableFormats objects have been created

CREATE ROLE SAVETODB_XLS.USERS;
CREATE ROLE SAVETODB_DEV.DEVELOPERS;

GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_QUERY_LIST          TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_EVENT_HANDLERS      TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_PARAMETER_VALUES    TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_COLUMN_TRANSLATION  TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_OBJECT_TRANSLATION  TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_OBJECT_DESCRIPTION  TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_TABLE_FORMATS       TO ROLE SAVETODB_XLS.USERS;

GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_QUERY_LIST          TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_EVENT_HANDLERS      TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_PARAMETER_VALUES    TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_COLUMN_TRANSLATION  TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_OBJECT_TRANSLATION  TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_OBJECT_DESCRIPTION  TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT ON TABLE SAVETODB_XLS.VIEW_TABLE_FORMATS       TO ROLE SAVETODB_DEV.DEVELOPERS;

GRANT SELECT ON TABLE SAVETODB_DEV.COLUMN_TRANSLATION       TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_DEV.EVENT_HANDLERS           TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_DEV.OBJECT_TRANSLATION       TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_DEV.PARAMETER_VALUES         TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_DEV.QUERY_LIST               TO ROLE SAVETODB_XLS.USERS;
GRANT SELECT ON TABLE SAVETODB_DEV.TABLE_FORMATS            TO ROLE SAVETODB_XLS.USERS;

GRANT SELECT ON TABLE SAVETODB_DEV.VIEW_QUERY_LIST          TO ROLE SAVETODB_XLS.USERS;

GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE SAVETODB_DEV.COLUMN_TRANSLATION   TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE SAVETODB_DEV.EVENT_HANDLERS       TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE SAVETODB_DEV.OBJECT_TRANSLATION   TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE SAVETODB_DEV.PARAMETER_VALUES     TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE SAVETODB_DEV.QUERY_LIST           TO ROLE SAVETODB_DEV.DEVELOPERS;
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE SAVETODB_DEV.TABLE_FORMATS        TO ROLE SAVETODB_DEV.DEVELOPERS;

GRANT SELECT ON TABLE SAVETODB_DEV.VIEW_QUERY_LIST                              TO ROLE SAVETODB_DEV.DEVELOPERS;

GRANT EXECUTE ON PROCEDURE SAVETODB_DEV.TABLE_FORMAT_SAVE                       TO ROLE SAVETODB_DEV.DEVELOPERS;

-- print Roles have been created

INSERT INTO SAVETODB_DEV.QUERY_LIST (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_PROCEDURE, UPDATE_PROCEDURE, DELETE_PROCEDURE, PROCEDURE_TYPE) VALUES ('SAVETODB_XLS', 'VIEW_TABLE_FORMATS',      'VIEW', NULL, 'SAVETODB_DEV.TABLE_FORMATS', 'SAVETODB_DEV.TABLE_FORMATS', 'SAVETODB_DEV.TABLE_FORMATS', 'TABLE');
INSERT INTO SAVETODB_DEV.QUERY_LIST (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_PROCEDURE, UPDATE_PROCEDURE, DELETE_PROCEDURE, PROCEDURE_TYPE) VALUES ('SAVETODB_XLS', 'VIEW_PARAMETER_VALUES',   'VIEW', NULL, 'SAVETODB_DEV.PARAMETER_VALUES', 'SAVETODB_DEV.PARAMETER_VALUES', 'SAVETODB_DEV.PARAMETER_VALUES', 'TABLE');
INSERT INTO SAVETODB_DEV.QUERY_LIST (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_PROCEDURE, UPDATE_PROCEDURE, DELETE_PROCEDURE, PROCEDURE_TYPE) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS',     'VIEW', NULL, 'SAVETODB_DEV.EVENT_HANDLERS', 'SAVETODB_DEV.EVENT_HANDLERS', 'SAVETODB_DEV.EVENT_HANDLERS', 'TABLE');
INSERT INTO SAVETODB_DEV.QUERY_LIST (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_PROCEDURE, UPDATE_PROCEDURE, DELETE_PROCEDURE, PROCEDURE_TYPE) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_DESCRIPTION', 'VIEW', NULL, 'SAVETODB_DEV.OBJECT_TRANSLATION', 'SAVETODB_DEV.OBJECT_TRANSLATION', 'SAVETODB_DEV.OBJECT_TRANSLATION', 'TABLE');
INSERT INTO SAVETODB_DEV.QUERY_LIST (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_PROCEDURE, UPDATE_PROCEDURE, DELETE_PROCEDURE, PROCEDURE_TYPE) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_TRANSLATION', 'VIEW', NULL, 'SAVETODB_DEV.OBJECT_TRANSLATION', 'SAVETODB_DEV.OBJECT_TRANSLATION', 'SAVETODB_DEV.OBJECT_TRANSLATION', 'TABLE');
INSERT INTO SAVETODB_DEV.QUERY_LIST (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_PROCEDURE, UPDATE_PROCEDURE, DELETE_PROCEDURE, PROCEDURE_TYPE) VALUES ('SAVETODB_XLS', 'VIEW_COLUMN_TRANSLATION', 'VIEW', NULL, 'SAVETODB_DEV.COLUMN_TRANSLATION', 'SAVETODB_DEV.COLUMN_TRANSLATION', 'SAVETODB_DEV.COLUMN_TRANSLATION', 'TABLE');

INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_DEV','COLUMN_TRANSLATION',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-translation.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_DEV','OBJECT_TRANSLATION',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-translation.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_DEV','EVENT_HANDLERS',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-event-handlers.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_DEV','PARAMETER_VALUES',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-parameter-values.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_DEV','QUERY_LIST',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-query-list.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_DEV','TABLE_FORMATS',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-table-formats.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_XLS','VIEW_COLUMN_TRANSLATION',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-translation.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_XLS','VIEW_OBJECT_TRANSLATION',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-translation.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_XLS','VIEW_EVENT_HANDLERS',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-event-handlers.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_XLS','VIEW_PARAMETER_VALUES',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-parameter-values.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_DEV','VIEW_QUERY_LIST',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-query-list.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_XLS','VIEW_QUERY_LIST',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-query-list.htm',NULL,11,NULL);
INSERT INTO SAVETODB_DEV.EVENT_HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('SAVETODB_XLS','VIEW_TABLE_FORMATS',NULL,'Actions','HTTP','Online Help','HTTP','http://www.savetodb.com/savetodb/configuring-table-formats.htm',NULL,11,NULL);

INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_DEV', 'COLUMN_TRANSLATION', 'en', NULL, 'Configuration Table of Column Translation', 'The table defines translation for Excel table headers and ribbon fields');
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'en', NULL, 'Configuration Table of Excel Event Handlers', 'The table defines Excel event handlers and Actions and Context Menu items');
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_DEV', 'OBJECT_TRANSLATION', 'en', NULL, 'Configuration Table of Object Translation', 'The table defines translation for Query List, Actions and Context Menu items');
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_DEV', 'PARAMETER_VALUES', 'en', NULL, 'Configuration Table of Parameter Value Queries', 'The table defines objects to select ribbon parameter values');
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_DEV', 'QUERY_LIST', 'en', NULL, 'Configuration Table of Query List', 'The table defines configuration of Query List objects');
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_DEV', 'TABLE_FORMAT_SAVE', 'en', NULL, 'Configuration Procedure to Save Table Formats', 'The procedure is used to save Excel formats of database objects');
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_DEV', 'TABLE_FORMATS', 'en', NULL, 'Configuration Table of Excel Table Formats', 'The table contains Excel formats of database objects');
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_XLS', 'VIEW_COLUMN_TRANSLATION', 'en', NULL, 'Configuration View of Column Translation', NULL);
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'en', NULL, 'Configuration View of Excel Event Handlers', NULL);
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_DESCRIPTION', 'en', 'Object Descriptions', 'Database Object Descriptions', 'You may use Query List to connect to these objects');
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_TRANSLATION', 'en', NULL, 'Configuration View of Object Translation', NULL);
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_XLS', 'VIEW_PARAMETER_VALUES', 'en', NULL, 'Configuration View of Parameter Value Queries', NULL);
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_XLS', 'VIEW_QUERY_LIST', 'en', 'Framework Query List', 'Configuration View of Query List', NULL);
INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('SAVETODB_XLS', 'VIEW_TABLE_FORMATS', 'en', NULL, 'Configuration View of Excel Table Formats', NULL);

INSERT INTO SAVETODB_DEV.OBJECT_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('HTTP', 'Online Help', 'en', 'Online Help', NULL, NULL);

INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'QUERY_LIST', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'QUERY_LIST', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'QUERY_LIST', 'TABLE_NAME', 'en', NULL, 'Database object name or unique name for SQL-code and http-query');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'QUERY_LIST', 'TABLE_TYPE', 'en', NULL, 'Database object type, CODE for SQL-code, or HTTP for http-query');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'QUERY_LIST', 'INSERT_PROCEDURE', 'en', NULL, 'Insert procedure or destination table to save changes of views, procedures, or SQL-codes');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'QUERY_LIST', 'UPDATE_PROCEDURE', 'en', NULL, 'Update procedure or destination table to save changes of views, procedures, or SQL-codes');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'QUERY_LIST', 'DELETE_PROCEDURE', 'en', NULL, 'Delete procedure or destination table to save changes of views, procedures, or SQL-codes');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'TABLE_NAME', 'en', NULL, 'Database object name, SQL-code name, or http-query name');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'COLUMN_NAME', 'en', NULL, 'Column name or NULL to use the handler for entire table');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'EVENT_NAME', 'en', NULL, 'Actions, Change, ContextMenu, DoubleClick, or SelectionChange');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'HANDLER_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'HANDLER_NAME', 'en', NULL, 'Database object name, unique name for SQL-code and http-query, or MenuSeparator');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'HANDLER_TYPE', 'en', NULL, 'Database object type, CODE for SQL-code, HTTP for http-query, or NULL for MenuSeparator');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'TARGET_WORKSHEET', 'en', NULL, 'NULL, sheet name, _new, _self, _reload. Additions: _RightWindow, _LeftWindow, _TopWindow, _BottomWindow');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'MENU_ORDER', 'en', NULL, 'Sort order of Actions and Context Menu items');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', 'EDIT_PARAMETERS', 'en', NULL, 'Show the Edit Parameters dialog box for Actions and Context Menu items: True/False/NULL (default behavior)');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'PARAMETER_VALUES', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'PARAMETER_VALUES', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'PARAMETER_VALUES', 'TABLE_NAME', 'en', NULL, 'Database object name');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'PARAMETER_VALUES', 'PARAMETER_NAME', 'en', NULL, 'Column or parameter name');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'PARAMETER_VALUES', 'SELECT_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'PARAMETER_VALUES', 'SELECT_NAME', 'en', NULL, 'Database object name or name for SQL-code');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'PARAMETER_VALUES', 'SELECT_TYPE', 'en', NULL, 'Database object type or CODE for SQL-code');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'OBJECT_TRANSLATION', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'OBJECT_TRANSLATION', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'OBJECT_TRANSLATION', 'TABLE_NAME', 'en', NULL, 'Database object name or a name of SQL-code or http-query specified in Query List and Event Handlers');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'OBJECT_TRANSLATION', 'LANGUAGE_NAME', 'en', NULL, 'Two-char language code');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'OBJECT_TRANSLATION', 'TRANSLATED_NAME', 'en', NULL, 'Translated name for Query List, Actions and Context Menus');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'OBJECT_TRANSLATION', 'TRANSLATED_DESC', 'en', NULL, 'Text for ScreenTips and SaveToDB_Desc named cells');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'OBJECT_TRANSLATION', 'TRANSLATED_COMMENT', 'en', NULL, 'Text for SuperTips and SaveToDB_Comment named cells');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'COLUMN_TRANSLATION', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'COLUMN_TRANSLATION', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema or NULL for entire database scope');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'COLUMN_TRANSLATION', 'TABLE_NAME', 'en', NULL, 'Database object name or NULL for entire schema scope');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'COLUMN_TRANSLATION', 'COLUMN_NAME', 'en', NULL, 'Column or parameter name');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'COLUMN_TRANSLATION', 'LANGUAGE_NAME', 'en', NULL, 'Two-char language code');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'COLUMN_TRANSLATION', 'TRANSLATED_NAME', 'en', NULL, 'Translated name for Excel table headers and ribbon parameters');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_DEV', 'COLUMN_TRANSLATION', 'TRANSLATED_DESC', 'en', NULL, 'Text for Excel table header comments and ribbon parameter SuperTips');

INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_QUERY_LIST', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_QUERY_LIST', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_QUERY_LIST', 'TABLE_NAME', 'en', NULL, 'Database object name or unique name for SQL-code and http-query');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_QUERY_LIST', 'TABLE_TYPE', 'en', NULL, 'Database object type, CODE for SQL-code, or HTTP for http-query');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_QUERY_LIST', 'INSERT_PROCEDURE', 'en', NULL, 'Insert procedure or destination table to save changes of views, procedures, or SQL-codes');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_QUERY_LIST', 'UPDATE_PROCEDURE', 'en', NULL, 'Update procedure or destination table to save changes of views, procedures, or SQL-codes');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_QUERY_LIST', 'DELETE_PROCEDURE', 'en', NULL, 'Delete procedure or destination table to save changes of views, procedures, or SQL-codes');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'TABLE_NAME', 'en', NULL, 'Database object name, SQL-code name, or http-query name');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'COLUMN_NAME', 'en', NULL, 'Column name or NULL to use the handler for entire table');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'EVENT_NAME', 'en', NULL, 'Actions, Change, ContextMenu, DoubleClick, or SelectionChange');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'HANDLER_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'HANDLER_NAME', 'en', NULL, 'Database object name, unique name for SQL-code and http-query, or MenuSeparator');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'HANDLER_TYPE', 'en', NULL, 'Database object type, CODE for SQL-code, HTTP for http-query, or NULL for MenuSeparator');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'TARGET_WORKSHEET', 'en', NULL, 'NULL, sheet name, _new, _self, _reload. Additions: _RightWindow, _LeftWindow, _TopWindow, _BottomWindow');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'MENU_ORDER', 'en', NULL, 'Sort order of Actions and Context Menu items');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_EVENT_HANDLERS', 'EDIT_PARAMETERS', 'en', NULL, 'Show the Edit Parameters dialog box for Actions and Context Menu items: True/False/NULL (default behavior)');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_PARAMETER_VALUES', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_PARAMETER_VALUES', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_PARAMETER_VALUES', 'TABLE_NAME', 'en', NULL, 'Database object name');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_PARAMETER_VALUES', 'PARAMETER_NAME', 'en', NULL, 'Column or parameter name');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_PARAMETER_VALUES', 'SELECT_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_PARAMETER_VALUES', 'SELECT_NAME', 'en', NULL, 'Database object name or name for SQL-code');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_PARAMETER_VALUES', 'SELECT_TYPE', 'en', NULL, 'Database object type or CODE for SQL-code');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_TRANSLATION', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_TRANSLATION', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_TRANSLATION', 'TABLE_NAME', 'en', NULL, 'Database object name or a name of SQL-code or http-query specified in Query List and Event Handlers');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_TRANSLATION', 'LANGUAGE_NAME', 'en', NULL, 'Two-char language code');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_TRANSLATION', 'TRANSLATED_NAME', 'en', NULL, 'Translated name for Query List, Actions and Context Menus');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_TRANSLATION', 'TRANSLATED_DESC', 'en', NULL, 'Text for ScreenTips and SaveToDB_Desc named cells');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_TRANSLATION', 'TRANSLATED_COMMENT', 'en', NULL, 'Text for SuperTips and SaveToDB_Comment named cells');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_COLUMN_TRANSLATION', 'ID', 'en', NULL, NULL);
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_COLUMN_TRANSLATION', 'TABLE_SCHEMA', 'en', NULL, 'Database object schema or NULL for entire database scope');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_COLUMN_TRANSLATION', 'TABLE_NAME', 'en', NULL, 'Database object name or NULL for entire schema scope');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_COLUMN_TRANSLATION', 'COLUMN_NAME', 'en', NULL, 'Column or parameter name');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_COLUMN_TRANSLATION', 'LANGUAGE_NAME', 'en', NULL, 'Two-char language code');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_COLUMN_TRANSLATION', 'TRANSLATED_NAME', 'en', NULL, 'Translated name for Excel table headers and ribbon parameters');
INSERT INTO SAVETODB_DEV.COLUMN_TRANSLATION (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC) VALUES ('SAVETODB_XLS', 'VIEW_COLUMN_TRANSLATION', 'TRANSLATED_DESC', 'en', NULL, 'Text for Excel table header comments and ribbon parameter SuperTips');

INSERT INTO "SAVETODB_DEV"."TABLE_FORMATS" ("TABLE_SCHEMA", "TABLE_NAME", "TABLE_EXCEL_FORMAT_XML") VALUES ('SAVETODB_XLS', 'VIEW_OBJECT_DESCRIPTION', '<?xml version="1.0" encoding="Unicode"?>
<table name="SAVETODB_XLS.VIEW_OBJECT_DESCRIPTION">
<columnFormats>
<column name="" property="ListObjectName" value="UserObjects_Table1" type="String" />
<column name="" property="ShowTotals" value="False" type="Boolean" />
<column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" />
<column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" />
<column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" />
<column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" />
<column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" />
<column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" />
<column name="_RowNum" property="Address" value="$B$7" type="String" />
<column name="_RowNum" property="VerticalAlignment" value="-4160" type="Double" />
<column name="ID" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="ID" property="Address" value="$C$7" type="String" />
<column name="ID" property="ColumnWidth" value="8.43" type="Double" />
<column name="LANGUAGE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="LANGUAGE_NAME" property="Address" value="$D$7" type="String" />
<column name="LANGUAGE_NAME" property="ColumnWidth" value="0.08" type="Double" />
<column name="LANGUAGE_NAME" property="VerticalAlignment" value="-4160" type="Double" />
<column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="Address" value="$E$7" type="String" />
<column name="TABLE_SCHEMA" property="VerticalAlignment" value="-4160" type="Double" />
<column name="TABLE_NAME" property="EntireColumn.Hidden" value="True" type="Boolean" />
<column name="TABLE_NAME" property="Address" value="$F$7" type="String" />
<column name="TABLE_NAME" property="VerticalAlignment" value="-4160" type="Double" />
<column name="TRANSLATED_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TRANSLATED_NAME" property="Address" value="$G$7" type="String" />
<column name="TRANSLATED_NAME" property="ColumnWidth" value="24" type="Double" />
<column name="TRANSLATED_NAME" property="VerticalAlignment" value="-4160" type="Double" />
<column name="TRANSLATED_COMMENT" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TRANSLATED_COMMENT" property="Address" value="$H$7" type="String" />
<column name="TRANSLATED_COMMENT" property="ColumnWidth" value="108.86" type="Double" />
<column name="TRANSLATED_COMMENT" property="VerticalAlignment" value="-4160" type="Double" />
<column name="TRANSLATED_COMMENT" property="WrapText" value="True" type="Boolean" />
<column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" />
<column name="" property="ActiveWindow.FreezePanes" value="False" type="Boolean" />
<column name="" property="ActiveWindow.Split" value="False" type="Boolean" />
<column name="" property="ActiveWindow.SplitRow" value="-6" type="Double" />
<column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" />
<column name="" property="PageSetup.Orientation" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" />
</columnFormats>
</table>
');
INSERT INTO "SAVETODB_DEV"."TABLE_FORMATS" ("TABLE_SCHEMA", "TABLE_NAME", "TABLE_EXCEL_FORMAT_XML") VALUES ('SAVETODB_DEV', 'QUERY_LIST', '<?xml version="1.0" encoding="Unicode"?>
<table name="SAVETODB_DEV.QUERY_LIST">
<columnFormats>
<column name="" property="ListObjectName" value="QueryList_Table1" type="String" />
<column name="" property="ShowTotals" value="False" type="Boolean" />
<column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" />
<column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" />
<column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" />
<column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" />
<column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" />
<column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" />
<column name="_RowNum" property="Address" value="$B$7" type="String" />
<column name="ID" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="ID" property="Address" value="$C$7" type="String" />
<column name="ID" property="ColumnWidth" value="4.43" type="Double" />
<column name="ID" property="Validation.Type" value="1" type="Double" />
<column name="ID" property="Validation.AlertStyle" value="1" type="Double" />
<column name="ID" property="Validation.Operator" value="1" type="Double" />
<column name="ID" property="Validation.Formula1" value="-2147483648" type="String" />
<column name="ID" property="Validation.Formula2" value="2147483647" type="String" />
<column name="ID" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="ID" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="ID" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="ID" property="Validation.ErrorMessage" value="The column requires values of the integer datatype." type="String" />
<column name="ID" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="ID" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_SCHEMA" property="Address" value="$D$7" type="String" />
<column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double" />
<column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_NAME" property="Address" value="$E$7" type="String" />
<column name="TABLE_NAME" property="ColumnWidth" value="27.86" type="Double" />
<column name="TABLE_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_TYPE" property="Address" value="$F$7" type="String" />
<column name="TABLE_TYPE" property="ColumnWidth" value="13.14" type="Double" />
<column name="TABLE_TYPE" property="Validation.Type" value="3" type="Double" />
<column name="TABLE_TYPE" property="Validation.AlertStyle" value="1" type="Double" />
<column name="TABLE_TYPE" property="Validation.Operator" value="8" type="Double" />
<column name="TABLE_TYPE" property="Validation.Formula1" value="TABLE;VIEW;PROCEDURE;CODE;HTTP;TEXT" type="String" />
<column name="TABLE_TYPE" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="TABLE_TYPE" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="TABLE_TYPE" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="TABLE_TYPE" property="Validation.ErrorMessage" value="The column requires values of the varchar(64) datatype." type="String" />
<column name="TABLE_TYPE" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="TABLE_TYPE" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_CODE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_CODE" property="Address" value="$G$7" type="String" />
<column name="TABLE_CODE" property="ColumnWidth" value="13.71" type="Double" />
<column name="INSERT_PROCEDURE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="INSERT_PROCEDURE" property="Address" value="$H$7" type="String" />
<column name="INSERT_PROCEDURE" property="ColumnWidth" value="37" type="Double" />
<column name="UPDATE_PROCEDURE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="UPDATE_PROCEDURE" property="Address" value="$I$7" type="String" />
<column name="UPDATE_PROCEDURE" property="ColumnWidth" value="37" type="Double" />
<column name="DELETE_PROCEDURE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="DELETE_PROCEDURE" property="Address" value="$J$7" type="String" />
<column name="DELETE_PROCEDURE" property="ColumnWidth" value="37" type="Double" />
<column name="PROCEDURE_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="PROCEDURE_TYPE" property="Address" value="$K$7" type="String" />
<column name="PROCEDURE_TYPE" property="ColumnWidth" value="18.86" type="Double" />
<column name="PROCEDURE_TYPE" property="Validation.Type" value="3" type="Double" />
<column name="PROCEDURE_TYPE" property="Validation.AlertStyle" value="1" type="Double" />
<column name="PROCEDURE_TYPE" property="Validation.Operator" value="8" type="Double" />
<column name="PROCEDURE_TYPE" property="Validation.Formula1" value="TABLE;PROCEDURE;CODE;MERGE" type="String" />
<column name="PROCEDURE_TYPE" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="PROCEDURE_TYPE" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="PROCEDURE_TYPE" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="PROCEDURE_TYPE" property="Validation.ErrorMessage" value="The column requires values of the varchar(64) datatype." type="String" />
<column name="PROCEDURE_TYPE" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="PROCEDURE_TYPE" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).AppliesTo.Address" value="$D$7:$D$12" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Priority" value="1" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Formula1" value="=ISBLANK(D7)" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).AppliesTo.Address" value="$E$7:$E$12" type="String" />
<column name="TABLE_NAME" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Priority" value="2" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(E7)" type="String" />
<column name="TABLE_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="TABLE_TYPE" property="FormatConditions(1).AppliesTo.Address" value="$F$7:$F$12" type="String" />
<column name="TABLE_TYPE" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_TYPE" property="FormatConditions(1).Priority" value="3" type="Double" />
<column name="TABLE_TYPE" property="FormatConditions(1).Formula1" value="=ISBLANK(F7)" type="String" />
<column name="TABLE_TYPE" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_TYPE" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" />
<column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" />
<column name="" property="ActiveWindow.Split" value="True" type="Boolean" />
<column name="" property="ActiveWindow.SplitRow" value="0" type="Double" />
<column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" />
<column name="" property="PageSetup.Orientation" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" />
</columnFormats>
</table>
');
INSERT INTO "SAVETODB_DEV"."TABLE_FORMATS" ("TABLE_SCHEMA", "TABLE_NAME", "TABLE_EXCEL_FORMAT_XML") VALUES ('SAVETODB_DEV', 'EVENT_HANDLERS', '<?xml version="1.0" encoding="Unicode"?>
<table name="SAVETODB_DEV.EVENT_HANDLERS">
<columnFormats>
<column name="" property="ListObjectName" value="EventHandlers_Table1" type="String" />
<column name="" property="ShowTotals" value="False" type="Boolean" />
<column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" />
<column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" />
<column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" />
<column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" />
<column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" />
<column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" />
<column name="_RowNum" property="Address" value="$B$7" type="String" />
<column name="ID" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="ID" property="Address" value="$C$7" type="String" />
<column name="ID" property="ColumnWidth" value="4.43" type="Double" />
<column name="ID" property="Validation.Type" value="1" type="Double" />
<column name="ID" property="Validation.AlertStyle" value="1" type="Double" />
<column name="ID" property="Validation.Operator" value="1" type="Double" />
<column name="ID" property="Validation.Formula1" value="-2147483648" type="String" />
<column name="ID" property="Validation.Formula2" value="2147483647" type="String" />
<column name="ID" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="ID" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="ID" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="ID" property="Validation.ErrorMessage" value="The column requires values of the integer datatype." type="String" />
<column name="ID" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="ID" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_SCHEMA" property="Address" value="$D$7" type="String" />
<column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double" />
<column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_NAME" property="Address" value="$E$7" type="String" />
<column name="TABLE_NAME" property="ColumnWidth" value="27.86" type="Double" />
<column name="COLUMN_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="COLUMN_NAME" property="Address" value="$F$7" type="String" />
<column name="COLUMN_NAME" property="ColumnWidth" value="17.43" type="Double" />
<column name="EVENT_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="EVENT_NAME" property="Address" value="$G$7" type="String" />
<column name="EVENT_NAME" property="ColumnWidth" value="15" type="Double" />
<column name="EVENT_NAME" property="Validation.Type" value="3" type="Double" />
<column name="EVENT_NAME" property="Validation.AlertStyle" value="1" type="Double" />
<column name="EVENT_NAME" property="Validation.Operator" value="8" type="Double" />
<column name="EVENT_NAME" property="Validation.Formula1" value="Actions;Change;ContextMenu;DoubleClick;SelectionChange;ConvertFormulas;DoNotConvertFormulas;DoNotSelect;DoNotSave;DoNotChange;ProtectRows;Formula;FormulaValue;ValidationList;SelectionList" type="String" />
<column name="EVENT_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="EVENT_NAME" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="EVENT_NAME" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="EVENT_NAME" property="Validation.ErrorMessage" value="The column requires values of the varchar(50) datatype." type="String" />
<column name="EVENT_NAME" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="EVENT_NAME" property="Validation.ShowError" value="True" type="Boolean" />
<column name="HANDLER_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="HANDLER_SCHEMA" property="Address" value="$H$7" type="String" />
<column name="HANDLER_SCHEMA" property="ColumnWidth" value="26.14" type="Double" />
<column name="HANDLER_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="HANDLER_NAME" property="Address" value="$I$7" type="String" />
<column name="HANDLER_NAME" property="ColumnWidth" value="26.14" type="Double" />
<column name="HANDLER_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="HANDLER_TYPE" property="Address" value="$J$7" type="String" />
<column name="HANDLER_TYPE" property="ColumnWidth" value="16.29" type="Double" />
<column name="HANDLER_TYPE" property="Validation.Type" value="3" type="Double" />
<column name="HANDLER_TYPE" property="Validation.AlertStyle" value="1" type="Double" />
<column name="HANDLER_TYPE" property="Validation.Operator" value="8" type="Double" />
<column name="HANDLER_TYPE" property="Validation.Formula1" value="TABLE;VIEW;PROCEDURE;FUNCTION;CODE;HTTP;TEXT;MACRO;CMD;VALUES;RANGE;REFRESH;MENUSEPARATOR" type="String" />
<column name="HANDLER_TYPE" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="HANDLER_TYPE" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="HANDLER_TYPE" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="HANDLER_TYPE" property="Validation.ErrorMessage" value="The column requires values of the varchar(64) datatype." type="String" />
<column name="HANDLER_TYPE" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="HANDLER_TYPE" property="Validation.ShowError" value="True" type="Boolean" />
<column name="HANDLER_CODE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="HANDLER_CODE" property="Address" value="$K$7" type="String" />
<column name="HANDLER_CODE" property="ColumnWidth" value="70.71" type="Double" />
<column name="TARGET_WORKSHEET" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TARGET_WORKSHEET" property="Address" value="$L$7" type="String" />
<column name="TARGET_WORKSHEET" property="ColumnWidth" value="21.71" type="Double" />
<column name="TARGET_WORKSHEET" property="Validation.Type" value="6" type="Double" />
<column name="TARGET_WORKSHEET" property="Validation.AlertStyle" value="1" type="Double" />
<column name="TARGET_WORKSHEET" property="Validation.Operator" value="8" type="Double" />
<column name="TARGET_WORKSHEET" property="Validation.Formula1" value="256" type="String" />
<column name="TARGET_WORKSHEET" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="TARGET_WORKSHEET" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="TARGET_WORKSHEET" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="TARGET_WORKSHEET" property="Validation.ErrorMessage" value="The column requires values of the varchar(256) datatype." type="String" />
<column name="TARGET_WORKSHEET" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="TARGET_WORKSHEET" property="Validation.ShowError" value="True" type="Boolean" />
<column name="MENU_ORDER" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="MENU_ORDER" property="Address" value="$M$7" type="String" />
<column name="MENU_ORDER" property="ColumnWidth" value="15.43" type="Double" />
<column name="MENU_ORDER" property="Validation.Type" value="1" type="Double" />
<column name="MENU_ORDER" property="Validation.AlertStyle" value="1" type="Double" />
<column name="MENU_ORDER" property="Validation.Operator" value="1" type="Double" />
<column name="MENU_ORDER" property="Validation.Formula1" value="-32768" type="String" />
<column name="MENU_ORDER" property="Validation.Formula2" value="32767" type="String" />
<column name="MENU_ORDER" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="MENU_ORDER" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="MENU_ORDER" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="MENU_ORDER" property="Validation.ErrorMessage" value="The column requires values of the smallint datatype." type="String" />
<column name="MENU_ORDER" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="MENU_ORDER" property="Validation.ShowError" value="True" type="Boolean" />
<column name="EDIT_PARAMETERS" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="EDIT_PARAMETERS" property="Address" value="$N$7" type="String" />
<column name="EDIT_PARAMETERS" property="ColumnWidth" value="19.57" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).AppliesTo.Address" value="$D$7:$D$12" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Priority" value="1" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Formula1" value="=ISBLANK(D7)" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" />
<column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" />
<column name="" property="ActiveWindow.Split" value="True" type="Boolean" />
<column name="" property="ActiveWindow.SplitRow" value="0" type="Double" />
<column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" />
<column name="" property="PageSetup.Orientation" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" />
</columnFormats>
</table>
');
INSERT INTO "SAVETODB_DEV"."TABLE_FORMATS" ("TABLE_SCHEMA", "TABLE_NAME", "TABLE_EXCEL_FORMAT_XML") VALUES ('SAVETODB_DEV', 'OBJECT_TRANSLATION', '<?xml version="1.0" encoding="Unicode"?>
<table name="SAVETODB_DEV.OBJECT_TRANSLATION">
<columnFormats>
<column name="" property="ListObjectName" value="ObjectTranslation_Table1" type="String" />
<column name="" property="ShowTotals" value="False" type="Boolean" />
<column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" />
<column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" />
<column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" />
<column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" />
<column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" />
<column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" />
<column name="_RowNum" property="Address" value="$B$7" type="String" />
<column name="ID" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="ID" property="Address" value="$C$7" type="String" />
<column name="ID" property="ColumnWidth" value="4.43" type="Double" />
<column name="ID" property="Validation.Type" value="1" type="Double" />
<column name="ID" property="Validation.AlertStyle" value="1" type="Double" />
<column name="ID" property="Validation.Operator" value="1" type="Double" />
<column name="ID" property="Validation.Formula1" value="-2147483648" type="String" />
<column name="ID" property="Validation.Formula2" value="2147483647" type="String" />
<column name="ID" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="ID" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="ID" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="ID" property="Validation.ErrorMessage" value="The column requires values of the integer datatype." type="String" />
<column name="ID" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="ID" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_SCHEMA" property="Address" value="$D$7" type="String" />
<column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double" />
<column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_NAME" property="Address" value="$E$7" type="String" />
<column name="TABLE_NAME" property="ColumnWidth" value="27.86" type="Double" />
<column name="LANGUAGE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="LANGUAGE_NAME" property="Address" value="$F$7" type="String" />
<column name="LANGUAGE_NAME" property="ColumnWidth" value="19.57" type="Double" />
<column name="LANGUAGE_NAME" property="Validation.Type" value="6" type="Double" />
<column name="LANGUAGE_NAME" property="Validation.AlertStyle" value="1" type="Double" />
<column name="LANGUAGE_NAME" property="Validation.Operator" value="8" type="Double" />
<column name="LANGUAGE_NAME" property="Validation.Formula1" value="2" type="String" />
<column name="LANGUAGE_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="LANGUAGE_NAME" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="LANGUAGE_NAME" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="LANGUAGE_NAME" property="Validation.ErrorMessage" value="The column requires values of the char(2) datatype." type="String" />
<column name="LANGUAGE_NAME" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="LANGUAGE_NAME" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TRANSLATED_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TRANSLATED_NAME" property="Address" value="$G$7" type="String" />
<column name="TRANSLATED_NAME" property="ColumnWidth" value="20.71" type="Double" />
<column name="TRANSLATED_DESC" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TRANSLATED_DESC" property="Address" value="$H$7" type="String" />
<column name="TRANSLATED_DESC" property="ColumnWidth" value="43.86" type="Double" />
<column name="TRANSLATED_COMMENT" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TRANSLATED_COMMENT" property="Address" value="$I$7" type="String" />
<column name="TRANSLATED_COMMENT" property="ColumnWidth" value="69.71" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).AppliesTo.Address" value="$D$7:$D$24" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Priority" value="1" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Formula1" value="=ISBLANK(D7)" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).AppliesTo.Address" value="$E$7:$E$24" type="String" />
<column name="TABLE_NAME" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Priority" value="2" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(E7)" type="String" />
<column name="TABLE_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="LANGUAGE_NAME" property="FormatConditions(1).AppliesTo.Address" value="$F$7:$F$24" type="String" />
<column name="LANGUAGE_NAME" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="LANGUAGE_NAME" property="FormatConditions(1).Priority" value="3" type="Double" />
<column name="LANGUAGE_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(F7)" type="String" />
<column name="LANGUAGE_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="LANGUAGE_NAME" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="SortFields(1)" property="KeyfieldName" value="LANGUAGE_NAME" type="String" />
<column name="SortFields(1)" property="SortOn" value="0" type="Double" />
<column name="SortFields(1)" property="Order" value="1" type="Double" />
<column name="SortFields(1)" property="DataOption" value="0" type="Double" />
<column name="SortFields(2)" property="KeyfieldName" value="TABLE_SCHEMA" type="String" />
<column name="SortFields(2)" property="SortOn" value="0" type="Double" />
<column name="SortFields(2)" property="Order" value="1" type="Double" />
<column name="SortFields(2)" property="DataOption" value="0" type="Double" />
<column name="SortFields(3)" property="KeyfieldName" value="TABLE_NAME" type="String" />
<column name="SortFields(3)" property="SortOn" value="0" type="Double" />
<column name="SortFields(3)" property="Order" value="1" type="Double" />
<column name="SortFields(3)" property="DataOption" value="0" type="Double" />
<column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" />
<column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" />
<column name="" property="ActiveWindow.Split" value="True" type="Boolean" />
<column name="" property="ActiveWindow.SplitRow" value="0" type="Double" />
<column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" />
<column name="" property="PageSetup.Orientation" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" />
</columnFormats>
</table>
');
INSERT INTO "SAVETODB_DEV"."TABLE_FORMATS" ("TABLE_SCHEMA", "TABLE_NAME", "TABLE_EXCEL_FORMAT_XML") VALUES ('SAVETODB_DEV', 'COLUMN_TRANSLATION', '<?xml version="1.0" encoding="Unicode"?>
<table name="SAVETODB_DEV.COLUMN_TRANSLATION">
<columnFormats>
<column name="" property="ListObjectName" value="ColumnTranslation_Table1" type="String" />
<column name="" property="ShowTotals" value="False" type="Boolean" />
<column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" />
<column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" />
<column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" />
<column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" />
<column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" />
<column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" />
<column name="_RowNum" property="Address" value="$B$7" type="String" />
<column name="ID" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="ID" property="Address" value="$C$7" type="String" />
<column name="ID" property="ColumnWidth" value="4.43" type="Double" />
<column name="ID" property="Validation.Type" value="1" type="Double" />
<column name="ID" property="Validation.AlertStyle" value="1" type="Double" />
<column name="ID" property="Validation.Operator" value="1" type="Double" />
<column name="ID" property="Validation.Formula1" value="-2147483648" type="String" />
<column name="ID" property="Validation.Formula2" value="2147483647" type="String" />
<column name="ID" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="ID" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="ID" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="ID" property="Validation.ErrorMessage" value="The column requires values of the integer datatype." type="String" />
<column name="ID" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="ID" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_SCHEMA" property="Address" value="$D$7" type="String" />
<column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double" />
<column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_NAME" property="Address" value="$E$7" type="String" />
<column name="TABLE_NAME" property="ColumnWidth" value="27.86" type="Double" />
<column name="COLUMN_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="COLUMN_NAME" property="Address" value="$F$7" type="String" />
<column name="COLUMN_NAME" property="ColumnWidth" value="22.43" type="Double" />
<column name="LANGUAGE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="LANGUAGE_NAME" property="Address" value="$G$7" type="String" />
<column name="LANGUAGE_NAME" property="ColumnWidth" value="19.57" type="Double" />
<column name="LANGUAGE_NAME" property="Validation.Type" value="6" type="Double" />
<column name="LANGUAGE_NAME" property="Validation.AlertStyle" value="1" type="Double" />
<column name="LANGUAGE_NAME" property="Validation.Operator" value="8" type="Double" />
<column name="LANGUAGE_NAME" property="Validation.Formula1" value="2" type="String" />
<column name="LANGUAGE_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="LANGUAGE_NAME" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="LANGUAGE_NAME" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="LANGUAGE_NAME" property="Validation.ErrorMessage" value="The column requires values of the char(2) datatype." type="String" />
<column name="LANGUAGE_NAME" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="LANGUAGE_NAME" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TRANSLATED_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TRANSLATED_NAME" property="Address" value="$H$7" type="String" />
<column name="TRANSLATED_NAME" property="ColumnWidth" value="20.71" type="Double" />
<column name="TRANSLATED_DESC" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TRANSLATED_DESC" property="Address" value="$I$7" type="String" />
<column name="TRANSLATED_DESC" property="ColumnWidth" value="70.71" type="Double" />
<column name="COLUMN_NAME" property="FormatConditions(1).ColumnsCount" value="2" type="Double" />
<column name="COLUMN_NAME" property="FormatConditions(1).AppliesTo.Address" value="$F$7:$G$16" type="String" />
<column name="COLUMN_NAME" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="COLUMN_NAME" property="FormatConditions(1).Priority" value="1" type="Double" />
<column name="COLUMN_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(F7)" type="String" />
<column name="COLUMN_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="COLUMN_NAME" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" />
<column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" />
<column name="" property="ActiveWindow.Split" value="True" type="Boolean" />
<column name="" property="ActiveWindow.SplitRow" value="0" type="Double" />
<column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" />
<column name="" property="PageSetup.Orientation" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" />
</columnFormats>
</table>
');
INSERT INTO "SAVETODB_DEV"."TABLE_FORMATS" ("TABLE_SCHEMA", "TABLE_NAME", "TABLE_EXCEL_FORMAT_XML") VALUES ('SAVETODB_DEV', 'PARAMETER_VALUES', '<?xml version="1.0" encoding="Unicode"?>
<table name="SAVETODB_DEV.PARAMETER_VALUES">
<columnFormats>
<column name="" property="ListObjectName" value="ParameterValues_Table1" type="String" />
<column name="" property="ShowTotals" value="False" type="Boolean" />
<column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" />
<column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" />
<column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" />
<column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" />
<column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" />
<column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" />
<column name="_RowNum" property="Address" value="$B$7" type="String" />
<column name="ID" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="ID" property="Address" value="$C$7" type="String" />
<column name="ID" property="ColumnWidth" value="4.43" type="Double" />
<column name="ID" property="Validation.Type" value="1" type="Double" />
<column name="ID" property="Validation.AlertStyle" value="1" type="Double" />
<column name="ID" property="Validation.Operator" value="1" type="Double" />
<column name="ID" property="Validation.Formula1" value="-2147483648" type="String" />
<column name="ID" property="Validation.Formula2" value="2147483647" type="String" />
<column name="ID" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="ID" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="ID" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="ID" property="Validation.ErrorMessage" value="The column requires values of the integer datatype." type="String" />
<column name="ID" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="ID" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_SCHEMA" property="Address" value="$D$7" type="String" />
<column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double" />
<column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_NAME" property="Address" value="$E$7" type="String" />
<column name="TABLE_NAME" property="ColumnWidth" value="21.71" type="Double" />
<column name="PARAMETER_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="PARAMETER_NAME" property="Address" value="$F$7" type="String" />
<column name="PARAMETER_NAME" property="ColumnWidth" value="20.29" type="Double" />
<column name="SELECT_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="SELECT_SCHEMA" property="Address" value="$G$7" type="String" />
<column name="SELECT_SCHEMA" property="ColumnWidth" value="17.29" type="Double" />
<column name="SELECT_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="SELECT_NAME" property="Address" value="$H$7" type="String" />
<column name="SELECT_NAME" property="ColumnWidth" value="15.29" type="Double" />
<column name="SELECT_TYPE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="SELECT_TYPE" property="Address" value="$I$7" type="String" />
<column name="SELECT_TYPE" property="ColumnWidth" value="13.86" type="Double" />
<column name="SELECT_TYPE" property="Validation.Type" value="3" type="Double" />
<column name="SELECT_TYPE" property="Validation.AlertStyle" value="1" type="Double" />
<column name="SELECT_TYPE" property="Validation.Operator" value="8" type="Double" />
<column name="SELECT_TYPE" property="Validation.Formula1" value="TABLE;VIEW;PROCEDURE;FUNCTION;CODE;HTTP;VALUES;RANGE" type="String" />
<column name="SELECT_TYPE" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="SELECT_TYPE" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="SELECT_TYPE" property="Validation.ErrorTitle" value="Datatype Control" type="String" />
<column name="SELECT_TYPE" property="Validation.ErrorMessage" value="The column requires values of the varchar(64) datatype." type="String" />
<column name="SELECT_TYPE" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="SELECT_TYPE" property="Validation.ShowError" value="True" type="Boolean" />
<column name="SELECT_CODE" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="SELECT_CODE" property="Address" value="$J$7" type="String" />
<column name="SELECT_CODE" property="ColumnWidth" value="58" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).AppliesTo.Address" value="$D$7:$D$8" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Priority" value="1" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Formula1" value="=ISBLANK(D7)" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).AppliesTo.Address" value="$E$7:$E$8" type="String" />
<column name="TABLE_NAME" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Priority" value="2" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(E7)" type="String" />
<column name="TABLE_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="PARAMETER_NAME" property="FormatConditions(1).AppliesTo.Address" value="$F$7:$F$8" type="String" />
<column name="PARAMETER_NAME" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="PARAMETER_NAME" property="FormatConditions(1).Priority" value="3" type="Double" />
<column name="PARAMETER_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(F7)" type="String" />
<column name="PARAMETER_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="PARAMETER_NAME" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="SELECT_NAME" property="FormatConditions(1).AppliesTo.Address" value="$H$7:$H$8" type="String" />
<column name="SELECT_NAME" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="SELECT_NAME" property="FormatConditions(1).Priority" value="4" type="Double" />
<column name="SELECT_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(H7)" type="String" />
<column name="SELECT_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="SELECT_NAME" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" />
<column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean" />
<column name="" property="ActiveWindow.Split" value="True" type="Boolean" />
<column name="" property="ActiveWindow.SplitRow" value="0" type="Double" />
<column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" />
<column name="" property="PageSetup.Orientation" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" />
</columnFormats>
</table>
');
INSERT INTO "SAVETODB_DEV"."TABLE_FORMATS" ("TABLE_SCHEMA", "TABLE_NAME", "TABLE_EXCEL_FORMAT_XML") VALUES ('SAVETODB_DEV', 'TABLE_FORMATS', '<?xml version="1.0" encoding="Unicode"?>
<table name="SAVETODB_DEV.TABLE_FORMATS">
<columnFormats>
<column name="" property="ListObjectName" value="UserObjects_Table1" type="String" />
<column name="" property="ShowTotals" value="False" type="Boolean" />
<column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String" />
<column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean" />
<column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean" />
<column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean" />
<column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean" />
<column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean" />
<column name="_RowNum" property="Address" value="$B$7" type="String" />
<column name="ID" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="ID" property="Address" value="$C$7" type="String" />
<column name="ID" property="ColumnWidth" value="4.43" type="Double" />
<column name="ID" property="Validation.Type" value="1" type="Double" />
<column name="ID" property="Validation.AlertStyle" value="1" type="Double" />
<column name="ID" property="Validation.Operator" value="1" type="Double" />
<column name="ID" property="Validation.Formula1" value="-2147483648" type="String" />
<column name="ID" property="Validation.Formula2" value="2147483647" type="String" />
<column name="ID" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="ID" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="ID" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="ID" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_SCHEMA" property="Address" value="$D$7" type="String" />
<column name="TABLE_SCHEMA" property="ColumnWidth" value="16.57" type="Double" />
<column name="TABLE_SCHEMA" property="Validation.Type" value="6" type="Double" />
<column name="TABLE_SCHEMA" property="Validation.AlertStyle" value="1" type="Double" />
<column name="TABLE_SCHEMA" property="Validation.Operator" value="8" type="Double" />
<column name="TABLE_SCHEMA" property="Validation.Formula1" value="64" type="String" />
<column name="TABLE_SCHEMA" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="TABLE_SCHEMA" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_NAME" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_NAME" property="Address" value="$E$7" type="String" />
<column name="TABLE_NAME" property="ColumnWidth" value="22.57" type="Double" />
<column name="TABLE_NAME" property="Validation.Type" value="6" type="Double" />
<column name="TABLE_NAME" property="Validation.AlertStyle" value="1" type="Double" />
<column name="TABLE_NAME" property="Validation.Operator" value="8" type="Double" />
<column name="TABLE_NAME" property="Validation.Formula1" value="64" type="String" />
<column name="TABLE_NAME" property="Validation.IgnoreBlank" value="True" type="Boolean" />
<column name="TABLE_NAME" property="Validation.InCellDropdown" value="True" type="Boolean" />
<column name="TABLE_NAME" property="Validation.ShowInput" value="True" type="Boolean" />
<column name="TABLE_NAME" property="Validation.ShowError" value="True" type="Boolean" />
<column name="TABLE_EXCEL_FORMAT_XML" property="EntireColumn.Hidden" value="False" type="Boolean" />
<column name="TABLE_EXCEL_FORMAT_XML" property="Address" value="$F$7" type="String" />
<column name="TABLE_EXCEL_FORMAT_XML" property="ColumnWidth" value="70.71" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).AppliesTo.Address" value="$D$7:$D$13" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Priority" value="1" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Formula1" value="=ISBLANK(D7)" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(2).ColumnsCount" value="2" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(2).AppliesTo.Address" value="$D$7:$E$13" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(2).Type" value="2" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(2).Priority" value="4" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(2).Formula1" value="=ISBLANK(D7)" type="String" />
<column name="TABLE_SCHEMA" property="FormatConditions(2).Interior.Color" value="65535" type="Double" />
<column name="TABLE_SCHEMA" property="FormatConditions(2).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).AppliesTo.Address" value="$E$7:$E$13" type="String" />
<column name="TABLE_NAME" property="FormatConditions(1).Type" value="2" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Priority" value="3" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Formula1" value="=ISBLANK(E7)" type="String" />
<column name="TABLE_NAME" property="FormatConditions(1).Interior.Color" value="65535" type="Double" />
<column name="TABLE_NAME" property="FormatConditions(1).Interior.PatternThemeColor" value="-1" type="Double" />
<column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean" />
<column name="" property="ActiveWindow.FreezePanes" value="False" type="Boolean" />
<column name="" property="ActiveWindow.Split" value="False" type="Boolean" />
<column name="" property="ActiveWindow.SplitRow" value="-6" type="Double" />
<column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double" />
<column name="" property="PageSetup.Orientation" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesWide" value="1" type="Double" />
<column name="" property="PageSetup.FitToPagesTall" value="1" type="Double" />
</columnFormats>
</table>');

-- print SaveToDB Framework data have been added

-- print SaveToDB Framework has been installed
