-- =============================================
-- Application: Sample 01 - Basic SaveToDB Features
-- Version 10.8, January 9, 2023
--
-- Copyright 2014-2023 Gartle LLC
--
-- License: MIT
-- =============================================

CREATE SCHEMA S01;

CREATE TABLE S01.CASHBOOK (
    ID INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    DATE DATE DEFAULT NULL,
    ACCOUNT VARCHAR(50) DEFAULT NULL,
    ITEM VARCHAR(50) DEFAULT NULL,
    COMPANY VARCHAR(50) DEFAULT NULL,
    DEBIT DOUBLE DEFAULT NULL,
    CREDIT DOUBLE DEFAULT NULL,
    PRIMARY KEY (ID)
);

CREATE INDEX IX_DATE ON S01.CASHBOOK(DATE);

CREATE TABLE IF NOT EXISTS S01.FORMATS (
    ID INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    TABLE_SCHEMA STRING NOT NULL,
    TABLE_NAME STRING NOT NULL,
    TABLE_EXCEL_FORMAT_XML STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_FORMATS ON S01.FORMATS (TABLE_SCHEMA, TABLE_NAME);

CREATE TABLE S01.WORKBOOKS (
    ID INTEGER NOT NULL GENERATED BY DEFAULT AS IDENTITY,
    NAME STRING NOT NULL,
    TEMPLATE STRING,
    DEFINITION STRING NOT NULL,
    TABLE_SCHEMA STRING,
    PRIMARY KEY (ID)
);

CREATE UNIQUE INDEX IX_WORKBOOKS ON S01.WORKBOOKS (NAME);

CREATE OR REPLACE VIEW S01.VIEW_CASHBOOK
AS

SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT
    , p.ITEM
    , p.COMPANY
    , p.DEBIT
    , p.CREDIT
FROM
    S01.CASHBOOK p
;

SET DELIMITER @@

CREATE OR REPLACE PROCEDURE S01.USP_CASHBOOK (
    ACCOUNT VARCHAR(50)
    , ITEM VARCHAR(50)
    , COMPANY VARCHAR(50)
    )
RETURNS tmp_tab (
  ID INTEGER
  , DATE DATE
  , ACCOUNT VARCHAR(50)
  , ITEM VARCHAR(50)
  , COMPANY VARCHAR(50)
  , DEBIT DOUBLE
  , CREDIT DOUBLE
  )
AS
VAR P1 STRING = ACCOUNT;
VAR P2 STRING = ITEM;
VAR P3 STRING = COMPANY;

INSERT INTO tmp_tab
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT
    , p.ITEM
    , p.COMPANY
    , p.DEBIT
    , p.CREDIT
FROM
    S01.CASHBOOK p
WHERE
    COALESCE(P1, p.ACCOUNT, '') = COALESCE(p.ACCOUNT, '')
    AND COALESCE(P2, p.ITEM, '') = COALESCE(p.ITEM, '')
    AND COALESCE(P3, p.COMPANY, '') = COALESCE(p.COMPANY, '');

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASHBOOK2 (
    ACCOUNT VARCHAR(50)
    , ITEM VARCHAR(50)
    , COMPANY VARCHAR(50)
    )
RETURNS tmp_tab (
  ID INTEGER
  , DATE DATE
  , ACCOUNT VARCHAR(50)
  , ITEM VARCHAR(50)
  , COMPANY VARCHAR(50)
  , DEBIT DOUBLE
  , CREDIT DOUBLE
  )
AS
VAR P1 STRING = ACCOUNT;
VAR P2 STRING = ITEM;
VAR P3 STRING = COMPANY;

INSERT INTO tmp_tab
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT
    , p.ITEM
    , p.COMPANY
    , p.DEBIT
    , p.CREDIT
FROM
    S01.CASHBOOK p
WHERE
    COALESCE(P1, p.ACCOUNT, '') = COALESCE(p.ACCOUNT, '')
    AND COALESCE(P2, p.ITEM, '') = COALESCE(p.ITEM, '')
    AND COALESCE(P3, p.COMPANY, '') = COALESCE(p.COMPANY, '');

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASHBOOK3 (
    ACCOUNT VARCHAR(50)
    , ITEM VARCHAR(50)
    , COMPANY VARCHAR(50)
    )
RETURNS tmp_tab (
  ID INTEGER
  , DATE DATE
  , ACCOUNT VARCHAR(50)
  , ITEM VARCHAR(50)
  , COMPANY VARCHAR(50)
  , DEBIT DOUBLE
  , CREDIT DOUBLE
  )
AS
VAR P1 STRING = ACCOUNT;
VAR P2 STRING = ITEM;
VAR P3 STRING = COMPANY;

INSERT INTO tmp_tab
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT
    , p.ITEM
    , p.COMPANY
    , p.DEBIT
    , p.CREDIT
FROM
    S01.CASHBOOK p
WHERE
    COALESCE(P1, p.ACCOUNT, '') = COALESCE(p.ACCOUNT, '')
    AND COALESCE(P2, p.ITEM, '') = COALESCE(p.ITEM, '')
    AND COALESCE(P3, p.COMPANY, '') = COALESCE(p.COMPANY, '');

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASHBOOK4 (
    ACCOUNT VARCHAR(50)
    , ITEM VARCHAR(50)
    , COMPANY VARCHAR(50)
    )
RETURNS tmp_tab (
  ID INTEGER
  , DATE DATE
  , ACCOUNT VARCHAR(50)
  , ITEM VARCHAR(50)
  , COMPANY VARCHAR(50)
  , DEBIT DOUBLE
  , CREDIT DOUBLE
  )
AS
VAR P1 STRING = ACCOUNT;
VAR P2 STRING = ITEM;
VAR P3 STRING = COMPANY;

INSERT INTO tmp_tab
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT
    , p.ITEM
    , p.COMPANY
    , p.DEBIT
    , p.CREDIT
FROM
    S01.CASHBOOK p
WHERE
    COALESCE(P1, p.ACCOUNT, '') = COALESCE(p.ACCOUNT, '')
    AND COALESCE(P2, p.ITEM, '') = COALESCE(p.ITEM, '')
    AND COALESCE(P3, p.COMPANY, '') = COALESCE(p.COMPANY, '');

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASHBOOK2_INSERT (
    ID INTEGER,
    DATE DATE,
    ACCOUNT VARCHAR(50),
    ITEM VARCHAR(50),
    COMPANY VARCHAR(50),
    DEBIT DOUBLE,
    CREDIT DOUBLE
    )
AS

INSERT INTO S01.CASHBOOK
    ( DATE
    , ACCOUNT
    , COMPANY
    , ITEM
    , DEBIT
    , CREDIT
    )
VALUES
    ( DATE
    , ACCOUNT
    , COMPANY
    , ITEM
    , DEBIT
    , CREDIT
    );

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASHBOOK2_UPDATE (
    ID INTEGER,
    DATE DATE,
    ACCOUNT VARCHAR(50),
    ITEM VARCHAR(50),
    COMPANY VARCHAR(50),
    DEBIT DOUBLE,
    CREDIT DOUBLE
    )
AS

VAR ID1 INTEGER = ID;
VAR DATE1 DATE = DATE;
VAR ACCOUNT1 VARCHAR(50) = ACCOUNT;
VAR ITEM1 VARCHAR(50) = ITEM;
VAR COMPANY1 VARCHAR(50) = COMPANY;
VAR DEBIT1 DOUBLE = DEBIT;
VAR CREDIT1 DOUBLE = CREDIT;

UPDATE S01.CASHBOOK
SET
    DATE = DATE1
    , ACCOUNT = ACCOUNT1
    , COMPANY = COMPANY1
    , ITEM = ITEM1
    , DEBIT = DEBIT1
    , CREDIT = CREDIT1
WHERE
    ID = ID1;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASHBOOK2_DELETE (
    ID INTEGER
    )
AS

VAR ID1 INTEGER = ID;

DELETE FROM S01.CASHBOOK
WHERE
    ID = ID1;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASHBOOK3_CHANGE (
    COLUMN_NAME VARCHAR(255)
    , CELL_VALUE VARCHAR(255)
    , CELL_NUMBER_VALUE DOUBLE
    , CELL_DATETIME_VALUE DATE
    , ID INTEGER
    )
AS

VAR ID1 INTEGER = ID;

IF (COLUMN_NAME = 'DATE')
    UPDATE S01.CASHBOOK
    SET DATE = CELL_DATETIME_VALUE WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'DEBIT')
    UPDATE S01.CASHBOOK
    SET DEBIT = CELL_NUMBER_VALUE WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'CREDIT')
    UPDATE S01.CASHBOOK
    SET CREDIT = CELL_NUMBER_VALUE WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'ACCOUNT')
    UPDATE S01.CASHBOOK
    SET ACCOUNT = CELL_VALUE WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'COMPANY')
    UPDATE S01.CASHBOOK
    SET COMPANY = CELL_VALUE WHERE ID = ID1;
    RETURN;
END_IF;

IF (COLUMN_NAME = 'ITEM')
    UPDATE S01.CASHBOOK
    SET ITEM = CELL_VALUE WHERE ID = ID1;
    RETURN;
END_IF;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASHBOOK4_MERGE (
    ID INTEGER,
    DATE DATE,
    ACCOUNT VARCHAR(50),
    ITEM VARCHAR(50),
    COMPANY VARCHAR(50),
    DEBIT DOUBLE,
    CREDIT DOUBLE
    )
AS

VAR ID1 INTEGER = ID;
VAR DATE1 DATE = DATE;
VAR ACCOUNT1 VARCHAR(50) = ACCOUNT;
VAR ITEM1 VARCHAR(50) = ITEM;
VAR COMPANY1 VARCHAR(50) = COMPANY;
VAR DEBIT1 DOUBLE = DEBIT;
VAR CREDIT1 DOUBLE = CREDIT;

UPDATE S01.CASHBOOK
SET
    DATE = DATE1
    , ACCOUNT = ACCOUNT1
    , COMPANY = COMPANY1
    , ITEM = ITEM1
    , DEBIT = DEBIT1
    , CREDIT = CREDIT1
WHERE
    ID = ID1;

IF ((SELECT GETUPDATECOUNT() FROM DUAL) = 0)
    INSERT INTO S01.CASHBOOK
        ( DATE
        , ACCOUNT
        , COMPANY
        , ITEM
        , DEBIT
        , CREDIT
        )
    VALUES
        ( DATE
        , ACCOUNT
        , COMPANY
        , ITEM
        , DEBIT
        , CREDIT
        );
END_IF;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASH_BY_MONTHS (
    IN "Year" SMALLINT
    )
RETURNS tmp_tab (
    SORT_ORDER BIGINT
    , SECTION INTEGER
    , LEVEL INTEGER
    , ITEM STRING
    , COMPANY STRING
    , "Name" STRING
    , "Total" DOUBLE
    , "Jan" DOUBLE
    , "Feb" DOUBLE
    , "Mar" DOUBLE
    , "Apr" DOUBLE
    , "May" DOUBLE
    , "Jun" DOUBLE
    , "Jul" DOUBLE
    , "Aug" DOUBLE
    , "Sep" DOUBLE
    , "Oct" DOUBLE
    , "Nov" DOUBLE
    , "Dec" DOUBLE
    )
AS
INSERT INTO tmp_tab
SELECT
    ROW_NUMBER() OVER(ORDER BY p.SECTION, p.ITEM, p.COMPANY) AS SORT_ORDER
    , p.SECTION
    , MAX(p.LEVEL) AS LEVEL
    , p.ITEM
    , p.COMPANY
    , CASE WHEN p.COMPANY IS NOT NULL THEN CONCAT('  ', MAX(p.NAME)) ELSE MAX(p.NAME) END AS "Name"
    , CASE WHEN p.SECTION = 1 THEN SUM(COALESCE(p."Jan", 0)) WHEN p.SECTION = 5 THEN SUM(COALESCE(p."Dec", 0)) ELSE SUM(COALESCE(p.TOTAL, 0)) END AS "Total"
    , SUM(p."Jan") AS "Jan"
    , SUM(p."Feb") AS "Feb"
    , SUM(p."Mar") AS "Mar"
    , SUM(p."Apr") AS "Apr"
    , SUM(p."May") AS "May"
    , SUM(p."Jun") AS "Jun"
    , SUM(p."Jul") AS "Jul"
    , SUM(p."Aug") AS "Aug"
    , SUM(p."Sep") AS "Sep"
    , SUM(p."Oct") AS "Oct"
    , SUM(p."Nov") AS "Nov"
    , SUM(p."Dec") AS "Dec"
FROM
    (
    -- Opening balance
    SELECT
        1 AS SECTION
        , 0 AS LEVEL
        , NULL AS ITEM
        , NULL AS COMPANY
        , 'Opening Balance' AS NAME
        , p.PERIOD
        , NULL AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            p.PERIOD
            , p.AMOUNT
        FROM
            (
            SELECT
                d.PERIOD
                , p."DATE" AS DATE1
                , COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) AS AMOUNT
            FROM
                (
                    SELECT 1 AS PERIOD FROM DUAL
                    UNION ALL SELECT 2 FROM DUAL
                    UNION ALL SELECT 3 FROM DUAL
                    UNION ALL SELECT 4 FROM DUAL
                    UNION ALL SELECT 5 FROM DUAL
                    UNION ALL SELECT 6 FROM DUAL
                    UNION ALL SELECT 7 FROM DUAL
                    UNION ALL SELECT 8 FROM DUAL
                    UNION ALL SELECT 9 FROM DUAL
                    UNION ALL SELECT 10 FROM DUAL
                    UNION ALL SELECT 11 FROM DUAL
                    UNION ALL SELECT 12 FROM DUAL
                ) d
                , S01.CASHBOOK p
            ) p
        WHERE
            YEAR(p.DATE1) < "Year"
            OR (YEAR(p.DATE1) = "Year" AND MONTH(p.DATE1) < p.PERIOD)
        ) p

    -- Closing balance
    UNION ALL
    SELECT
        5 AS SECTION
        , 0 AS LEVEL
        , NULL AS ITEM
        , NULL AS COMPANY
        , 'Closing Balance' AS NAME
        , p.PERIOD
        , NULL AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            p.PERIOD
            , p.AMOUNT
        FROM
            (
            SELECT
                d.PERIOD
                , p."DATE" AS DATE1
                , COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) AS AMOUNT
            FROM
                (
                    SELECT 1 AS PERIOD FROM DUAL
                    UNION ALL SELECT 2 FROM DUAL
                    UNION ALL SELECT 3 FROM DUAL
                    UNION ALL SELECT 4 FROM DUAL
                    UNION ALL SELECT 5 FROM DUAL
                    UNION ALL SELECT 6 FROM DUAL
                    UNION ALL SELECT 7 FROM DUAL
                    UNION ALL SELECT 8 FROM DUAL
                    UNION ALL SELECT 9 FROM DUAL
                    UNION ALL SELECT 10 FROM DUAL
                    UNION ALL SELECT 11 FROM DUAL
                    UNION ALL SELECT 12 FROM DUAL
                ) d
                , S01.CASHBOOK p
            ) p
        WHERE
            YEAR(p.DATE1) < "Year"
            OR (YEAR(p.DATE1) = "Year" AND MONTH(p.DATE1) <= p.PERIOD)
        ) p

    -- Companies
    UNION ALL
    SELECT
        p.SECTION
        , 2 AS LEVEL
        , p.ITEM
        , p.COMPANY
        , p.COMPANY AS NAME
        , p.PERIOD
        , p.AMOUNT AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            CAST(CASE WHEN p.CREDIT IS NOT NULL THEN 3 ELSE 2 END AS INTEGER) AS SECTION
            , p.ITEM
            , p.COMPANY
            , MONTH(p."DATE") AS PERIOD
            , CASE WHEN p.CREDIT IS NOT NULL THEN COALESCE(p.CREDIT, 0) - COALESCE(p.DEBIT, 0) ELSE COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) END AS AMOUNT
        FROM
            S01.CASHBOOK p
        WHERE
            p.COMPANY IS NOT NULL
            AND YEAR(p."DATE") = "Year"
        ) p

    -- Total Items
    UNION ALL
    SELECT
        p.SECTION
        , 1 AS LEVEL
        , p.ITEM
        , NULL AS COMPANY
        , p.ITEM AS NAME
        , p.PERIOD
        , p.AMOUNT AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            CAST(CASE WHEN p.CREDIT IS NOT NULL THEN 3 ELSE 2 END AS INTEGER) AS SECTION
            , p.ITEM
            , MONTH(p."DATE") AS PERIOD
            , CASE WHEN p.CREDIT IS NOT NULL THEN COALESCE(p.CREDIT, 0) - COALESCE(p.DEBIT, 0) ELSE COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) END AS AMOUNT
        FROM
            S01.CASHBOOK p
        WHERE
            p.ITEM IS NOT NULL
            AND YEAR(p."DATE") = "Year"
        ) p

    -- Total Income/Expenses
    UNION ALL
    SELECT
        p.SECTION
        , 0 AS LEVEL
        , NULL AS ITEM
        , NULL AS COMPANY
        , p.ITEM_TYPE AS NAME
        , p.PERIOD
        , p.AMOUNT AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            CAST(CASE WHEN p.CREDIT IS NOT NULL THEN 3 ELSE 2 END AS INTEGER) AS SECTION
            , CAST(CASE WHEN p.CREDIT IS NOT NULL THEN 'Total Expenses' ELSE 'Total Income' END AS STRING) AS ITEM_TYPE
            , MONTH(p."DATE") AS PERIOD
            , CASE WHEN p.CREDIT IS NOT NULL THEN COALESCE(p.CREDIT, 0) - COALESCE(p.DEBIT, 0) ELSE COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) END AS AMOUNT
        FROM
            S01.CASHBOOK p
        WHERE
            YEAR(p."DATE") = "Year"
        ) p

    -- Net Chanhge
    UNION ALL
    SELECT
        4 AS SECTION
        , 0 AS LEVEL
        , NULL AS ITEM
        , NULL AS COMPANY
        , 'Net Change' AS NAME
        , p.PERIOD
        , p.AMOUNT AS TOTAL
        , CASE p.PERIOD WHEN  1 THEN p.AMOUNT ELSE 0 END AS "Jan"
        , CASE p.PERIOD WHEN  2 THEN p.AMOUNT ELSE 0 END AS "Feb"
        , CASE p.PERIOD WHEN  3 THEN p.AMOUNT ELSE 0 END AS "Mar"
        , CASE p.PERIOD WHEN  4 THEN p.AMOUNT ELSE 0 END AS "Apr"
        , CASE p.PERIOD WHEN  5 THEN p.AMOUNT ELSE 0 END AS "May"
        , CASE p.PERIOD WHEN  6 THEN p.AMOUNT ELSE 0 END AS "Jun"
        , CASE p.PERIOD WHEN  7 THEN p.AMOUNT ELSE 0 END AS "Jul"
        , CASE p.PERIOD WHEN  8 THEN p.AMOUNT ELSE 0 END AS "Aug"
        , CASE p.PERIOD WHEN  9 THEN p.AMOUNT ELSE 0 END AS "Sep"
        , CASE p.PERIOD WHEN 10 THEN p.AMOUNT ELSE 0 END AS "Oct"
        , CASE p.PERIOD WHEN 11 THEN p.AMOUNT ELSE 0 END AS "Nov"
        , CASE p.PERIOD WHEN 12 THEN p.AMOUNT ELSE 0 END AS "Dec"
    FROM
        (
        SELECT
            MONTH(p."DATE") AS PERIOD
            , COALESCE(p.DEBIT, 0) - COALESCE(p.CREDIT, 0) AS AMOUNT
        FROM
            S01.CASHBOOK p
        WHERE
            YEAR(p."DATE") = "Year"
        ) p

    ) p
GROUP BY
    p.SECTION
    , p.ITEM
    , p.COMPANY
ORDER BY
    SORT_ORDER;

END_PROCEDURE
@@

CREATE OR REPLACE PROCEDURE S01.USP_CASH_BY_MONTHS_CHANGE (
    COLUMN_NAME VARCHAR(255)
    , CELL_NUMBER_VALUE DOUBLE
    , SECTION INTEGER
    , ITEM STRING
    , COMPANY STRING
    , YEAR INTEGER
    )
AS

VAR ITEM1 STRING = ITEM;
VAR COMPANY1 STRING = COMPANY;

VAR month INTEGER;
VAR start_date DATE;
VAR end_date DATE;
VAR id1 INTEGER;
VAR count1 INTEGER;
VAR date1 DATE;
VAR account1 STRING;

month = POSITION(' ' || COLUMN_NAME || ' ' IN '    Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec ') / 4;

IF (month < 1) RETURN; END_IF;

IF (year = NULL) year = YEAR(CURRENT DATE); END_IF;

IF (year = NULL) year = (SELECT YEAR(MAX(DATE)) FROM S01.CASHBOOK); END_IF;

start_date = DATE_ADD(CAST(year AS string) || '-01-01', interval (month - 1) MONTH);

end_date = DATE_ADD(DATE_ADD(CAST(year AS string) || '-01-01', interval month MONTH), interval - 1 DAY);

count1 = (SELECT COUNT(*) FROM S01.CASHBOOK WHERE ITEM = ITEM1 AND COALESCE(COMPANY, '') = COALESCE(COMPANY1, '') AND DATE BETWEEN start_date AND end_date);

IF (count1 = 0)

    IF (ITEM1 IS NULL)
        RETURN;
    END_IF;

    id1 = (SELECT MAX(ID) FROM S01.CASHBOOK WHERE ITEM = ITEM1 AND COALESCE(COMPANY, '') = COALESCE(COMPANY1, '') AND DATE < end_date);

    IF (id1 IS NOT NULL)

        date1 = (SELECT DATE FROM S01.CASHBOOK WHERE ID = id1);
        account1 = (SELECT ACCOUNT FROM S01.CASHBOOK WHERE ID = id1);

        IF (DAY(date1) > DAY(end_date))
            date1 = end_date;
        ELSE
            date1 = DATE_ADD(DATE_ADD(date1, interval -MONTH(date1) MONTH), interval month MONTH);
        END_IF;
    ELSE
        date1 = end_date;
    END_IF;

    INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT)
        VALUES (date1, account1, ITEM, COMPANY,
            CASE WHEN SECTION = 3 THEN NULL ELSE CELL_NUMBER_VALUE END,
            CASE WHEN SECTION = 3 THEN CELL_NUMBER_VALUE ELSE NULL END);

    RETURN;
END_IF;

IF (count1 > 1)
    RETURN;
END_IF;

id1 = (SELECT MAX(id) FROM S01.CASHBOOK WHERE ITEM = ITEM1 AND COALESCE(COMPANY, '') = COALESCE(COMPANY1, '') AND DATE BETWEEN start_date AND end_date);

UPDATE S01.CASHBOOK
SET
    DEBIT = CASE WHEN SECTION = 3 THEN NULL ELSE CELL_NUMBER_VALUE END
    , CREDIT = CASE WHEN SECTION = 3 THEN CELL_NUMBER_VALUE ELSE NULL END
WHERE
    ID = id1;

END_PROCEDURE
@@

SET DELIMITER ;

INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-01-10', 'Bank', 'Revenue', 'Customer C1', 200000, NULL);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-01-10', 'Bank', 'Expenses', 'Supplier S1', NULL, 50000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-01-31', 'Bank', 'Payroll', NULL, NULL, 85000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-01-31', 'Bank', 'Taxes', 'Individual Income Tax', NULL, 15000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-01-31', 'Bank', 'Taxes', 'Payroll Taxes', NULL, 15000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-02-10', 'Bank', 'Revenue', 'Customer C1', 300000, NULL);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-02-10', 'Bank', 'Revenue', 'Customer C2', 100000, NULL);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-02-10', 'Bank', 'Expenses', 'Supplier S1', NULL, 100000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-02-10', 'Bank', 'Expenses', 'Supplier S2', NULL, 50000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-02-28', 'Bank', 'Payroll', NULL, NULL, 85000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-02-28', 'Bank', 'Taxes', 'Individual Income Tax', NULL, 15000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-02-28', 'Bank', 'Taxes', 'Payroll Taxes', NULL, 15000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-03-10', 'Bank', 'Revenue', 'Customer C1', 300000, NULL);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-03-10', 'Bank', 'Revenue', 'Customer C2', 200000, NULL);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-03-10', 'Bank', 'Revenue', 'Customer C3', 100000, NULL);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-03-15', 'Bank', 'Taxes', 'Corporate Income Tax', NULL, 100000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-03-31', 'Bank', 'Payroll', NULL, NULL, 170000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-03-31', 'Bank', 'Taxes', 'Individual Income Tax', NULL, 30000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-03-31', 'Bank', 'Taxes', 'Payroll Taxes', NULL, 30000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-03-31', 'Bank', 'Expenses', 'Supplier S1', NULL, 100000);
INSERT INTO S01.CASHBOOK (DATE, ACCOUNT, ITEM, COMPANY, DEBIT, CREDIT) VALUES ('2023-03-31', 'Bank', 'Expenses', 'Supplier S2', NULL, 50000);

INSERT INTO S01.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S01', 'USP_CASH_BY_MONTHS', '<table name="s01.usp_cash_by_months"><columnFormats><column name="" property="ListObjectName" value="usp_cash_by_months" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium15" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="sort_order" property="Address" value="$C$4" type="String"/><column name="sort_order" property="NumberFormat" value="General" type="String"/><column name="section" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="section" property="Address" value="$D$4" type="String"/><column name="section" property="NumberFormat" value="General" type="String"/><column name="level" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="level" property="Address" value="$E$4" type="String"/><column name="level" property="NumberFormat" value="General" type="String"/><column name="item" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="item" property="Address" value="$F$4" type="String"/><column name="item" property="NumberFormat" value="General" type="String"/><column name="company" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="company" property="Address" value="$G$4" type="String"/><column name="company" property="NumberFormat" value="General" type="String"/><column name="Name" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Name" property="Address" value="$H$4" type="String"/><column name="Name" property="ColumnWidth" value="21.43" type="Double"/><column name="Name" property="NumberFormat" value="General" type="String"/><column name="Jan" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jan" property="Address" value="$I$4" type="String"/><column name="Jan" property="ColumnWidth" value="10" type="Double"/><column name="Jan" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Feb" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Feb" property="Address" value="$J$4" type="String"/><column name="Feb" property="ColumnWidth" value="10" type="Double"/><column name="Feb" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Mar" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Mar" property="Address" value="$K$4" type="String"/><column name="Mar" property="ColumnWidth" value="10" type="Double"/><column name="Mar" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Apr" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Apr" property="Address" value="$L$4" type="String"/><column name="Apr" property="ColumnWidth" value="10" type="Double"/><column name="Apr" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="May" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="May" property="Address" value="$M$4" type="String"/><column name="May" property="ColumnWidth" value="10" type="Double"/><column name="May" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Jun" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jun" property="Address" value="$N$4" type="String"/><column name="Jun" property="ColumnWidth" value="10" type="Double"/><column name="Jun" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Jul" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jul" property="Address" value="$O$4" type="String"/><column name="Jul" property="ColumnWidth" value="10" type="Double"/><column name="Jul" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Aug" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Aug" property="Address" value="$P$4" type="String"/><column name="Aug" property="ColumnWidth" value="10" type="Double"/><column name="Aug" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Sep" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Sep" property="Address" value="$Q$4" type="String"/><column name="Sep" property="ColumnWidth" value="10" type="Double"/><column name="Sep" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Oct" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Oct" property="Address" value="$R$4" type="String"/><column name="Oct" property="ColumnWidth" value="10" type="Double"/><column name="Oct" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Nov" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Nov" property="Address" value="$S$4" type="String"/><column name="Nov" property="ColumnWidth" value="10" type="Double"/><column name="Nov" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="Dec" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Dec" property="Address" value="$T$4" type="String"/><column name="Dec" property="ColumnWidth" value="10" type="Double"/><column name="Dec" property="NumberFormat" value="#,##0;[Red]-#,##0;" type="String"/><column name="_RowNum" property="FormatConditions(1).AppliesToTable" value="True" type="Boolean"/><column name="_RowNum" property="FormatConditions(1).AppliesTo.Address" value="$B$4:$T$20" type="String"/><column name="_RowNum" property="FormatConditions(1).Type" value="2" type="Double"/><column name="_RowNum" property="FormatConditions(1).Priority" value="3" type="Double"/><column name="_RowNum" property="FormatConditions(1).Formula1" value="=$E4&lt;2" type="String"/><column name="_RowNum" property="FormatConditions(1).Font.Bold" value="True" type="Boolean"/><column name="_RowNum" property="FormatConditions(2).AppliesToTable" value="True" type="Boolean"/><column name="_RowNum" property="FormatConditions(2).AppliesTo.Address" value="$B$4:$T$20" type="String"/><column name="_RowNum" property="FormatConditions(2).Type" value="2" type="Double"/><column name="_RowNum" property="FormatConditions(2).Priority" value="4" type="Double"/><column name="_RowNum" property="FormatConditions(2).Formula1" value="=AND($E4=0,$D4&gt;1,$D4&lt;5)" type="String"/><column name="_RowNum" property="FormatConditions(2).Font.Bold" value="True" type="Boolean"/><column name="_RowNum" property="FormatConditions(2).Font.Color" value="16777215" type="Double"/><column name="_RowNum" property="FormatConditions(2).Font.ThemeColor" value="1" type="Double"/><column name="_RowNum" property="FormatConditions(2).Font.TintAndShade" value="0" type="Double"/><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="6773025" type="Double"/><column name="_RowNum" property="FormatConditions(2).Interior.Color" value="6773025" type="Double"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All columns"><column name="" property="ListObjectName" value="cash_by_month" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="sort_order" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="section" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="level" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Name" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jan" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Feb" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Mar" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Apr" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="May" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jun" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jul" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Aug" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Sep" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Oct" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Nov" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Dec" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Default"><column name="" property="ListObjectName" value="cash_by_month" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="sort_order" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="section" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="level" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="Name" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jan" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Feb" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Mar" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Apr" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="May" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jun" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Jul" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Aug" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Sep" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Oct" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Nov" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="Dec" property="EntireColumn.Hidden" value="False" type="Boolean"/></view></views></table>');
INSERT INTO S01.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S01', 'CASHBOOK', '<table name="s01.cashbook"><columnFormats><column name="" property="ListObjectName" value="cashbook" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="Address" value="$E$4" type="String"/><column name="account" property="ColumnWidth" value="12.14" type="Double"/><column name="account" property="NumberFormat" value="General" type="String"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="Address" value="$F$4" type="String"/><column name="item" property="ColumnWidth" value="20.71" type="Double"/><column name="item" property="NumberFormat" value="General" type="String"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="Address" value="$G$4" type="String"/><column name="company" property="ColumnWidth" value="20.71" type="Double"/><column name="company" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO S01.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S01', 'VIEW_CASHBOOK', '<table name="s01.view_cashbook"><columnFormats><column name="" property="ListObjectName" value="view_cashbook" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="Address" value="$E$4" type="String"/><column name="account" property="ColumnWidth" value="12.14" type="Double"/><column name="account" property="NumberFormat" value="General" type="String"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="Address" value="$F$4" type="String"/><column name="item" property="ColumnWidth" value="20.71" type="Double"/><column name="item" property="NumberFormat" value="General" type="String"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="Address" value="$G$4" type="String"/><column name="company" property="ColumnWidth" value="20.71" type="Double"/><column name="company" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO S01.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S01', 'USP_CASHBOOK', '<table name="s01.usp_cashbook"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="Address" value="$E$4" type="String"/><column name="account" property="ColumnWidth" value="12.14" type="Double"/><column name="account" property="NumberFormat" value="General" type="String"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="Address" value="$F$4" type="String"/><column name="item" property="ColumnWidth" value="20.71" type="Double"/><column name="item" property="NumberFormat" value="General" type="String"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="Address" value="$G$4" type="String"/><column name="company" property="ColumnWidth" value="20.71" type="Double"/><column name="company" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO S01.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S01', 'USP_CASHBOOK2', '<table name="s01.usp_cashbook2"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook2" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="Address" value="$E$4" type="String"/><column name="account" property="ColumnWidth" value="12.14" type="Double"/><column name="account" property="NumberFormat" value="General" type="String"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="Address" value="$F$4" type="String"/><column name="item" property="ColumnWidth" value="20.71" type="Double"/><column name="item" property="NumberFormat" value="General" type="String"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="Address" value="$G$4" type="String"/><column name="company" property="ColumnWidth" value="20.71" type="Double"/><column name="company" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO S01.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S01', 'USP_CASHBOOK3', '<table name="s01.usp_cashbook3"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook3" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="Address" value="$E$4" type="String"/><column name="account" property="ColumnWidth" value="12.14" type="Double"/><column name="account" property="NumberFormat" value="General" type="String"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="Address" value="$F$4" type="String"/><column name="item" property="ColumnWidth" value="20.71" type="Double"/><column name="item" property="NumberFormat" value="General" type="String"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="Address" value="$G$4" type="String"/><column name="company" property="ColumnWidth" value="20.71" type="Double"/><column name="company" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');
INSERT INTO S01.FORMATS (TABLE_SCHEMA, TABLE_NAME, TABLE_EXCEL_FORMAT_XML) VALUES ('S01', 'USP_CASHBOOK4', '<table name="s01.usp_cashbook4"><columnFormats><column name="" property="ListObjectName" value="usp_cashbook4" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="" property="TableStyle.Name" value="TableStyleMedium2" type="String"/><column name="" property="ShowTableStyleColumnStripes" value="False" type="Boolean"/><column name="" property="ShowTableStyleFirstColumn" value="False" type="Boolean"/><column name="" property="ShowShowTableStyleLastColumn" value="False" type="Boolean"/><column name="" property="ShowTableStyleRowStripes" value="True" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="_RowNum" property="Address" value="$B$4" type="String"/><column name="_RowNum" property="NumberFormat" value="General" type="String"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="id" property="Address" value="$C$4" type="String"/><column name="id" property="ColumnWidth" value="4.29" type="Double"/><column name="id" property="NumberFormat" value="General" type="String"/><column name="id" property="Validation.Type" value="1" type="Double"/><column name="id" property="Validation.Operator" value="1" type="Double"/><column name="id" property="Validation.Formula1" value="-2147483648" type="String"/><column name="id" property="Validation.Formula2" value="2147483647" type="String"/><column name="id" property="Validation.AlertStyle" value="1" type="Double"/><column name="id" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="id" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="id" property="Validation.ShowInput" value="True" type="Boolean"/><column name="id" property="Validation.ShowError" value="True" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="Address" value="$D$4" type="String"/><column name="date" property="ColumnWidth" value="11.43" type="Double"/><column name="date" property="NumberFormat" value="m/d/yyyy" type="String"/><column name="date" property="Validation.Type" value="4" type="Double"/><column name="date" property="Validation.Operator" value="5" type="Double"/><column name="date" property="Validation.Formula1" value="12/31/1899" type="String"/><column name="date" property="Validation.AlertStyle" value="1" type="Double"/><column name="date" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="date" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="date" property="Validation.ShowInput" value="True" type="Boolean"/><column name="date" property="Validation.ShowError" value="True" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="Address" value="$E$4" type="String"/><column name="account" property="ColumnWidth" value="12.14" type="Double"/><column name="account" property="NumberFormat" value="General" type="String"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="Address" value="$F$4" type="String"/><column name="item" property="ColumnWidth" value="20.71" type="Double"/><column name="item" property="NumberFormat" value="General" type="String"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="Address" value="$G$4" type="String"/><column name="company" property="ColumnWidth" value="20.71" type="Double"/><column name="company" property="NumberFormat" value="General" type="String"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="Address" value="$H$4" type="String"/><column name="debit" property="ColumnWidth" value="11.43" type="Double"/><column name="debit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="debit" property="Validation.Type" value="2" type="Double"/><column name="debit" property="Validation.Operator" value="4" type="Double"/><column name="debit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="debit" property="Validation.AlertStyle" value="1" type="Double"/><column name="debit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="debit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="debit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="debit" property="Validation.ShowError" value="True" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="Address" value="$I$4" type="String"/><column name="credit" property="ColumnWidth" value="11.43" type="Double"/><column name="credit" property="NumberFormat" value="#,##0.00_ ;[Red]-#,##0.00 " type="String"/><column name="credit" property="Validation.Type" value="2" type="Double"/><column name="credit" property="Validation.Operator" value="4" type="Double"/><column name="credit" property="Validation.Formula1" value="-1.11222333444555E+29" type="String"/><column name="credit" property="Validation.AlertStyle" value="1" type="Double"/><column name="credit" property="Validation.IgnoreBlank" value="True" type="Boolean"/><column name="credit" property="Validation.InCellDropdown" value="True" type="Boolean"/><column name="credit" property="Validation.ShowInput" value="True" type="Boolean"/><column name="credit" property="Validation.ShowError" value="True" type="Boolean"/><column name="" property="ActiveWindow.DisplayGridlines" value="False" type="Boolean"/><column name="" property="ActiveWindow.FreezePanes" value="True" type="Boolean"/><column name="" property="ActiveWindow.Split" value="True" type="Boolean"/><column name="" property="ActiveWindow.SplitRow" value="0" type="Double"/><column name="" property="ActiveWindow.SplitColumn" value="-2" type="Double"/><column name="" property="PageSetup.Orientation" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesWide" value="1" type="Double"/><column name="" property="PageSetup.FitToPagesTall" value="1" type="Double"/></columnFormats><views><view name="All rows"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/></view><view name="Incomes"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view><view name="Expenses"><column name="" property="ListObjectName" value="cash_book" type="String"/><column name="" property="ShowTotals" value="False" type="Boolean"/><column name="_RowNum" property="EntireColumn.Hidden" value="True" type="Boolean"/><column name="id" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="date" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="account" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="item" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="company" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="debit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="EntireColumn.Hidden" value="False" type="Boolean"/><column name="credit" property="AutoFilter.Criteria1" value="&lt;&gt;" type="String"/></view></views></table>');

INSERT INTO S01.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 01 - Basic Features - User1.xlsx', 'https://www.savetodb.com/downloads/v10/sample01-user1.xlsx', '
cashbook=S01.CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"cashbook"}
view_cashbook=S01.VIEW_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"view_cashbook"}
usp_cashbook=S01.USP_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"usp_cashbook"}
usp_cashbook2=S01.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"usp_cashbook2"}
usp_cashbook3=S01.USP_CASHBOOK3,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"usp_cashbook3"}
usp_cashbook4=S01.USP_CASHBOOK4,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"usp_cashbook4"}
cash_by_months=S01.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2021},"ListObjectName":"cash_by_months"}
', 'S01');

INSERT INTO S01.WORKBOOKS (NAME, TEMPLATE, DEFINITION, TABLE_SCHEMA) VALUES ('Sample 01 - Basic Features - User2 (Restricted).xlsx', 'https://www.savetodb.com/downloads/v10/sample01-user2.xlsx', '
cashbook=S01.CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"cashbook"}
view_cashbook=S01.VIEW_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"view_cashbook"}
usp_cashbook=S01.USP_CASHBOOK,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"usp_cashbook"}
usp_cashbook2=S01.USP_CASHBOOK2,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"usp_cashbook2"}
usp_cashbook3=S01.USP_CASHBOOK3,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"usp_cashbook3"}
usp_cashbook4=S01.USP_CASHBOOK4,(Default),False,$B$3,,{"Parameters":{"account":null,"item":null,"company":null},"ListObjectName":"usp_cashbook4"}
cash_by_months=S01.USP_CASH_BY_MONTHS,(Default),False,$B$3,,{"Parameters":{"year":2021},"ListObjectName":"cash_by_months"}
', 'S01');

-- print Application installed
