-- =============================================
-- SaveToDB Sample 02 for Snowflake - Advanced Features
-- Version 10.8, January 9, 2023
--
-- Copyright 2018-2023 Gartle LLC
--
-- License: MIT
--
-- Prerequisites: SaveToDB Framework 8.19 or higher
-- =============================================

CREATE SCHEMA S02;

CREATE TABLE S02.ITEMS (
      ID int IDENTITY(1,1) NOT NULL
    , NAME varchar(50) NOT NULL
    , CONSTRAINT PK_ITEMS PRIMARY KEY (ID)
    , CONSTRAINT IX_ITEMS_NAME UNIQUE (NAME)
);

CREATE TABLE S02.ACCOUNTS (
      ID int IDENTITY(1,1) NOT NULL
    , NAME varchar(50) NOT NULL
    , CONSTRAINT PK_ACCOUNTS PRIMARY KEY (ID)
    , CONSTRAINT IX_ACCOUNTS_NAME UNIQUE (NAME)
);

CREATE TABLE S02.COMPANIES (
      ID int IDENTITY(1,1) NOT NULL
    , NAME VARCHAR(50) NOT NULL
    , CONSTRAINT PK_COMPANIES PRIMARY KEY (ID)
);

CREATE TABLE S02.ITEM_COMPANIES (
    ITEM_ID int,
    COMPANY_ID int,
    CONSTRAINT PK_ITEM_COMPANIES PRIMARY KEY (ITEM_ID, COMPANY_ID)
);

ALTER TABLE S02.ITEM_COMPANIES ADD CONSTRAINT FK_ITEM_COMPANIES_ITEMS FOREIGN KEY (ITEM_ID) REFERENCES S02.ITEMS (ID) ON UPDATE CASCADE;

ALTER TABLE S02.ITEM_COMPANIES ADD CONSTRAINT FK_ITEM_COMPANIES_COMPANIES FOREIGN KEY (COMPANY_ID) REFERENCES S02.COMPANIES (ID) ON UPDATE CASCADE;

CREATE TABLE S02.CASHBOOK (
      ID int IDENTITY(1,1) NOT NULL
    , DATE date NOT NULL
    , ACCOUNT_ID int NOT NULL
    , ITEM_ID int NOT NULL
    , COMPANY_ID int NULL
    , DEBIT number(15,2) NULL
    , CREDIT number(15,2) NULL
    , CHECKED boolean NULL
    , CONSTRAINT PK_CASHBOOK PRIMARY KEY (ID)
);

ALTER TABLE S02.CASHBOOK ADD CONSTRAINT FK_CASHBOOK_ITEMS FOREIGN KEY (ITEM_ID) REFERENCES S02.ITEMS (ID) ON UPDATE CASCADE;

ALTER TABLE S02.CASHBOOK ADD CONSTRAINT FK_CASHBOOK_ACCOUNTS FOREIGN KEY (ACCOUNT_ID) REFERENCES S02.ACCOUNTS (ID) ON UPDATE CASCADE;

ALTER TABLE S02.CASHBOOK ADD CONSTRAINT FK_CASHBOOK_COMPANIES FOREIGN KEY (COMPANY_ID) REFERENCES S02.COMPANIES (ID) ON UPDATE CASCADE;

CREATE OR REPLACE VIEW S02.VIEW_CASHBOOK
AS
SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT_ID
    , p.ITEM_ID
    , p.COMPANY_ID
    , p.DEBIT
    , p.CREDIT
    , p.CHECKED
FROM
    S02.CASHBOOK p;

CREATE OR REPLACE VIEW S02.VIEW_CASHBOOK2
AS
SELECT
    *
FROM
    S02.CASHBOOK p;

CREATE OR REPLACE VIEW S02.VIEW_CASHBOOK3
AS
SELECT
    p.*
FROM
    S02.CASHBOOK p;

CREATE OR REPLACE FUNCTION S02.USP_CASHBOOK (
    ACCOUNT_ID int
    , ITEM_ID int
    , COMPANY_ID int
)
RETURNS TABLE (
    ID int,
    DATE date,
    ACCOUNT_ID int,
    ITEM_ID int,
    COMPANY_ID int,
    DEBIT number(15,2),
    CREDIT number(15,2),
    CHECKED boolean
)
AS
'SELECT
    p.ID
    , p.DATE
    , p.ACCOUNT_ID
    , p.ITEM_ID
    , p.COMPANY_ID
    , p.DEBIT
    , p.CREDIT
    , p.CHECKED
FROM
    S02.CASHBOOK p
WHERE
    COALESCE(p.ACCOUNT_ID, -1) = COALESCE(ACCOUNT_ID, p.ACCOUNT_ID, -1)
    AND COALESCE(p.ITEM_ID, -1) = COALESCE(ITEM_ID, p.ITEM_ID, -1)
    AND COALESCE(p.COMPANY_ID, -1) = COALESCE(COMPANY_ID, p.COMPANY_ID, -1)';

INSERT INTO S02.ACCOUNTS (ID, NAME) VALUES (1, 'Bank');

INSERT INTO S02.ITEMS (ID, NAME) VALUES (1, 'Revenue');
INSERT INTO S02.ITEMS (ID, NAME) VALUES (2, 'Expenses');
INSERT INTO S02.ITEMS (ID, NAME) VALUES (3, 'Payroll');
INSERT INTO S02.ITEMS (ID, NAME) VALUES (4, 'Taxes');

INSERT INTO S02.COMPANIES (ID, NAME) VALUES (1, 'Customer C1');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (2, 'Customer C2');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (3, 'Customer C3');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (4, 'Customer C4');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (5, 'Customer C5');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (6, 'Customer C6');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (7, 'Customer C7');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (8, 'Supplier S1');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (9, 'Supplier S2');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (10, 'Supplier S3');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (11, 'Supplier S4');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (12, 'Supplier S5');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (13, 'Supplier S6');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (14, 'Supplier S7');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (15, 'Corporate Income Tax');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (16, 'Individual Income Tax');
INSERT INTO S02.COMPANIES (ID, NAME) VALUES (17, 'Payroll Taxes');

INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 1);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 2);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 3);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 4);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 5);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 6);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (1, 7);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 8);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 9);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 10);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 11);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 12);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 13);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (2, 14);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (4, 15);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (4, 16);
INSERT INTO S02.ITEM_COMPANIES (ITEM_ID, COMPANY_ID) VALUES (4, 17);

INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-10', 1, 1, 1, 200000, NULL, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-10', 1, 2, 8, NULL, 50000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-31', 1, 3, NULL, NULL, 85000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-31', 1, 4, 16, NULL, 15000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-01-31', 1, 4, 17, NULL, 15000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-10', 1, 1, 1, 300000, NULL, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-10', 1, 1, 2, 100000, NULL, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-10', 1, 2, 9, NULL, 50000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-10', 1, 2, 8, NULL, 100000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-28', 1, 3, NULL, NULL, 85000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-28', 1, 4, 16, NULL, 15000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-02-28', 1, 4, 17, NULL, 15000, 1);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-10', 1, 1, 1, 300000, NULL, 0);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-10', 1, 1, 2, 200000, NULL, 0);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-10', 1, 1, 3, 100000, NULL, 0);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-15', 1, 4, 15, NULL, 100000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 3, NULL, NULL, 170000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 4, 16, NULL, 30000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 4, 17, NULL, 30000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 2, 9, NULL, 50000, NULL);
INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT, CHECKED) VALUES ('2023-03-31', 1, 2, 8, NULL, 100000, NULL);

GRANT USAGE ON SCHEMA S02 TO ROLE PUBLIC;
GRANT SELECT, INSERT, UPDATE, DELETE ON ALL TABLES IN SCHEMA S02 TO ROLE PUBLIC;
GRANT SELECT ON ALL VIEWS IN SCHEMA S02 TO ROLE PUBLIC;


INSERT INTO XLS.OBJECTS (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_OBJECT, UPDATE_OBJECT, DELETE_OBJECT) VALUES ('S02', 'VIEW_CASHBOOK', 'VIEW', NULL, 'S02.VIEW_CASHBOOK', 'S02.VIEW_CASHBOOK', 'S02.VIEW_CASHBOOK');
INSERT INTO XLS.OBJECTS (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_OBJECT, UPDATE_OBJECT, DELETE_OBJECT) VALUES ('S02', 'USP_CASHBOOK', 'PROCEDURE', NULL, 'S02.CASHBOOK', 'S02.CASHBOOK', 'S02.CASHBOOK');
INSERT INTO XLS.OBJECTS (TABLE_SCHEMA, TABLE_NAME, TABLE_TYPE, TABLE_CODE, INSERT_OBJECT, UPDATE_OBJECT, DELETE_OBJECT) VALUES ('S02', 'CODE_CASHBOOK', 'CODE', 'SELECT
    t.ID
    , t.DATE
    , t.ACCOUNT_ID
    , t.ITEM_ID
    , t.COMPANY_ID
    , t.DEBIT
    , t.CREDIT
FROM
    S02.CASHBOOK t
WHERE
    COALESCE(:ACCOUNT_ID, t.ACCOUNT_ID, -1) = COALESCE(t.ACCOUNT_ID, -1)
    AND COALESCE(:ITEM_ID, t.ITEM_ID, -1) = COALESCE(t.ITEM_ID, -1)
    AND COALESCE(:COMPANY_ID, t.COMPANY_ID, -1) = COALESCE(t.COMPANY_ID, -1)', 'INSERT INTO S02.CASHBOOK (DATE, ACCOUNT_ID, ITEM_ID, COMPANY_ID, DEBIT, CREDIT) VALUES (:DATE, :ACCOUNT_ID, :ITEM_ID, :COMPANY_ID, :DEBIT, :CREDIT)', 'UPDATE S02.CASHBOOK SET DATE = :DATE, ACCOUNT_ID = :ACCOUNT_ID, ITEM_ID = :ITEM_ID, COMPANY_ID = :COMPANY_ID, DEBIT = :DEBIT, CREDIT = :CREDIT WHERE ID = :ID', 'DELETE FROM S02.CASHBOOK WHERE ID = :ID');

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CASHBOOK', 'DATE', 'SelectPeriod', NULL, NULL, 'ATTRIBUTE', NULL, 'HideWeeks HideYears', NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CASHBOOK', 'ACCOUNT_ID', 'ValidationList', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CASHBOOK', 'COMPANY_ID', 'ValidationList', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CASHBOOK', 'ITEM_ID', 'ValidationList', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK', 'ACCOUNT_ID', 'ValidationList', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK', 'COMPANY_ID', 'ValidationList', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK', 'ITEM_ID', 'ValidationList', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', 'ACCOUNT_ID', 'ValidationList', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', 'COMPANY_ID', 'ValidationList', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', 'ITEM_ID', 'ValidationList', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', NULL, 'Change', 'S02', 'VIEW_CASHBOOK2', 'VIEW', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', NULL, 'DoNotSave', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK2', NULL, 'ProtectRows', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'ACCOUNT_ID', 'ValidationList', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'COMPANY_ID', 'ValidationList', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'ITEM_ID', 'ValidationList', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'DEBIT', 'Change', 'S02', 'VIEW_CASHBOOK3_DEBIT_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET DEBIT = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'CREDIT', 'Change', 'S02', 'VIEW_CASHBOOK3_CREDIT_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET CREDIT = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'ACCOUNT_ID', 'Change', 'S02', 'VIEW_CASHBOOK3_ACCOUNT_ID_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET ACCOUNT_ID = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'COMPANY_ID', 'Change', 'S02', 'VIEW_CASHBOOK3_COMPANY_ID_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET COMPANY_ID = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'ITEM_ID', 'Change', 'S02', 'VIEW_CASHBOOK3_ITEM_ID_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET ITEM_ID = :cell_number_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', 'DATE', 'Change', 'S02', 'VIEW_CASHBOOK3_DATE_CHANGE', 'CODE', 'UPDATE S02.CASHBOOK SET DATE = :cell_datetime_value WHERE ID = :ID', '_Commit', NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', NULL, 'DoNotSave', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'VIEW_CASHBOOK3', NULL, 'ProtectRows', NULL, NULL, 'ATTRIBUTE', NULL, NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'ACCOUNT_ID', 'ValidationList', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'COMPANY_ID', 'ValidationList', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'ITEM_ID', 'ValidationList', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'ACCOUNT_ID', 'ParameterValues', 'S02', 'ACCOUNTS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'COMPANY_ID', 'ParameterValues', 'S02', 'COMPANIES', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'USP_CASHBOOK', 'ITEM_ID', 'ParameterValues', 'S02', 'ITEMS', 'TABLE', 'ID, +NAME', NULL, NULL, NULL);

INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', NULL, 'Actions', 'S02', 'See Online Help', 'HTTP', 'https://www.savetodb.com/samples/sample02-code_cashbook', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'ACCOUNT_ID', 'ParameterValues', 'S02', 'XL_LIST_ACCOUNT_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ACCOUNTS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'COMPANY_ID', 'ParameterValues', 'S02', 'XL_LIST_COMPANY_ID_FOR_ITEM_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.COMPANIES m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
WHERE
    :ITEM_ID IS NULL
UNION ALL
SELECT
    c.ID
    , COALESCE(t.TRANSLATED_NAME, c.NAME) AS NAME
FROM
    S02.ITEM_COMPANIES ic
    INNER JOIN S02.COMPANIES c ON c.ID = ic.COMPANY_ID
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = c.NAME
WHERE
    ic.ITEM_ID = :ITEM_ID
ORDER BY
    NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'ITEM_ID', 'ParameterValues', 'S02', 'XL_LIST_ITEM_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ITEMS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'ACCOUNT_ID', 'ValidationList', 'S02', 'XL_LIST_ACCOUNT_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ACCOUNTS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'COMPANY_ID', 'ValidationList', 'S02', 'XL_LIST_COMPANY_ID_WITH_ITEM_ID_CODE', 'CODE', 'SELECT
    c.ID
    , COALESCE(t.TRANSLATED_NAME, c.NAME) AS NAME
    , ic.ITEM_ID
FROM
    S02.ITEM_COMPANIES ic
    INNER JOIN S02.COMPANIES c ON c.ID = ic.COMPANY_ID
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = c.NAME
ORDER BY
    ic.ITEM_ID
    , NAME', NULL, NULL, NULL);
INSERT INTO XLS.HANDLERS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, EVENT_NAME, HANDLER_SCHEMA, HANDLER_NAME, HANDLER_TYPE, HANDLER_CODE, TARGET_WORKSHEET, MENU_ORDER, EDIT_PARAMETERS) VALUES ('S02', 'CODE_CASHBOOK', 'ITEM_ID', 'ValidationList', 'S02', 'XL_LIST_ITEM_ID_CODE', 'CODE', 'SELECT
    m.ID
    , COALESCE(t.TRANSLATED_NAME, m.NAME) AS NAME
FROM
    S02.ITEMS m
    LEFT OUTER JOIN XLS.TRANSLATIONS t ON t.TABLE_SCHEMA = ''S02'' AND t.TABLE_NAME = ''strings''
            AND t.LANGUAGE_NAME = :DATA_LANGUAGE AND t.COLUMN_NAME = m.NAME
ORDER BY
    NAME', NULL, NULL, NULL);

INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','de','Konto','','');
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','de','Konto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','de','Apr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','de','Aug.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','de','Balance', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','de','Überprüft', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','de','Unternehmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','de','Unternehmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','de','Kosten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','de','Datum', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','de','Tag', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','de','Einkommen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','de','Dez.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','de','Endtermin', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','de','Feb.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','de','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','de','Artikel', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','de','Artikel', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','de','Jan.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','de','Juli', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','de','Juni', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','de','Niveau', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','de','März', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','de','Mai', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','de','Monat', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','de','Name', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','de','Nov.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','de','Okt.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','de','Sektion', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','de','Sept.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','de','Sortierung', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','de','Startdatum', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','de','Jahr', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'de','Konten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'de','Kassenbuch', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'de','Kassenbuch (SQL-Code)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'de','Unternehmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'de','Artikel und Firmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'de','Artikel', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','de','Bank', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','de','Schlussbilanz', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','de','Körperschaftssteuer', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','de','Kunde C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','de','Kunde C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','de','Kunde C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','de','Kunde C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','de','Kunde C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','de','Kunde C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','de','Kunde C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','de','Kosten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','de','Lohnsteuer', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','de','Nettoveränderung', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','de','Anfangsbestand', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','de','Lohn-und Gehaltsabrechnung', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','de','Sozialabgaben', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','de','Einnahmen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','de','Lieferant S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','de','Lieferant S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','de','Lieferant S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','de','Lieferant S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','de','Lieferant S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','de','Lieferant S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','de','Lieferant S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','de','Steuern', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','de','Gesamtausgaben', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','de','Gesamteinkommen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'de','Bargeld nach Monaten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','de','Firmen-ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','de','Artikel-ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'de','Kassenbuch (Prozedur)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'de','Kassenbuch (Prozedur, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'de','Kassenbuch (Prozedur, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'de','Kassenbuch (Prozedur, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'de','Kassenbuch (Formeln)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'de','Kassenbuch (Ansicht)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'de','Kassenbuch (Ansicht, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'de','Kassenbuch (Ansicht, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'de','Translationen', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'de','Einzelheiten', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','en','Account', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','en','Account', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','en','Apr', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','en','Aug', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','en','Balance', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','en','Checked', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','en','Company', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','en','Company', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','en','Expenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','en','Date', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','en','Day', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','en','Income', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','en','Dec', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','en','End Date', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','en','Feb', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','en','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','en','Item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','en','Item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','en','Jan', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','en','Jul', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','en','Jun', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','en','Level', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','en','Mar', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','en','May', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','en','Month', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','en','Name', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','en','Nov', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','en','Oct', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','en','Section', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','en','Sep', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','en','Sort Order', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','en','Start Date', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','en','Year', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'en','Accounts', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'en','Cashbook', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'en','Cashbook (SQL code)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'en','Companies', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'en','Item and Companies', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'en','Items', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','en','Bank', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','en','Closing Balance', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','en','Corporate Income Tax', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','en','Customer C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','en','Customer C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','en','Customer C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','en','Customer C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','en','Customer C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','en','Customer C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','en','Customer C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','en','Expenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','en','Individual Income Tax', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','en','Net Change', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','en','Opening Balance', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','en','Payroll', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','en','Payroll Taxes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','en','Revenue', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','en','Supplier S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','en','Supplier S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','en','Supplier S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','en','Supplier S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','en','Supplier S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','en','Supplier S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','en','Supplier S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','en','Taxes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','en','Total Expenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','en','Total Income', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'en','Cash by Months', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','en','Company Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','en','Item Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'en','Cashbook (procedure)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'en','Cashbook (procedure, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'en','Cashbook (procedure, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'en','Cashbook (procedure, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'en','Cashbook (formulas)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'en','Cashbook (view)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'en','Cashbook (view, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'en','Cashbook (view, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'en','Translations', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'en','Details', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','es','Cuenta', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','es','Cuenta', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','es','Abr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','es','Agosto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','es','Equilibrio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','es','Comprobado', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','es','Empresa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','es','Empresa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','es','Gasto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','es','Fecha', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','es','Día', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','es','Ingresos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','es','Dic.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','es','Fecha final', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','es','Feb.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','es','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','es','Artículo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','es','Artículo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','es','Enero', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','es','Jul.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','es','Jun.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','es','Nivel', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','es','Marzo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','es','Mayo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','es','Mes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','es','Nombre', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','es','Nov.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','es','Oct.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','es','Sección', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','es','Sept.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','es','Orden', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','es','Fecha de inicio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','es','Año', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'es','Cuentas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'es','Libro de caja', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'es','Libro de caja (código SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'es','Compañías', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'es','Artículo y empresas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'es','Artículos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','es','Banco', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','es','Balance de cierre', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','es','Impuesto sobre Sociedades', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','es','Cliente C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','es','Cliente C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','es','Cliente C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','es','Cliente C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','es','Cliente C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','es','Cliente C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','es','Cliente C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','es','Gasto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','es','IRPF', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','es','Cambio neto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','es','Saldo de apertura', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','es','Salario', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','es','Cargas sociales', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','es','Ingresos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','es','Abastecedor A1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','es','Abastecedor A2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','es','Abastecedor A3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','es','Abastecedor A4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','es','Abastecedor A5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','es','Abastecedor A6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','es','Abastecedor A7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','es','Impuestos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','es','Gasto total', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','es','Ingresos totales', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'es','Efectivo por meses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','es','ID de empresa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','es','ID del artículo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'es','Libro de caja (proc)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'es','Libro de caja (proc, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'es','Libro de caja (proc, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'es','Libro de caja (proc, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'es','Libro de caja (fórmulas)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'es','Libro de caja (ver)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'es','Libro de caja (ver, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'es','Libro de caja (ver, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'es','Traducciones', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'es','Detalles', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','fr','Compte', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','fr','Compte', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','fr','Avril', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','fr','Août', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','fr','Solde', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','fr','Vérifié', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','fr','Entreprise', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','fr','Entreprise', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','fr','Dépenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','fr','Date', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','fr','Journée', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','fr','Revenu', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','fr','Déc.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','fr','Date de fin', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','fr','Févr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','fr','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','fr','Article', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','fr','Article', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','fr','Janv.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','fr','Juil.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','fr','Juin', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','fr','Niveau', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','fr','Mars', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','fr','Mai', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','fr','Mois', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','fr','Prénom', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','fr','Nov.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','fr','Oct.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','fr','Section', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','fr','Sept.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','fr','Ordre de tri', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','fr','Date de début', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','fr','Année', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'fr','Comptes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'fr','Livre de caisse', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'fr','Livre de caisse (code SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'fr','Entreprises', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'fr','Article et sociétés', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'fr','Articles', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','fr','Banque', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','fr','Solde de clôture', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','fr','Impôt sur les sociétés', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','fr','Client 01', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','fr','Client 02', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','fr','Client 03', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','fr','Client 04', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','fr','Client 05', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','fr','Client 06', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','fr','Client 07', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','fr','Dépenses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','fr','Impôt sur le revenu', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','fr','Changement net', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','fr','Solde d''ouverture', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','fr','Paie', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','fr','Charges sociales', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','fr','Revenu', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','fr','Fournisseur 01', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','fr','Fournisseur 02', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','fr','Fournisseur 03', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','fr','Fournisseur 04', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','fr','Fournisseur 05', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','fr','Fournisseur 06', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','fr','Fournisseur 07', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','fr','Taxes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','fr','Dépenses totales', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','fr','Revenu total', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'fr','Cash par mois', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','fr','ID de l''entreprise', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','fr','ID de l''article', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'fr','Livre de caisse (procédure)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'fr','Livre de caisse (procédure, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'fr','Livre de caisse (procédure, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'fr','Livre de caisse (procédure, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'fr','Livre de caisse (formules)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'fr','Livre de caisse (vue)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'fr','Livre de caisse (vue, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'fr','Livre de caisse (vue, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'fr','Traductions', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'fr','Détails', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','it','Conto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','it','Conto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','it','Apr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','it','Ag.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','it','Saldo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','it','Controllato', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','it','Azienda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','it','Azienda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','it','Credito', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','it','Data', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','it','Giorno', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','it','Debito', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','it','Dic.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','it','Data di fine', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','it','Febbr.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','it','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','it','Articolo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','it','Articolo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','it','Genn.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','it','Luglio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','it','Giugno', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','it','Livello', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','it','Mar.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','it','Magg.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','it','Mese', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','it','Conome', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','it','Nov.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','it','Ott.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','it','Sezione', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','it','Sett.', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','it','Ordinamento', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','it','Data d''inizio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','it','Anno', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'it','Conti', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'it','Cashbook', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'it','Cashbook (codice SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'it','Aziende', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'it','Articolo e società', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'it','Elementi', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','it','Banca', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','it','Saldo finale', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','it','Imposta sul reddito delle società', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','it','Cliente C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','it','Cliente C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','it','Cliente C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','it','Cliente C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','it','Cliente C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','it','Cliente C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','it','Cliente C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','it','Spese', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','it','IRPEF', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','it','Cambio netto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','it','Saldo iniziale', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','it','Paga', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','it','Imposte sui salari', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','it','Reddito', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','it','Fornitore F1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','it','Fornitore F2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','it','Fornitore F3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','it','Fornitore F4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','it','Fornitore F5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','it','Fornitore F6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','it','Fornitore F7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','it','Tasse', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','it','Spese totale', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','it','Reddito totale', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'it','Contanti per mesi', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','it','ID dell''azienda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','it','ID articolo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'it','Cashbook (procedura)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'it','Cashbook (procedura, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'it','Cashbook (procedura, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'it','Cashbook (procedura, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'it','Cashbook (formule)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'it','Cashbook (visualizza)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'it','Cashbook (visualizza, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'it','Cashbook (visualizza, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'it','Traduzioni', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'it','Dettagli', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','ja','アカウント', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','ja','アカウント', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','ja','4月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','ja','8月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','ja','バランス', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','ja','チェック済み', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','ja','会社', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','ja','会社', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','ja','経費', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','ja','日付', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','ja','日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','ja','所得', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','ja','12月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','ja','終了日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','ja','2月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','ja','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','ja','アイテム', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','ja','アイテム', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','ja','1月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','ja','7月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','ja','6月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','ja','レベル', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','ja','3月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','ja','5月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','ja','月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','ja','名', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','ja','11月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','ja','10月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','ja','セクション', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','ja','9月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','ja','並べ替え順序', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','ja','開始日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','ja','年', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'ja','アカウント', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'ja','キャッシュブック', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'ja','キャッシュブック（SQLコード）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'ja','会社', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'ja','アイテムと会社', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'ja','アイテム', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','ja','銀行', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','ja','クロージングバランス', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','ja','法人所得税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','ja','顧客C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','ja','顧客C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','ja','顧客C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','ja','顧客C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','ja','顧客C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','ja','顧客C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','ja','顧客C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','ja','経費', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','ja','個人所得税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','ja','正味変化', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','ja','オープニングバランス', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','ja','給与', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','ja','給与税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','ja','収益', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','ja','サプライヤーS1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','ja','サプライヤーS2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','ja','サプライヤーS3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','ja','サプライヤーS4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','ja','サプライヤーS5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','ja','サプライヤーS6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','ja','サプライヤーS7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','ja','税金', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','ja','総費用', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','ja','総収入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'ja','月ごとの現金', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','ja','会社ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','ja','アイテムID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'ja','キャッシュブック（手順）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'ja','キャッシュブック（手順、_編集）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'ja','キャッシュブック（手順、_変更）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'ja','キャッシュブック（手順、_マージ）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'ja','キャッシュブック（数式）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'ja','キャッシュブック（表示）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'ja','キャッシュブック（表示、_変更）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'ja','キャッシュブック（表示、_変更、SQL）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'ja','翻訳', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'ja','詳細', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','ko','계정', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','ko','계정', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','ko','4월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','ko','8월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','ko','균형', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','ko','확인됨', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','ko','회사', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','ko','회사', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','ko','소요 경비', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','ko','날짜', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','ko','하루', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','ko','소득', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','ko','12월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','ko','종료일', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','ko','2월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','ko','ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','ko','아이템', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','ko','아이템', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','ko','1월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','ko','7월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','ko','6월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','ko','수준', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','ko','3월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','ko','5월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','ko','월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','ko','이름', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','ko','11월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','ko','10월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','ko','섹션', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','ko','9월', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','ko','정렬 순서', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','ko','시작일', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','ko','년', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'ko','계정', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'ko','캐쉬북', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'ko','캐쉬북(SQL 코드)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'ko','회사', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'ko','아이템 및 회사', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'ko','아이템', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','ko','은행', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','ko','폐쇄 균형', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','ko','법인 소득세', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','ko','고객 C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','ko','고객 C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','ko','고객 C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','ko','고객 C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','ko','고객 C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','ko','고객 C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','ko','고객 C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','ko','소요 경비', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','ko','개인소득세', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','ko','순 변화', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','ko','기초 잔액', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','ko','월급', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','ko','급여 세금', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','ko','수익', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','ko','공급 업체 S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','ko','공급 업체 S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','ko','공급 업체 S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','ko','공급 업체 S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','ko','공급 업체 S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','ko','공급 업체 S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','ko','공급 업체 S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','ko','세금', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','ko','총 경비', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','ko','총 수입', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'ko','월별 현금', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','ko','회사 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','ko','항목 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'ko','캐쉬북(절차)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'ko','캐쉬북(절차, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'ko','캐쉬북(절차, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'ko','캐쉬북(절차, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'ko','캐쉬북(공식)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'ko','캐쉬북(보기)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'ko','캐쉬북(보기, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'ko','캐쉬북(보기, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'ko','번역', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'ko','세부', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','pt','Conta', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','pt','Conta', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','pt','Abr', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','pt','Agosto', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','pt','Saldo', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','pt','Verificado', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','pt','Companhia', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','pt','Companhia', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','pt','Despesas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','pt','Encontro', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','pt','Dia', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','pt','Renda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','pt','Dez', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','pt','Data final', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','pt','Fev', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','pt','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','pt','Item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','pt','Item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','pt','Jan', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','pt','Julho', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','pt','Junho', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','pt','Nível', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','pt','Março', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','pt','Maio', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','pt','Mês', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','pt','Nome', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','pt','Nov', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','pt','Out', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','pt','Seção', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','pt','Set', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','pt','Ordem de classificação', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','pt','Data de início', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','pt','Ano', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'pt','Contas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'pt','Livro caixa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'pt','Livro caixa (código SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'pt','Empresas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'pt','Item e Empresas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'pt','Itens', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','pt','Banco', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','pt','Saldo final', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','pt','Imposto de Renda', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','pt','Cliente C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','pt','Cliente C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','pt','Cliente C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','pt','Cliente C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','pt','Cliente C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','pt','Cliente C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','pt','Cliente C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','pt','Despesas', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','pt','Imposto de renda individual', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','pt','Mudança de rede', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','pt','Saldo inicial', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','pt','Folha de pagamento', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','pt','Impostos sobre os salários', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','pt','Receita', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','pt','Fornecedor S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','pt','Fornecedor S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','pt','Fornecedor S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','pt','Fornecedor S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','pt','Fornecedor S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','pt','Fornecedor S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','pt','Fornecedor S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','pt','Impostos', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','pt','Despesas totais', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','pt','Renda total', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'pt','Dinheiro por meses', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','pt','ID da empresa', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','pt','ID do item', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'pt','Livro caixa (proc)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'pt','Livro caixa (proc, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'pt','Livro caixa (proc, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'pt','Livro caixa (proc, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'pt','Livro caixa (fórmulas)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'pt','Livro caixa (ver)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'pt','Livro caixa (ver, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'pt','Livro caixa (ver, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'pt','Traduções', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'pt','Detalhes', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','ru','Счет', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','ru','Счет', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','ru','Апр', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','ru','Авг', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','ru','Остаток', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','ru','Проверено', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','ru','Компания', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','ru','Компания', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','ru','Расходы', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','ru','Дата', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','ru','Число', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','ru','Доходы', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','ru','Дек', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','ru','Конечная дата', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','ru','Фев', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','ru','Id', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','ru','Статья', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','ru','Статья', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','ru','Янв', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','ru','Июл', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','ru','Июн', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','ru','Уровень', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','ru','Мар', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','ru','Май', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','ru','Месяц', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','ru','Наименование', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','ru','Ноя', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','ru','Окт', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','ru','Секция', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','ru','Сен', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','ru','Порядок', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','ru','Начальная дата', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','ru','Год', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'ru','Счета', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'ru','Кассовая книга', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'ru','Кассовая книга (SQL код)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'ru','Компании', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'ru','Статьи и компании', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'ru','Статьи', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','ru','Банк', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','ru','Конечное сальдо', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','ru','Налог на прибыль', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','ru','Покупатель C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','ru','Покупатель C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','ru','Покупатель C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','ru','Покупатель C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','ru','Покупатель C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','ru','Покупатель C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','ru','Покупатель C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','ru','Расходы', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','ru','НДФЛ', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','ru','Изменение', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','ru','Начальное сальдо', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','ru','Заработная плата', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','ru','Страховые взносы', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','ru','Выручка', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','ru','Поставщик S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','ru','Поставщик S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','ru','Поставщик S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','ru','Поставщик S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','ru','Поставщик S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','ru','Поставщик S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','ru','Поставщик S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','ru','Налоги', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','ru','Всего выплаты', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','ru','Всего поступления', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'ru','ДДС по месяцам', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','ru','Id компании', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','ru','Id статьи', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'ru','Кассовая книга (процедура)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'ru','Кассовая книга (процедура, _edit)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'ru','Кассовая книга (процедура, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'ru','Кассовая книга (процедура, _merge)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'ru','Кассовая книга (с формулами)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'ru','Кассовая книга (view)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'ru','Кассовая книга (view, _change)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'ru','Кассовая книга (view, _change, SQL)', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'ru','Исходные данные', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','zh-hans','帐户', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','zh-hans','帐户', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','zh-hans','四月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','zh-hans','八月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','zh-hans','平衡', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','zh-hans','已检查', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','zh-hans','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','zh-hans','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','zh-hans','花费', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','zh-hans','日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','zh-hans','日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','zh-hans','收入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','zh-hans','十二月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','zh-hans','结束日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','zh-hans','二月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','zh-hans','ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','zh-hans','项', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','zh-hans','项', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','zh-hans','一月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','zh-hans','七月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','zh-hans','六月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','zh-hans','水平', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','zh-hans','三月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','zh-hans','五月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','zh-hans','月份', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','zh-hans','名称', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','zh-hans','十一月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','zh-hans','十月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','zh-hans','部分', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','zh-hans','九月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','zh-hans','排序顺序', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','zh-hans','开始日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','zh-hans','年', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'zh-hans','账户', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'zh-hans','现金簿', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'zh-hans','现金簿（SQL 代码）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'zh-hans','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'zh-hans','物品和公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'zh-hans','物品', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','zh-hans','银行', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','zh-hans','期末余额', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','zh-hans','公司所得税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','zh-hans','顾客C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','zh-hans','顾客C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','zh-hans','顾客C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','zh-hans','顾客C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','zh-hans','顾客C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','zh-hans','顾客C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','zh-hans','顾客C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','zh-hans','花费', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','zh-hans','个人所得税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','zh-hans','净变化', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','zh-hans','期初余额', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','zh-hans','工资单', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','zh-hans','工资税', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','zh-hans','营收', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','zh-hans','供应商 S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','zh-hans','供应商 S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','zh-hans','供应商 S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','zh-hans','供应商 S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','zh-hans','供应商 S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','zh-hans','供应商 S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','zh-hans','供应商 S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','zh-hans','税收', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','zh-hans','总费用', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','zh-hans','总收入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'zh-hans','每月现金', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','zh-hans','公司 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','zh-hans','项目 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'zh-hans','现金簿（程序）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'zh-hans','现金簿（程序，_edit）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'zh-hans','现金簿（程序，_change）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'zh-hans','现金簿（程序，_merge）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'zh-hans','现金簿（公式）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'zh-hans','现金簿（查看）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'zh-hans','现金簿（查看，_change）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'zh-hans','现金簿（视图、_change、SQL）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'zh-hans','翻译', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'zh-hans','详情', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account','zh-hant','帳戶', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'account_id','zh-hant','帳戶', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Apr','zh-hant','四月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Aug','zh-hant','八月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'balance','zh-hant','平衡', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'checked','zh-hant','已檢查', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company','zh-hant','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'company_id','zh-hant','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'credit','zh-hant','花費', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'date','zh-hant','日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'day','zh-hant','日', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'debit','zh-hant','收入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Dec','zh-hant','十二月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'end_date','zh-hant','結束日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Feb','zh-hant','二月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'id','zh-hant','ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item','zh-hant','項', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'item_id','zh-hant','項', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jan','zh-hant','一月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jul','zh-hant','七月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Jun','zh-hant','六月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'level','zh-hant','水平', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Mar','zh-hant','三月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'May','zh-hant','五月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'month','zh-hant','月份', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Name','zh-hant','名稱', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Nov','zh-hant','十一月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Oct','zh-hant','十月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'section','zh-hant','部分', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'Sep','zh-hant','九月', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'sort_order','zh-hant','排序順序', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'start_date','zh-hant','開始日期', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02', NULL,'year','zh-hant','年', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ACCOUNTS', NULL,'zh-hant','賬戶', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','CASHBOOK', NULL,'zh-hant','現金簿', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','code_CASHBOOK', NULL,'zh-hant','現金簿（SQL 代碼）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','COMPANIES', NULL,'zh-hant','公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','item_COMPANIES', NULL,'zh-hant','物品和公司', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','ITEMS', NULL,'zh-hant','物品', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Bank','zh-hant','銀行', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Closing Balance','zh-hant','期末餘額', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Corporate Income Tax','zh-hant','公司所得稅', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C1','zh-hant','顧客C1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C2','zh-hant','顧客C2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C3','zh-hant','顧客C3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C4','zh-hant','顧客C4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C5','zh-hant','顧客C5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C6','zh-hant','顧客C6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Customer C7','zh-hant','顧客C7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Expenses','zh-hant','花費', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Individual Income Tax','zh-hant','個人所得稅', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Net Change','zh-hant','淨變化', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Opening Balance','zh-hant','期初餘額', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll','zh-hant','工資單', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Payroll Taxes','zh-hant','工資稅', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Revenue','zh-hant','營收', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S1','zh-hant','供應商 S1', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S2','zh-hant','供應商 S2', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S3','zh-hant','供應商 S3', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S4','zh-hant','供應商 S4', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S5','zh-hant','供應商 S5', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S6','zh-hant','供應商 S6', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Supplier S7','zh-hant','供應商 S7', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Taxes','zh-hant','稅收', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Expenses','zh-hant','總費用', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','strings','Total Income','zh-hant','總收入', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS', NULL,'zh-hant','每月現金', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','company_id','zh-hant','公司 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASH_BY_MONTHS','item_id','zh-hant','項目 ID', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK', NULL,'zh-hant','现金簿（程序）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK2', NULL,'zh-hant','现金簿（程序，_edit）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK3', NULL,'zh-hant','现金簿（程序，_change）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK4', NULL,'zh-hant','现金簿（程序，_merge）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','USP_CASHBOOK5', NULL,'zh-hant','现金簿（公式）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK', NULL,'zh-hant','现金簿（查看）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK2', NULL,'zh-hant','现金簿（查看，_change）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_CASHBOOK3', NULL,'zh-hant','现金簿（视图、_change、SQL）', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','VIEW_TRANSLATIONS', NULL,'zh-hant','翻译', NULL, NULL);
INSERT INTO XLS.TRANSLATIONS (TABLE_SCHEMA, TABLE_NAME, COLUMN_NAME, LANGUAGE_NAME, TRANSLATED_NAME, TRANSLATED_DESC, TRANSLATED_COMMENT) VALUES ('S02','XL_DETAILS_CASH_BY_MONTHS', NULL,'zh-hant','詳情', NULL, NULL);

-- print Application installed
